/* ndisasm.c   the Netwide Disassembler main module
 *
 * The Netwide Assembler is copyright (C) 1996 Simon Tatham and
 * Julian Hall. All rights reserved. The software is
 * redistributable under the licence given in the file "Licence"
 * distributed in the NASM archive.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "nasm.h"
#include "nasmlib.h"
#include "sync.h"
#include "disasm.h"

#define BPL 8			       /* bytes per line of hex dump */

static const char *help = 
"usage: %s [-a] [-i] [-h] [-r] [-u] [-b bits] [-o origin] [-s sync...] file\n"
"   -a or -i activates auto (intelligent) sync\n"
"   -u sets USE32 (32-bit mode)\n"
"   -b 16 or -b 32 sets number of bits too\n"
"   -h displays this text\n"
"   -r displays the version number\n";

static void output_ins (unsigned long, unsigned char *, int, char *);

int main(int argc, char **argv) {
    unsigned char buffer[INSN_MAX * 2], *p, *q;
    char outbuf[256];
    char *pname = *argv;
    char *filename = NULL;
    int lenread, lendis;
    int autosync = FALSE;
    int bits = 16;
    int rn_error;
    long offset;
    FILE *fp;

    offset = 0;
    init_sync();

    while (--argc) {
	char *v, *p = *++argv;
	if (*p == '-') {
	    p++;
	    while (*p) switch (tolower(*p)) {
	      case 'a':		       /* auto or intelligent sync */
	      case 'i':
		autosync = TRUE;
		p++;
		break;
	      case 'h':
		fprintf(stderr, help, pname);
		return 0;
		break;
	      case 'r':
		fprintf(stderr, "NDISASM version " NASM_VER "\n");
		return 0;
		break;
	      case 'u':		       /* USE32 */
		bits = 32;
		p++;
		break;
	      case 'b':		       /* bits */
		v = p[1] ? p+1 : --argc ? *++argv : NULL;
		if (!v) {
		    fprintf(stderr, "%s: `-b' requires an argument\n", pname);
		    return 1;
		}
		if (!strcmp(v, "16"))
		    bits = 16;
		else if (!strcmp(v, "32"))
		    bits = 32;
		else {
		    fprintf(stderr, "%s: argument to `-b' should"
			    " be `16' or `32'\n", pname);
		}
		p = "";		       /* force to next argument */
		break;
	      case 'o':		       /* origin */
		v = p[1] ? p+1 : --argc ? *++argv : NULL;
		if (!v) {
		    fprintf(stderr, "%s: `-o' requires an argument\n", pname);
		    return 1;
		}
		offset = readnum (v, &rn_error);
		if (rn_error) {
		    fprintf(stderr, "%s: `-o' requires a numeric argument\n",
			    pname);
		    return 1;
		}
		p = "";		       /* force to next argument */
		break;
	      case 's':		       /* sync point */
		v = p[1] ? p+1 : --argc ? *++argv : NULL;
		if (!v) {
		    fprintf(stderr, "%s: `-s' requires an argument\n", pname);
		    return 1;
		}
		add_sync (readnum (v, &rn_error));
		if (rn_error) {
		    fprintf(stderr, "%s: `-s' requires a numeric argument\n",
			    pname);
		    return 1;
		}
		p = "";		       /* force to next argument */
		break;
	    }
	} else if (!filename) {
	    filename = p;
	} else {
	    fprintf(stderr, "%s: more than one filename specified\n", pname);
	    return 1;
	}
    }

    if (!filename) {
	fprintf(stderr, help, pname);
	return 0;
    }

    fp = fopen(filename, "rb");
    p = q = buffer;
    do {
	lenread = fread (p, 1, buffer+sizeof(buffer)-p, fp);
	p += lenread;
	while (p > q && (p - q >= INSN_MAX || lenread == 0)) {
	    lendis = disasm (q, outbuf, bits, offset, autosync);
	    if (!lendis || lendis > (p - q) ||
		lendis > next_sync(offset)-offset)
		lendis = eatbyte (q, outbuf);
	    output_ins (offset, q, lendis, outbuf);
	    q += lendis;
	    offset += lendis;
	}
	if (q >= buffer+INSN_MAX) {
	    memmove (buffer, q, p - q);
	    p -= (q - buffer);
	    q = buffer;
	}
    } while (lenread > 0);
    fclose (fp);
    return 0;
}

static void output_ins (unsigned long offset, unsigned char *data,
			int datalen, char *insn) {
    int bytes;
    printf("%08lX  ", offset);

    bytes = 0;
    while (datalen > 0 && bytes < BPL) {
	printf("%02X", *data++);
	bytes++;
	datalen--;
    }

    printf("%*s%s\n", (BPL+1-bytes)*2, "", insn);

    while (datalen > 0) {
	printf("         -");
	bytes = 0;
	while (datalen > 0 && bytes < BPL) {
	    printf("%02X", *data++);
	    bytes++;
	    datalen--;
	}
	printf("\n");
    }
}
