/* The Netwide Assembler main program module
 *
 * The Netwide Assembler is copyright (C) 1996 Simon Tatham and
 * Julian Hall. All rights reserved. The software is
 * redistributable under the licence given in the file "Licence"
 * distributed in the NASM archive.
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "nasm.h"
#include "nasmlib.h"
#include "parser.h"
#include "assemble.h"
#include "labels.h"
#include "outform.h"

static void report_error (int, char *, ...);
static void parse_cmdline (int, char **);
static void assemble_file (char *);
static int getkw (char *buf, char **value);
static void register_output_formats(void);
static void usage(void);

static char *obuf;
static char inname[FILENAME_MAX];
static char outname[FILENAME_MAX];
static char realout[FILENAME_MAX];
static int lineno;		       /* for error reporting */
static int pass;
static struct ofmt *ofmt = NULL;

static FILE *ofile = NULL;
static int sb = 16;		       /* by default */

static long current_seg;
static long *offsets;
static long offsetsize;
static long abs_offset;
#define OFFSET_DELTA 256

/*
 * lvalue translating to the current offset...
 */
#define current_ofs  (*(current_seg==NO_SEG?&abs_offset:offsets+current_seg))

static int want_usage;
static int terminate_after_phase;

int main(int argc, char **argv) {
    want_usage = terminate_after_phase = FALSE;

    nasm_set_malloc_error (report_error);
    offsets = nasm_malloc(OFFSET_DELTA*sizeof(*offsets));
    for (offsetsize = 0; offsetsize < OFFSET_DELTA; offsetsize++)
	offsets[offsetsize] = 0L;

    seg_init();

    register_output_formats();

    parse_cmdline(argc, argv);

    if (terminate_after_phase) {
	if (want_usage)
	    usage();
	return 1;
    }

    if (!*outname) {
	ofmt->filename (inname, realout, report_error);
	strcpy(outname, realout);
    }

    ofile = fopen(outname, "wb");
    if (!ofile) {
	report_error (ERR_FATAL | ERR_NOFILE,
		      "unable to open output file `%s'", outname);
    }
    ofmt->init (ofile, report_error, define_label);
    assemble_file (inname);
    if (!terminate_after_phase) {
	ofmt->cleanup ();
	cleanup_labels ();
    }
    fclose (ofile);
    if (terminate_after_phase)
	remove(outname);

    if (want_usage)
	usage();

    return 0;
}

static void parse_cmdline(int argc, char **argv) {
    char *param;

    *inname = *outname = '\0';
    while (--argc) {
	char *p = *++argv;
	if (p[0]=='-') {
	    switch (p[1]) {
	      case 'o':		       /* these parameters take values */
	      case 'f':
		if (p[2])	       /* the parameter's in the option */
		    param = p+2;
		else if (!argv[1]) {
		    report_error (ERR_NONFATAL | ERR_NOFILE | ERR_USAGE,
				  "option `-%c' requires an argument",
				  p[1]);
		    break;
		} else
		    --argc, param = *++argv;
		if (p[1]=='o') {       /* output file */
		    strcpy (outname, param);
		} else if (p[1]=='f') { /* output format */
		    ofmt = ofmt_find(param);
		    if (!ofmt) {
			report_error (ERR_FATAL | ERR_NOFILE | ERR_USAGE,
				      "nasm: unrecognised output format `%s'",
				      param);
		    }
		}
		break;
	      case 'h':
		fprintf(stderr,
			"usage: nasm [-o outfile] [-f format] filename\n");
		fprintf(stderr,
			"    or nasm -r   for version info\n\n");
		fprintf(stderr,
			"valid output formats for -f are"
			" (`*' denotes default):\n");
		ofmt_list(ofmt);
		exit (0);	       /* never need usage message here */
		break;
	      case 'r':
		fprintf(stderr, "NASM version %s\n", NASM_VER);
		exit (0);	       /* never need usage message here */
		break;
	      default:
		report_error (ERR_NONFATAL | ERR_NOFILE | ERR_USAGE,
			      "unrecognised option `-%c'",
			      p[1]);
		break;
	    }
	} else {
	    if (*inname) {
		report_error (ERR_NONFATAL | ERR_NOFILE | ERR_USAGE,
			      "more than one input file specified");
	    } else
		strcpy(inname, p);
	}
    }
    if (!*inname)
	report_error (ERR_NONFATAL | ERR_NOFILE | ERR_USAGE,
		      "no input file specified");
}

/* used by error function to report location */
static char currentfile[FILENAME_MAX];

static void assemble_file (char *fname) {
    FILE *fp = fopen (fname, "r");
    FILE *oldfile = NULL;     /* jrh - used when processing include files */
    int oldfileline = 0;
    char *value, *p, buffer[256];
    insn output_ins;
    int i, seg, rn_error;

    if (!fp) {			       /* couldn't open file */
	report_error (ERR_FATAL | ERR_NOFILE,
		      "unable to open input file `%s'", fname);
	return;
    }

    init_labels ();
    strcpy(currentfile,fname);

    /* pass one */
    pass = 1;
    current_seg = ofmt->section(NULL, pass);
    lineno = 0;
    while (1) {
        if (! fgets(buffer, sizeof(buffer), fp)) {   /* EOF on current file */
	    if (oldfile) {
		fclose(fp);
		fp = oldfile;
		lineno = oldfileline;
		strcpy(currentfile,fname);
		oldfile = NULL;
		continue;
	    }
	    else
		break;
	}
	lineno++;
	strip_newline (buffer);

	/* here we parse our directives; this is not handled by the 'real'
	 * parser. */

	if ( (i = getkw (buffer, &value)) ) {
	    switch (i) {
	      case 1:	       /* [SEGMENT n] */
		seg = ofmt->section (value, pass);
		if (seg == NO_SEG) {
		    report_error (ERR_NONFATAL,
				  "segment name `%s' not recognised",
				  value);
		} else {
		    current_seg = seg;
		    if (current_seg >= offsetsize) {
			int newsize = current_seg + OFFSET_DELTA;
			offsets = nasm_realloc(offsets,
					       newsize*sizeof(*offsets));
			while (offsetsize < newsize)
			    offsets[offsetsize++] = 0;
		    }
		}
		break;
	      case 2:	       /* [EXTERN label] */
		define_label (value, seg_alloc(), 0L, ofmt, report_error);
		break;
	      case 3:	       /* [BITS bits] */
		switch (atoi(value)) {
		  case 16:
		  case 32:
		    sb = atoi(value);
		    break;
		  default:
		    report_error(ERR_NONFATAL,
				 "`%s' is not a valid argument to [BITS]",
				 value);
		    break;
		}
		break;
	      case 4:		/* [INC file] */
		oldfile = fp;
		oldfileline = lineno;
		lineno = 0;
		strcpy(currentfile,value);
		fp = fopen(value,"r");
		if (!fp)  {
		    lineno = oldfileline;
		    fp = oldfile;
		    strcpy(currentfile,fname);
		    report_error (ERR_FATAL,
				  "unable to open include file `%s'\n",
				  value);
		}
		break;
	      case 5:	       /* [GLOBAL symbol] */
		declare_as_global (value, report_error);
		break;
	      case 6:	       /* [COMMON symbol size] */
		p = value;
		while (*p && !isspace(*p))
		    p++;
		if (*p) {
		    long size;

		    while (*p && isspace(*p))
			*p++ = '\0';
		    size = readnum (p, &rn_error);
		    if (rn_error)
			report_error (ERR_NONFATAL, "invalid size specified"
				      " in COMMON declaration");
		    else
			define_common (value, seg_alloc(), size,
				       ofmt, report_error);
		} else
		    report_error (ERR_NONFATAL, "no size specified in"
				  " COMMON declaration");
		break;
	      case 7:		       /* [ABSOLUTE address] */
		current_seg = NO_SEG;
		abs_offset = readnum(value, &rn_error);
		if (rn_error) {
		    report_error (ERR_NONFATAL, "invalid address specified"
				  " for ABSOLUTE directive");
		    abs_offset = 0x100;/* don't go near zero in case of / */
		}
		break;
	      default:
		if (!ofmt->directive (buffer+1, value, 1))
		    report_error (ERR_NONFATAL, "unrecognised directive [%s]",
				  buffer+1);
		break;
	    }
	} else {
	    parse_line (current_seg, current_ofs, lookup_label,
			1, buffer, &output_ins, ofmt, report_error);
	    if (output_ins.opcode == I_EQU) {
		/*
		 * Special `..' EQUs get processed in pass two.
		 */
		if (!output_ins.label)
		    report_error (ERR_NONFATAL,
				  "EQU not preceded by label");
		else if (output_ins.label[0] != '.' ||
			 output_ins.label[1] != '.') {
		    if (output_ins.operands == 1 &&
			(output_ins.oprs[0].type & IMMEDIATE)) {
			define_label (output_ins.label,
				      output_ins.oprs[0].segment,
				      output_ins.oprs[0].offset,
				      ofmt, report_error);
		    } else if (output_ins.operands == 2 &&
			       (output_ins.oprs[0].type & IMMEDIATE) &&
			       (output_ins.oprs[0].type & COLON) &&
			       output_ins.oprs[0].segment == NO_SEG &&
			       (output_ins.oprs[1].type & IMMEDIATE) &&
			       output_ins.oprs[1].segment == NO_SEG) {
			define_label (output_ins.label,
				      output_ins.oprs[0].offset | SEG_ABS,
				      output_ins.oprs[1].offset,
				      ofmt, report_error);
		    } else
			report_error(ERR_NONFATAL, "bad syntax for EQU");
		}
	    } else {
		if (output_ins.label)
		    define_label (output_ins.label,
				  current_seg, current_ofs,
				  ofmt, report_error);
		current_ofs += insn_size (current_seg, current_ofs, sb,
					  &output_ins, report_error);
	    }
	    cleanup_insn (&output_ins);
	}
    }

    if (terminate_after_phase) {
	fclose(ofile);
	remove(outname);
	if (want_usage)
	    usage();
	exit (1);
    }

    /* pass two */
    pass = 2;
    rewind (fp);
    current_seg = ofmt->section(NULL, pass);
    for (i=0; i<offsetsize; i++)
	offsets[i] = 0;
    lineno = 0;
    while (1) {
        if (!fgets(buffer, sizeof(buffer), fp)) {
	    if (oldfile) {
		fclose(fp);
		fp = oldfile;
		lineno = oldfileline;
		strcpy(currentfile,fname);
		oldfile = NULL;
		continue;
	    } else
		break;
        }
	lineno++;
	strip_newline (buffer);

	/* here we parse our directives; this is not handled by
	 * the 'real' parser. */

	if ( (i = getkw (buffer, &value)) ) {
	    switch (i) {
	      case 1:	       /* [SEGMENT n] */
		seg = ofmt->section (value, pass);
		if (seg == NO_SEG) {
		    report_error (ERR_PANIC,
				  "invalid segment name on pass two");
		} else {
		    current_seg = seg;
		    if (current_seg >= offsetsize)
			report_error (ERR_PANIC,
				      "segment table confusion on pass two");
		}
		break;
	      case 2:	       /* [EXTERN label] */
		break;
	      case 3:	       /* [BITS bits] */
		switch (atoi(value)) {
		  case 16:
		  case 32:
		    sb = atoi(value);
		    break;
		  default:
		    report_error(ERR_PANIC,
				 "invalid [BITS] value on pass two",
				 value);
		    break;
		}
		break;
	      case 4:
		oldfile = fp;
		oldfileline = lineno;
		lineno = 0;
		strcpy(currentfile,value);
		fp = fopen(value,"r");
		if (!fp) {
		    lineno = oldfileline;
		    fp = oldfile;
		    strcpy(currentfile,fname);
		    /*
		     * We don't report this error in the PANIC
		     * class, even though we might expect to have
		     * already picked it up during pass one,
		     * because of the tiny chance that some other
		     * process may have removed the include file
		     * between the passes.
		     */
		    report_error (ERR_FATAL,
				  "unable to open include file `%s'\n",
				  value);
		}
		break;
	      case 5:		       /* [GLOBAL symbol] */
		break;
	      case 6:		       /* [COMMON symbol size] */
		break;
	      case 7:		       /* [ABSOLUTE addr] */
		current_seg = NO_SEG;
		abs_offset = readnum(value, &rn_error);
		if (rn_error)
		    report_error (ERR_PANIC, "invalid ABSOLUTE address "
				  "in pass two");
		break;
	      default:
		if (!ofmt->directive (buffer+1, value, 2))
		    report_error (ERR_PANIC, "invalid directive on pass two");
		break;
	    }
	} else {
	    parse_line (current_seg, current_ofs, lookup_label, 2,
			buffer, &output_ins, ofmt, report_error);
	    obuf = buffer;
	    if (output_ins.label)
		define_label_stub (output_ins.label, report_error);
	    if (output_ins.opcode == I_EQU) {
		/*
		 * Special `..' EQUs get processed here.
		 */
		if (output_ins.label[0] == '.' &&
		    output_ins.label[1] == '.') {
		    if (output_ins.operands == 1 &&
			(output_ins.oprs[0].type & IMMEDIATE)) {
			define_label (output_ins.label,
				      output_ins.oprs[0].segment,
				      output_ins.oprs[0].offset,
				      ofmt, report_error);
		    } else if (output_ins.operands == 2 &&
			       (output_ins.oprs[0].type & IMMEDIATE) &&
			       (output_ins.oprs[0].type & COLON) &&
			       output_ins.oprs[0].segment == NO_SEG &&
			       (output_ins.oprs[1].type & IMMEDIATE) &&
			       output_ins.oprs[1].segment == NO_SEG) {
			define_label (output_ins.label,
				      output_ins.oprs[0].offset | SEG_ABS,
				      output_ins.oprs[1].offset,
				      ofmt, report_error);
		    } else
			report_error(ERR_NONFATAL, "bad syntax for EQU");
		}
	    }
	    current_ofs += assemble (current_seg, current_ofs, sb,
				     &output_ins, ofmt, report_error);
	    cleanup_insn (&output_ins);
	}
    }
}

static int getkw (char *buf, char **value) {
    char *p, *q;

    if (*buf!='[')
    	return 0;
    p = buf;
    while (*p && *p != ']') p++;
    if (!*p)
	return 0;
    q = p++;
    while (*p && *p != ';') {
	if (!isspace(*p))
	    return 0;
	p++;
    }
    q[1] = '\0';

    p = buf+1;
    while (*buf && *buf!=' ' && *buf!=']' && *buf!='\t')
    	buf++;
    if (*buf==']') {
	*buf = '\0';
	*value = buf;
    } else {
	*buf++ = '\0';
	*value = buf;
	while (*buf!=']') buf++;
	*buf++ = '\0';
    }
    for (q=p; *q; q++)
	*q = tolower(*q);
    if (!strcmp(p, "segment") || !strcmp(p, "section"))
    	return 1;
    if (!strcmp(p, "extern"))
    	return 2;
    if (!strcmp(p, "bits"))
    	return 3;
    if (!strcmp(p, "inc") || !strcmp(p, "include"))
        return 4;
    if (!strcmp(p, "global"))
    	return 5;
    if (!strcmp(p, "common"))
    	return 6;
    if (!strcmp(p, "absolute"))
    	return 7;
    return -1;
}

static void report_error (int severity, char *fmt, ...) {
    va_list ap;

    if (severity & ERR_NOFILE)
	fputs ("nasm: ", stderr);
    else
	fprintf (stderr, "%s:%d: ", currentfile, lineno);

    if ( (severity & ERR_MASK) == ERR_WARNING)
	fputs ("warning: ", stderr);
    else if ( (severity & ERR_MASK) == ERR_PANIC)
	fputs ("panic: ", stderr);

    va_start (ap, fmt);
    vfprintf (stderr, fmt, ap);
    fputc ('\n', stderr);

    if (severity & ERR_USAGE)
	want_usage = TRUE;

    switch (severity & ERR_MASK) {
      case ERR_WARNING:
	/* no further action, by definition */
	break;
      case ERR_NONFATAL:
	terminate_after_phase = TRUE;
	break;
      case ERR_FATAL:
	fclose(ofile);
	remove(outname);
	if (want_usage)
	    usage();
	exit(1);		       /* instantly die */
	break;			       /* placate silly compilers */
      case ERR_PANIC:
	abort();		       /* panic and dump core */
	break;
    }
}

static void usage(void) {
    fputs("type `nasm -h' for help\n", stderr);
}

static void register_output_formats(void) {
    /* Flat-form binary format */
    extern struct ofmt of_bin;
    /* Unix formats: a.out, COFF, ELF */
    extern struct ofmt of_aout, of_coff, of_elf;
    /* Linux strange format: as86 */
    extern struct ofmt of_as86;
    /* DOS formats: OBJ, Win32 */
    extern struct ofmt of_obj, of_win32;

    ofmt_register (&of_bin);
    ofmt_register (&of_aout);
    ofmt_register (&of_coff);
    ofmt_register (&of_elf);
    ofmt_register (&of_as86);
    ofmt_register (&of_obj);
    ofmt_register (&of_win32);

    /*
     * set the default format
     */
    ofmt = &of_bin;
}
