/* pam_handlers.c -- pam config file parsing and module loading */

/* created by Marc Ewing.
 *
 * $Id: pam_handlers.c,v 1.9 1996/06/02 08:00:18 morgan Exp $
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dlfcn.h>
#include <sys/file.h>

#include "pam_private.h"

#define BUF_SIZE      1024
#define MODULE_CHUNK  4

#ifdef DEBUG
#define D(x) _pam_debug x
#else
#define D(x)
#endif

int _pam_assemble_line(FILE *f, char *buf, int buf_len);

void _pam_free_handlers_aux(struct handler **hp);

int _pam_add_handler(pam_handle_t *pamh, int other, int type,
		     int control_flag, char *mod_path,
		     int argc, char **argv, int argvlen);

/* Values for type */
#define PAM_AUTH    1
#define PAM_SESS    2
#define PAM_ACCT    4
#define PAM_PASS    8

/* Parse config file, allocate handler structures, dlopen() */
int _pam_init_handlers(pam_handle_t *pamh)
{
    FILE *f;
    char buf[BUF_SIZE];
    char *mod_path;
    int x, module_type, control_flag;
    int other;
    char *service_name = pamh->service_name;
    int res;
#ifdef DEBUG
    int y;
#endif

    int argc;
    char **argv;
    int argvlen;

    D(("_pam_init_handlers called"));
    IF_NO_PAMH("_pam_init_handlers",pamh,PAM_SYSTEM_ERR);

    /* Return immediately if everything is already loaded */
    if (pamh->handlers.handlers_loaded) {
	return PAM_SUCCESS;
    }

    D(("_pam_init_handlers: initializing"));
    
    /* First clean the service structure */

    _pam_free_handlers(pamh);
    if (! pamh->handlers.module) {
	if ((pamh->handlers.module =
	     malloc(MODULE_CHUNK * sizeof(struct loaded_module))) == NULL) {
	    _pam_log_error("_pam_init_handlers: no memory loading module");
	    return PAM_BUF_ERR;
	}
	pamh->handlers.modules_allocated = MODULE_CHUNK;
	pamh->handlers.modules_used = 0;
    }
    
    /* Now parse the config file and add handlers */

    if ((f = fopen(PAM_CONFIG, "r")) == NULL || flock(fileno(f),LOCK_SH)) {
	_pam_log_error("_pam_init_handlers: could not open " PAM_CONFIG );
	return PAM_ABORT;
    }
    while ((x = _pam_assemble_line(f, buf, BUF_SIZE)) > 0) {
	char *tok, *nexttok=NULL;

	D(("_pam_init_handler: LINE: %s", buf));
	tok = StrTok(buf, " \n\t", &nexttok);
	other = !strCMP(tok, "OTHER");
	if (!strCMP(tok, service_name) || other) {
	    /* This is the service we are looking for */
	    D(("_pam_init_handlers: Found pam.conf entry for: %s", tok));

	    tok = StrTok(NULL, " \n\t", &nexttok);
	    if (!strCMP("auth", tok)) {
		module_type = PAM_AUTH;
	    } else if (!strCMP("session", tok)) {
		module_type = PAM_SESS;
	    } else if (!strCMP("account", tok)) {
		module_type = PAM_ACCT;
	    } else if (!strCMP("password", tok)) {
		module_type = PAM_PASS;
	    } else {
		/* Illegal module type */
		D(("_pam_init_handlers: bad module type: %s", tok));
		_pam_log_error("illegal module type: %s", tok);
		continue;                          /* skip this line */
	    }
	    D(("Using pam.conf entry: %s", tok));

	    /* accept "service name" or OTHER of this module_type */
	    
	    tok = StrTok(NULL, " \n\t", &nexttok);
	    if (!strCMP("required", tok)) {
		control_flag = PAM_REQUIRED;
		D(("*PAM_REQUIRED*"));
	    } else if (!strCMP("optional", tok)) {
		control_flag = PAM_OPTIONAL;
		D(("*PAM_OPTIONAL*"));
	    } else if (!strCMP("sufficient", tok)) {
		control_flag = PAM_SUFFICIENT;
		D(("*PAM_SUFFICIENT*"));
	    } else {
		/* Illegal flag */
		D(("_pam_init_handlers: bad control flag: %s", tok));
		_pam_log_error("illegal control flag: %s", tok);
		continue;                         /* skip to next line */
	    }

	    tok = StrTok(NULL, " \n\t", &nexttok);
	    if (tok != NULL) {
		mod_path = tok;
		D(("mod_path = %s",mod_path));
	    } else {
		/* no module name given */
		D(("_pam_init_handlers: no module name supplied"));
		_pam_log_error("no module name supplied");
		continue;                               /* ignore this line */
	    }

	    /* nexttok points to remaining arguments... */

	    if (nexttok != NULL) {
		D(("list: %s",nexttok));
	        argvlen = _pam_mkargv(nexttok, &argv, &argc);
		D(("argvlen = %d",argvlen));
	    } else {               /* there are no arguments so fix by hand */
		D(("_pam_init_handlers: empty argument list"));
		argvlen = argc = 0;
		argv = NULL;
	    }

#ifdef DEBUG
	    D(("CONF: %s %d %d %s %d", service_name, module_type,
	       control_flag, mod_path, argc));
	    for (y = 0; y < argc; y++) {
		D(("CONF: %s", argv[y]));
	    }
#endif

	    res = _pam_add_handler(pamh, other, module_type, control_flag,
				   mod_path, argc, argv, argvlen);
	    if (res != PAM_SUCCESS) {
		_pam_log_error("error loading %s", mod_path);
		continue;
	    }
	}
    }
    (void) flock(fileno(f),LOCK_UN);        /* unlock */
    fclose(f);                              /* close */

    D(("closed configuration file"));

    if (x < 0) {
	/* Read error */
	_pam_log_error("error reading pam.conf");
	return PAM_ABORT;
    }

    pamh->handlers.handlers_loaded = 1;

    D(("_pam_init_handlers exiting"));
    return PAM_SUCCESS;
}

int _pam_assemble_line(FILE *f, char *buffer, int buf_len)
{
    char *p = buffer;
    char *s, *os;
    int used = 0;

    /* loop broken with a 'break' when a non-'\\' ending line is read */

    for (;;) {
	if (used >= buf_len) {
	    /* Overflow */
	    D(("_pam_assemble_line: overflow"));
	    return -1;
	}
	if (fgets(p, buf_len - used, f) == NULL) {
	    if (used) {
		/* Incomplete read */
		return -1;
	    } else {
		/* EOF */
		return 0;
	    }
	}

	/* skip leading spaces --- line may be blank */

	s = p + strspn(p, " \n\t");
	if (*s && (*s != '#')) {
	    os = s;

	    /* Check for backslash by scanning back from the end of
	     * the entered line, the '\n' has been included since
	     * normally a line is terminated with this
	     * character. fgets() should only return one though! */

	    s += strlen(s);
	    while (s > os && ((*--s == ' ') || (*s == '\t')
			      || (*s == '\n')));

	    /* check if it ends with a backslash */
	    if (*s == '\\') {
		*s++ = ' ';             /* replace backslash with ' ' */
		*s = '\0';              /* truncate the line here */
		used += strlen(os);
		p = s;                  /* there is more ... */
	    } else {
		/* End of the line! */
		used += strlen(os);
		break;                  /* this is the complete line */
	    }

	} else {
	    /* Nothing in this line */
	    /* Don't move p         */
	}
    }

    return used;
}

/* XXX - _pam_add_handler() is where we would handle the static libs */

typedef int (*servicefn)(pam_handle_t *, int, int, char **);

int _pam_add_handler(pam_handle_t *pamh, int other, int type,
		     int control_flag, char *mod_path,
		     int argc, char **argv, int argvlen)
{
    struct loaded_module *mod;
    int x = 0;
    struct handler **handler_p;
    struct handler **handler_p2;
    struct handlers *the_handlers;
    const char *sym, *sym2;
    servicefn func, func2;
    int success;

    IF_NO_PAMH("_pam_add_handler",pamh,PAM_SYSTEM_ERR);

    D(("_pam_add_handler: adding type %d, module `%s'",type,mod_path));
    mod  = pamh->handlers.module;

    /* First, ensure the module is loaded */
    while (x < pamh->handlers.modules_used) {
	if (!strcmp(mod[x].name, mod_path)) {  /* case sensitive ! */
	    break;
	}
	x++;
    }
    if (x == pamh->handlers.modules_used) {
	/* Not found */
	if (pamh->handlers.modules_allocated == pamh->handlers.modules_used) {
	    /* will need more memory */
	    void *tmp = realloc(pamh->handlers.module,
				pamh->handlers.modules_allocated+
				(MODULE_CHUNK*sizeof(struct loaded_module)));
	    if (tmp == NULL) {
		D(("cannot enlarge module pointer memory"));
		_pam_log_error("realloc returned NULL in _pam_add_handler");
		return PAM_ABORT;
	    }
	    pamh->handlers.module = tmp;
	    pamh->handlers.modules_allocated += MODULE_CHUNK;
	}
	mod = &(pamh->handlers.module[x]);
	/* Be pessimistic... */
	success = PAM_ABORT;
#ifdef PAM_DYNAMIC
	D(("_pam_add_handler: dlopen(%s) -> %lx", mod_path, &mod->dl_handle));
	mod->dl_handle = dlopen(mod_path, RTLD_NOW);
	D(("_pam_add_handler: dlopen'ed"));
	if (mod->dl_handle == NULL) {
	    D(("_pam_add_handler: dlopen(%s) failed", mod_path));
	    _pam_log_error("unable to dlopen(%s)", mod_path);
	    /* Don't abort yet; static code may be able to find function.
	     * But defaults to abort if nothing found below... */
	} else {
	    D(("module added successfully"));
	    success = PAM_SUCCESS;
	    mod->type = PAM_DYNAMIC_MOD;
	    pamh->handlers.modules_used++;
	}
#endif
#ifdef PAM_STATIC
	/* Only load static function if function was not found dynamically.
	 * This code should work even if no dynamic loading is available. */
	if (success != PAM_SUCCESS) {
	    D(("_pam_add_handler: open static handler %s", mod_path));
	    mod->dl_handle = _pam_open_static_handler(mod_path);
	    if (mod->dl_handle == NULL) {
	        D(("_pam_add_handler: unable to find static handler %s",
		   mod_path));
		_pam_log_error("unable to open static handler %s", mod_path);
		/* Didn't find module in dynamic or static */
		return(PAM_ABORT);
	    } else {
	        D(("static module added successfully"));
		success = PAM_SUCCESS;
		mod->type = PAM_STATIC_MOD;
		pamh->handlers.modules_used++;
	    }
	}
#endif
	if ((mod->name = strdup(mod_path)) == NULL) {
	    D(("_pam_handler: couldn't get memory for mod_path"));
	    _pam_log_error("no memory for module path", mod_path);
	    return(PAM_ABORT);
	}

	/* Now return error if necessary after trying all possible ways... */
        if (success != PAM_SUCCESS)
	     return(success);
    } else {    /* x != pamh->handlers.modules_used */
	 mod += x;
    }

    /* mod points to the struct loaded_module we care about */
    /* Now add the handler based on mod->dlhandle and type  */

    /* decide which list of handlers to use */
    the_handlers = (other) ? &pamh->handlers.other : &pamh->handlers.conf;

    handler_p = handler_p2 = NULL;
    func = func2 = NULL;
    sym2 = NULL;
    switch (type) {
    case PAM_AUTH:
	handler_p = &the_handlers->authenticate;
	sym = "pam_sm_authenticate";
	handler_p2 = &the_handlers->setcred;
	sym2 = "pam_sm_setcred";
	break;
    case PAM_SESS:
	handler_p = &the_handlers->open_session;
	sym = "pam_sm_open_session";
	handler_p2 = &the_handlers->close_session;
	sym2 = "pam_sm_close_session";
	break;
    case PAM_ACCT:
	handler_p = &the_handlers->acct_mgmt;
	sym = "pam_sm_acct_mgmt";
	break;
    case PAM_PASS:
	handler_p = &the_handlers->chauthtok;
	sym = "pam_sm_chauthtok";
	break;
    default:
	/* Illegal module type */
	D(("_pam_add_handler: illegal module type %d", type));
	return PAM_ABORT;
    }

    if (
#ifdef PAM_DYNAMIC
	 mod->type != PAM_DYNAMIC_MOD
#ifdef PAM_STATIC
	 &&
#endif /* PAM_STATIC */
#endif /* PAM_DYNAMIC */
#ifdef PAM_STATIC
	 mod->type != PAM_STATIC_MOD
#endif /* PAM_STATIC */
	) {
	D(("_pam_add_handlers: illegal module library type; %d", mod->type));
	_pam_log_error("internal error: module libraray type not known: %s;%d"
		       , sym, mod->type);
	return PAM_ABORT;
    }

#ifdef PAM_DYNAMIC
    if ((mod->type == PAM_DYNAMIC_MOD) &&
        (func = (servicefn) dlsym(mod->dl_handle, sym)) == NULL) {
	_pam_log_error("unable to resolve symbol: %s", sym);
	return(PAM_ABORT);
    }
#endif
#ifdef PAM_STATIC
    if ((mod->type == PAM_STATIC_MOD) &&
        (func = (servicefn)_pam_get_static_sym(mod->dl_handle, sym)) == NULL) {
	_pam_log_error("unable to resolve static symbol: %s", sym);
	return(PAM_ABORT);
    }
#endif
    if (sym2) {
#ifdef PAM_DYNAMIC
	if ((mod->type == PAM_DYNAMIC_MOD) &&
	    (func2 = (servicefn) dlsym(mod->dl_handle, sym2)) == NULL) {
	    _pam_log_error("unable to resolve symbol: %s", sym2);
	    return(PAM_ABORT);
	}
#endif
#ifdef PAM_STATIC
	if ((mod->type == PAM_STATIC_MOD) &&
	    (func2 = (servicefn)_pam_get_static_sym(mod->dl_handle, sym2))
	    == NULL) {
	    _pam_log_error("unable to resolve symbol: %s", sym2);
	    return(PAM_ABORT);
	}
#endif
    }
    
    /* add new handler to end of existing list */
    while (*handler_p != NULL) {
	handler_p = &((*handler_p)->next);
    }

    if ((*handler_p = malloc(sizeof(struct handler))) == NULL) {
	_pam_log_error("cannot malloc struct handler #1");
	return (PAM_ABORT);
    }

    (*handler_p)->func = func;
    (*handler_p)->control_flag = control_flag;
    (*handler_p)->argc = argc;
    (*handler_p)->argv = argv;                       /* not a copy */
    (*handler_p)->next = NULL;

    /* some of the modules have a second calling function */
    if (handler_p2) {
	/* add new handler to end of existing list */
	while (*handler_p2) {
	    handler_p2 = &((*handler_p2)->next);
	}

	if ((*handler_p2 = malloc(sizeof(struct handler))) == NULL) {
	    _pam_log_error("cannot malloc struct handler #2");
	    return (PAM_ABORT);
	}

	(*handler_p2)->func = func2;
	(*handler_p2)->control_flag = control_flag;
	(*handler_p2)->argc = argc;
	if (argv) {
	    if (((*handler_p2)->argv = malloc(argvlen)) == NULL) {
		_pam_log_error("cannot malloc argv for handler #2");
		return (PAM_ABORT);
	    }
	    memcpy((*handler_p2)->argv, argv, argvlen);
	} else {
	    (*handler_p2)->argv = NULL;              /* no arguments */
	}
	(*handler_p2)->next = NULL;
    }

    D(("_pam_add_handler: returning successfully"));

    return PAM_SUCCESS;
}

/* Free various allocated structures and dlclose() the libs */
int _pam_free_handlers(pam_handle_t *pamh)
{
    struct loaded_module *mod;

    IF_NO_PAMH("_pam_free_handlers",pamh,PAM_SYSTEM_ERR);

    mod = pamh->handlers.module;

    /* Close all loaded modules */

    while (pamh->handlers.modules_used) {
	D(("_pam_free_handlers: dlclose(%s)", mod->name));
	free(mod->name);
#ifdef PAM_DYNAMIC
	if (mod->type == PAM_DYNAMIC_MOD) dlclose(mod->dl_handle);
#endif
	mod++;
	pamh->handlers.modules_used--;
    }

    /* Free all the handlers */
    
    _pam_free_handlers_aux(&(pamh->handlers.conf.authenticate));
    _pam_free_handlers_aux(&(pamh->handlers.conf.setcred));
    _pam_free_handlers_aux(&(pamh->handlers.conf.acct_mgmt));
    _pam_free_handlers_aux(&(pamh->handlers.conf.open_session));
    _pam_free_handlers_aux(&(pamh->handlers.conf.close_session));
    _pam_free_handlers_aux(&(pamh->handlers.conf.chauthtok));

    _pam_free_handlers_aux(&(pamh->handlers.other.authenticate));
    _pam_free_handlers_aux(&(pamh->handlers.other.setcred));
    _pam_free_handlers_aux(&(pamh->handlers.other.acct_mgmt));
    _pam_free_handlers_aux(&(pamh->handlers.other.open_session));
    _pam_free_handlers_aux(&(pamh->handlers.other.close_session));
    _pam_free_handlers_aux(&(pamh->handlers.other.chauthtok));

    /* Indicate that handlers are not initialized for this pamh */
    pamh->handlers.handlers_loaded = 0;

    return PAM_SUCCESS;
}

void _pam_start_handlers(pam_handle_t *pamh)
{
    /* NB. There is no check for a NULL pamh here, since no return
     * value to communicate the fact!  */

    /* Indicate that handlers are not initialized for this pamh */
    pamh->handlers.handlers_loaded = 0;

    pamh->handlers.modules_allocated = 0;
    pamh->handlers.modules_used = 0;
    pamh->handlers.module = NULL;

    /* initialize the .conf and .other entries */
    
    pamh->handlers.conf.authenticate = NULL;
    pamh->handlers.conf.setcred = NULL;
    pamh->handlers.conf.acct_mgmt = NULL;
    pamh->handlers.conf.open_session = NULL;
    pamh->handlers.conf.close_session = NULL;
    pamh->handlers.conf.chauthtok = NULL;

    pamh->handlers.other.authenticate = NULL;
    pamh->handlers.other.setcred = NULL;
    pamh->handlers.other.acct_mgmt = NULL;
    pamh->handlers.other.open_session = NULL;
    pamh->handlers.other.close_session = NULL;
    pamh->handlers.other.chauthtok = NULL;
}

void _pam_free_handlers_aux(struct handler **hp)
{
    struct handler *h = *hp;
    struct handler *last;

    while (h) {
	last = h;
	DROP(h->argv);  /* This is all alocated in a single chunk */
	h = h->next;
	free(last);
    }

    *hp = NULL;
}
