/* texparse.cc
   Definitions of classes for parsing TeX.
   First part, the basics.

   ?? Need to fix up ^^A etc and stuff for ASCII characters

    Author: John Collins, collins@phys.psu.edu.
    21 Jan 96

    (C) John Collins & Penn State University.

*/

#include "texparse.h"
#include "utils.h"
#include <string.h>


catcode chcat[256];

void set_TeX_cat(void){
  // Set the array chcat to the cat codes I am using.
  int i;
  for (i=0; i<=31;  i++) chcat[i] = Control;  // ctrl chars special
  for (i=32; i<=255; i++) chcat[i] = Other;  // nothing special
  for (i='A'; i<='Z'; i++) chcat[i] = Letter;
  for (i='a'; i<='z'; i++) chcat[i] = Letter;
  for (i='0'; i<='9'; i++) chcat[i] = Digit;
  chcat[0] = Null;
  chcat['\t'] = White;  //tab
  chcat['\n'] = NewLine;
  chcat['\f'] = White; //form feed ^L
  chcat[' '] = White; //space
  // Now the TeX specials
  chcat['^'] = SuperscriptCh;
  chcat['_'] = SubscriptCh;
  chcat['\\'] = EscapeCh;
  chcat['{'] = BeginGroupCh;
  chcat['}'] = EndGroupCh;
  chcat['%'] = CommentCh;
  chcat['$'] = MathShiftCh;
  chcat['#'] = ParameterCh;
  chcat['&'] = AlignTabCh;

}


/* ==============================================================
   class BaseTeXObject
   Base class for all objects in a parsed structure for a TeX file.
*/


BaseTeXObject::BaseTeXObject() { Init();}


BaseTeXObject::~BaseTeXObject() {
   // Unlink me.
   if (prev) (prev->next) = next;
   if (next) (next->prev) = prev;
   if (prevlex) (prevlex->nextlex) = nextlex;
   if (nextlex) (nextlex->prevlex) = prevlex;
}



BaseTeXObject * BaseTeXObject::head() {
  BaseTeXObject *p = this;
  while (p->prev) p= p->prev;
  return p;
}

BaseTeXObject * BaseTeXObject::tail() {
  BaseTeXObject *p = this;
  while (p->next) p= p->next;
  return p;
}

char *BaseTeXObject::GetName() {
   //Name of environment I control, or documentclass's style, etc.
   return NULL;
}

void BaseTeXObject::Show (sos &f) const {
   // For debugging.  Show what kind of an object we've got.
   f << "BaseTeXObject\n";
}


void BaseTeXObject::ShowStartEnd (sos &f) const {
   f << "start at ";
   start.Show(f);
   f << ", end at ";
   end.Show(f);
}

void BaseTeXObject::WriteTeX (sos &f) const {
   // Write the TeX representation of this object.
}



int BaseTeXObject::IsTeXText()   { return FALSE;}
int BaseTeXObject::IsBlankLine() { return FALSE;}
int BaseTeXObject::IsComment()   { return FALSE;}

void BaseTeXObject::Extend(TeXPosn &NewEnd){
   /* Extend the ending position of my text.  Error if I am not pointing
      to anything.
   */
   if (end.l) {
      end = NewEnd;
   } else {
     // I am not pointing to anything.
     lite_cerr <<
     "Bug: attempt to Extend a BaseTeXObject that is not pointing to text."
     << endl;
   }
}


//===========================================================



/* ==============================================================
   class TeXObject
   Base for TeXObjects that point to something.
*/

TeXObject::TeXObject(const TeXPosn &p1, const unsigned int i2) {
   // p1 is starting point of the text, i2 is index of end in same line.
   start = p1;
   end.l = p1.l;
   end.index = i2;
}

TeXObject::TeXObject(const TeXPosn &p1, const TeXPosn &p2) {
   // p1 is starting point of the text, p2 is ending point of the text.
   start = p1;
   end   = p2;
}

void TeXObject::Show (sos &f) const {
   // For debugging.  Show what kind of an object we've got.
   f << "TeXObject; ";
//??   ShowStartEnd(f);
//??   f << ".\n";
}



void TeXObject::WriteMyText (sos &f) const {
   // Write out the text I point to

   Line *curline = start.l;
   unsigned int i1 = start.index;
   while (curline){
      if (curline->len > 0) {
         /* There's something to print on this line.
	    Print chars [i1] to [i2] of current line.
            i1 is determined.
            i2 is either end of line, or the end of the buffer, whichever is
            earlier.
         */
         unsigned int i2 = (curline->len)-1;
         // Note that i2 is the index into the file, and is zero-based.
         if ((curline == end.l) && (i2 > end.index))
            i2 = end.index;
         for (unsigned int i = i1; i <= i2; i++)
            f.put ((*curline)[i]);
         // The following is faster:
         //write (fileno(f), curline->data+i1, i2-i1+1);
      }
      if (curline == end.l) {
         curline = NULL; // Flag that we are done
      } else {
         curline = curline->next;
         i1 = 0;
      }
   }
}






/* ==============================================================
   class TeXText
   This points to a piece of text that does not contain any special
   characters.  It contains the terminating \n if there is one.
   It may span more than one line.
*/



TeXText::TeXText(const TeXPosn &p1, const unsigned int i2)
   : TeXObject(p1, i2) {
   // p1 is starting point of the text, i2 is index of end in same line.
   ID = IDText;
}


TeXText::TeXText(const TeXPosn &p1, const TeXPosn &p2) : TeXObject(p1, p2){
   // p1 is starting point of the text, p2 is ending point.
   ID = IDText;
}


int TeXText::IsTeXText()   { return TRUE;}
int TeXText::IsBlankLine() { return FALSE;}
int TeXText::IsComment()   { return FALSE;}


void TeXText::Show (sos &f) const {
   // For debugging.  Show what kind of an object we've got.
   f << "TT; ";
//   ShowStartEnd(f);
//   f << ".\n";
}


void TeXText::WriteTeX (sos &f) const {
   // Write the TeX representation of this object.
   WriteMyText (f);
}



/* ==============================================================
   class BlankLine
   This points to a piece of text that is a blank line, plus all following
   white space/newlines.
   It may span more than one line.
*/


BlankLine::BlankLine(const TeXPosn &p1) : TeXText (p1, p1.l->len -1) {
   /* p1 is starting point of the text; end is necessarily end-of-line.
      But a BlankLine may be extended later beyond the end-of-line,
      to include white space and more blank lines.
   */
   ID = IDPar;
}


int BlankLine::IsTeXText()   { return FALSE;}
int BlankLine::IsBlankLine() { return TRUE;}

void BlankLine::Show (sos &f) const {
   // For debugging.  Show what kind of an object we've got.
   f << "BlankLine; ";
//   ShowStartEnd(f);
//   f << ".\n";
}





/* ==============================================================
   class Comment

   This points to a piece of text that is TeX comment (i.e., '%'
   followed by all characters up to and including a new line.
   Add in any following white space.
   It may NOT span more than one line.

   It should always point to some text. Hence there is no Comment()
   constructor.  Since this is just a specialization of the TeXText
   class, the destructors are just call those for TeXText.

   No destructor is needed.
*/


Comment::Comment(const TeXPosn &p1) : TeXText (p1, p1.l->len -1) {
   /* p1 is starting point of the text; end is necessarily end-of-line.
      But a Comment may be extended later beyond the end-of-line,
      to include white space.
   */
   ID=IDComment;
}


int Comment::IsTeXText()   { return FALSE;}
int Comment::IsComment()   { return TRUE;}


void Comment::Show (sos &f) const {
   // For debugging.  Show what kind of an object we've got.
   f << "Comment; ";
//   ShowStartEnd(f);
//   f << ".\n";
}



//===========================================================


/* ==========================================================
   class BaseTeXSpecial
*/

void BaseTeXSpecial::Show (sos &f) const {
   f << "BaseTeXSpecial\n";
}



/* ===========================================================
   class DummyTeX
   For debugging.  Used for TeX CS's I haven't defined yet.
*/

void DummyTeX::Show (sos &f) const {
   f << "==========DummyTeX; ";
//   ShowStartEnd(f);
//   f << ".\n";
}

void DummyTeX::WriteTeX (sos &f) const{
   f << "*****";
}


/* ==========================================================
   class TeXCS
*/

TeXCS::TeXCS(const TeXPosn &p1): BaseTeXSpecial (p1){
   ID = IDCS;
   endCS = end;
   lenCS = 1;
   if (chcat[p1.c()] == SuperscriptCh) {
      // Nothing to do for now
      end.SkipCSTrail();
   } else if (chcat[p1.c()] == SubscriptCh) {
      // Nothing to do for now
      end.SkipCSTrail();
   } else {
      // Assume p1 points to escape char (normally '\\')
      if (chcat[p1[1]] != Letter) {
         // '\\' followed by single character
         endCS.inc();
         lenCS = 2;
         end = endCS;
      } else {
         while (chcat[endCS[1]] == Letter) {
            endCS.inc();
            lenCS++;
	  }
         // endCS now points to the end of the CS
         end = endCS;
         end.SkipCSTrail();
      }
   }
}

void TeXCS::Show (sos &f) const {
   f << "CS; ";
//   ShowStartEnd(f);
//   f << "; ";
   for (int i = 1; i<=lenCS; i++)
       f.put(start[i-1]);
   f << "\n";
}

void TeXCS::WriteTeX (sos &f) const {
    WriteMyText(f);
}

/* ==========================================================
   class DocClass
*/

DocClass::DocClass(const TeXPosn &p1, const TeXPosn &p2)
: BaseTeXSpecial(p1){
  ID = IDDocClass;
  end = p2;
  end.inc();
  options = NULL;
  int lensty;
  if (end.c() == '[') {
     end.dec();
     style = strdup ("article");
  } else {
     GetSimpleArg(end, style, lensty);
     if (style) {
        // Make it into a null-terminated string and skip space.
        style = MakeString (style, lensty);
        end.SkipCSTrail();
        lite_cerr << style <<endl;
     } else {
       // Go back where I started.
       end.dec();
     }
  }
}

DocClass::~DocClass() {
   if (options) delete options;
   if (style) delete style;
}

char *DocClass::GetName() {
   //Name of environment I control, or documentclass's style, etc.
   return style;
}

void DocClass::Show (sos &f) const {
   f << " DC{" << style << "} ";
}

void DocClass::WriteTeX (sos &f) const {
   f<< "\\documentclass";
   if (options)
      f << '[' << options << ']';
   f <<'{' << style << "}\n";
}


/* ==========================================================
   class BeginEnv
*/

BeginEnv::BeginEnv(const TeXPosn &p1, const TeXPosn &p2)
: BaseTeXSpecial(p1){
  ID = IDBegin;
  Body = NULL;
  end = p2;
  end.inc();
  Opens = 1;
  int len;
  GetSimpleArg(end, name, len);
  if (name) {
     // Make it into a null-terminated string and skip space.
     name = MakeString (name, len);
     end.SkipCSTrail();
  } else {
     end.dec();
  }
}

BeginEnv::~BeginEnv() {
   if (name) delete name;
   if (Body) delete Body;
}

char *BeginEnv::GetName() {
   //Name of environment I control, or documentclass's style, etc.
   return name;
}

void BeginEnv::Show (sos &f) const {
   if (Body) {
      f << "\nEnv{" << name << "}\n";
      if (strcmp(name, "document") == 0)
         Body->ShowAll();
   } else {
      f << "\nBE{" << name << "}\n";
   }
}

void BeginEnv::WriteTeX (sos &f) const {
   if (Body) {
      f << "\\begin{" << name << "}\n";
      Body->WriteTeXAll(f);
      f << "\\end{" << name << "}\n";
   } else {
      f << "\\begin{" << name << "}\n";
   }
}


/* ==========================================================
   class EndEnv
*/

EndEnv::EndEnv(const TeXPosn &p1, const TeXPosn &p2)
: BaseTeXSpecial(p1){
  ID = IDEnd;
  end = p2;
  end.inc();
  Closes = 1;
  int len;
  GetSimpleArg(end, name, len);
  if (name) {
     // Make it into a null-terminated string and skip space.
     name = MakeString (name, len);
     end.SkipCSTrail();
  }
}

EndEnv::~EndEnv() {
   if (name) delete name;
}

char *EndEnv::GetName() {
   //Name of environment I control, or documentclass's style, etc.
   return name;
}

void EndEnv::Show (sos &f) const {
   f << "EE{" << name << "}\n";
}

void EndEnv::WriteTeX (sos &f) const {
   f << "\\end{" << name << "}\n";
}


//===========================================================


/* ==========================================================
   class Parameter
*/

Parameter::Parameter (const TeXPosn &p1): BaseTeXSpecial(p1) {
   //Assume p1 points to '#' (or equivalent)
   if ((chcat[p1[1]] == ParameterCh) && (chcat[p1[2]] == Digit)) {
      // Then we have ##
      end.inc();
      end.inc();
   } else if (chcat[p1[1]] == Digit) {
      end.inc();
   } else {
      lite_cerr << "Bad parameter: ";
      p1.ShowContext(lite_cerr);
   }
}

void Parameter::Show (sos &f) const {
   f << "Parameter; ";
//   ShowStartEnd(f);
//   f << ".\n";
}

void Parameter::WriteTeX (sos &f) const {
    WriteMyText(f);
}


/* ==========================================================
   class BaseSimple
*/

void BaseSimple::Show (sos &f) const {
   f.put(name);
}

void BaseSimple::WriteTeX (sos &f) const {
   f.put(name);
}





//===============================================================
//===============================================================

/* ==============================================================
      class ListHead
     Used to mark the head of lists.
      It is the only BaseTeXObject that should have a pointer to it that
      is not via the prev, etc pointers.
      Thus it is dangerous to delete it except from the controlling routine
      that owns all the pointers to it.
*/

ListHead::ListHead(BaseTeXObject *head)
   // Extract a list and attach it to me
: BaseTeXObject(){
   ID=IDListHead;
   Extract (head);
   next = head;
   if (head) head->prev = this;
}

ListHead::ListHead(BaseTeXObject *head, BaseTeXObject *tail)
: BaseTeXObject() {
   // Extract a list and attach it to me
   ID=IDListHead;
   if (head && tail) {
      MoveTo(*this, head, tail);
   } else {
     Error ("I tried to make a ListHead from a bad list.", Nowhere);
   }
}

void ListHead::Show (sos &f) const {
   f << "ListHead\n";
}


void ListHead::ShowAll (sos &f) const {
   for (BaseTeXObject const *t = this; t; t = t->next)
      t->Show(f);
}



void ListHead::WriteTeXAll (sos &f) const {
   for (BaseTeXObject const *t = this; t; t = t->next)
      t->WriteTeX (f);
}



int ListHead::Tokenize (const Buffer &LineList){
   /* The real thing, starting out for now.

      Construct a list of TeXObjects attached to me, using the data
      in LineList.

      Return 0 if I have success, an error code otherwise.
      The error code is that of the first error detected, for a non-fatal
      error, or the error code of a fatal error that prevented me from
      continuing.
   */

   if (!LineList.first) {
      // There is nothing to do.
      return 0;
   }
   BaseTeXObject *Last = this;
   TeXPosn FirstChar(LineList.first, 0);
   TeXPosn CurPos = FirstChar;
   int RetCode = 0, CountObjects = 0;
   set_TeX_cat();

   while (!CurPos.eof()) {
      int NewRet = CurPos.AddOneObject(Last);
      CountObjects++;
/*???
      if (CountObjects % 10 == 0)
         // Allow user on MSDOS to ctrl/C out:
         lite_cerr << ".";
*/
      if (NewRet != 0) {
         if (RetCode == 0)
            // RetCode is normally the first error.
            RetCode = NewRet;
         if (NewRet == OUT_OF_MEMORY) {
            // I'm in trouble
            lite_cerr << "Repeat message.  Out of memory.\n";
            return NewRet;
         }
      }
   }
   return RetCode;
}


int ListHead::RawTokenize (const Buffer &LineList){
   /* Dummy for now.  Convert everthing to text.
      Return 0 if I have success, an error code otherwise.
   */

   if (!LineList.first) {
      // There is nothing to do.
      return 0;
   }

   BaseTeXObject *last = this;
   TeXPosn FirstChar(LineList.first, 0);
   TeXPosn CurPos = FirstChar;
   TeXText *t = NULL;

   set_TeX_cat();

   goto version3;


//===================================================================
version1:
// All the lines in separate TeXTexts

   while (CurPos.l) {
      t = new TeXText(CurPos, CurPos.l->len -1);
      if (!t) {
         // Something went wrong.
         return 1;
      }
      AddToList (last, t);
      CurPos.l = CurPos.l->next;
   }
   // Now set the lexical list:
   for (last = this; last; last = last->next)
      SetLexFromLogical ();

   return 0;


//===================================================================
version2:
// All the lines in a single TeXText


   while (CurPos.l->next)
      CurPos.l = CurPos.l->next;
   CurPos.index = CurPos.l->len -1;
   t = new TeXText(FirstChar, CurPos);
   if (!t) {
      // Something went wrong.
      return 1;
   }
   lite_cerr << (int)FirstChar.index << " " << (int)CurPos.index << endl;
   AddToList (last, t);
   // Now set the lexical list:
   for (last = this; last; last = last->next)
      SetLexFromLogical ();
   return 0;

version3:
// All the lines in a single TeXText.  Exercise the inc() routine to do it.

lite_cerr << "Starting to run through rawtokenize by chars...\n";
   while (!CurPos.eof())
      CurPos.inc();
   t = new TeXText(FirstChar, CurPos);
   if (!t) {
      // Something went wrong.
      return 1;
   }
   AddToList (last, t);
lite_cerr << "I have done the rawtokenize by chars.\n";
lite_cerr << (int)FirstChar.index << " " << (int)CurPos.index << endl;
   // Now set the lexical list:
   for (last = this; last; last = last->next)
      SetLexFromLogical ();
   return 0;

}





//===============================================================
//===============================================================
//===============================================================


/* =================================================================
   class TeXPosn
   Used for position in a TeX file.  Its methods are used to do
      the parsing of the file.
*/

Line EmptyLine;

int TeXPosn::AddOneObject (BaseTeXObject *&Last) {
   /* Construct the TeX object that starts at my position,
      add it to a list of TeXObjects, after the element Last,
      update Last to point to the newly added TeXObject,
      and move myself to the next character.

      Normally I will only get a construct from the current line.  If
      necessary, I will append it to the previous construct (for TeXText,
      and for a blank line).

      Return 0 if I succeeded, a positive error code if I failed, and
      EOF if I got to the end of the file.
   */

   if (index >= l->len) {
      /* I am pointing at end of the line, so I am not pointing to a
         character.  Advance to the next line which is non-empty,
         if any.
      */
      lite_cerr <<
   "?? I think this is an error.  Outside a line in TeXPosn::AddOneObject.\n";
      ShowContext(lite_cerr);
      return EOF;
   }

   /* Now I point to a character.  Now find the last character of the
      construct I begin.
   */
   if (chcat[c()] > NewLine) {
      return AddTeXToken (Last);
   }
   TeXPosn EndChar = *this;
   if (index == 0) {
      // Check for blank line
      EndChar.SkipWhite();
      if (chcat[EndChar.c()] == NewLine) {
         // My construct is a blank line.
         if (Last && (Last->IsBlankLine())) {
            /* If the previous construct was a blank line, then we
               concatenate this blank line with the previous one, since
               TeX treats multiple blank lines the same as one.
            */
            Last->Extend(EndChar);
         } else {
            TeXObject *t = new BlankLine (*this);
            if (!t) {
               // Something went wrong.
               lite_cerr <<
                     "In TeXPosn::AddOneObject, I ran out of memory.\n";
               ShowContext(lite_cerr);
               return OUT_OF_MEMORY;
            }
            AddToList (Last, t);
         }
         goto ExitPoint;
      }
      /* Now I know I am not the first character of a blank line.
         But I first must append my blank space to the end of the previous
         construct.  Since it ended with a newline, TeX will ignore the
         white space at the beginning of this line.
      */
      if ((EndChar.index > 0) && (Last)) {
        // My line starts with white space.  Point EndChar to its end:
        EndChar.dec();
        Last->Extend(EndChar);
        goto ExitPoint;
      }
   }
   /* If I arrive here, I know that I point to a character that can be part
      of a TeXText, and that I am not on a blank line.  Moreover, any blank
      space at the beginning of the line has been appended to the previous
      construct.
      So I need to find the first non-text character in the line,
      or the end-of-line, whichever is first.
   */
   EndChar.FindTeXSpecial();
   if (EndChar.index == 0) {
     // This shouldn't happen.  It's a bug if I arrive here.
     lite_cerr <<
     "Something is wrong in TeXPosn::AddOneObject\n";
     ShowContext(lite_cerr);
     return 1;
   }
   if (EndChar.cat() != NewLine)
      EndChar.dec();
   if (Last && (Last->IsTeXText())) {
      /* If the previous construct was TeXText, then we just add the new
         stuff to it.
      */
      Last->Extend(EndChar);
   } else {
      TeXObject *t = new TeXText(*this, EndChar);
      if (!t) {
         // Something went wrong.
         Show(lite_cerr);
         lite_cerr <<
                  ".  In TeXPosn::AddOneObject, I ran out of memory.\n";
         return OUT_OF_MEMORY;
      }
      AddToList (Last, t);
   }

   ExitPoint:
   // Now EndChar points to the last character of my construct.
   *this = EndChar;
   inc();

   return 0;
}




void TeXPosn::MakeCS (TeXObject *&t) {
   /* Make a Control Sequence object t from the TeXCS I point to.
   */

   //Default 1 character CS:
   TeXPosn endCS = *this, end = endCS;
   int lenCS = 1;
   if (chcat[c()] == SuperscriptCh) {
      end.SkipCSTrail();
      t = new TeXCS (*this);
   } else if (chcat[c()] == SubscriptCh) {
      end.SkipCSTrail();
      t = new TeXCS (*this);
   } else {
      // Assume p1 points to escape char (normally '\')
      if (chcat[(*this)[1]] != Letter) {
         // '\' followed by single character
         endCS.inc();
         lenCS = 2;
         end = endCS;
      } else {
         while (chcat[endCS[1]] == Letter) {
            endCS.inc();
            lenCS++;
	  }
         // endCS now points to the end of the CS
         end = endCS;
         end.SkipCSTrail();
      }
      //I point to a CS of length lenCS
      if ( eqstr1("\\documentclass", l->data+index, lenCS)) {
         t = new DocClass (*this, end);
      } else if ( eqstr1("\\documentstyle", l->data+index, lenCS)) {
         // This will need different handling from documentclass
         t = new DocClass (*this, end);
      } else if (eqstr1("\\begin", l->data+index, lenCS)) {
         t = new BeginEnv (*this, end);
      } else if (eqstr1("\\end", l->data+index, lenCS)) {
         t = new EndEnv (*this, end);
      } else if ( eqstr1("\\par", l->data+index, lenCS)) {
         t = new TeXCS (*this);
         t->ID = IDPar;
      } else {
         t = new TeXCS (*this);
      }
   }
}




int TeXPosn::AddTeXToken (BaseTeXObject *&Last) {
   /* Assume I am pointing at a TeX special character.
      Construct the TeX token that starts at my position,
      add it to a list of TeXObjects, after the element Last,
      update Last to point to the newly added TeXObject,
      and move myself to the next character.

      Return 0 if I succeeded, a positive error code if I failed, and
      -1 if I got to the end of the file.
   */

   TeXObject *t;
   switch (cat()){
      case CommentCh:
         t = new Comment (*this);
         break;
      case EscapeCh:
         MakeCS (t);
         break;
      case BeginGroupCh:
         t = new BeginGroup (*this);
         break;
      case EndGroupCh:
         t = new EndGroup (*this);
         break;
      case MathShiftCh:
         t = new MathShift (*this);
         break;
      case AlignTabCh:
         t = new AlignTab (*this);
         break;
      case ParameterCh:
         t = new Parameter (*this);
         break;
      case SuperscriptCh:
         t = new TeXCS (*this);
         break;
      case SubscriptCh:
         t = new TeXCS (*this);
         break;
      case Other:
      case Control:
      case Invalid:
         lite_cerr << "Incorrect character\n";
         ShowContext(lite_cerr);
         t = new DummyTeX(*this);
         break;
      default:
         lite_cerr << "BUG: unknown character\n";
         ShowContext(lite_cerr);
         t = new DummyTeX(*this);
   }
   if (!t) {
      // Something went wrong.
      lite_cerr <<
               "In TeXPosn::AddTeXToken, I ran out of memory.\n";
      ShowContext(lite_cerr);
      return OUT_OF_MEMORY;
   }
   AddToList (Last, t);
   *this = t->end;
   inc();
   return 0;
}


void TeXPosn::SkipWhite() {
   /* Point myself to the first non-white-space character starting from my
      current position.  Don't count newline as white.
   */
   while (chcat[c()] == White)
      inc();
}


void TeXPosn::SkipWhiteAndNL() {
   /* Point me to the first non-white-space character starting from my
      current position.  Count newline as white.
   */

   while ( (chcat[c()] == White) || (chcat[c()] == NewLine))
      inc();
}


void TeXPosn::SkipCSTrail() {
   /* I am pointing at the last character of a CS.  If there is following
      white space point me at the last space, else leave me where I am.
      Here white space is a sequence of white space characters followed by
      zero or one newline.
   */
   while (chcat[(*this)[1]] == White)
      inc();
   if (chcat[(*this)[1]] == NewLine)
      inc();
}


void TeXPosn::FindTeXSpecial() {
   /* Point myself to the first TeX special character forward from
      the current position.  A TeX special character is one that is not
      regular text.
   */
   while (chcat[c()] < NewLine)
      inc();
}


/* =================================================================
   =================================================================

   Other routines.
*/



void AddToList (BaseTeXObject *&last, TeXObject *newitem){
   /* Add the new item to the list after the item given by last.
      Update last to point to the new object.
      Do not necessarily assume that last is actually the last item
      in the list.
      If the list is empty (last == NULL) just set last = newitem.
      Don't mess with the lexical structure.
   */
   if (!newitem){
     lite_cerr <<
     "Bug?? AddToList was asked to add a NULL item to a list.\n";
     return;
   }
   if (last) {
      newitem->prev = last;
      newitem->next = last->next;
      last->next = newitem;
      if (newitem->next)
         (newitem->next)->prev = newitem;
   }
   last = newitem;
}


void GetSimpleArg (TeXPosn & p, char *& s, int & len) {
   s=NULL; len = 0;
   if (p.cat() == BeginGroupCh) {
      TeXPosn end = p;
      end.inc();
      while (end.cat() < NewLine)
         end.inc();
      // So end points to the first special character after the '{'
      if (end.cat() == EndGroupCh) {
         s = p.l->data + p.index + 1;
         len = end.index - p.index - 1;
         p = end;
      } else {
         Error ("Couldn't find end of simple argument", p);
      }
   } else if (p.cat() < NewLine) {
      s = p.l->data + p.index;
      len = 1;
      p.inc();
   } else {
      Error ("Couldn't find simple argument", p);
   }
}

char *MakeString (char *s, int len) {
   // Make null terminated string
   if (!s) return NULL;
   char *s1 = new char [len+1];
   strncpy (s1, s, len);
   s1[len] = 0;
   return s1;
}

int SameName(const BeginEnv &b, const EndEnv &e) {
   return (strcmp (b.name, e.name) == 0);
}


