// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <ctype.h>
}
#include <qpainter.h>
#include "BlockRenderer.h"
#include "Canvas.h"
#include "FormRenderer.h"
#include "HLineRenderer.h"
#include "Options.h"
#include "PreformatRenderer.h"
#include "Style.h"
#include "StyleSheet.h"
#include "TableRenderer.h"
#include "VerticalRenderer.h"

VerticalRenderer::ChildInfo::ChildInfo( Renderer* _renderer,
                                        int       _marginLeft,
                                        int       _marginRight,
                                        int       _marginTop,
                                        int       _marginBottom )
        : renderer( _renderer ),
          marginLeft( _marginLeft ),
          marginRight( _marginRight ),
          marginTop( _marginTop ),
          marginBottom( _marginBottom )
{
}

VerticalRenderer::VerticalRenderer( Canvas*     canvas,
                                    SgmlParser* parser,
                                    int         clipWidth,
                                    QObject*    parent,
                                    const char* name )
        : SgmlRenderer( canvas, parser, clipWidth, parent, name ),
          _sgmlRenderer( 0 ),
          _textRenderer( 0 ),
          _marginLeft( 0 ),
          _marginRight( 0 ),
          _vspace( 0 ),
          _sgmlDepth( 0 )
{
    resize( clipWidth, 0 );
}

VerticalRenderer::~VerticalRenderer()
{
//    printf( "VerticalRenderer::~VerticalRenderer()\n" );
    while ( _renderers.first() ) {
        delete _renderers.first()->renderer;
	delete _renderers.first();
	_renderers.remove();
    }
}

void VerticalRenderer::content( QString text )
{
    if ( _sgmlRenderer ) {
        _sgmlRenderer->content( text );
        return;
    }

    if ( !style() ) return;
    
    if ( style()->displayType() == Style::Title ) {
        _titleString += text;
    } else if ( style()->displayType() == Style::Image ) {
        // Do nothing.
        printf( "ignoring content '%s'\n", text.data() );
    } else if ( _textRenderer ) {
        _textRenderer->content( text );
    } else {
        int tmp;
        
        // Is the content empty?
        if ( ( style()->enumValue( "white-space", tmp ) ) && ( tmp  != Style::Pre ) ) {
            bool empty = TRUE;
            for ( int i = 0; text[i]; i++ ) {
                if ( !isspace( text[i] ) ) {
                    empty = FALSE;
                }
            }
            if ( empty ) {
                return;
            }
        }
    
        if ( ( style()->enumValue( "white-space", tmp ) ) && ( tmp == Style::Normal ) ) {
            _textRenderer = new BlockRenderer( canvas(), parser(), clipWidth() - _marginLeft - _marginRight, this );
        } else {
            _textRenderer = new PreformatRenderer( canvas(), parser(), clipWidth() - _marginLeft - _marginRight, this );
        }
        _renderers.append( new ChildInfo( _textRenderer, _marginLeft, _marginRight, _vspace, 0 ) );
        _vspace = 0;
        connect( _textRenderer, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
        _textRenderer->startTag();
        _textRenderer->content( text );
//        _textRenderer->show();
    }
}

void VerticalRenderer::endOfData()
{
    if ( _sgmlRenderer ) {
        _sgmlRenderer->endOfData();
        _sgmlRenderer = 0;
    }
    
    if ( _textRenderer ) {
        _textRenderer->endOfData();
        _textRenderer = 0;
    }
    
    if ( _renderers.last() ) {
        _renderers.last()->marginBottom = _vspace;
    }
    
//    redraw();
    needRedraw();
}

void VerticalRenderer::endTag()
{
    if ( _sgmlRenderer ) {
        _sgmlRenderer->endTag();
        if ( _sgmlDepth == int( tagStack().count() ) ) {
            _sgmlRenderer->endOfData();
            _sgmlRenderer = 0;
        }
        return;
    }
    
    if ( style()->displayType() == Style::Block ) {
        if ( _textRenderer ) {
            _textRenderer->endOfData();
            _textRenderer = 0;
        }
    }

    int tmp;
    
    if ( ( style()->numberValue( "margin-bottom", tmp ) ) && ( tmp > _vspace ) ) {
        _vspace = tmp;
    }

    if ( style()->numberValue( "margin-left", tmp ) ) {
        _marginLeft  -= tmp;
    }
    if ( style()->numberValue( "margin-right", tmp ) ) {
        _marginRight -= tmp;
    }
    
    if ( _textRenderer ) {
        _textRenderer->endTag();
    } else if ( style()->displayType() == Style::Title ) {
        canvas()->setTitle( _titleString );
    }
}

void VerticalRenderer::startTag()
{
    if ( _sgmlRenderer ) {
        _sgmlRenderer->startTag();
        return;
    }

    int tmp;
    
    if ( ( style()->numberValue( "margin-top", tmp ) ) && ( tmp > _vspace ) ) {
        _vspace = tmp;
    }

    if ( style()->numberValue( "margin-left", tmp ) ) {
        _marginLeft  += tmp;
    }
    if ( style()->numberValue( "margin-right", tmp ) ) {
        _marginRight += tmp;
    }

    if ( style()->displayType() == Style::Block ) {
        if ( _textRenderer ) {
            _textRenderer->endOfData();
            _textRenderer = 0;
        }
    } else if ( ( style()->displayType() == Style::Inline    ) ||
                ( style()->displayType() == Style::Hyperlink ) ){
        if ( _textRenderer ) {
            _textRenderer->startTag();
        }
    } else if ( style()->displayType() == Style::Title ) {
        _titleString = "";
    } else if ( style()->displayType() == Style::Base ) {
        const QString* tmp = 0;
        QString tmpStr;
        if ( style()->stringValue( "link-attr", tmpStr ) ) {
            tmp = tag()->find( tmpStr );
        }
        if ( tmp ) {
            canvas()->setBaseUrl( Url( *tmp ) );
        }
    } else if ( style()->displayType() == Style::Table ) {
        if ( _textRenderer ) {
            _textRenderer->endOfData();
            _textRenderer = 0;
        }
        _sgmlRenderer = new TableRenderer( canvas(), parser(), clipWidth() - _marginLeft - _marginRight, this );
        _renderers.append( new ChildInfo( _sgmlRenderer, _marginLeft, _marginRight, _vspace, 0 ) );
        _vspace = 0;
        connect( _sgmlRenderer, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
//        _sgmlRenderer->show();
        _sgmlDepth = tagStack().count();
        _sgmlRenderer->startTag();
    } else if ( style()->displayType() == Style::HLine ) {
        if ( _textRenderer ) {
            _textRenderer->endOfData();
            _textRenderer = 0;
        }
        HLineRenderer* hline = new HLineRenderer( canvas(), clipWidth() - _marginLeft - _marginRight, this );
        _renderers.append( new ChildInfo( hline, _marginLeft, _marginRight, _vspace, 0 ) );
        _vspace = 0;
        connect( hline, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
//        hline->show();
    } else if ( ( style()->displayType() == Style::Anchor         ) ||
                ( style()->displayType() == Style::Image          ) ||
                ( style()->displayType() == Style::FormButton     ) ||
                ( style()->displayType() == Style::FormCheckBox   ) ||
                ( style()->displayType() == Style::FormEntry      ) ||
                ( style()->displayType() == Style::FormHidden     ) ||
                ( style()->displayType() == Style::FormImage      ) ||
                ( style()->displayType() == Style::FormListBox    ) ||
                ( style()->displayType() == Style::FormMLE        ) ||
                ( style()->displayType() == Style::FormOption     ) ||
                ( style()->displayType() == Style::FormOptionMenu ) ||
                ( style()->displayType() == Style::FormRadio      ) ) {
        if ( !_textRenderer ) {
            int tmp;
            if ( ( style()->enumValue( "white-space", tmp ) ) && ( tmp == Style::Normal ) ) {
                _textRenderer = new BlockRenderer( canvas(), parser(), clipWidth() - _marginLeft - _marginRight, this );
            } else {
                _textRenderer = new PreformatRenderer( canvas(), parser(), clipWidth() - _marginLeft - _marginRight, this );
            }
            _renderers.append( new ChildInfo( _textRenderer, _marginLeft, _marginRight, _vspace, 0 ) );
            _vspace = 0;
            connect( _textRenderer, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
//            _textRenderer->show();
        }
        _textRenderer->startTag();
    } else if ( ( style()->displayType() == Style::Form ) && ( strcmp( className(), "FormRenderer" ) ) ) {
        if ( _textRenderer ) {
            _textRenderer->endOfData();
            _textRenderer = 0;
        }
        _sgmlRenderer = new FormRenderer( canvas(), parser(), clipWidth() - _marginLeft - _marginRight, this );
        _renderers.append( new ChildInfo( _sgmlRenderer, _marginLeft, _marginRight, _vspace, 0 ) );
        _vspace = 0;
        connect( _sgmlRenderer, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
//        _sgmlRenderer->show();
        _sgmlDepth = tagStack().count();
        _sgmlRenderer->startTag();
    } else {
        // Unsupported display style.
        if ( _textRenderer ) {
            _textRenderer->endOfData();
            _textRenderer = 0;
        }
    }
}

void VerticalRenderer::widthChanged( int w )
{
    if ( _clipWidth != w ) {
        _clipWidth = w;
    
        QListIterator<ChildInfo> i( _renderers );
        for ( i.toFirst(); i.current(); ++i ) {
            i.current()->renderer->blockSignals( TRUE );
            i.current()->renderer->widthChanged( _clipWidth - i.current()->marginLeft - i.current()->marginRight );
            i.current()->renderer->blockSignals( FALSE );
        }
//        redraw();
        needRedraw();
    }
}

bool VerticalRenderer::redraw()
{
    // Give my kids a chance to redraw.
    bool b = FALSE;
    ChildInfo* r;
    for ( r = _renderers.first(); r; r = _renderers.next() ) {
        if ( r->renderer->redraw() ) {
            b = TRUE;
        }
    }
    
    // Do I need to redraw?
    if ( !_needRedraw ) return b;

    int        bm = 0;
    int        y = 4;
    int        w = 0;

    _minimumWidth = 0;
    _maximumWidth = 0;
    
    for ( r = _renderers.first(); r; r = _renderers.next() ) {
        if ( _minimumWidth < r->renderer->minimumWidth() ) {
            _minimumWidth = r->renderer->minimumWidth();
        }
        if ( _maximumWidth < r->renderer->maximumWidth() ) {
            _maximumWidth = r->renderer->maximumWidth();
        }
        
        // Padding above.
        if ( bm > r->marginTop ) {
            y += bm;
        } else {
            y += r->marginTop;
        }
        bm = r->marginBottom;
        
        r->renderer->move( r->marginLeft, y );
        y += r->renderer->height();
        
        if ( ( r->renderer->width() + r->marginLeft + r->marginRight ) > w ) w = r->renderer->width() + r->marginLeft + r->marginRight;
    }
    y += bm;

    resize( w, y );

    _needRedraw = FALSE;

    return TRUE;
}

bool VerticalRenderer::findAnchor( const QString& name, int& x, int& y )
{
    ChildInfo* i;

    for ( i = _renderers.first(); i; i = _renderers.next() ) {
        if ( i->renderer->inherits( "SgmlRenderer" ) ) {
            if ( ((SgmlRenderer*)i->renderer)->findAnchor( name, x, y ) ) {
                x += i->renderer->x();
                y += i->renderer->y();
                return TRUE;
            }
        }
    }

    return FALSE;
}

void VerticalRenderer::repaint( QPainter& p, const Rect& r )
{
    ChildInfo* i;

    p.translate( x(), y() );
    for ( i = _renderers.first(); i; i = _renderers.next() ) {
        if ( i->renderer->intersects( r ) ) {
            Rect r1 = i->renderer->intersect( r );
            r1.moveBy( -i->renderer->x(), -i->renderer->y() );
//            r1.moveTopLeft( QPoint( 0, 0 ) );
            i->renderer->repaint( p, r1 );
        }
    }
    p.translate( -x(), -y() );
}
