// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qapp.h>
#include <qdatetm.h>
#include <qmsgbox.h>
#include <qpopmenu.h>
#include "Bookmarks.h"
#include "Browser.h"
#include "Options.h"
#include "Request.h"

Link::Link( Bookmarks* bookmarks )
        : _bookmarks( bookmarks )
{
}

Link::~Link()
{
    while ( _links.first() ) {
	delete _links.first();
	_links.remove();
    }
}

QPopupMenu* Link::menu()
{
    if ( _links.count() == 0 ) {
        return 0;
    }

    QPopupMenu* m = new QPopupMenu;
    m->setFont( QFont( "helvetica", 12, QFont::Bold ) );

    QPopupMenu* s;
    Link* l;
    for ( l = _links.first(); l; l = _links.next() ) {
        if ( ( s = l->menu() ) ) {
            m->insertItem( l->name(), s );
        } else {
            m->insertItem( l->name(), l, SLOT( selected() ) );
        }
    }

    return m;
}

void Link::save( FILE* fptr, int indent )
{
    int i;
    
    if ( _links.count() == 0 ) {
        if ( _url.length() ) {
            for ( i = 0; i < indent; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "<ITEM>\n" );
            for ( i = 0; i < indent + 4; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "<LINK URL=\"%s\">\n", _url.data() );
            for ( i = 0; i < indent + 8; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "%s\n", _name.data() );
            for ( i = 0; i < indent + 4; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "</LINK>\n" );
            for ( i = 0; i < indent; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "</ITEM>\n" );
        } else {
            for ( i = 0; i < indent; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "<ITEM>%s\n", _name.data() );
            for ( i = 0; i < indent; i++ ) fprintf( fptr, " " );
            fprintf( fptr, "</ITEM>\n" );
        }
        return;
    }

    for ( i = 0; i < indent; i++ ) fprintf( fptr, " " );
    fprintf( fptr, "<ITEM>%s\n", _name.data() );
    
    Link* l;
    for ( l = _links.first(); l; l = _links.next() ) {
        l->save( fptr, indent + 4 );
    }

    for ( i = 0; i < indent; i++ ) fprintf( fptr, " " );
    fprintf( fptr, "</ITEM>\n" );
}

void Link::selected()
{
    _bookmarks->active()->stop();
    _bookmarks->active()->open( _url );
}

Bookmarks::Bookmarks()
        : _link( 0 ),
          _active( 0 )
{
    read();
    connect( options, SIGNAL( updateBookmarksUrl() ), this, SLOT( updateBookmarksUrl() ) );
}

Bookmarks::~Bookmarks()
{
    delete _link;
}

void Bookmarks::read()
{
    _link = new Link( this );
    _stack.push( _link );
    
    Request* request = new Request( options->bookmarksUrl() );

    _parser = new SgmlParser;
    
    connect( request, SIGNAL( data( const char*, int ) ), _parser, SLOT( data( const char*, int ) ) );
    connect( request, SIGNAL( endOfData() )             , _parser, SLOT( endOfData() ) );
    connect( _parser, SIGNAL( dtdSelected( QString ) )  , this   , SLOT( dtdSelected( QString ) ) );
    
    request->open();
}

QPopupMenu* Bookmarks::menu()
{
    QPopupMenu* m = 0;

    if ( _link ) {
        m = _link->menu();
    }
    
    if ( !m ) {
        m = new QPopupMenu;
        m->setFont( QFont( "helvetica", 12, QFont::Bold ) );
    }
    m->insertItem( "&Add Bookmark", 10000, 0 );
    m->insertSeparator( 1 );
    m->connectItem( 10000, this, SLOT( addBookmark() ) );
    
    return m;
}

void Bookmarks::activeBrowser( Browser* browser )
{
    _active = browser;
}

void Bookmarks::dtdSelected( QString dtdName )
{
    if ( dtdName == "-//SMV//DTD BOOKMARK 1.0//EN" ) {
        connect( _parser, SIGNAL( startTag() )        , this, SLOT( startTag10() ) );
        connect( _parser, SIGNAL( endTag() )          , this, SLOT( endTag10() ) );
        connect( _parser, SIGNAL( content( QString ) ), this, SLOT( content10( QString ) ) );
    } else if ( dtdName == "-//SMV//DTD BOOKMARK 1.1//EN" ) {
        connect( _parser, SIGNAL( startTag() )        , this, SLOT( startTag11() ) );
        connect( _parser, SIGNAL( endTag() )          , this, SLOT( endTag11() ) );
        connect( _parser, SIGNAL( content( QString ) ), this, SLOT( content11( QString ) ) );
    } else if ( dtdName == "-//W3C//DTD HTML 3.2//EN" ) {
        // Ignore because this is probably a File Not Found error message.
    } else {
        QMessageBox msgBox;
        msgBox.setCaption( "QWeb: Error" );
        QString error;
        error.sprintf( "Unrecognized bookmark DTD '%s'", dtdName.data() );
        msgBox.setText( error );
        msgBox.show();
    }
}

void Bookmarks::startTag10()
{
    if ( _text.stripWhiteSpace() != "" ) {
        _stack.top()->setName( _text.stripWhiteSpace() );
        _text = "";
    }
    
    if ( _parser->tagStack().getLast()->name == "LINK" ) {
        Link* l = new Link( this );
        const QString* url = 0;
        if ( ( url = _parser->tagStack().getLast()->find( "URL" ) ) ) {
            l->setUrl( *url );
        }
        _stack.push( l );
    }
}

void Bookmarks::startTag11()
{
    if ( _text.stripWhiteSpace() != "" ) {
        _stack.top()->setName( _text.stripWhiteSpace() );
        _text = "";
    }
    
    if ( _parser->tagStack().getLast()->name == "ITEM" ) {
        Link* l = new Link( this );
        _stack.push( l );
    } else if ( _parser->tagStack().getLast()->name == "LINK" ) {
        const QString* url = 0;
        if ( ( url = _parser->tagStack().getLast()->find( "URL" ) ) ) {
            _stack.top()->setUrl( *url );
        }
    }
}

void Bookmarks::endTag10()
{
    if ( _text.stripWhiteSpace() != "" ) {
        _stack.top()->setName( _text.stripWhiteSpace() );
        _text = "";
    }
    
    if ( _parser->tagStack().getLast()->name == "LINK" ) {
        Link* l = _stack.pop();
        _stack.top()->appendLink( l );
    } else if ( _parser->tagStack().getLast()->name == "BOOKMARK" ) {
        _stack.clear();
        emit bookmarksChanged();
    }
}

void Bookmarks::endTag11()
{
    if ( _text.stripWhiteSpace() != "" ) {
        _stack.top()->setName( _text.stripWhiteSpace() );
        _text = "";
    }
    
    if ( _parser->tagStack().getLast()->name == "ITEM" ) {
        Link* l = _stack.pop();
        _stack.top()->appendLink( l );
    } else if ( _parser->tagStack().getLast()->name == "BOOKMARK" ) {
        _stack.clear();
        emit bookmarksChanged();
    }
}

void Bookmarks::content10( QString text )
{
    _text += text;
}

void Bookmarks::content11( QString text )
{
    _text += text;
}

void Bookmarks::addBookmark()
{
    if ( _active->baseUrl().url().length() == 0 ) return;
    
    QString tmp = _active->caption();
    tmp.remove( 0, 6 );
    tmp = tmp.stripWhiteSpace();
    if ( tmp.length() == 0 ) {
        tmp = _active->baseUrl().url();
    }

    Link* l = new Link( this );
    l->setName( tmp );
    l->setUrl( _active->baseUrl().url() );
    _link->appendLink( l );

    save();
    
    emit bookmarksChanged();
}

void Bookmarks::save()
{
    // Open the bookmarks file for writing.
    Url url( options->bookmarksUrl() );
    if ( url.method() != "file" ) {
        QString error;
        error.sprintf( "Don't known how to write bookmarks with method '%s'.", url.method().data() );
        QMessageBox::message( "QWeb: Error", error );
        return;
    }

    // Open the file for writing.
    FILE* fptr = fopen( url.path().data(), "w" );
    if ( !fptr ) {
        QString error;
        error.sprintf( "Failed to open file '%s' for writing.", url.path().data() );
        QMessageBox::message( "QWeb: Error", error );
        return;
    }
    
    fprintf( fptr, "<!DOCTYPE bookmark PUBLIC \"-//SMV//DTD BOOKMARK 1.1//EN\">\n" );
    fprintf( fptr, "<BOOKMARK>\n" );

    Link* l;
    for ( l = _link->links().first(); l; l = _link->links().next() ) {
        l->save( fptr, 4 );
    }

    fprintf( fptr, "    <LASTMOD>\n" );
    fprintf( fptr, "        %s\n", QDateTime::currentDateTime().toString().data() );
    fprintf( fptr, "    </LASTMOD>\n" );

    fprintf( fptr, "</BOOKMARK>\n" );
    fclose( fptr );
}

void Bookmarks::updateBookmarksUrl()
{
    delete _link;
    _link = 0;
    emit bookmarksChanged();
    read();
}
