/*
 * crtc.c - A line-based CRTC emulation (under construction).
 *
 * Written by
 *   Ettore Perazzoli (ettore@comm2000.it)
 *   Andre' Fachat (fachat@physik.tu-chemnitz.de)
 *
 * 16/24bpp support added by
 *   Steven Tieu (stieu@physics.ubc.ca)
 *   Teemu Rantanen (tvr@cs.hut.fi)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef PET
#define PET
#endif

#define _CRTC_C

#include "vice.h"

#include <stdlib.h>
#include <stdio.h>

#include "crtc.h"
#include "raster.h"
#include "vmachine.h"
#include "interrupt.h"
#include "mem.h"

/* ------------------------------------------------------------------------- */

static void crtc_init_dwg_tables(void);
static void crtc_update_memory_ptrs(void);
static void crtc_arrange_window(int width, int height);
static int fill_cache(struct line_cache *l, int *xs, int *xe, int r);
static void draw_line(void);
static void draw_line_2x(void);
static void draw_line_cached(struct line_cache *l, int xs, int xe);
static void draw_line_cached_2x(struct line_cache *l, int xs, int xe);

static BYTE crtc[19];
static ADDRESS scraddr;
static BYTE *chargen_ptr;
static BYTE *screenmem;

PIXEL4 dwg_table_0[256], dwg_table_1[256];
PIXEL4 dwg_table2x_0[256], dwg_table2x_1[256];
PIXEL4 dwg_table2x_2[256], dwg_table2x_3[256];

#include "raster.c"

/* -------------------------------------------------------------------------- */

canvas_t crtc_init(void)
{
#ifdef __MSDOS__
    if (SCREEN_XPIX > 320) {
	app_resources.doubleSize = 1;
	app_resources.vgaMode = VGA_640x480;
    } else {
	app_resources.doubleSize = 0;
	app_resources.vgaMode = VGA_320x200;
    }
    app_resources.doubleScan = 1;
#endif
    
    init_raster(1, 2, 2);
    video_resize();
    if (open_output_window(CRTC_WINDOW_TITLE,
			   SCREEN_XPIX + 10, SCREEN_YPIX + 10,
			   color_defs,
			   (canvas_redraw_t) crtc_arrange_window)) {
	fprintf(stderr, "fatal error: can't open window for CRTC emulation.\n");
	return NULL;
    }
    video_mode = CRTC_STANDARD_MODE;
    refresh_all();
    chargen_ptr = chargen_rom;
    screenmem = ram;
    border_color = 0;
    background_color = 0;

    crtc_init_dwg_tables();
    display_ystart = CRTC_SCREEN_BORDERHEIGHT;
    display_ystop = CRTC_SCREEN_BORDERHEIGHT + CRTC_SCREEN_YPIX;
    return canvas;
}

static void crtc_init_dwg_tables(void)
{
    int byte, p;
    BYTE msk;

    for (byte = 0; byte < 0x0100; byte++) {
	*((PIXEL *) (dwg_table2x_0 + byte))
	    = *((PIXEL *) (dwg_table2x_0 + byte) + 1)
	    = PIXEL(byte & 0x80 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_0 + byte) + 2)
	    = *((PIXEL *) (dwg_table2x_0 + byte) + 3)
	    = PIXEL(byte & 0x40 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_1 + byte))
	    = *((PIXEL *) (dwg_table2x_1 + byte) + 1)
	    = PIXEL(byte & 0x20 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_1 + byte) + 2)
	    = *((PIXEL *) (dwg_table2x_1 + byte) + 3)
	    = PIXEL(byte & 0x10 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_2 + byte))
	    = *((PIXEL *) (dwg_table2x_2 + byte) + 1)
	    = PIXEL(byte & 0x08 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_2 + byte) + 2)
	    = *((PIXEL *) (dwg_table2x_2 + byte) + 3)
	    = PIXEL(byte & 0x04 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_3 + byte))
	    = *((PIXEL *) (dwg_table2x_3 + byte) + 1)
	    = PIXEL(byte & 0x02 ? 1 : 0);
	*((PIXEL *) (dwg_table2x_3 + byte) + 2)
	    = *((PIXEL *) (dwg_table2x_3 + byte) + 3)
	    = PIXEL(byte & 0x01 ? 1 : 0);
    }

    for (byte = 0; byte < 0x0100; byte++) {
	for (msk = 0x80, p = 0; p < 4; msk >>= 1, p++)
	    *((PIXEL *)(dwg_table_0 + byte) + p) = PIXEL(byte & msk ? 1 : 0);
	for (p = 0; p < 4; msk >>= 1, p++)
	    *((PIXEL *)(dwg_table_1 + byte) + p) = PIXEL(byte & msk ? 1 : 0);
    }
}

/* Set proper functions and constants for the current video settings. */
void video_resize(void)
{
    static int old_size = 0;

    if (app_resources.doubleSize) {
	pixel_height = 2;
	if (crtc_cols == 40) {
	    pixel_width = 2;
	    video_modes[CRTC_STANDARD_MODE].fill_cache = fill_cache;
	    video_modes[CRTC_STANDARD_MODE].draw_line_cached = draw_line_cached_2x;
	    video_modes[CRTC_STANDARD_MODE].draw_line = draw_line_2x;
	    if (old_size == 1) {
		window_width *= 2;
		window_height *= 2;
		if (canvas)
		    canvas_resize(canvas, window_width, window_height);
	    }
	} else {
	    /* When in 80 column mode, only the height is doubled. */
	    pixel_width = 1;
	    video_modes[CRTC_STANDARD_MODE].fill_cache = fill_cache;
	    video_modes[CRTC_STANDARD_MODE].draw_line_cached = draw_line_cached;
	    video_modes[CRTC_STANDARD_MODE].draw_line = draw_line;
	    if (old_size == 1)
		window_height *= 2;
	}
    } else {
	pixel_width = 1;
	pixel_height = 1;
	video_modes[CRTC_STANDARD_MODE].fill_cache = fill_cache;
	video_modes[CRTC_STANDARD_MODE].draw_line_cached = draw_line_cached;
	video_modes[CRTC_STANDARD_MODE].draw_line = draw_line;
	if (old_size == 2) {
	    if (crtc_cols == 40)
		window_width /= 2;
	    window_height /= 2;
	}
    }
    old_size = app_resources.doubleSize ? 2 : 1;

    if (canvas) {
	resize(window_width, window_height);
	frame_buffer_clear(&frame_buffer, pixel_table[0]);
	force_repaint();
	refresh_all();
    }
}

void video_free(void)
{
    frame_buffer_free(&frame_buffer);
}

static void crtc_arrange_window(int width, int height)
{
    resize(width, height);
    refresh_all();
}

/* -------------------------------------------------------------------------- */

/* CRTC interface functions. */

void REGPARM2 store_crtc(ADDRESS addr, BYTE value)
{
    crtc[addr] = value;

    if (app_resources.debugFlag)
	printf("store crtc[%x] %x\n", addr, (int) value);

    switch (addr) {
      case 0:			/* R00  Horizontal total (characters + 1) */
      case 1:			/* R01  Horizontal characters displayed */
	/* crtc_update_memory_ptrs (); FIXME: not implemented */
	break;

      case 2:			/* R02  Columns Displayed  (+1) */

      case 3:			/* R03  Horizontal/Vertical Sync widths */
	/* crtc_update_memory_ptrs (); FIXME: not implemented */
	break;

      case 4:			/* R04  Raster line count */
	break;

      case 5:			/* R05  Vertical Screen position */
	/* crtc_update_memory_ptrs (); FIXME: not implemented */
	break;

      case 6:			/* R06  Number of display lines on screen */
      case 7:			/* R07  Lines displayed */
	break;

      case 8:			/* R08  unused: Interlace and Skew */
	break;

      case 9:			/* R09  Rasters between two display lines */
	break;

      case 10:			/* R10  Cursor (not implemented on the PET) */
      case 11:			/* R11  Cursor (not implemented on the PET) */
	break;

      case 12:			/* R12  Control register */

	/* Bit 0: 1=add 256 to screen start address ( 512 for 80-columns)
	 * Bit 1: 1=add 512 to screen start address (1024 for 80-columns)
	 * Bit 2: invert flyback
	 * Bit 3: invert video signal
	 * Bit 4: use top half of 4K character generator
	 * Bit 5: (not implemented on the PET)
	 * Bit 6: (not implemented on the PET)
	 * Bit 7: not used.
	 */

	break;

      case 13:			/* R13  Address of first character */
	/* Value + 32786 (8000) */
	crtc_update_memory_ptrs();
	break;

      case 14:
      case 15:			/* R14-5 Cursor location HI/LO -- unused */
	break;

      case 16:
      case 17:			/* R16-7 Light Pen HI/LO -- read only */
	break;

      case 18:
      case 19:			/* R18-9 Update address HI/LO (only 6545)  */
	break;
    }
}

BYTE REGPARM1 read_crtc(ADDRESS addr)
{
    switch (addr) {
      case 14:
      case 15:			/* Cursor location HI/LO */
	return crtc[addr];

      case 16:
      case 17:			/* Light Pen X,Y */
	return 0xff;

      default:
	return 0;		/* All the rest are write-only registers */
    }
}

void store_colorram(ADDRESS addr, BYTE value)
{
    /* No color RAM. */
}

BYTE read_colorram(ADDRESS addr)
{
    /* Bogus. */
    return 0;
}

void crtc_set_char(int crom)
{
    chargen_ptr = chargen_rom + (crom ? 0x800 : 0);
}

void reset_crtc(void)
{
    /* spec says to initialize "all internal scan counter circuits.
     * When /RES is low, all internal counters stop and clear and all
     * scan and video output go low; control registers are unaffected.
     * All scan timing initiates when /RES goes high. "
     * "In this way, /RES can synchronize display frame timing with 
     * line frequency."
     *
     * Well, we just emulate...
     */

    /* store_crtc (5, 240); */

    /* This is for PET 3032 - to get it working without any crtc register
     * setting, as the older PETs don't know about them.
     */
    screenmem = ram + 0x8000;

    maincpu_set_alarm_clk(A_RASTERDRAW, CYCLES_PER_LINE);
    return;
}

int crtc_offscreen(void)
{
    return rasterline >= YPIX;
}

void crtc_set_screen_width(int num_cols)
{
    int w, h;

    crtc_cols = num_cols;
    w = crtc_cols * 8 * pixel_width + 2 * SCREEN_BORDERWIDTH;
    h = 25 * pixel_width + 2 * SCREEN_BORDERHEIGHT;

    display_xstart = SCREEN_BORDERWIDTH;
    display_xstop = SCREEN_BORDERWIDTH + SCREEN_XPIX;
    display_ystart = SCREEN_BORDERHEIGHT;
    display_ystop = SCREEN_BORDERHEIGHT + SCREEN_YPIX;

    if (canvas) {
	canvas_resize(canvas, w, h);
	crtc_arrange_window(w, h);
	video_resize();
    }
}

static void crtc_update_memory_ptrs(void)
{
    scraddr = 0x8000;

#ifdef DEBUG
    if (app_resources.debugFlag)
	printf("scraddr:%x\n", scraddr);
#endif

    screenmem = ram + scraddr;
}

/* -------------------------------------------------------------------------- */

int int_rasterdraw(long offset)
{
    maincpu_set_alarm(A_RASTERDRAW, CYCLES_PER_LINE - offset);
    emulate_line();

    /* This generates one raster interrupt per frame. */
    if (!rasterline && clk > 1000)
	strobe_irq(&maincpu_int_status);

    return 0;
}

static int fill_cache(struct line_cache *l, int *xs, int *xe, int r)
{
    return _fill_cache_text(l->fgdata, screenmem + memptr, chargen_ptr,
			    crtc_cols, ycounter, xs, xe, r);
}

static void draw_line(void)
{
    PIXEL *p = frame_buffer_ptr + SCREEN_BORDERWIDTH;
    register int i, d;

    for (i = 0; i < crtc_cols; i++, p += 8) {
	d = GET_CHAR_DATA(chargen_ptr, (screenmem + memptr)[i], ycounter);
	*((PIXEL4 *) p) = dwg_table_0[d];
	*((PIXEL4 *) p + 1) = dwg_table_1[d];
    }
}

static void draw_line_2x(void)
{
    PIXEL *p = (frame_buffer_ptr + SCREEN_BORDERWIDTH * pixel_width);
    register int i, d;

    for (i = 0; i < crtc_cols; i++, p += 16) {
	d = GET_CHAR_DATA(chargen_ptr, (screenmem + memptr)[i], ycounter);
	*((PIXEL4 *) p) = dwg_table2x_0[d];
	*((PIXEL4 *) p + 1) = dwg_table2x_1[d];
	*((PIXEL4 *) p + 2) = dwg_table2x_2[d];
	*((PIXEL4 *) p + 3) = dwg_table2x_3[d];
    }
}

static void draw_line_cached(struct line_cache *l, int xs, int xe)
{
    PIXEL *p = frame_buffer_ptr + SCREEN_BORDERWIDTH + xs * 8;
    register int i;

    for (i = xs; i <= xe; i++, p += 8) {
	*((PIXEL4 *) p) = dwg_table_0[l->fgdata[i]];
	*((PIXEL4 *) p + 1) = dwg_table_1[l->fgdata[i]];
    }
}

static void draw_line_cached_2x(struct line_cache *l, int xs, int xe)
{
    PIXEL *p = frame_buffer_ptr + 2 * (SCREEN_BORDERWIDTH + xs * 8);
    register int i;

    for (i = xs; i <= xe; i++, p += 16) {
	*((PIXEL4 *) p) = dwg_table2x_0[l->fgdata[i]];
	*((PIXEL4 *) p + 1) = dwg_table2x_1[l->fgdata[i]];
	*((PIXEL4 *) p + 2) = dwg_table2x_2[l->fgdata[i]];
	*((PIXEL4 *) p + 3) = dwg_table2x_3[l->fgdata[i]];
    }
}

/* -------------------------------------------------------------------------- */

void crtc_prevent_clk_overflow(void)
{
    oldclk -= PREVENT_CLK_OVERFLOW_SUB;
}
