/*
 * ui_xaw.c - Simple Xaw-based graphical user interface.  It uses widgets
 * from the Free Widget Foundation and Robert W. McMullen.
 *
 * Written by
 *  Ettore Perazzoli (ettore@comm2000.it)
 *  Andre' Fachat    (fachat@physik.tu-chemnitz.de)
 *
 * Support for multiple visuals and depths by
 *  Teemu Rantanen (tvr@cs.hut.fi)
 * Joystick options by
 *  Bernhard Kuhn  (kuhn@eikon.e-technik.tu-muenchen.de)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#define _UI_XAW_C

#include "vice.h"

#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>	/* XtResizeWidget() */
#include <X11/Shell.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/AsciiText.h>

#ifdef XPM
#include <X11/xpm.h>
#include "icon.h"
#endif

#include <X11/keysym.h>

#ifdef HAVE_X11_SUNKEYSYM_H
#include <X11/Sunkeysym.h>
#endif

#include "widgets/Canvas.h"
#include "widgets/FileSel.h"
#include "widgets/TextField.h"

#include "checkmark.xbm"
#include "right_arrow.xbm"

#include "ui_xaw.h"
#include "serial.h"
#include "interrupt.h"
#include "patchlevel.h"
#include "info.h"
#include "vsync.h"
#include "mem.h"
#include "sid.h"
#include "drive.h"
#include "tape.h"
#include "resources.h"
#include "mon.h"
#include "utils.h"

#ifdef HAS_JOYSTICK
#include "joystick.h"
#endif

#ifdef HAVE_TRUE1541
#include "true1541.h"
#endif

#ifdef REU
#include "reu.h"
#endif

extern void video_resize(void);

/* These are used by video_x.[ch]. */
Display *display;
int screen;
Visual *visual;
int depth = X_DISPLAY_DEPTH;

/* Application context. */
static XtAppContext app_context;

/* This is needed to catch the `Close' command from the Window Manager. */
static Atom WM_delete_window;

/* Useful images. */
static Pixmap CheckmarkBitmap, RightArrowBitmap;
#ifdef XPM
static Pixmap IconPixmap;
#endif

/* Toplevel widget. */
static Widget TopLevel = NULL;

/* XDebugger, Jarkko's X11-based 6502 debugger. */
static Widget XDebugger;

/* Our colormap. */
static Colormap colormap;

/* This keeps a list of the menus with a checkmark on the left.  Each time
   some setting is changed, we have to update them. */
#define MAX_UPDATE_MENU_LIST_SIZE 1024
static Widget CheckmarkMenuItems[MAX_UPDATE_MENU_LIST_SIZE];
static int NumCheckmarkMenuItems = 0;

/* This allows us to pop up the transient shells centered to the last visited
   shell. */
static Widget LastVisitedAppShell = NULL;
#define MAX_APP_SHELLS 256
static struct {
    Widget widget;
    String title;
} AppShells[MAX_APP_SHELLS];
static int NumAppShells = 0;

/* This is for our home-made menu implementation.  Not very clever, but it
   works. */
static int PoppedUpCnt = 0;
static int MenuPopup = 0;
#define MAX_SUBMENUS 256
static struct {
    Widget widget;
    Widget parent;
} SubMenus[MAX_SUBMENUS];
static int NumSubMenus = 0;

/* If != 0, we should save the settings. */
static int resources_have_changed = 0;

/* ------------------------------------------------------------------------- */

static int UiAllocColormap(void);
static int UiAllocColors(int num_colors, const UiColorDef colors[],
			 PIXEL pixel_return[]);
static void UiPopup(Widget w, const char *title, Boolean wait_popdown);
static void UiPopdown(Widget w);
static Widget UiBuildFileSelector(Widget parent, UiButton *button_return);
static Widget UiBuildErrorDialog(Widget parent, UiButton *button_return,
				 const String message);
static Widget UiBuildInputDialog(Widget parent, UiButton *button_return,
				 Widget *InputDialogLabel,
				 Widget *InputDialogField);
static Widget UiBuildShowText(Widget parent, UiButton *button_return,
			      const String text, int width, int height);
static Widget UiBuildConfirmDialog(Widget parent, UiButton *button_return,
				   Widget *ConfirmDialogMessage);
static Widget UiCreateTransientShell(Widget parent, const char *name);
static Widget UiBuildInfoDialog(Widget parent, UiButton *button_return,...);
static void UiPositionSubMenu(Widget w, Widget parent);
static void UiCloseAction(Widget w, XEvent *event, String *params,
			  Cardinal *num_params);
static void UiPositionSubMenuAction(Widget w, XEvent *event, String *params,
				    Cardinal *num_params);
static void UiPopdownSubMenusAction(Widget w, XEvent *event, String *params,
				   Cardinal *num_params);
static void UiMenuUnhighlightAction(Widget w, XEvent *event, String *params,
				    Cardinal *num_params);
static void UiHandleSpecialKeys(Widget w, XtPointer closure, XEvent *x_event,
				Boolean *continue_to_dispatch);
#define CallbackFunc(name)  static void name(Widget w, XtPointer client_data, \
					     XtPointer call_data)

CallbackFunc(UiEnterWindowCallback);
CallbackFunc(UiExposureCallback);
CallbackFunc(UiMenuPopupCallback);
CallbackFunc(UiMenuPopdownCallback);
CallbackFunc(UiSubMenuPopupCallback);
CallbackFunc(UiSubMenuPopdownCallback);
CallbackFunc(UiAttachDisk);
CallbackFunc(UiDetachDisk);
CallbackFunc(UiChangeWorkingDir);
CallbackFunc(UiActivateXDebugger);
CallbackFunc(UiActivateMonitor);
CallbackFunc(UiRunC1541);
CallbackFunc(UiReset);
CallbackFunc(UiPowerUpReset);
CallbackFunc(UiBrowseManual);
CallbackFunc(UiExit);
CallbackFunc(UiInfo);
CallbackFunc(UiSetRefreshRate);
CallbackFunc(UiSetCustomRefreshRate);
CallbackFunc(UiSetMaximumSpeed);
CallbackFunc(UiSetCustomMaximumSpeed);
CallbackFunc(UiInfoDialogNoWarrantyCallback);
CallbackFunc(UiInfoDialogContribCallback);
CallbackFunc(UiInfoDialogLicenseCallback);
CallbackFunc(UiCloseButtonCallback);
CallbackFunc(UiOkButtonCallback);
CallbackFunc(UiCancelButtonCallback);
CallbackFunc(UiYesButtonCallback);
CallbackFunc(UiNoButtonCallback);
CallbackFunc(UiResetButtonCallback);
CallbackFunc(UiMonButtonCallback);
CallbackFunc(UiDebugButtonCallback);
CallbackFunc(UiContentsButtonCallback);
CallbackFunc(UiToggleVideoCache);
CallbackFunc(UiToggleDoubleSize);
CallbackFunc(UiToggleDoubleScan);
CallbackFunc(UiToggleUseXSync);
CallbackFunc(UiTogglePause);
CallbackFunc(UiSaveResources);
CallbackFunc(UiLoadResources);
CallbackFunc(UiSetDefaultResources);
CallbackFunc(UiToggleSaveResourcesOnExit);

#if defined(CBM64) || defined(C128)
CallbackFunc(UiAttachTape);
CallbackFunc(UiDetachTape);
CallbackFunc(UiToggleSpriteToSpriteCollisions);
CallbackFunc(UiToggleSpriteToBackgroundCollisions);
#ifdef HAS_JOYSTICK
CallbackFunc(UiSetJoystickDevice1);
CallbackFunc(UiSetJoystickDevice2);
#else
CallbackFunc(UiSetNumpadJoystickPort);
#endif
CallbackFunc(UiSwapJoystickPorts);
CallbackFunc(UiToggleEmuID);
CallbackFunc(UiToggleIEEE488);
CallbackFunc(UiToggleREU);
#endif

#ifdef HAVE_TRUE1541
CallbackFunc(UiToggleTrue1541);
CallbackFunc(UiSet1541SyncFactor);
CallbackFunc(UiSet1541IdleMethod);
CallbackFunc(UiSetCustom1541SyncFactor);
#endif

#ifdef SOUND
CallbackFunc(UiToggleSound);
CallbackFunc(UiToggleSoundSpeedAdjustment);
CallbackFunc(UiSetSoundSampleRate);
CallbackFunc(UiSetSoundBufferSize);
CallbackFunc(UiSetSoundSuspendTime);
#endif

/* This one needs the menu functions to be prototyped first, and that's why we
   include it now. */
#include "menu.h"

/* This one needs the MenuEntry typedef from menu.h */
static Widget UiBuildPopupMenu(Widget parent, const char *name,
			       MenuEntry list[]);

/* ------------------------------------------------------------------------- */

static String fallback_resources[] = {
    "*font:					   -*-lucida-bold-r-*-*-12-*",
    "*Command.font:			           -*-lucida-bold-r-*-*-12-*",
    "*fileSelector.width:			     380",
    "*fileSelector.height:			     300",
    "*inputDialog.inputForm.borderWidth:	     0",
    "*inputDialog.inputForm.field.width:	     300",
    "*inputDialog.inputForm.field.scrollHorizontal:  True",
    "*inputDialog.inputForm.label.width:	     250",
    "*inputDialog.inputForm.label.borderWidth:	     0",
    "*inputDialog.inputForm.label.internalWidth:     0",
    "*inputDialog.buttonBox.borderWidth:	     0",
    "*errorDialog.messageForm.borderWidth:	     0",
    "*errorDialog.buttonBox.borderWidth:	     0",
    "*errorDialog.messageForm.label.borderWidth:     0",
    "*jamDialog.messageForm.borderWidth:	     0",
    "*jamDialog.buttonBox.borderWidth:		     0",
    "*jamDialog.messageForm.label.borderWidth:       0",
    "*infoDialogShell.width:			     380",
    "*infoDialogShell.height:			     290",
    "*infoDialog.textForm.infoString.borderWidth:    0",
    "*infoDialog.textForm.borderWidth:		     0",
    "*infoDialog.textForm.defaultDistance:	     0",
    "*infoDialog.buttonBox.borderWidth:		     0",
    "*infoDialog.buttonBox.internalWidth:	     5",
    "*infoDialog.textForm.infoString.internalHeight: 0",
    "*confirmDialogShell.width:			     300",
    "*confirmDialog.messageForm.message.borderWidth: 0",
    "*confirmDialog.messageForm.message.height:      20",
    "*showText.textBox.text.width:		     480",
    "*showText.textBox.text.height:		     305",
    "*showText.textBox.text*font:       -*-lucidatypewriter-medium-r-*-*-12-*",
    "*okButton.label:				     Confirm",
    "*cancelButton.label:			     Cancel",
    "*closeButton.label:			     Dismiss",
    "*yesButton.label:				     Yes",
    "*resetButton.label:			     Reset",
    "*monButton.label:			   	     Monitor",
    "*debugButton.label:		   	     Debug",
    "*noButton.label:				     No",
    "*licenseButton.label:			     License...",
    "*noWarrantyButton.label:			     No warranty!",
    "*contribButton.label:			     Contributors...",
    "*Text.translations:			     #override \\n"
    "                                                <Key>Return: no-op()\\n"
    "						     <Key>Linefeed: no-op()\\n"
    "						     Ctrl<Key>J: no-op() \\n",

    /* Default color settings (suggestions are welcome...) */
    "*foreground:				     black",
    "*background:				     gray80",
    "*borderColor:				     black",
    "*internalBorderColor:			     black",
    "*TransientShell*Dialog.background:		     gray80",
    "*TransientShell*Label.background:		     gray80",
    "*TransientShell*Box.background:		     gray80",
    "*fileSelector.background:			     gray80",
    "*Command.background:			     gray90",
    "*Menubutton.background:		             gray80",
    "*Scrollbar.background:		             gray80",
    "*Form.background:				     gray80",

    XDEBUG_FALLBACK_RESOURCES,
    NULL
};

/* ------------------------------------------------------------------------- */

/* Initialize the GUI and parse the command line. */
int UiInit(int *argc, char **argv)
{
    /* Create the toplevel. */
    TopLevel = XtAppInitialize(&app_context, "VICE", NULL, 0, argc, argv,
			       fallback_resources, NULL, 0);
    if (!TopLevel)
	return -1;
    display = XtDisplay(TopLevel);
    screen = XDefaultScreen(display);
    atexit(UiAutoRepeatOn);
    return 0;
}

typedef struct {
    char		*name;
    int			 class;
} namedvisual_t;

/* Continue GUI initialization after resources are set */
int UiInitFinish(void)
{
    static XtActionsRec actions[] = {
	{ "Close", UiCloseAction },
	{ "PositionSubMenu", UiPositionSubMenuAction },
	{ "PopdownSubMenus", UiPopdownSubMenusAction },
	{ "Unhighlight", UiMenuUnhighlightAction }
    };
    static namedvisual_t classes[] = {
	{ "PseudoColor", PseudoColor },
	{ "TrueColor", TrueColor },
	{ "StaticGray", StaticGray },
	{ NULL }
    };
    XVisualInfo visualinfo;

#if X_DISPLAY_DEPTH == 0
    depth = app_resources.displayDepth;
#endif

    if (depth != 0) {
	int i;
	
	for (i = 0; classes[i].name != NULL; i++) {
	    if (XMatchVisualInfo(display, screen, depth, classes[i].class,
				 &visualinfo))
		break;
	}
	if (!classes[i].name) {
	    fprintf(stderr,
		    "\nThis display does not support suitable %dbit visuals.\n"
#if X_DISPLAY_DEPTH == 0
		    "Please select a bit depth supported by your display.\n",
#else
		    "Please recompile the program for a supported bit depth.\n",
#endif
		    depth);
	    return -1;
	} else
	    printf("Found %dbit/%s visual.\n", depth, classes[i].name);
    } else {
	/* Autodetect. */
	int i, j, done;
	int depths[8];

	depths[0] = DefaultDepth(display, screen);
	depths[1] = 0;

	for (i = done = 0; depths[i] != 0 && !done; i++)
	    for (j = 0; classes[j].name != NULL; j++) {
		if (XMatchVisualInfo(display, screen, depths[i],
				     classes[j].class, &visualinfo)) {
		    depth = depths[i];
		    printf("Found %dbit/%s visual.\n",
			   depth, classes[j].name);
		    done = 1;
		    break;
		}
	    }
	if (!done) {
	    fprintf(stderr, "Couldn't autodetect a proper visual.\n");
	    return -1;
	} 
    }

    visual = visualinfo.visual;

    /* Allocate the colormap. */
    UiAllocColormap();

    /* Recreate TopLevel to support non-default display depths. */
    XtDestroyWidget(TopLevel);
    TopLevel = XtVaAppCreateShell(EMULATOR, "VICE",
				  applicationShellWidgetClass,
				  display, XtNvisual, visual,
				  XtNdepth, depth, XtNcolormap, colormap,
				  NULL);

    /* Create the `tick' bitmap. */
    CheckmarkBitmap = XCreateBitmapFromData(display,
					    DefaultRootWindow(display),
					    checkmark_bits, checkmark_width,
					    checkmark_height);
    /* Create the right arrow bitmap. */
    RightArrowBitmap = XCreateBitmapFromData(display,
					     DefaultRootWindow(display),
					     right_arrow_bits,
					     right_arrow_width,
					     right_arrow_height);

#ifdef XPM
    /* Create the icon pixmap. */
    XpmCreatePixmapFromData(display, DefaultRootWindow(display), icon_data,
			    &IconPixmap, NULL, NULL);
#endif

    WM_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);

    XtAppAddActions(app_context, actions, XtNumber(actions));

    return 0;
}

/* Create a shell with a canvas widget in it. */
UiWindow UiOpenCanvasWindow(const char *title, int width, int height,
			    int no_autorepeat, UiExposureHandler exposure_proc,
			    int num_colors, const UiColorDef colors[],
			    PIXEL pixel_return[])
{
    /* Note: this is correct because we never destroy CanvasWindows. */
    static int menus_created = 0;
    XtTranslations translations;
    Widget canvas, w;
    XSetWindowAttributes attr;

    /* Create the pop-up menus. */
    if (!menus_created) {
	UiBuildPopupMenu(TopLevel, "LeftMenu", LeftMenu);
	UiBuildPopupMenu(TopLevel, "RightMenu", RightMenu);
	UiUpdateMenus();
	menus_created = 1;
    }

    if (++NumAppShells > MAX_APP_SHELLS) {
	fprintf(stderr,
	  	"UiOpenCanvasWindow(): maximum number of open windows reached."
		"\n");
	return NULL;
    }
    w = XtVaCreatePopupShell("CanvasWindow", applicationShellWidgetClass,
			     TopLevel, XtNinput, True, XtNtitle, title,
			     XtNiconName, title, NULL);

#ifdef XPM
    XtVaSetValues(w, XtNiconPixmap, IconPixmap, NULL);
#endif

    canvas = XtVaCreateManagedWidget("Canvas", xfwfcanvasWidgetClass,
				     w, XtNwidth, width, XtNheight, height,
				     NULL);

    /* Assign proper translations to open the menus. */

    translations = XtParseTranslationTable
	("<Btn1Down>: XawPositionSimpleMenu(LeftMenu) MenuPopup(LeftMenu)\n"
	 "<Btn3Down>: XawPositionSimpleMenu(RightMenu) MenuPopup(RightMenu)\n");
    XtOverrideTranslations(canvas, translations);

    if (no_autorepeat) {
	XtAddEventHandler(canvas, EnterWindowMask, False,
			  (XtEventHandler) UiAutoRepeatOff, NULL);
	XtAddEventHandler(canvas, LeaveWindowMask, False,
			  (XtEventHandler) UiAutoRepeatOn, NULL);
	XtAddEventHandler(w, KeyPressMask, False,
			  (XtEventHandler) UiHandleSpecialKeys, NULL);
	XtAddEventHandler(canvas, KeyPressMask, False,
			  (XtEventHandler) UiHandleSpecialKeys, NULL);
    }
    XtAddEventHandler(w, EnterWindowMask, False,
		      (XtEventHandler) UiEnterWindowCallback, NULL);
    XtAddEventHandler(canvas, ExposureMask | StructureNotifyMask, False,
		      (XtEventHandler) UiExposureCallback, exposure_proc);

    XtRealizeWidget(w);
    XtPopup(w, XtGrabNone);

    attr.backing_store = Always;
    XChangeWindowAttributes(display, XtWindow(canvas), 
    	 		    CWBackingStore, &attr);

    XSetWMProtocols(display, XtWindow(w), &WM_delete_window, 1);
    XtOverrideTranslations(w, XtParseTranslationTable
			        ("<Message>WM_PROTOCOLS: Close()"));

    AppShells[NumAppShells - 1].widget = w;
    AppShells[NumAppShells - 1].title = XtMalloc(strlen(title) + 1);
    strcpy(AppShells[NumAppShells - 1].title, title);

    if (UiAllocColors(num_colors, colors, pixel_return) == -1)
	return NULL;

    return canvas;
}

/* Set the colormap variable. The user must tell us whether he wants the
   default one or not using the `privateColormap' resource. */
static int UiAllocColormap(void)
{
    if (colormap)
	return 0;
    
    if (!app_resources.privateColormap
	&& depth == DefaultDepth(display, screen)) {
	colormap = DefaultColormap(display, screen);
    } else {
	colormap = XCreateColormap(display, RootWindow(display, screen),
				   visual, AllocNone);
    }

    XtVaSetValues(TopLevel, XtNcolormap, colormap, NULL);
    return 0;
}

/* Allocate colors in the colormap. */
static int UiAllocColors(int num_colors, const UiColorDef color_defs[],
			 PIXEL pixel_return[])
{
    static int allocated_colors;
    int i, failed;
    XColor color;
    XImage *im;
    PIXEL *data = (PIXEL *)xmalloc(4);

    /* This is a kludge to map pixels to zimage values. Is there a better
       way to do this? //tvr */
    im = XCreateImage(display, visual, depth, ZPixmap, 0, (char *)data,
		      1, 1, 8, 0);
    if (!im)
	return -1;
    
    color.flags = DoRed | DoGreen | DoBlue;
    for (i = 0, failed = 0; i < num_colors; allocated_colors++, i++) {
	color.red = color_defs[i].red;
	color.green = color_defs[i].green;
	color.blue = color_defs[i].blue;
	if (!XAllocColor(display, colormap, &color)) {
	    failed = 1;
	    fprintf(stderr, 
		    "Warning: cannot allocate color \"#%04X%04X%04X\"\n",
		    color.red, color.green, color.blue);
	}
	XPutPixel(im, 0, 0, color.pixel);
#if X_DISPLAY_DEPTH == 0
	{
	    /* XXX: prototypes where? */
	    extern PIXEL  real_pixel1[];
	    extern PIXEL2 real_pixel2[];
	    extern PIXEL4 real_pixel4[];
	    extern long   real_pixel[];
	    extern BYTE   shade_table[];
	    pixel_return[i] = i;
	    if (depth == 8)
		pixel_return[i] = *data;
	    else if (im->bits_per_pixel == 8)
		real_pixel1[i] = *(PIXEL *)data;
	    else if (im->bits_per_pixel == 16)
		real_pixel2[i] = *(PIXEL2 *)data;
	    else if (im->bits_per_pixel == 32)
		real_pixel4[i] = *(PIXEL4 *)data;
	    else
		real_pixel[i] = color.pixel;
	    if (im->bits_per_pixel == 1)
		shade_table[i] = color_defs[i].dither;
	}
#else
	pixel_return[i] = *data;
#endif
    }

    XDestroyImage(im);

    if (failed)
	printf("\nHint: use the `-install' option to install a private colormap.\n");
    
    return 0;
}

/* Return the drawable for the canvas in the specified UiWindow. */
Window UiCanvasDrawable(UiWindow w)
{
    return XtWindow(w);
}

/* Show the speed index to the user.  For Xaw, this is simply done by putting
   a proper string on the title bar. */
void UiDisplaySpeed(float percent, float framerate)
{
    int i;
    char str[1024];
    int percent_int = (int)(percent + 0.5);
    int framerate_int = (int)(framerate + 0.5);

    for (i = 0; i < NumAppShells; i++) {
	if (!percent)
	    XtVaSetValues(AppShells[i].widget, XtNtitle, AppShells[i].title,
			  NULL);
	else {
	    sprintf(str, "%s at %d%% speed, %d fps",
		    AppShells[i].title, percent_int, framerate_int);
	    XtVaSetValues(AppShells[i].widget, XtNtitle, str, NULL);
	}
    }
}

/* Display a message indicating that the emulation is paused.  As with speed,
   this is displayed on the title bar. */
void UiDisplayPaused(void)
{
    int i;
    char str[1024];

    for (i = 0; i < NumAppShells; i++) {
	sprintf(str, "%s (paused)", AppShells[i].title);
	XtVaSetValues(AppShells[i].widget, XtNtitle, str, NULL);
    }
}

/* Dispatch all the pending Xt events. */
void UiDispatchEvents(void)
{
    XEvent report;

    while (XtAppPending(app_context) || MenuPopup) {
	XtAppNextEvent(app_context, &report);
	XtDispatchEvent(&report);
    }
}

/* Resize one window. */
void UiResizeCanvasWindow(UiWindow w, int width, int height)
{
    XtResizeWidget(XtParent(w), width, height, 0);
}

/* Map one window. */
void UiMapCanvasWindow(UiWindow w)
{
    XtPopup(w, XtGrabNone);
}

/* Unmap one window. */
void UiUnmapCanvasWindow(UiWindow w)
{
    XtPopdown(w);
}

/* Enable autorepeat. */
void UiAutoRepeatOn(void)
{
    XAutoRepeatOn(display);
    XFlush(display);
}

/* Disable autorepeat. */
void UiAutoRepeatOff(void)
{
    XAutoRepeatOff(display);
    XFlush(display);
}

/* ------------------------------------------------------------------------- */

/* Handle all special keys. */
static void UiHandleSpecialKeys(Widget w, XtPointer closure,
				XEvent *x_event, Boolean *continue_to_dispatch)
{
    static char buffer[20];
    KeySym key;
    XComposeStatus compose;

    XLookupString(&x_event->xkey, buffer, 20, &key, &compose);
    
    switch (key) {
      case XK_F10:
	UiAttachDisk(NULL, (XtPointer) 8, NULL);
	break;
#if defined(CBM64) || defined(C128)
      case XK_F11:
#ifdef HAVE_X11_SUNKEYSYM_H
      case SunXK_F36:		/* SUN does it its own way... */
#endif
	UiSwapJoystickPorts(NULL, NULL, NULL);
	break;
#endif
      case XK_F12:
#ifdef HAVE_X11_SUNKEYSYM_H
      case SunXK_F37:		/* SUN does it its own way... */
#endif
	UiPowerUpReset(NULL, NULL, NULL);
	break;
      default:
	break;
    }
}

/* ------------------------------------------------------------------------- */

/* Report an error to the user. */
void UiError(const char *format,...)
{
    char str[1024];
    va_list ap;
    static Widget ErrorDialog;
    static UiButton button;

    va_start(ap, format);
    vsprintf(str, format, ap);
    ErrorDialog = UiBuildErrorDialog(TopLevel, &button, str);
    UiPopup(XtParent(ErrorDialog), "VICE Error!", False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    UiPopdown(XtParent(ErrorDialog));
    XtDestroyWidget(XtParent(ErrorDialog));
    UiDispatchEvents();
    suspend_speed_eval();
}

/* Report a message to the user. */
void UiMessage(const char *format,...)
{
    char str[1024];
    va_list ap;
    static Widget ErrorDialog;
    static UiButton button;

    va_start(ap, format);
    vsprintf(str, format, ap);
    ErrorDialog = UiBuildErrorDialog(TopLevel, &button, str);
    UiPopup(XtParent(ErrorDialog), "VICE", False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    UiPopdown(XtParent(ErrorDialog));
    XtDestroyWidget(XtParent(ErrorDialog));
    UiDispatchEvents();
    suspend_speed_eval();
}

void activate_xdebug_window(void)
{
    if (!XDebugger) {
	XDebugger = XtVaCreatePopupShell("XDebugger",
					 applicationShellWidgetClass, TopLevel,
					 NULL);
	XtOverrideTranslations(XDebugger,
			       (XtParseTranslationTable
				("<Message>WM_PROTOCOLS: Close()")));
	xdebug_create(XDebugger);
    }
    XtPopup(XDebugger, XtGrabNone);
    XSetWMProtocols(display, XtWindow(XDebugger), &WM_delete_window, 1);
    xdebug_enable();
}

/* Report a message to the user, allow different buttons. */
int UiJamDialog(const char *format, ...)
{
    char str[1024];
    va_list ap;
    static Widget JamDialog, shell, tmp, mform, bbox;
    static UiButton button;

    va_start(ap, format);

    shell = UiCreateTransientShell(TopLevel, "jamDialogShell");
    JamDialog = XtVaCreateManagedWidget
	("jamDialog", panedWidgetClass, shell, NULL);
    mform = XtVaCreateManagedWidget
	("messageForm", formWidgetClass, JamDialog, NULL);

    vsprintf(str, format, ap);
    tmp = XtVaCreateManagedWidget
	("label", labelWidgetClass, mform,
	 XtNresize, False, XtNjustify, XtJustifyCenter, XtNlabel, str,
	 NULL);

    bbox = XtVaCreateManagedWidget
	("buttonBox", boxWidgetClass, JamDialog,
	 XtNshowGrip, False, XtNskipAdjust, XtNorientation,
	 XtorientHorizontal, NULL);

    tmp = XtVaCreateManagedWidget
	("resetButton", commandWidgetClass, bbox, NULL);
    XtAddCallback(tmp, XtNcallback, UiResetButtonCallback,
		  (XtPointer) &button);

    tmp = XtVaCreateManagedWidget
	("monButton", commandWidgetClass, bbox, NULL);
    XtAddCallback(tmp, XtNcallback, UiMonButtonCallback,
		  (XtPointer) &button);

    tmp = XtVaCreateManagedWidget
	("debugButton", commandWidgetClass, bbox, NULL);
    XtAddCallback(tmp, XtNcallback, UiDebugButtonCallback,
		  (XtPointer) &button);

    UiPopup(XtParent(JamDialog), "VICE", False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    UiPopdown(XtParent(JamDialog));
    XtDestroyWidget(XtParent(JamDialog));

    if (button == Button_Debug) {
	activate_xdebug_window();
	XSync(display, False);
	UiDispatchEvents();
    }
    
    suspend_speed_eval();
    UiDispatchEvents();

    if (button==Button_Mon)
	return 1;
    if (button==Button_Debug)
	return 2;
    return 0;
}

/* File browser. */
char *UiFileSelect(const char *title,
		   char *(*read_contents_func)(const char *))
{
    static char buf[4096], curdir[4096];
    XfwfFileSelectorStatusStruct fs_status;
    static UiButton button;
    Widget FileSelector;

    /* We always rebuild the file selector from scratch (which is slow),
       because there seems to be a bug in the XfwfScrolledList that causes
       the file and directory listing to disappear randomly.  I hope this
       fixes the problem... */
    FileSelector = UiBuildFileSelector(TopLevel, &button);
    if (getcwd(curdir, 4096))
	XfwfFileSelectorChangeDirectory((XfwfFileSelectorWidget) FileSelector,
					curdir);
    UiPopup(XtParent(FileSelector), title, False);
    do {
	button = Button_None;
	while (button == Button_None)
	    UiDispatchEvents();
	XfwfFileSelectorGetStatus((XfwfFileSelectorWidget)FileSelector,
				  &fs_status);
	if (button == Button_Contents && fs_status.file_selected
	    && read_contents_func != NULL) {
	    char *contents;
		
	    sprintf(buf, "%s%s", fs_status.path, fs_status.file);
	    contents = read_contents_func(buf);
	    if (contents != NULL) {
		UiShowText(fs_status.file, contents, 250, 240);
		free(contents);
	    } else {
		UiError("Unknown image type.");
	    }
	}
    } while ((!fs_status.file_selected && button != Button_Cancel)
	     || button == Button_Contents);
    if (fs_status.file_selected) {
	strncpy(buf, fs_status.path, 2048);
	strcat(buf, fs_status.file);
    } else
	*buf = '\0';
    UiPopdown(XtParent(FileSelector));

    XtDestroyWidget(XtParent(FileSelector));
    return button == Button_Ok ? buf : NULL;
}

/* Ask for a string.  The user can confirm or cancel. */
UiButton UiInputString(const char *title, const char *prompt, char *buf,
		       unsigned int buflen)
{
    String str;
    static Widget InputDialog, InputDialogLabel, InputDialogField;
    static UiButton button;

    if (!InputDialog)
	InputDialog = UiBuildInputDialog(TopLevel, &button, &InputDialogLabel,
					 &InputDialogField);
    XtVaSetValues(InputDialogLabel, XtNlabel, prompt, NULL);
    XtVaSetValues(InputDialogField, XtNstring, buf, NULL);
    XtSetKeyboardFocus(InputDialog, InputDialogField);
    UiPopup(XtParent(InputDialog), title, False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    XtVaGetValues(InputDialogField, XtNstring, &str, NULL);
    strncpy(buf, str, buflen);
    UiPopdown(XtParent(InputDialog));
    return button;
}

/* Display a text to the user. */
void UiShowText(const char *title, const char *text, int width, int height)
{
    static UiButton button;
    Widget ShowText;

    ShowText = UiBuildShowText(TopLevel, &button, (String)text, width, height);
    UiPopup(XtParent(ShowText), title, False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    UiPopdown(XtParent(ShowText));
    XtDestroyWidget(XtParent(ShowText));
}

/* Ask for a confirmation. */
UiButton UiAskConfirmation(const char *title, const char *text)
{
    static Widget ConfirmDialog, ConfirmDialogMessage;
    static UiButton button;

    if (!ConfirmDialog)
	ConfirmDialog = UiBuildConfirmDialog(TopLevel, &button,
					     &ConfirmDialogMessage);
    XtVaSetValues(ConfirmDialogMessage, XtNlabel, text, NULL);
    UiPopup(XtParent(ConfirmDialog), title, False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    UiPopdown(XtParent(ConfirmDialog));
    return button;
}

/* Update the menu items with a checkmark according to the current resource
   values. */
void UiUpdateMenus(void)
{
    int i;

    for (i = 0; i < NumCheckmarkMenuItems; i++)
	XtCallCallbacks(CheckmarkMenuItems[i], XtNcallback, (XtPointer) ! NULL);
}

/* ------------------------------------------------------------------------- */

/* Pop up a popup shell and center it to the last visited AppShell */
static void UiPopup(Widget w, const char *title, Boolean wait_popdown)
{
    Widget s = NULL;

    /* Keep sure that we really know which was the last visited shell. */
    UiDispatchEvents();

    if (LastVisitedAppShell)
	s = LastVisitedAppShell;
    else {
	/* Choose one realized shell. */
	int i;
	for (i = 0; i < NumAppShells; i++)
	    if (XtIsRealized(AppShells[i].widget)) {
		s = AppShells[i].widget;
		break;
	    }
    }

    if (s) {
	/* Center the popup. */
	Dimension my_width, my_height, shell_width, shell_height;
	Dimension my_x, my_y;
	Position tlx, tly;

	XtRealizeWidget(w);
	XtVaGetValues(w, XtNwidth, &my_width, XtNheight, &my_height, NULL);
	XtVaGetValues(s, XtNwidth, &shell_width, XtNheight, &shell_height,
		      XtNx, &tlx, XtNy, &tly, NULL);
	XtTranslateCoords(XtParent(s), tlx, tly, &tlx, &tly);
	my_x = tlx + (shell_width - my_width) / 2;
	my_y = tly + (shell_height - my_height) / 2;
	XtVaSetValues(w, XtNx, my_x, XtNy, my_y, NULL);
    }
    XtVaSetValues(w, XtNtitle, title, NULL);
    XtPopup(w, XtGrabExclusive);
    XSetWMProtocols(display, XtWindow(w), &WM_delete_window, 1);

    /* If requested, wait for this widget to be popped down before
       returning. */
    if (wait_popdown) {
	int oldcnt = PoppedUpCnt++;
	while (oldcnt != PoppedUpCnt)
	    UiDispatchEvents();
    } else
	PoppedUpCnt++;
}

/* Pop down a popup shell. */
static void UiPopdown(Widget w)
{
    XtPopdown(w);
    if (--PoppedUpCnt < 0)
	PoppedUpCnt = 0;
}

/* ------------------------------------------------------------------------- */

/* These functions build all the widgets. */

static Widget UiBuildFileSelector(Widget parent, UiButton * button_return)
{
    Widget shell = UiCreateTransientShell(parent, "fileSelectorShell");
    Widget FileSelector = XtVaCreateManagedWidget("fileSelector",
						  xfwfFileSelectorWidgetClass,
						  shell,
						  XtNflagLinks, True, NULL);
    XtAddCallback((Widget) FileSelector, XtNokButtonCallback,
		  UiOkButtonCallback, (XtPointer) button_return);
    XtAddCallback((Widget) FileSelector,
		  XtNcancelButtonCallback, UiCancelButtonCallback,
		  (XtPointer) button_return);
    XtAddCallback((Widget) FileSelector,
		  XtNcontentsButtonCallback, UiContentsButtonCallback,
		  (XtPointer) button_return);
    return FileSelector;
}

static Widget UiBuildErrorDialog(Widget parent, UiButton * button_return,
				 const String message)
{
    Widget shell, ErrorDialog, tmp;

    shell = UiCreateTransientShell(parent, "errorDialogShell");
    ErrorDialog = XtVaCreateManagedWidget
	("errorDialog", panedWidgetClass, shell, NULL);
    tmp = XtVaCreateManagedWidget
	("messageForm", formWidgetClass, ErrorDialog, NULL);
    tmp = XtVaCreateManagedWidget
	("label", labelWidgetClass, tmp,
	 XtNresize, False, XtNjustify, XtJustifyCenter, XtNlabel, message,
	 NULL);
    tmp = XtVaCreateManagedWidget
	("buttonBox", boxWidgetClass, ErrorDialog,
	 XtNshowGrip, False, XtNskipAdjust, XtNorientation,
	 XtorientHorizontal, NULL);
    tmp = XtVaCreateManagedWidget
	("closeButton", commandWidgetClass, tmp, NULL);
    XtAddCallback(tmp, XtNcallback, UiCloseButtonCallback,
		  (XtPointer) button_return);
    return ErrorDialog;
}

static Widget UiBuildInputDialog(Widget parent, UiButton * button_return,
				 Widget * InputDialogLabel,
				 Widget * InputDialogField)
{
    Widget shell, InputDialog, tmp1, tmp2;

    shell = UiCreateTransientShell(parent, "inputDialogShell");
    InputDialog = XtVaCreateManagedWidget
	("inputDialog", panedWidgetClass, shell, NULL);
    tmp1 = XtVaCreateManagedWidget
	("inputForm", formWidgetClass, InputDialog, NULL);
    *InputDialogLabel = XtVaCreateManagedWidget
	("label", labelWidgetClass, tmp1, XtNresize, False, XtNjustify,
	 XtJustifyLeft, NULL);
    *InputDialogField = XtVaCreateManagedWidget
	("field", textfieldWidgetClass, tmp1, XtNfromVert, *InputDialogLabel,
	 NULL);
    XtAddCallback(*InputDialogField, XtNactivateCallback, UiOkButtonCallback,
		  (XtPointer) button_return);
    tmp1 = XtVaCreateManagedWidget
	("buttonBox", boxWidgetClass, InputDialog,
	 XtNshowGrip, False, XtNskipAdjust, True,
	 XtNorientation, XtorientHorizontal, NULL);
    tmp2 = XtVaCreateManagedWidget
	("okButton", commandWidgetClass, tmp1, NULL);
    XtAddCallback(tmp2,
		  XtNcallback, UiOkButtonCallback, (XtPointer) button_return);
    tmp2 = XtVaCreateManagedWidget
	("cancelButton", commandWidgetClass, tmp1, XtNfromHoriz, tmp2, NULL);
    XtAddCallback(tmp2,
		  XtNcallback, UiCancelButtonCallback, (XtPointer) button_return);
    return InputDialog;
}

static Widget UiBuildShowText(Widget parent, UiButton * button_return,
			      const String text, int width, int height)
{
    Widget shell, tmp;
    Widget ShowText;

    shell = UiCreateTransientShell(parent, "showTextShell");
    ShowText = XtVaCreateManagedWidget
	("showText", panedWidgetClass, shell, NULL);
    tmp = XtVaCreateManagedWidget
	("textBox", formWidgetClass, ShowText, NULL);
    tmp = XtVaCreateManagedWidget
	("text", asciiTextWidgetClass, tmp,
	 XtNtype, XawAsciiString, XtNeditType, XawtextRead,
	 XtNscrollVertical, XawtextScrollWhenNeeded, XtNdisplayCaret, False,
	 XtNstring, text, NULL);
    if (width > 0)
	XtVaSetValues(tmp, XtNwidth, (Dimension)width, NULL);
    if (height > 0)
	XtVaSetValues(tmp, XtNheight, (Dimension)height, NULL);
    tmp = XtVaCreateManagedWidget
	("buttonBox", boxWidgetClass, ShowText,
	 XtNshowGrip, False, XtNskipAdjust, True,
	 XtNorientation, XtorientHorizontal, NULL);
    tmp = XtVaCreateManagedWidget("closeButton", commandWidgetClass, tmp, NULL);
    XtAddCallback(tmp,
		  XtNcallback, UiCloseButtonCallback, (XtPointer) button_return);
    return ShowText;
}

static Widget UiBuildConfirmDialog(Widget parent, UiButton *button_return,
				   Widget *ConfirmDialogMessage)
{
    Widget shell, ConfirmDialog, tmp1, tmp2;

    shell = UiCreateTransientShell(parent, "confirmDialogShell");
    ConfirmDialog = XtVaCreateManagedWidget
	("confirmDialog", panedWidgetClass, shell, NULL);
    tmp1 = XtVaCreateManagedWidget("messageForm", formWidgetClass,
				   ConfirmDialog, NULL);
    *ConfirmDialogMessage = XtVaCreateManagedWidget
	("message", labelWidgetClass, tmp1, XtNresize, False,
	 XtNjustify, XtJustifyCenter, NULL);
    tmp1 = XtVaCreateManagedWidget
	("buttonBox", boxWidgetClass, ConfirmDialog,
	 XtNshowGrip, False, XtNskipAdjust, True,
	 XtNorientation, XtorientHorizontal, NULL);
    tmp2 = XtVaCreateManagedWidget
	("yesButton", commandWidgetClass, tmp1, NULL);
    XtAddCallback(tmp2,
		  XtNcallback, UiYesButtonCallback, (XtPointer) button_return);
    tmp2 = XtVaCreateManagedWidget
	("noButton", commandWidgetClass, tmp1, NULL);
    XtAddCallback(tmp2,
		  XtNcallback, UiNoButtonCallback, (XtPointer) button_return);
    tmp2 = XtVaCreateManagedWidget
	("cancelButton", commandWidgetClass, tmp1, NULL);
    XtAddCallback(tmp2,
		  XtNcallback, UiCancelButtonCallback, (XtPointer) button_return);
    return ConfirmDialog;
}

static Widget UiCreateTransientShell(Widget parent, const char *name)
{
    Widget w;

    w = XtVaCreatePopupShell
	(name, transientShellWidgetClass, parent, XtNinput, True, NULL);
    XtOverrideTranslations(w, XtParseTranslationTable
			   ("<Message>WM_PROTOCOLS: Close()"));
    return w;
}

static Widget UiBuildPopupMenu(Widget parent, const char *name,
			       MenuEntry list[])
{
    Widget w;
    unsigned int i, j;
    static int level = 0;

    level++;
    w = XtCreatePopupShell(name, simpleMenuWidgetClass, parent, NULL, 0);
    XtAddCallback(w, XtNpopupCallback, UiMenuPopupCallback, NULL);
    XtAddCallback(w, XtNpopdownCallback, UiMenuPopdownCallback, NULL);
    if (level == 1)
	XtOverrideTranslations(w, XtParseTranslationTable
			       ("<BtnMotion>: highlight() PositionSubMenu()\n"
				"<LeaveWindow>: Unhighlight()\n"
 				"<BtnUp>: PopdownSubMenus() MenuPopdown() " 
 				"         notify() unhighlight()")); 
    for (i = j = 0; list[i].string; i++) {
	Widget new_item;
	char name[256];
	
	sprintf(name, "MenuItem%d", j);	/* ugly... */
	switch (*list[i].string) {
	  case '-':		/* line */
	    new_item = XtCreateManagedWidget("separator", smeLineObjectClass,
					     w, NULL, 0);
	    break;
	  case '*':		/* toggle */
	    new_item = XtVaCreateManagedWidget(name, smeBSBObjectClass, w,
					       XtNrightMargin, 20,
					       XtNleftMargin, 20,
					       XtNlabel, list[i].string + 1,
					       NULL);
	    /* Add this item to the list of calls to perform to update the
	       menu status. */
	    if (list[i].callback) {
		if (NumCheckmarkMenuItems < MAX_UPDATE_MENU_LIST_SIZE)
		    CheckmarkMenuItems[NumCheckmarkMenuItems++] = new_item;
		else {
		    fprintf(stderr,
			    "Maximum number of menus reached! "
			    "Please fix the code.\n");
		    exit(-1);
		}
	    }
	    j++;
	    break;
	  default:
	    new_item = XtVaCreateManagedWidget(name, smeBSBObjectClass, w,
					       XtNleftMargin, 20,
					       XtNrightMargin, 20,
					       XtNlabel, list[i].string,
					       NULL);
	    j++;
	}
	if (list[i].callback)
	    XtAddCallback(new_item, XtNcallback,
			  (XtCallbackProc) list[i].callback,
			  list[i].callback_data);
	if (list[i].sub_menu) {
	    Widget sub;
	    XtVaSetValues(new_item, XtNrightBitmap, RightArrowBitmap, NULL);
	    sub = UiBuildPopupMenu(parent, "SUB", list[i].sub_menu);
	    SubMenus[NumSubMenus].widget = sub;
	    SubMenus[NumSubMenus].parent = new_item;
	    if (++NumSubMenus > MAX_SUBMENUS) {
		fprintf(stderr, "Maximum number of sub menus reached! "
			"Please fix the code.\n");
		exit(-1);
	    }
	    XtAddCallback(sub, XtNpopupCallback, UiSubMenuPopupCallback, NULL);
	    XtAddCallback(sub, XtNpopdownCallback, UiSubMenuPopdownCallback,
			  (XtPointer) w);
	}
    }
    level--;
    return w;
}

static Widget UiBuildInfoDialog(Widget parent, UiButton * button_return,...)
{
    Widget shell, pane, info_form, button_form, tmp, prevlabel = NULL;
    va_list arglist;
    String str;

    shell = UiCreateTransientShell(parent, "infoDialogShell");
    pane = XtVaCreateManagedWidget
	("infoDialog", panedWidgetClass, shell, NULL);
    info_form = XtVaCreateManagedWidget
	("textForm", formWidgetClass, pane, NULL);
    button_form = XtVaCreateManagedWidget
	("buttonBox", boxWidgetClass, pane, XtNshowGrip, False,
	 XtNskipAdjust, True, XtNorientation, XtorientHorizontal, NULL);
    va_start(arglist, button_return);
    while ((str = va_arg(arglist, String))) {
	tmp = XtVaCreateManagedWidget
	    ("infoString", labelWidgetClass, info_form,
	     XtNlabel, str, XtNjustify, XtJustifyCenter, XtNresize, False,
	     XtNwidth, 220, NULL);
	if (prevlabel)
	    XtVaSetValues(tmp, XtNfromVert, prevlabel, NULL);
	prevlabel = tmp;
    }
    tmp = XtVaCreateManagedWidget
	("closeButton", commandWidgetClass, button_form, NULL);
    XtAddCallback(tmp, XtNcallback,
		  UiCloseButtonCallback, (XtPointer)button_return);
    tmp = XtVaCreateManagedWidget
	("licenseButton", commandWidgetClass, button_form,
	 XtNfromHoriz, tmp, NULL);
    XtAddCallback(tmp, XtNcallback, UiInfoDialogLicenseCallback, NULL);
    tmp = XtVaCreateManagedWidget
	("noWarrantyButton", commandWidgetClass, button_form,
	 XtNfromHoriz, tmp, NULL);
    XtAddCallback(tmp, XtNcallback, UiInfoDialogNoWarrantyCallback, NULL);
    tmp = XtVaCreateManagedWidget
	("contribButton", commandWidgetClass, button_form,
	 XtNfromHoriz, tmp, NULL);
    XtAddCallback(tmp, XtNcallback, UiInfoDialogContribCallback, NULL);
    return pane;
}

CallbackFunc(UiInfoDialogLicenseCallback)
{
    UiShowText("VICE is FREE software!", license_text, -1, -1);
}

CallbackFunc(UiInfoDialogNoWarrantyCallback)
{
    UiShowText("No warranty!", warranty_text, -1, -1);
}

CallbackFunc(UiInfoDialogContribCallback)
{
    UiShowText("Contributors to the VICE project", contrib_text, -1, -1);
}

CallbackFunc(UiOkButtonCallback)
{
    *((UiButton *)client_data) = Button_Ok;
}

CallbackFunc(UiCancelButtonCallback)
{
    *((UiButton *)client_data) = Button_Cancel;
}

CallbackFunc(UiYesButtonCallback)
{
    *((UiButton *)client_data) = Button_Yes;
}

CallbackFunc(UiNoButtonCallback)
{
    *((UiButton *)client_data) = Button_No;
}

CallbackFunc(UiCloseButtonCallback)
{
    *((UiButton *)client_data) = Button_Close;
}

CallbackFunc(UiMonButtonCallback)
{
    *((UiButton *)client_data) = Button_Mon;
}

CallbackFunc(UiDebugButtonCallback)
{
    *((UiButton *)client_data) = Button_Debug;
}

CallbackFunc(UiResetButtonCallback)
{
    *((UiButton *)client_data) = Button_Reset;
}

CallbackFunc(UiContentsButtonCallback)
{
    *((UiButton *)client_data) = Button_Contents;
}

CallbackFunc(UiEnterWindowCallback)
{
    LastVisitedAppShell = w;
}

CallbackFunc(UiExposureCallback)
{
    Dimension width, height;

    suspend_speed_eval();
    XtVaGetValues(w, XtNwidth, (XtPointer) & width,
		  XtNheight, (XtPointer) & height, NULL);
    ((UiExposureHandler) client_data)((unsigned int)width,
				      (unsigned int)height);
}

CallbackFunc(UiMenuPopupCallback)
{
    MenuPopup++;
    suspend_speed_eval();
}

CallbackFunc(UiMenuPopdownCallback)
{
    MenuPopup--;
    if (MenuPopup < 0) {
	MenuPopup = 0;
    }
}

/* FIXME: this does not handle multiple application shells. */
static void UiCloseAction(Widget w, XEvent * event, String * params,
			  Cardinal * num_params)
{
    suspend_speed_eval();
    if (w == XDebugger) {
	xdebug_disable();
	XtPopdown(XDebugger);
    } else {
	int i;
	for (i = 0; i < NumAppShells; i++)
	    if (AppShells[i].widget == w)
		UiExit(w, NULL, NULL);
	LastVisitedAppShell = NULL;
    }
}

/* ------------------------------------------------------------------------- */

/* Here is a quick & dirty implementation of cascade submenus.  This is not
   good Xt style, but it is surely faster to do than writing a new menu widget
   supporting submenus.  Any patches, suggestions, or even a new better
   implementation are welcome at any time.  */

static Widget UiFindSubMenu(Widget w, Widget *parent_return)
{
    int i;
    Widget active_entry;

    active_entry = XawSimpleMenuGetActiveEntry(w);
    for (i = 0; i < NumSubMenus; i++)
	if (SubMenus[i].parent == active_entry) {
	    if (parent_return)
		*parent_return = active_entry;
	    return SubMenus[i].widget;
	}
    *parent_return = NULL;
    return NULL;
}

static Widget old_active_submenu, old_active_entry, old_widget;
static int SubMenuPoppedUp = 0;

static void UiPositionSubMenuAction(Widget w, XEvent * event, String * params,
				    Cardinal * num_params)
{
    Widget active_submenu, active_entry;

    active_submenu = UiFindSubMenu(w, &active_entry);
    if (old_active_submenu && old_active_submenu != active_submenu)
	XtPopdown(old_active_submenu);
    if (active_submenu)
	UiPositionSubMenu(active_submenu, active_entry);
    old_active_submenu = active_submenu;
    old_active_entry = active_entry;
    old_widget = w;
}

static void UiPopdownSubMenusAction(Widget w, XEvent * event, String * params,
				    Cardinal * num_params)
{
    int i;

    for (i = 0; i < NumSubMenus; i++) {
	/* XtCallActionProc(SubMenus[i].parent, "unhighlight", event, params,
           *num_params); */
	XtPopdown(SubMenus[i].widget);
    }
    MenuPopup = 0;
}

static void UiMenuUnhighlightAction(Widget w, XEvent * event, String * params,
				    Cardinal * num_params)
{
    if (1 || !SubMenuPoppedUp)
	XtCallActionProc(w, "unhighlight", event, params, *num_params);
}

static void UiPositionSubMenu(Widget w, Widget parent)
{
    Position parent_x, parent_y, my_x, my_y;
    Dimension parent_width, my_width, my_height;
    int root_width, root_height, foo;
    Window foowin;

    XtVaGetValues(parent, XtNx, &parent_x, XtNy, &parent_y,
		  XtNwidth, &parent_width, NULL);
    XtVaGetValues(w, XtNwidth, &my_width, XtNheight, &my_height, NULL);
    XtTranslateCoords(XtParent(parent), parent_x, parent_y,
		      &parent_x, &parent_y);
    my_x = parent_x + parent_width - 2;
    my_y = parent_y + 1;
    XGetGeometry(display, RootWindow(display, screen), &foowin, &foo,
		 &foo, &root_width, &root_height, &foo, &foo);
    if ((my_x + my_width) > root_width)
	my_x -= my_width + parent_width - 2;
    if ((my_y + my_height) > root_height)
	my_y = root_height - my_height;
    XtVaSetValues(w, XtNx, my_x, XtNy, my_y, NULL);
    XtPopup(w, XtGrabNonexclusive);
}

CallbackFunc(UiSubMenuPopupCallback)
{
    SubMenuPoppedUp++;
}

CallbackFunc(UiSubMenuPopdownCallback)
{
    SubMenuPoppedUp--;
    if (XawSimpleMenuGetActiveEntry(w))
	XtPopdown((Widget)client_data);
}

/* ------------------------------------------------------------------------- */

/* These functions simply perform the actions requested in the menus. */

CallbackFunc(UiAttachDisk)
{
    int unit = (int)client_data;
    char *filename;
    char title[1024];

    suspend_speed_eval();
    sprintf(title, "Attach Disk Image as unit #%d", unit);
    if ((filename = UiFileSelect(title, read_disk_image_contents)))
	if (serial_select_file(DT_DISK | DT_1541, unit, filename) < 0)
	    UiError("Invalid Disk Image");
}

CallbackFunc(UiDetachDisk)
{
    int unit = (int)client_data;

    suspend_speed_eval();
    if (unit < 0) {
	remove_serial(8);
	remove_serial(9);
	remove_serial(10);
    } else {
	remove_serial(unit);
    }
}

CallbackFunc(UiAttachTape)
{
    char *filename;

    suspend_speed_eval();
    if ((filename = UiFileSelect("Attach a tape image",
				 read_tape_image_contents)))
	if (serial_select_file(DT_TAPE, 1, filename) < 0)
	    UiError("Invalid Tape Image");
}

CallbackFunc(UiDetachTape)
{
    remove_serial(1);
}

CallbackFunc(UiChangeWorkingDir)
{
    static char buf[2048];

    suspend_speed_eval();
    if (UiInputString("VICE setting", "Change current working directory",
		      buf, 2047) != Button_Ok)
	return;
    else if (chdir(buf) < 0)
	UiError("Directory not found");
}

CallbackFunc(UiActivateXDebugger)
{
    suspend_speed_eval();
    activate_xdebug_window();
}

CallbackFunc(UiActivateMonitor)
{
    suspend_speed_eval();
    UiDispatchEvents();		/* popdown the menu */
    UiAutoRepeatOn();
    maincpu_trigger_trap(mon);
}

CallbackFunc(UiRunC1541)
{
    suspend_speed_eval();
    switch (system("xterm -sb -e c1541 &")) {
      case 127:
	  UiError("Couldn't run /bin/sh???");
	  break;
      case -1:
	  UiError("Couldn't run xterm");
	  break;
      case 0:
	  break;
      default:
	  UiError("Unknown error while running c1541");
    }
}

CallbackFunc(UiReset)
{
    suspend_speed_eval();
    maincpu_trigger_reset();
}

CallbackFunc(UiPowerUpReset)
{
    suspend_speed_eval();
    mem_powerup();
    maincpu_trigger_reset();
}

CallbackFunc(UiBrowseManual)
{
    if (app_resources.htmlBrowserCommand == NULL
	|| *app_resources.htmlBrowserCommand == '\0') {
	UiError("No HTML browser is defined.");
    } else {
#define BROWSE_CMD_BUF_MAX 16384
	char buf[BROWSE_CMD_BUF_MAX];
	static const char manual_path[] = LIBDIR "/" DOCDIR "/MANUAL.html";
	char *res_ptr;
	int manual_path_len, cmd_len;

	cmd_len = strlen(app_resources.htmlBrowserCommand);
	manual_path_len = strlen(manual_path);

	res_ptr = strstr(app_resources.htmlBrowserCommand, "%s");
	if (res_ptr == NULL) {
	    /* No substitution. */
	    if (cmd_len + 2 > BROWSE_CMD_BUF_MAX - 1) {
		UiError("Browser command too long.");
		return;
	    }
	    sprintf(buf, "%s &", app_resources.htmlBrowserCommand);
	} else {
	    char *tmp_ptr, *cmd_ptr;
	    int offs;
	    
	    /* Replace each occurrence of "%s" with the path of the HTML
               manual. */

	    cmd_len += manual_path_len - 2;
	    cmd_len += 2;	/* Trailing " &". */
	    if (cmd_len > BROWSE_CMD_BUF_MAX - 1) {
		UiError("Browser command too long.");
		return;
	    }

	    offs = res_ptr - app_resources.htmlBrowserCommand;
	    memcpy(buf, app_resources.htmlBrowserCommand, offs);
	    strcpy(buf + offs, manual_path);
	    cmd_ptr = buf + offs + manual_path_len;
	    res_ptr += 2;
	    
	    while ((tmp_ptr = strstr(res_ptr, "%s")) != NULL) {
		cmd_len += manual_path_len - 2;
		if (cmd_len > BROWSE_CMD_BUF_MAX - 1) {
		    UiError("Browser command too long.");
		    return;
		}
		offs = tmp_ptr - res_ptr;
		memcpy(cmd_ptr, res_ptr, offs);
		strcpy(cmd_ptr + offs, manual_path);
		cmd_ptr += manual_path_len + offs;
		res_ptr = tmp_ptr + 2;
	    }

	    sprintf(cmd_ptr, "%s &", res_ptr);
	}
	
	printf("Executing `%s'...\n", buf);
	if (system(buf) != 0)
	    UiError("Cannot run HTML browser.");
    }
}

CallbackFunc(UiExit)
{
    UiButton b;
	    
    b = UiAskConfirmation("Exit " EMULATOR " emulator",
			  "Do you really want to exit?");

    if (b == Button_Yes) {
	if (app_resources.saveResourcesOnExit && resources_have_changed) {
	    b = UiAskConfirmation("Exit " EMULATOR " emulator",
				  "Save the current settings?");
	    if (b == Button_Yes)
		UiSaveResources(NULL, NULL, NULL);
	    else if (b == Button_Cancel)
		return;
	}
	UiAutoRepeatOn();
	exit(-1);
    }
}

CallbackFunc(UiInfo)
{
    static Widget InfoDialog;
    static UiButton button;

    if (!InfoDialog) {
	InfoDialog = UiBuildInfoDialog
	    (TopLevel, &button,
	     "", "V I C E", "the Versatile Commodore Emulator", "",
	     "Version " VERSION, "", "Copyright (c) 1993-1997",
	     "E. Perazzoli, T. Rantanen, A. Fachat,",
	     "J. Valta, D. Sladic and J. Sonninen", "",
	     "Official VICE homepage:",
	     "http://www.tu-chemnitz.de/~fachat/vice/vice.html", "", NULL);
    }
    suspend_speed_eval();
    UiPopup(XtParent(InfoDialog), "VICE Information", False);
    button = Button_None;
    do
	UiDispatchEvents();
    while (button == Button_None);
    UiPopdown(XtParent(InfoDialog));
}

CallbackFunc(UiSetRefreshRate)
{
    if (!call_data) {
	if (app_resources.refreshRate != (int)client_data) {
	    app_resources.refreshRate = (int)client_data;
	    resources_have_changed = 1;
	    UiUpdateMenus();
	}
    } else {
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.refreshRate == (int)client_data ?
		      CheckmarkBitmap : 0, NULL);
	if (client_data == 0 && app_resources.speed == 0) {
	    /* Cannot enable the `automatic' setting if a speed limit is not
	       specified. */
	    XtVaSetValues(w, XtNsensitive, False, NULL);
	} else {
	    XtVaSetValues(w, XtNsensitive, True, NULL);
	}
    }
}

CallbackFunc(UiSetCustomRefreshRate)
{
    static char input_string[32];
    char msg_string[256];
    UiButton button;
    int i, found;
    MenuEntry *m = &SetRefreshRateSubMenu[0];

    if (!*input_string)
	sprintf(input_string, "%d", app_resources.refreshRate);

    if (call_data) {
	for (found = i = 0; m[i].callback == UiSetRefreshRate; i++) {
	    if (app_resources.refreshRate == (int)m[i].callback_data)
		found++;
	}
	XtVaSetValues(w, XtNleftBitmap, found ? 0 : CheckmarkBitmap, NULL);
    } else {
	suspend_speed_eval();
	sprintf(msg_string, "Enter refresh rate");
	button = UiInputString("Refresh rate", msg_string,
			       input_string, 32);
	if (button == Button_Ok) {
	    i = atoi(input_string);
	    if (!(app_resources.speed <= 0 && i <= 0) && i >= 0
		&& app_resources.refreshRate != i) {
		resources_have_changed = 1;
		app_resources.refreshRate = i;
		UiUpdateMenus();
	    }
	}
    }
}

CallbackFunc(UiSetMaximumSpeed)
{
    if (!call_data) {
	if (app_resources.speed != (int)client_data) {
	    app_resources.speed = (int)client_data;
	    resources_have_changed = 1;
	    UiUpdateMenus();
	}
    } else {
	XtVaSetValues(w, XtNleftBitmap, app_resources.speed == (int) client_data
		      ? CheckmarkBitmap : 0, NULL);
	if (client_data == 0 && app_resources.refreshRate == 0) {
	    /* Cannot enable the `no limit' setting if the refresh rate is set
	       to `automatic'. */
	    XtVaSetValues(w, XtNsensitive, False, NULL);
	} else {
	    XtVaSetValues(w, XtNsensitive, True, NULL);
	}
    }
}

CallbackFunc(UiSetCustomMaximumSpeed)
{
    static char input_string[32];
    char msg_string[256];
    UiButton button;
    int i, found;
    MenuEntry *m = &SetMaximumSpeedSubMenu[0];

    if (!*input_string)
	sprintf(input_string, "%d", app_resources.speed);

    if (call_data) {
	for (found = i = 0; m[i].callback == UiSetMaximumSpeed; i++) {
	    if (app_resources.speed == (int)m[i].callback_data)
		found++;
	}
	XtVaSetValues(w, XtNleftBitmap, found ? 0 : CheckmarkBitmap, NULL);
    } else {
	suspend_speed_eval();
	sprintf(msg_string, "Enter speed");
	button = UiInputString("Maximum run speed", msg_string, input_string,
			       32);
	if (button == Button_Ok) {
	    i = atoi(input_string);
	    if (!(app_resources.refreshRate <= 0 && i <= 0) && i >= 0
		&& app_resources.speed != i) {
		resources_have_changed = 1;
		app_resources.speed = i;
		UiUpdateMenus();
	    } else
		UiError("Invalid speed value");
	}
    }
}

CallbackFunc(UiSaveResources)
{
    suspend_speed_eval();
    if (resources_save(NULL, EMULATOR) < 0)
	UiError("Cannot save settings.");
    else {
	if (w != NULL)
	    UiMessage("Settings successfully saved.");
	resources_have_changed = 0;
    }
    UiUpdateMenus();
}

CallbackFunc(UiLoadResources)
{
    suspend_speed_eval();
    if (resources_load(NULL, EMULATOR) < 0)
	UiError("Cannot load settings.");
#if 0
    else if (w != NULL) 
            UiMessage("Settings loaded.");
#endif
    UiUpdateMenus();
}

CallbackFunc(UiSetDefaultResources)
{
    suspend_speed_eval();
    resources_set_defaults();
    resources_have_changed = 1;
    UiUpdateMenus();
}

/* ------------------------------------------------------------------------- */

/* These are the callbacks for the toggle menus (the ones with a checkmark on
   the left).  If call_data is NULL, they simply set/unset the checkmark
   according to the value of the corrisponding resource.  If not NULL, they
   set the value of the corresponding resource before doing so.  For this
   reason, to update the checkmarks, we simply have to call all the callbacks
   with a NULL call_data parameter. */

#define DEFINE_TOGGLE(f, resource, update_func)				\
    CallbackFunc(f)							\
    {									\
	if (!call_data) {						\
            void (*ff)() = update_func;					\
	    app_resources.resource = !app_resources.resource;	        \
	    resources_have_changed = 1;					\
	    if (ff)							\
		ff();							\
	    UiUpdateMenus();						\
	} else {							\
	    XtVaSetValues(w, XtNleftBitmap,				\
			  app_resources.resource ? CheckmarkBitmap	\
			                         : 0, NULL);		\
	}								\
    }

DEFINE_TOGGLE(UiToggleVideoCache, videoCache, video_resize)
    
DEFINE_TOGGLE(UiToggleDoubleSize, doubleSize, video_resize)
    
DEFINE_TOGGLE(UiToggleDoubleScan, doubleScan, video_resize)
    
DEFINE_TOGGLE(UiToggleUseXSync, useXSync, NULL)
    
DEFINE_TOGGLE(UiToggleSaveResourcesOnExit, saveResourcesOnExit, NULL)
    
CallbackFunc(UiTogglePause)
{
    static int paused;

    if (paused) {
	if (call_data == NULL)
	    paused = 0;
    } else {			/* !paused */
	if (call_data == NULL) {
	    paused = 1;
	    XtVaSetValues(w, XtNleftBitmap, CheckmarkBitmap, NULL);
	    UiDisplayPaused();
	    suspend_speed_eval();
	    while (paused)
		UiDispatchEvents();
	}
    }
    XtVaSetValues(w, XtNleftBitmap, paused ? CheckmarkBitmap : 0, NULL);
    UiDisplaySpeed(0.0, 0.0);
}

/* ------------------------------------------------------------------------- */

/* C64/128 specific menu items. */

#if defined(CBM64) || defined(C128)

DEFINE_TOGGLE(UiToggleSpriteToSpriteCollisions, checkSsColl, NULL)
DEFINE_TOGGLE(UiToggleSpriteToBackgroundCollisions, checkSbColl, NULL)

#ifdef HAS_JOYSTICK

CallbackFunc(UiSetJoystickDevice1)
{
    suspend_speed_eval();
    if (!call_data) {
	app_resources.joyDevice1 = (int) client_data;
	UiUpdateMenus();
    } else
	XtVaSetValues(w, XtNleftBitmap,
		      (app_resources.joyDevice1 == (int) client_data
		       ? CheckmarkBitmap : 0), NULL);
    joyport1select(app_resources.joyDevice1);
}

CallbackFunc(UiSetJoystickDevice2)
{
    suspend_speed_eval();
    if (!call_data) {
	app_resources.joyDevice2 = (int) client_data;
	UiUpdateMenus();
    } else
	XtVaSetValues(w, XtNleftBitmap,
		      (app_resources.joyDevice2 == (int) client_data
		       ? CheckmarkBitmap : 0), NULL);
    joyport2select(app_resources.joyDevice2);
}

CallbackFunc(UiSwapJoystickPorts)
{
    int tmp;

    if (w != NULL)
	suspend_speed_eval();
    tmp = app_resources.joyDevice1;
    app_resources.joyDevice1 = app_resources.joyDevice2;
    app_resources.joyDevice2 = tmp;
    resources_have_changed = 1;
    UiUpdateMenus();
}

#else  /* !HAS_JOYSTICK */

CallbackFunc(UiSetNumpadJoystickPort)
{
    suspend_speed_eval();
    if (!call_data) {
	if (app_resources.joyPort != (int)client_data) {
	    app_resources.joyPort = (int)client_data;
	    resources_have_changed = 1;
	    UiUpdateMenus();
	}
    } else
	XtVaSetValues(w, XtNleftBitmap,
		      (app_resources.joyPort == (int) client_data
		       ? CheckmarkBitmap : 0), NULL);
}

CallbackFunc(UiSwapJoystickPorts)
{
    suspend_speed_eval();
    app_resources.joyPort = 3 - app_resources.joyPort;
    printf("Numpad joystick now in port #%d.\n", app_resources.joyPort);
    UiUpdateMenus();
}

#endif

DEFINE_TOGGLE(UiToggleEmuID, emuID, NULL)

CallbackFunc(UiToggleIEEE488)
{
    if (!call_data) {
        app_resources.ieee488 = !app_resources.ieee488;
	/* The REU and the IEEE488 interface share the same address space, so
	   they cannot be enabled at the same time. */
	if (app_resources.ieee488)
	    app_resources.reu = 0;
	resources_have_changed = 1;
	UiUpdateMenus();
    } else {
	XtVaSetValues(w, XtNleftBitmap, 
		      app_resources.ieee488 ? CheckmarkBitmap : 0, NULL);
    }
}

CallbackFunc(UiToggleREU)
{
    if (!call_data) {
        app_resources.reu = !app_resources.reu;
	/* The REU and the IEEE488 interface share the same address space, so
	   they cannot be enabled at the same time. */
	if (app_resources.reu) {
	    app_resources.ieee488 = 0;
	    activate_reu();
	}
	resources_have_changed = 1;
	UiUpdateMenus();
    } else {
	XtVaSetValues(w, XtNleftBitmap, 
		      app_resources.reu ? CheckmarkBitmap : 0, NULL);
    }
}

#endif /* CBM64 || C128 */

/* ------------------------------------------------------------------------- */

/* True 1541 support items. */

#ifdef HAVE_TRUE1541

DEFINE_TOGGLE(UiToggleTrue1541, true1541, true1541_ack_switch)

CallbackFunc(UiSetCustom1541SyncFactor)
{
    static char input_string[256];
    char msg_string[256];
    UiButton button;

    if (!*input_string)
	sprintf(input_string, "%d", app_resources.true1541SyncFactor);

    if (call_data) {
	if (app_resources.true1541SyncFactor != TRUE1541_PAL_SYNC_FACTOR
	    && app_resources.true1541SyncFactor != TRUE1541_NTSC_SYNC_FACTOR)
	    XtVaSetValues(w, XtNleftBitmap, CheckmarkBitmap, NULL);
	else
	    XtVaSetValues(w, XtNleftBitmap, 0, NULL);
    } else {
	suspend_speed_eval();
	sprintf(msg_string, "Enter factor (PAL %d, NTSC %d)",
		TRUE1541_PAL_SYNC_FACTOR, TRUE1541_NTSC_SYNC_FACTOR);
	button = UiInputString("1541 Sync Factor", msg_string, input_string,
			       256);
	if (button == Button_Ok) {
	    int v;

	    v = atoi(input_string);
	    if (v != app_resources.true1541SyncFactor) {
		true1541_set_sync_factor(atoi(input_string));
		resources_have_changed = 1;
		UiUpdateMenus();
	    }
	}
    }
}

CallbackFunc(UiSet1541SyncFactor)
{
    if (!call_data) {
	if (app_resources.true1541SyncFactor != (int)client_data) {
	    true1541_set_sync_factor((int) client_data);
	    UiUpdateMenus();
	    resources_have_changed = 1;
	}
    } else {
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.true1541SyncFactor == (int) client_data ?
		      CheckmarkBitmap : 0, NULL);
    }
}

CallbackFunc(UiSet1541IdleMethod)
{
    if (!call_data) {
	/* XXX: not sure this is needed */
	if (app_resources.true1541)
	    true1541_cpu_execute();
	app_resources.true1541IdleMethod = (int)client_data;
	resources_have_changed = 1;
	UiUpdateMenus();
    } else {
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.true1541IdleMethod == (int) client_data ?
		      CheckmarkBitmap : 0, NULL);
    }
}

#endif /* HAVE_TRUE1541 */

/* ------------------------------------------------------------------------- */

/* Sound support. */

#ifdef SOUND
CallbackFunc(UiToggleSound)
{
    suspend_speed_eval();
    if (!call_data) {
	app_resources.sound = !app_resources.sound;
	if (app_resources.sound) {
	    /* Disable the vsync timer, or SIGALRM might happen while writing
	       to the sound device. */
	    vsync_disable_timer();
	} else
	    close_sound();
	resources_have_changed = 1;
	UiUpdateMenus();
    } else
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.sound ? CheckmarkBitmap : 0, NULL);
}

CallbackFunc(UiToggleSoundSpeedAdjustment)
{
    if (!call_data) {
	app_resources.soundSpeedAdjustment =
	    !app_resources.soundSpeedAdjustment;
	resources_have_changed = 1;
	UiUpdateMenus();
    } else
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.soundSpeedAdjustment ? CheckmarkBitmap : 0,
		      NULL);
}

CallbackFunc(UiSetSoundSampleRate)
{
    suspend_speed_eval();
    if (!call_data) {
	app_resources.soundSampleRate = (int) client_data;
	close_sound();
	resources_have_changed = 1;
	UiUpdateMenus();
    } else {
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.soundSampleRate == (int) client_data ?
		      CheckmarkBitmap : 0, NULL);
    }
}

CallbackFunc(UiSetSoundBufferSize)
{
    suspend_speed_eval();
    if (!call_data) {
	app_resources.soundBufferSize = (int) client_data;
	close_sound();
	resources_have_changed = 1;
	UiUpdateMenus();
    } else {
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.soundBufferSize == (int) client_data ?
		      CheckmarkBitmap : 0, NULL);
    }
}

CallbackFunc(UiSetSoundSuspendTime)
{
    suspend_speed_eval();
    if (!call_data) {
	app_resources.soundSuspendTime = (int) client_data;
	resources_have_changed = 1;
	UiUpdateMenus();
    } else {
	XtVaSetValues(w, XtNleftBitmap,
		      app_resources.soundSuspendTime == (int) client_data ?
		      CheckmarkBitmap : 0, NULL);
    }
}

#endif /* SOUND */
