/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <forms.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <varargs.h>
#include <time.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <dirent.h>

#include "common.h"
#include "xisp.h"
#include "pcode.h"
#include "network.h"
#include "rcio.h"
#include "xisp.xpm"
#include "version.h"
#include "help.c"

#ifdef XPMANIMATE
#include "online1.xpm"
#include "online2.xpm"
#include "online3.xpm"
#include "online4.xpm"
#include "online5.xpm"
#include "online6.xpm"
#include "online7.xpm"
#include "online8.xpm"
#include "online9.xpm"
#include "online10.xpm"
#include "online11.xpm"
#include "online12.xpm"
#endif

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* All application objects ("forms") are declared global */

FD_topFrame		*fd_topFrame;		/* The main application form */
FD_accountInfo	*fd_accountInfo;	/* Account information */
FD_dialInfo		*fd_dialInfo;		/* Dialing options */
FD_CBInfo		*fd_CBInfo;			/* Call-back script */
FD_aboutInfo	*fd_aboutInfo;		/* Application "About" */
FD_exitDialog	*fd_exitDialog;		/* Exit dialog */
FD_ISPInfo		*fd_ISPInfo;		/* ISP selection */
FD_renameISP	*fd_renameISP;		/* Modify ISP description */
FD_helpInfo		*fd_helpInfo;		/* General GUI information */
FD_commInfo		*fd_commInfo;		/* Communication options */
FD_tcpipInfo	*fd_tcpipInfo;		/* TCP/IP options */

/* The options from .xisprc go in here; defaults are held in [0] and
   all further option records are saved from [1] and unwards */

xisprc_t xispOptions[MAXNUM_ISP+1] =	/* table of options */
		 { XISPRC_DEFAULT };
int currentRC = 1;						/* sequence number of, and */
xisprc_t *p_xisprc = &xispOptions[1];	/* pointer to current option block */

/* Files */

char *rcfname;		/* file holding program parameters */
char *pipefname;	/* named pipe used for reading dialer output */
char *envfname;		/* temp file for passing the dialing environment */
char *papfname;		/* temp file with PAP username and password for +ua */
#ifdef RUNDIR
char *runfname;		/* the /var/run (or equivalent) ppp?.pid file name */
char *runfp;		/* pointer to the first filename character in runfname */
#endif

/* Global variables */

int dialerFD;		/* file descriptor used for reading dialer output */
int dialerON = 0;	/* indicates whether or not a dialer process is active */
int ipUDON = 0;		/* set when either ip-up or ip-down are active*/
char ipWhich[8] =
	{0};			/* the active script name; "ip-up" or "ip-down" */
int pppdPID = 0;	/* saved process id of initiated pppd process */
int ipUDPID = 0;	/* saved process id of the ip-up or ip-down process */
int pppdPPID = 0;	/* the parent PID of the above, after pppd forks */
int pppdStat = 0;	/* set to 1 when pppd process status has been collected */
int dialerPID = 0;	/* process id returned by dialer via named pipe */
int	connected = 0;	/* flag indicating connection status */
time_t connTime;	/* ammount of time connected */
char connSpeed[8];	/* formatted string of connection speed */
int linkOK = 0;		/* flag indicating link status */
int linkPoll = 0;	/* starts link polling after dialing is done */
char connIP[16];	/* string IP address of PPP interface when up */
int userDiscon = 0;	/* indicates user initiated disconnection request */
#ifdef XPMANIMATE
int frame = 0;		/* frame number of xpm animation */
unsigned netpkts=0;	/* sum of net packets */
int transfer = 0;	/* indicates data is coming over the modem line */
int minimized = 0;	/* flag indicating master form mapstate */
Pixmap pixmap;		/* animation pixmaps */
Pixmap online[12];
#endif

/* Timers for periodic events */

FL_OBJECT *btimer;		/* triggers browser updates with dialer output */
FL_OBJECT *ctimer;		/* measures connection time */
FL_OBJECT *ltimer;		/* used for polling link status */
#ifdef XPMANIMATE
FL_OBJECT *xpmtimer;	/* times XPM icon animation */
#endif


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |           Command line options & Resources - Data structures            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

FL_CMD_OPT options[] = 
{
	{"-bgcol", ".bgcol", XrmoptionSepArg, BGCOL_STRING},
	{"-iconic", ".iconic", XrmoptionNoArg, "True"},
	{"-geometry", ".geometry", XrmoptionSepArg, ""}
};

char bgcols[16+1] = {0};				/* the background color string */
color_t bgcol = BGCOL_DEFAULT;			/* default background for all forms */
int iconic = 0,							/* window placement flag */
	winPosX,							/* hint to WM for position */
	winPosY,
	winWidth,							/* request for different size */
	winHeight;
char geoms[32+1] = {0};					/* the geometry string */
int placementMethod = FL_PLACE_FREE;	/* top form placement method */

FL_resource resources[] =
{
	{"bgcol", "BGColor", FL_STRING, bgcols, BGCOL_STRING, 16},
	{"iconic", "Iconic", FL_BOOL, &iconic, "False"},
	{"geometry", "Geometry", FL_STRING, geoms, "", 32}
};

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                            Utility routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

	if (errno < sys_nerr)
		sprintf(emsg, "xISP: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xISP: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	exit(1);
}


/* Returns pointer to string with current time in 12-hour format */

char *strtime(void)
{
	struct timeval tv;
	time_t time;
	static char tout[16];

	gettimeofday(&tv, NULL);
	time = tv.tv_sec;
	strftime(tout, 16, "%I:%M:%S %p", localtime(&time));
	return tout;
}


/* Prints out-of-memory diagnostic and aborts program */

void outofMem(void)
{
	fprintf(stderr, "xISP: out of memory!\n");
	exit(1);
}


/* Waits for desired number of miliseconds using usleep(3). Thus, we avoid
   using sleep(3), which causes problems when mixed with calls to alarm(3). */

void mSleep(unsigned int msec)
{
	unsigned long usec;

	usec = msec * 1000;
	usleep(usec);
}


/* Assembles the variables holding the xisprc file name, the filename of
   the named pipe used for communications with the dialer process, as
   well as the name of the dialing environment file . */

void initFnames(void)
{
	struct passwd *user;

	user = getpwuid(getuid());
	rcfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(RCFNAME)+1);
	pipefname = (char *)malloc(6+strlen(user->pw_name)+strlen(PIPEFNAME)+1);
	envfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(ENVFNAME)+1);
	papfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(PAPFNAME)+1);
#ifdef RUNDIR
	runfname = (char *)malloc(strlen(RUNDIR)+8+4+1);
#endif
	if (rcfname != NULL && pipefname != NULL &&
#ifdef RUNDIR
		runfname != NULL &&
#endif
		envfname != NULL && papfname != NULL) {
		strcpy(rcfname, user->pw_dir); strcat(rcfname, "/");
		strcat(rcfname, RCFNAME);
		strcpy(pipefname, "/tmp/"); strcat(pipefname, PIPEFNAME);
		strcat(pipefname, "."); strcat(pipefname, user->pw_name);
		strcpy(envfname, user->pw_dir); strcat(envfname, "/");
		strcat(envfname, ENVFNAME);
		strcpy(papfname, user->pw_dir); strcat(papfname, "/");
		strcat(papfname, PAPFNAME);
#ifdef RUNDIR
		strcpy(runfname, RUNDIR);
		strcat(runfname, "/");
		runfp = &runfname[strlen(runfname)];
#endif
	}
	else outofMem();
}


/* Create and/or open a named pipe for reading only */

int namedPipe(char *fname)
{
	struct stat st;
	int create, fd;

	if (access(fname, F_OK) == -1)				/* check to see if it exists */
		create = 1;								/* nope, creation required */
	else {
		stat(fname, &st);						/* yes, get the node status */
		if (!S_ISFIFO(st.st_mode)) {			/* is it a FIFO? */
			unlink(fname);						/* nope, delete it */
			create = 1;							/* indicate creation required */
		}
		else create = 0;						/* it's a FIFO, all OK */
	}
	if (create)									/* was creation requested? */
		if (mknod(pipefname, S_IFIFO+0600, 0))	/* yes create the FIFO node */
			doErr("namedPipe: mknod");
	fd = open(fname, O_RDONLY|O_NDELAY);		/* and open it for reading */
	if (fd < 0)									/* bail out on error */
		doErr("namedPipe: open");
	return fd;									/* return the descriptor */
}


/* Search a string for multiple characters returning first occurence */

static char *strfstr(char *haystack, char *needles)
{
	char cn, *hp;

	while ((cn=*needles)) {						/* search for all needles */
		hp = strchr(haystack, cn);				/* found in haystack? */
		if (hp != NULL)							/* yes, return pointer to */
			return(hp);							/* location of matched char */
		++needles;								/* nope, get next needle */
	}
	return(NULL);								/* nothing found */
}


/* Write printf style in the browser object. Note the implicit
   total length of MSGLEN_BROWSER bytes for the resulting string.
   A routine to flush the buffer used by bprintf is also provided. */

#define MSGLEN_BROWSER (MAXBUF_CHILD*2)

char bmsg[MSGLEN_BROWSER+1] = {0};	/* the string buffer used by bprintf() */
char btmp[MAXBUF_CHILD+1] = {0};	/* temporary buffer for new strings */
char *where = bmsg;					/* incomplete line continuation pointer */
int needNL = 0;						/* and the continuation indicator */

void bDoBuf(void)
{
	char *nl;

	while ((nl= strfstr(bmsg,"\r\n"))!=NULL) {	/* string contains CR or LF? */
		*nl = 0;								/* yes, mark it as last char */
		if (needNL) {							/* redisplay string if the */
			fl_replace_browser_line(			/* last line input lacked */
				fd_topFrame->lstBrowser,		/* an accepted new line char */
				fl_get_browser_maxline(			/* so replace instead */
					fd_topFrame->lstBrowser),	/* of adding the line */
				bmsg);
			needNL = 0;							/* done with line with no NL */
		}
		else
			fl_addto_browser(fd_topFrame->		/* display string normally */
							 lstBrowser, bmsg);	/* on the browser object */
		strcpy(bmsg, nl+1);						/* move rest to beginning */
	}
}

int bprintf(va_alist) va_dcl
{
	int bw, pending = 0;
	va_list ap;
	char *fmt;
	static int tot = 0;

	va_start(ap);								/* start variable arg list */
	fmt = va_arg(ap, char*);					/* first string is format */
	bw = vsprintf(btmp, fmt, ap);				/* pass the rest to vsprintf */
	va_end(ap);									/* end variable arg list */
	if ((tot+bw) < (MSGLEN_BROWSER-1))			/* do we have space for new? */
		strcat(where, btmp);					/* yup, tack it on the end */
	else {										/* nope, so */
		strcat(where, "\n");					/* end the string here */
		pending = 1;							/* and indicate new pending */
	}
	bDoBuf();									/* process the message buf */
	if (pending) {								/* pending new string? */
		strcpy(bmsg, btmp);						/* yup, copy it in the buffer */
		bDoBuf();								/* process the buffer again */
	}
	tot = strlen(bmsg);							/* total chars so far */
	where = bmsg + tot;							/* pick up from where we left */
	if (tot) {									/* any trailing characters? */
		if (needNL)
			fl_replace_browser_line(			/* last line input lacked */
				fd_topFrame->lstBrowser,		/* an accepted new line char */
				fl_get_browser_maxline(			/* so replace instead */
					fd_topFrame->lstBrowser),	/* of adding the line */
				bmsg);
		else									/* unless it just happened */
			fl_addto_browser(fd_topFrame->		/* in which case add the */ 
							 lstBrowser, bmsg);	/* new line at the end */
		needNL = 1;								/* indicate we need NL */
	}
	return bw;									/* return bytes written */
}

void bflush(void)
{
	if ( *bmsg ) {								/* if leftover chars exist */
		fl_replace_browser_line(				/* last line input lacked */
			fd_topFrame->lstBrowser,			/* an accepted new line char */
			fl_get_browser_maxline(				/* so replace it */
				fd_topFrame->lstBrowser),
				bmsg);
		*bmsg = 0;								/* indicate nothing here */
		needNL = 0;								/* no incomplete line */
	}
	where = bmsg;								/* and start all over again */
}


/* Argument list building functions for pppd. freePppdArgs() is called
   to free previously used arguments, while pppdArg() gradually builds
   the pppd argument list up to MAXARGS_CHILD arguments, having itself
   a zero terminated variable argument list. */

#define MAXARGS_CHILD 32

char *pppd_arg[MAXARGS_CHILD+1];				/* the argument list strings */
int npppd_args = 0;								/* current number of args */

void freePppdArgs(void)
{
	int i;

	for (i=0; i<npppd_args; i++)				/* free all current args */
		free(pppd_arg[i]);
	npppd_args = 0;								/* and restart counter */
}

void pppdArg(va_alist) va_dcl
{
	va_list ap;
	char *arg;

	va_start(ap);								/* start variable arg list */
	while ((arg=va_arg(ap,char*))!=(char*)0) {	/* get next argument */
		if (npppd_args > MAXARGS_CHILD-1) {		/* next argument overflows ? */
			fprintf(stderr, "xISP: %s %s\n",	/* yup, bail out */
					"pppdArg",
					"argument overflow!");
			exit(1);
		}
		if ((pppd_arg[npppd_args] =				/* nope, try to allocate it */
			 malloc(strlen(arg)+1)) == NULL)	/* if allocation failed */
			doErr("pppdArg");					/* then bail out */
		strcpy(pppd_arg[npppd_args], arg);		/* all OK, copy it */
		++npppd_args;							/* and increment the counter */
	}
	va_end(ap);									/* end variable arg list */
}

/* Executes pppd process with the argument list given. The argument
   list is terminated by a (char *)0 pointer. It also initializes
   the browser-update timer. */

void pppd(char *args[])
{
	pppdPPID = pppdPID = fork();				/* fork to create child */
	if (pppdPID < 0)							/* ret < 0 : fork failed */
		doErr("pppd: fork");
	if (pppdPID) {								/* in parrent process */
		freePppdArgs();							/* free used arguments */
		dialerON = 1;							/* indicate dial in progress */
		pppdStat = 0;							/* status not collected yet */
		fl_set_timer(btimer, BU_INTERVAL);		/* start callback timer */
	}
	else {										/* in pppd process */
		umask(022);								/* ensure readable pid file */
		args[npppd_args] = (char *)0;			/* terminate the arg list */
		execv(args[0], args);					/* in /var/run and exec pppd */
		doErr("pppd: execv");					/* return here means error */
		fprintf(stderr,
			"xISP: pppd: execv returned %d!\n",
			errno);
	}
}


/* Check script-line syntax to comply with printf() and xispdial */

int lineCheck(char *sline)
{
	char *pc = sline;
	int len = strlen(sline), fs = 0;

	for (; *pc; pc++) {							/* check all '%' characters */
		for(; *pc!='%'; pc++)					/* by skipping the rest */
			if (*pc == 0) return fs;
		if (pc++ > sline+len-2)					/* is '%' the last char? */
			if (*pc != '%')						/* yes, this is no good */
				return -1;						/* unless last char is '%' */
		if (*pc != 's' && *pc != '%')			/* next char 's' or '%'? */
			return -1;							/* nope, no good either */
		if (*pc == 's')							/* if 's', increment counter */
			++fs;
	}
	return fs;									/* done, return counter */
}

/* Create dialing environment file */

int writeISPenv(xisprc_t *p)
{
	FILE *envfp;
	char ppasswd[MAXLEN_PASSWD+1] = {0};
	int i, fsdi = 0, fscb = 0, fs = 0;
	FL_Coord x, y;

	envfp = fopen(envfname, "w");				/* create the file */
	if (envfp == NULL)							/* bail out on error */
		doErr("writeISPenv, fopen");
	fprintf(envfp, "%d\n", p->maxAttempts);		/* write all needed */
	fprintf(envfp, "%d\n", p->sleepDelay);		/* dialing variables */
	fprintf(envfp, "%d\n", p->connectWait);
	fprintf(envfp, "%d\n", p->numPhones);
	for (i=0; i<p->numPhones; i++)
		fprintf(envfp, "%s\n", p->phone[i]);
	if (p->operOpts & (PAP_LOGIN | PAPS_LOGIN |
					   CHAPS_LOGIN)) {			/* if PAP/PAPS/CHAPS login */
		fprintf(envfp, "AUTH\n");				/* then let xispdial know */
	}
	else if (p->operOpts & MANUAL_LOGIN) {		/* if manual login */
		fprintf(envfp, "MANUAL\n");				/* let xispdial know */
		fprintf(envfp, "%s\n",					/* tell it the DISPLAY */
				XDisplayName(NULL));
		fprintf(envfp, "#%02X%02X%02X\n",		/* the background color */
				bgcol.r, bgcol.g, bgcol.b);
		fl_get_winorigin(						/* and a geometry position */
			FL_ObjWin(fd_topFrame->topIcon),
			&x, &y);
		fprintf(envfp, "+%d+%d\n", x, y-55+14);
	}
	else {
		if (p->operOpts & CALL_BACK)			/* if call-back desired */
			fprintf(envfp, "AUTO-CB\n");		/* let xispdial know */
		else
			fprintf(envfp, "AUTO\n");			/* else, simple auto-login */
		pdecode(ppasswd, p->passwd);			/* decrypt password */
		fprintf(envfp, "%s\n", p->account);		/* account, password and */
		fprintf(envfp, "%s\n", ppasswd);		/* script lines only if */
		fprintf(envfp, "%d\n", p->numSlines);	/* we want automatic login */
		for (i=0; i<p->numSlines; i++) {
			fs = lineCheck(p->sline[i]);		/* check expect-line syntax */
			if (fs<0 || (fsdi += fs)>2) break;	/* bail out on error */
			fs = lineCheck(p->sline[
						   MAXNUM_SLINES+i]);	/* check send-line syntax */
			if (fs<0 || (fsdi += fs)>2) break;	/* bail out on error */
		}
		if (p->numSlines)
			if (fs < 0 || fsdi > 2) {			/* syntax problem? */
				fclose(envfp);					/* yes, close file */
				return (fs<0)? fs:fsdi;			/* and bail out */
			}
		for (i=0; i<p->numSlines; i++)			/* print expect-send pairs */
			fprintf(envfp,"'%s' '%s'\n",		/* single-quoted to make */
					p->sline[i],				/* users' life easier :) */
					p->sline[MAXNUM_SLINES+i]);
		if (p->operOpts & CALL_BACK) {			/* if call-back desired */
			fprintf(envfp,"%d\n",p->CBDelay);	/* print out call-back delay */
			fprintf(envfp,"%d\n",p->numCBSlns);	/* and the call-back */
			for (i=0; i<p->numCBSlns; i++) {	/* script lines */
				fs = lineCheck(p->CBsln[i]);	/* check expect-line syntax */
				if (fs<0 || (fscb += fs)>2)		/* bail out on error */
					break;
				fs = lineCheck(p->CBsln[
						MAXNUM_SLINES+i]);		/* check send-line syntax */
				if (fs<0 || (fscb += fs)>2)		/* bail out on error */
					break;
			}
			if (p->numCBSlns)
				if (fs < 0 || fscb > 2) {		/* syntax problem? */
					fclose(envfp);				/* yes, close file */
					return (fs<0)? fs:2+fscb;	/* and bail out */
				}
			for (i=0; i<p->numCBSlns; i++)		/* print expect-send pairs */
				fprintf(envfp,"'%s' '%s'\n",	/* for call-back lines */
					p->CBsln[i],
					p->CBsln[MAXNUM_SLINES+i]);
		}
	}
	fprintf(envfp, "%s\n", p->modemReset);		/* modem reset */
	fprintf(envfp, "%s\n", p->modemInit);		/* and init strings */
	if (p->operOpts & MODEM_TONEDIAL)			/* dialing method character */
		fprintf(envfp, "T\n");
	else
		fprintf(envfp, "P\n");
	fclose(envfp);								/* finally, close the file */
	return fsdi + fscb;							/* return total # of "%s"'s */
}

/* Create PAP authentication file */

void writeISPPAP(xisprc_t *p)
{
	FILE *papfp;
	char ppasswd[MAXLEN_PASSWD+1] = {0};

	papfp = fopen(papfname, "w");				/* create the file */
	if (papfp == NULL)							/* bail out on error */
		doErr("writeISPPAP, fopen");
	pdecode(ppasswd, p->passwd);				/* decrypt password */
	fprintf(papfp, "%s\n", p->account);			/* write username and */
	fprintf(papfp, "%s\n", ppasswd);			/* password for PAP */
	fclose(papfp);
}

/* Update all connection indicators */

void updateCONN(void)
{
	fl_set_object_label(fd_topFrame->statusText,
		(connected)? "CONN":"DISCON");
	fl_set_object_label(fd_topFrame->speedText,
		connSpeed);
	fl_set_object_label(fd_topFrame->IPText,
		connIP);
}


/* Retrieve PID of spawned pppd process from RUNDIR/ppp? or if RUNDIR is not
   defined by pppd.h, search /proc for the pppd entry in the process table */

int getPppdPID(char *IF)
{
#ifdef RUNDIR
	char buf[16];
	int fd, br;

	strcpy(runfp, IF);							/* form the PID file */
	strcat(runfp, ".pid");
	fd = open(runfname, O_RDONLY);				/* open it for reading */
	if (fd < 0)									/* if error, return 0 */
		return 0;
	br = read(fd, buf, 16);						/* read the PID string */
	close(fd);									/* close the file */
	if (br < 0)									/* if error in reading */
		return 0;								/* return 0 */
	buf[br] = 0;								/* mark end of string */
	return atoi(buf);							/* and return converted */
#else
	DIR *procd;
	struct dirent *direntp;
	char name[32], statbuf[64], *p;
	int statd, br, pid = 0, ppid;

	procd = opendir("/proc");					/* open the /proc tree */
	while ((direntp=readdir(procd)) != NULL &&	/* search all entries */
		   !pid ) {
		if (! atoi(direntp->d_name))			/* skip non numeric entries */
			continue;
		p = direntp->d_name;					/* file name is PID number */
		strcpy(name, "/proc/");					/* form status file name */
		strcat(name, p);
		strcat(name, "/status");
		if ((statd=open(name,O_RDONLY)) < 0) {	/* and open it */
			closedir(procd);					/* bail out on failure */
			return 0;
		}
		br = read(statd, statbuf, 63);			/* read first few lines */
		close(statd);							/* and close it */
		if (br < 0) {							/* bail out if read fails */
			closedir(procd);
			return 0;
		}
		statbuf[br] = 0;						/* zero terminate buffer */
		sscanf(statbuf, "Name: %s", name);		/* get process name */
		if (strcmp(name, "pppd"))				/* continue if not "pppd" */
			continue;
		p = strstr(statbuf, "Pid:");			/* find PID string */
		sscanf(p, "Pid: %d PPid: %d",			/* and read PID and PPID */
			   &pid, &ppid);
		if (ppid != 1 && ppid != pppdPPID)		/* parent is not init nor */
			pid = 0;							/* the pppd started by xisp */
	}											/* so it's somebody else's */
	closedir(procd);
	return pid;
#endif
}

/* Retrieve PID of ip-up script spawned by the child pppd process
   by searching /proc for the ip-up entry in the process table */

int getipUDPID()
{
	DIR *procd;
	struct dirent *direntp;
	char name[32], statbuf[64], *p;
	int statd, br, pid = 0, ppid;

	procd = opendir("/proc");					/* open the /proc tree */
	while ((direntp=readdir(procd)) != NULL &&	/* search all entries */
		   !pid ) {
		if (! atoi(direntp->d_name))			/* skip non numeric entries */
			continue;
		p = direntp->d_name;					/* file name is PID number */
		strcpy(name, "/proc/");					/* form status file name */
		strcat(name, p);
		strcat(name, "/status");
		if ((statd=open(name,O_RDONLY)) < 0) {	/* and open it */
			closedir(procd);					/* bail out on failure */
			return 0;
		}
		br = read(statd, statbuf, 63);			/* read first few lines */
		close(statd);							/* and close it */
		if (br < 0) {							/* bail out if read fails */
			closedir(procd);
			return 0;
		}
		statbuf[br] = 0;						/* zero terminate buffer */
		sscanf(statbuf, "Name: %s", name);		/* get process name */
		if (strcmp(name, ipWhich))				/* continue if no match */
			continue;
		p = strstr(statbuf, "Pid:");			/* find PID string */
		sscanf(p, "Pid: %d PPid: %d",			/* and read PID and PPID */
			   &pid, &ppid);
		if (ppid != pppdPID)					/* parent is not our pppd */
			pid = 0;							/* so it's somebody else's */
	}
	closedir(procd);
	return pid;
}


/* Wait for pppd to die, after sending SIGINT. Procedure provided because
   pppd appears not to honor SIGINT while in the connection phase, when
   sending LCP packets to its peer. New pppd started during this phase
   will terminate with error */

#define MAXWAIT_PPPD 8		/* number of iterations waiting for pppd to die */

void waitPppd(char *IF)
{
	int i, pid;

	for (i=0; i<MAXWAIT_PPPD &&					/* as long as pppd is there */
			  (pid = getPppdPID(IF)); i++) {	/* up to MAXWAIT_PPPD times */
		kill(pid, SIGINT);						/* send SIGINT to pppd */
		bprintf("%s%d%s\n",						/* print message on browser */
				"Waiting for pppd[", pid,
				"] to die.");
		mSleep(2000);							/* wait a couple of seconds */
	}
	if (i >= MAXWAIT_PPPD) {					/* if we timed out */
		bprintf("%s%d%s\n",						/* print message on browser */
				"Killing pppd[", pid, "].");
		kill(pid, SIGKILL);						/* send SIGKILL to pppd */
		mSleep(1000);							/* and wait a second */
	}											/* before returning */
}


/* String <-> IP address conversions */

int IPToStr(unsigned char *ip, char *str)		/* convert IP to string */
{
	return sprintf(str, "%u.%u.%u.%u",
		ip[0],ip[1],ip[2],ip[3]);
}

int StrToIP(char *str, unsigned char *ip)		/* convert string to IP */
{
	unsigned int iip[4];
	int n, i;

	n = sscanf(str, "%3u.%3u.%3u.%3u", &iip[0],
			   &iip[1], &iip[2], &iip[3]);
	if (n == 4)
		for (i=0; i<4; ip++, i++) *ip = iip[i];
	return n;
}


/* Check optional pppd options file */

int pppdOptsFileOK(void)
{
	struct stat st;

	if (!stat(PPPD_OPTIONS, &st))				/* does it exist ? */
		return (st.st_size > 0);				/* yup, if size > 0 all OK */
	else
		return 0;								/* no it doesn't */
}


/* Reduce colormap usage */

void colorSqueeze(void)
{
	int i;

	for (i=0; i<FL_FREE_COL1; i++) {
		switch (i) {

			case FL_BLACK:						/* except for these which */
			case FL_YELLOW:						/* are used in our code */
			case FL_WHITE:
			case FL_INDIANRED:
			case FL_COL1:
			case FL_RIGHT_BCOL:
			case FL_BOTTOM_BCOL:
			case FL_TOP_BCOL:
			case FL_LEFT_BCOL:
			case FL_MCOL:
			case FL_INACTIVE:
			case FL_DARKCYAN:
				break;

			default:
				fl_set_icm_color(i, 0,0,0);		/* reset all unused internal */
		}										/* colormap colors to black */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                  XPM animation routines and callbacks                   |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#ifdef XPMANIMATE
void updateAnim(int run)
{
	Pixmap id;
	Window win = FL_ObjWin(fd_topFrame->topIcon);

	if (run) {									/* if animation is running */
		id = online[frame];						/* use current frame */
		if (minimized)							/* if minimized */
			fl_winicon(win, id, 0);				/* set the window icon */
		else									/* if maximized */
			fl_set_pixmap_pixmap(				/* set the master form icon */
				fd_topFrame->topIcon, id, 0);
	}
	else {
		id = pixmap;							/* otherwise, reset icons */
		fl_winicon(win, id, 0);					/* for the main form */
		fl_set_pixmap_pixmap(fd_topFrame->		/* and the minimized window */
			topIcon, id, 0);
		frame = 0;								/* reset current frame */
	}
}

void prepAnimPixmaps(void)
{
	unsigned int width = 50, height = 50,
			 hotx = 1, hoty = 1;
 
	pixmap = fl_create_from_pixmapdata(fl_default_window(), xisp_xpm,
				&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[0] = fl_create_from_pixmapdata(fl_default_window(), online1_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[1] = fl_create_from_pixmapdata(fl_default_window(), online2_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[2] = fl_create_from_pixmapdata(fl_default_window(), online3_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[3] = fl_create_from_pixmapdata(fl_default_window(), online4_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[4] = fl_create_from_pixmapdata(fl_default_window(), online5_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[5] = fl_create_from_pixmapdata(fl_default_window(), online6_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[6] = fl_create_from_pixmapdata(fl_default_window(), online7_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[7] = fl_create_from_pixmapdata(fl_default_window(), online8_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[8] = fl_create_from_pixmapdata(fl_default_window(), online9_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[9] = fl_create_from_pixmapdata(fl_default_window(), online10_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[10] = fl_create_from_pixmapdata(fl_default_window(), online11_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
	online[11] = fl_create_from_pixmapdata(fl_default_window(), online12_xpm,
					&width, &height, NULL, &hotx, &hoty, FL_BLACK);
}

int mapquery(FL_FORM *form, void *xev)
{
	XEvent *ev = xev;

	switch (ev->type) {							/* figure out if we're */
		case MapNotify:							/* being minimized or not */
			minimized = 0;
			break;

		case UnmapNotify:
			minimized = 1;
			break;

		default: break;
	}
	return 0;
}

void doXPMUpdate(FL_OBJECT *obj, long param)
{
	unsigned newnetpkts;

	frame = ++frame % 12;						/* increment frame counter */
	if (frame) {								/* if not on frame 0 */
		if (transfer)							/* just spin the animation */
			updateAnim(1);
	}
	else {										/* if on frame 0 */
		newnetpkts = pppPkts("ppp0");			/* check for new packets */
		if (newnetpkts != netpkts) {			/* if new packets exist */
			netpkts = newnetpkts;				/* start next animation cycle */
			transfer = 1;
			updateAnim(1);
		}
		else transfer = 0;
	}
	fl_set_timer(xpmtimer, XPM_INTERVAL);		/* restart animation timer */
}
#endif


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |          Routine adjusting possible actions according to state          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void possibleActions(int state)
{
	switch (state) {

		case DISCONNECTED: {
			fl_activate_object(fd_topFrame->conButton);
			fl_deactivate_object(fd_topFrame->intButton);
			fl_deactivate_object(fd_topFrame->disButton);
			fl_activate_object(fd_topFrame->optMenu);
    		fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->hlpMenu);
    		fl_set_object_lcol(fd_topFrame->hlpMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->autoRedial);
			break;
		}

		case DIALING: {
			fl_deactivate_object(fd_topFrame->conButton);
			fl_activate_object(fd_topFrame->intButton);
			fl_deactivate_object(fd_topFrame->disButton);
			fl_deactivate_object(fd_topFrame->optMenu);
    		fl_set_object_lcol(fd_topFrame->optMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->hlpMenu);
    		fl_set_object_lcol(fd_topFrame->hlpMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->autoRedial);
			break;
		}

		case CONNECTED: {
			fl_deactivate_object(fd_topFrame->conButton);
			fl_deactivate_object(fd_topFrame->intButton);
			fl_activate_object(fd_topFrame->disButton);
			fl_activate_object(fd_topFrame->optMenu);
    		fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->hlpMenu);
    		fl_set_object_lcol(fd_topFrame->hlpMenu,FL_LCOL);
			fl_deactivate_object(fd_topFrame->autoRedial);
			break;
		}

		default: break;
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Dialer output parsing routines                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Note the maximum field length defined by
   MAXLEN_FIELD, used by the generic parser doutWait() */

#define MAXLEN_FIELD 64

char *doutWait(char *buf, char *field, char *ecs)
{
	char *p1, *p2;
	static int next = 0, nc;
	static char rstr[MAXLEN_FIELD];

	if (next) {									/* processing field string? */
		p2 = strfstr(buf, ecs);					/* yup, find ending char */
		if (p2 != NULL) {						/* if found */
			next = 0;							/* indicate processing done */
			strncpy(&rstr[nc], buf,				/* save it locally */
					(int)(p2-buf));
			rstr[nc+(int)(p2-buf)] = 0;			/* terminate it */
			return rstr;						/* and return its address */
		}
		else {
			bprintf("%s: %s!\n","doutWait",		/* ec not found: dialer */
		 "invalid output from dialer process");	/* output is screwed, or */
			return NULL;						/* something is very wrong */
		}
	}

	p1 = strstr(buf, field);					/* wanted field in there? */
	if (p1 != NULL) {							/* yup, parse to ending ec */
		p2 = strfstr(p1, ecs);
		if (p2 != NULL) {						/* if found */
			nc = (int)(p2-p1)-strlen(field);	/* string has so many chars */
			strncpy(rstr, p1+strlen(field),		/* save it locally */
					nc);
			rstr[nc] = 0;						/* terminate it */
			return rstr;						/* and return its address */
		}
		else {									/* oops, string ends */
			strcpy(rstr, p1+strlen(field));		/* copy what's there */
			next = 1;							/* take note for next call */
			nc = strlen(rstr);
			return NULL;
		}
	}
	return NULL;
}

/* Parse dialer output for its PID number */

int doutPID(char *buf)
{
	char pids[16], *p;

	p = doutWait(buf, "PID=", ".");				/* look for PID=?????. */
	if (p != NULL) {							/* found it? */
		strcpy(pids, p);						/* yup, copy the string */
		dialerPID = atoi(pids);					/* convert and save it */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else										/* nope */
		return 0;								/* indicate not found */
}

/* Parse dialer output for CONNECT string from modem */

int doutCONN(char *buf)
{
	char speeds[32], *p, *sp;
	unsigned speed;
	static int first = 1;

	p = doutWait(buf, "CONN", "/\r\n");			/* look for CONNECT */
	if (p != NULL) {							/* found it? */
		if ((p_xisprc->operOpts & CALL_BACK)	/* if call-back is enabled */
			&& first) {							/* and it's the 1st CONNECT */
			first = 0;							/* indicate it's gone by */
			return 0;							/* and wait for 2nd CONNECT */
		}
		first = 1;								/* else reset for next time */
		sp = strchr(p, ' ');					/* continue: look for space */
		if (sp != NULL) {						/* found it? */
			strcpy(speeds, sp);					/* yes, copy the string */
			speed = atoi(speeds);				/* convert and save speed */
			if (speed < 1000)					/* and format its string */
				sprintf(connSpeed,"%d",speed);
			else
				sprintf(connSpeed,"%d,%03d",
						speed/1000,speed%1000);
		}
		else {									/* oops, no speed! */
			if (strstr(p, "ECT") == NULL)		/* connect string OK? */
				return 0;						/* nope, indicate not found */
			else								/* yes */
				strcpy(connSpeed, "- NA -");	/* but no speed indication */
		}
		connected = 1;							/* update variables */
		connTime = 0;
		strcpy(connIP, EMPTY_IP);
		fl_set_object_label(					/* update indicators */
			fd_topFrame->timeText, EMPTY_TIME);
		updateCONN();
		fl_set_timer(ctimer, CT_INTERVAL);		/* start connection timer */
		if (p_xisprc->operOpts & CONNECT_BELL)	/* ring bell if desired */
			XBell(fl_get_display(), 50);
#ifdef XPMANIMATE
		netpkts = 0;							/* initialize net packets */
		updateAnim(1);							/* animation pixmap */		
		fl_winicon(								/* window icon, and */
			FL_ObjWin(fd_topFrame->topIcon),
			online[0], 0);
		fl_set_timer(xpmtimer, XPM_INTERVAL);	/* start xpm-animation timer */
#endif
		linkPoll = 1;							/* clear link-poll activation */
		possibleActions(CONNECTED);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else										/* nope */
		return 0;								/* indicate not found */
}

/* Parse dialer output for TIMEOUT string from xispdial */

int doutTIMEOUT(char *buf)
{
	char *p;

	p = doutWait(buf, "TIME", "\n");
	if (p != NULL) {
		connected = 0;							/* invalidate connection */
		linkPoll = 0;							/* no link-poll activation */
		fl_set_timer(ctimer, 0.0);				/* stop connection timer */
#ifdef XPMANIMATE
		XSetIconName(fl_get_display(),			/* reset icon label */
			fd_topFrame->topFrame->window,
			"X-ISP");
		updateAnim(0);							/* and animation pixmap */		
		fl_set_timer(xpmtimer, 0.0);			/* disable animation timer */
#endif
		strcpy(connSpeed, EMPTY_SPEED);			/* reset speed indicator */
		updateCONN();							/* update display */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else
		return 0;
}

/* Parse dialer output for ABORT string from xispterm */

int doutABORT(char *buf)
{
	char *p;

	p = doutWait(buf, "ABOR", "\n");
	if (p != NULL) {
		userDiscon = 1;							/* it's a user disconnect */
		connected = 0;							/* invalidate connection */
		linkPoll = 0;							/* no link-poll activation */
		fl_set_timer(ctimer, 0.0);				/* stop connection timer */
#ifdef XPMANIMATE
		XSetIconName(fl_get_display(),			/* reset icon label */
			fd_topFrame->topFrame->window,
			"X-ISP");
		updateAnim(0);							/* and animation pixmap */		
		fl_set_timer(xpmtimer, 0.0);			/* disable animation timer */
#endif
		strcpy(connSpeed, EMPTY_SPEED);			/* reset speed indicator */
		updateCONN();							/* update display */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else
		return 0;
}

/* Parse dialer output for usefull output */

void doutParse(char *buf)
{
	static char line[MAXBUF_CHILD+1] = {0}, *pline = line;
	char *p;

	if (MAXBUF_CHILD-(int)(pline-line)			/* take care of possible */
		< strlen(buf)) pline = line;			/* line buffer overflow */
	for (p=buf; *p; ) {							/* while chars are available */
		for (; *p && *p!='\n' && *p!='\r';		/* make local copy of buf */
			 p++, pline++) *pline = *p;			/* in line-buffered fashion */
		if (*p) {								/* end of line found? */
			*pline++ = *p;						/* yes, add special char */
			*pline = 0;							/* and terminate the line */
			if (! dialerPID) doutPID(line);		/* parse for dialer PID */
			if (! connected) doutCONN(line);	/* for CONNECTED info */
			if (dialerON && connected) {
				doutTIMEOUT(line);				/* for dialer TIMEOUT */
				if (p_xisprc->operOpts &		/* and if manual login */
					MANUAL_LOGIN)				/* is selected, */
					doutABORT(line);			/* parse also for ABORT */
			}
			pline = line;						/* start new line */
			++p;								/* from next input char */
		}
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                              Misc callbacks                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doCTupdate(FL_OBJECT *obj, long param)
{
	char tout[16];

	connTime += (int)CT_INTERVAL;				/* increment by interval */
	strftime(tout, 16, "%H:%M:%S",				/* format time HH:MM:SS */
			 gmtime(&connTime));
	fl_set_object_label(fd_topFrame->timeText,	/* print it on main form */
						tout);
	XSetIconName(fl_get_display(),				/* show it on the icon */ 
				 fd_topFrame->topFrame->window,
				 &tout[1]);
	fl_set_timer(ctimer, CT_INTERVAL);			/* and restart timer */
}

void doLPupdate(FL_OBJECT *obj, long param)
{
	char *p;
	static int justConnected = 1;

	if (pppdPID && (p=pppAddr("ppp0"))!=NULL) {	/* pppd running and link OK? */
		if (!linkOK) {							/* yes, just now? */
			strcpy(connIP, p);					/* yup, copy new IP address */
			updateCONN();						/* update indicators */
			linkOK = 1;							/* and mark link as OK */
			justConnected = 0;					/* init for next time */
			pppdPID = getPppdPID("ppp0");		/* get PID of detached pppd */
		}
		fl_set_timer(ltimer, SP_INTERVAL);		/* restart timer */
	}
	else {
		if (pppdPID && justConnected) {			/* give it a second chance */
			justConnected = 0;					/* but not a third :) */
			fl_set_timer(ltimer, SP_INTERVAL);	/* restart status poll timer */
			ipUDON = UD_WAIT/BU_INTERVAL;		/* and restart ip-up/ip-down */
			fl_set_timer(btimer, BU_INTERVAL);	/* output processing */
			return;
		}
		justConnected = 1;						/* init for next time */
		bprintf("PPP link is down.\n");			/* nope, print message */
		fl_set_timer(ctimer, 0.0);				/* disable timers */
		fl_set_timer(ltimer, 0.0);
#ifdef XPMANIMATE
		XSetIconName(fl_get_display(),			/* reset icon label */
			fd_topFrame->topFrame->window,
			"X-ISP");
		updateAnim(0);							/* and animation pixmap */
		fl_set_timer(xpmtimer, 0.0);			/* disable animation timer */
#endif
		if (pppdPID)							/* if not dead already */
			kill(pppdPID, SIGINT);				/* terminate pppd process */
		pppdPID = 0;							/* indicate that it's gone */
		connected = 0;							/* update variables */
		connTime = 0;
		linkOK = 0;
		linkPoll = 0;
		strcpy(connSpeed, EMPTY_SPEED);
		strcpy(connIP, EMPTY_IP);
		updateCONN();							/* update indicators */
		possibleActions(DISCONNECTED);			/* mark possible actions */
		if (!userDiscon &&						/* if not user disconnect */
			(p_xisprc->operOpts &				/* and auto-redial set, then */
			 AUTO_REDIAL)) {
			fl_call_object_callback(			/* "push Connect" again */
				fd_topFrame->conButton);
		}
		userDiscon = 0;							/* reset indicator */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |  Callback and support routines for browser updates from dialer output   |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#define MAXLEN_IOTBUF 11						/* I/O trigger buffer size */
char IOTBuf[MAXLEN_IOTBUF+1] = {0},
	 *pIOTB = IOTBuf;

void browserIOTrigger(int fd, void *data)		/* pipe I/O-triggered input */
{
	int space, br, dump = 0;
	char *pc;
	void doBUpdate();

	space = MAXLEN_IOTBUF-(int)(pIOTB-IOTBuf);	/* calculate space left, and */
	br = read(fd, pIOTB, space);				/* read max that many bytes */
	pIOTB += br;								/* increment pointer */
	for (pc=IOTBuf; !dump && pc<pIOTB; pc++)	/* scan for CR or LF */
		if (*pc=='\n' || *pc=='\r') dump = 1;	/* dump buffer if found */
	dump |= (pIOTB == MAXLEN_IOTBUF + IOTBuf);	/* or dump when buffer full */
	if (dump) {									/* dump confition met? */
		fl_set_timer(btimer, BU_INTERVAL);		/* yes, restart timer */
		fl_call_object_callback(btimer);		/* call update routine */
		pIOTB = IOTBuf;							/* and start all over again */
	}
}

int IOIdleCallback(XEvent *xev, void *data)		/* pipe input checking proc */
{
	fd_set rfds;
	struct timeval tv;
	int stat;

	FD_ZERO(&rfds);								/* initialize the fs set */
	FD_SET(dialerFD, &rfds);					/* stick the dialerFD in */
	tv.tv_sec = tv.tv_usec = 0;					/* return with no delay */
	stat = select(dialerFD+1, &rfds,			/* check the descriptor */
				  NULL, NULL, &tv);				/* for any input */
	if (stat < 0) {
		if (errno != EINTR)						/* non-blocking signal? */
			doErr("IOIdleCallback: select");	/* no, abort with diagnostic */
		return 0;								/* yes, do nothing */
	}
	if (stat)									/* data exists for reading? */
		browserIOTrigger(dialerFD, NULL);		/* yes, call the I/O proc */
	return 0;
}

int readT(int fd, void *buf, size_t cnt)		/* I/O-trigger aware read */
{												/* note that cnt is ignored */
	int br, bp = pIOTB - IOTBuf;				/* when IOTbuf has data, as */
												/* cnt always >MAXLEN_IOTBUF */
	if (bp > 0) {								/* are bytes from I/O */
		br = bp;								/* trigger still pending? */
		memcpy(buf, IOTBuf, br);				/* yes, put them in buffer */
		pIOTB = IOTBuf;							/* mark trigger buffer empty */
	}
	else
		br = read(dialerFD, buf, cnt);			/* else read from descriptor */
	return br;									/* return bytes read */
}

int pppdCleanup()								/* collect pppd status etc. */
{
	int i, stat;

	i = waitpid(pppdPID, &stat, 1);				/* get pppd return status */
	if (i < 0)									/* ret < 0 : wait failed */
		doErr("pppdCleanup: waitpid");			/* abort with diagnostic */
	pppdStat = 1;								/* indicate status collected */
	if (stat) {									/* if pppd didn't fare well */
		if (WIFEXITED(stat))					/* print messages */
			bprintf("pppd returned %d!\n",		/* for abnormal exit */
					WEXITSTATUS(stat));
		if (WIFSIGNALED(stat))					/* and for untrapped signal */
			bprintf("pppd got signal %d!\n",
					WTERMSIG(stat));
		if (dialerPID)							/* dialer still there? */
			kill(dialerPID, SIGTERM);			/* yup, terminate it */
		pppdPID = getPppdPID("ppp0");			/* PID of spawned pppd */
		if (pppdPID)							/* pppd still there? */
			kill(pppdPID, SIGINT);				/* yup, terminate it */
		pppdPID = 0;							/* indicate that it's gone */
		dialerON = 0;							/* indicate dial aborted */
		dialerPID = 0;							/* and no dialer process */
		fl_activate_object(						/* finally reactivate the */
			fd_topFrame->conButton);			/* connection button */
	}
	if (p_xisprc->operOpts & PAP_LOGIN)			/* PAP authentication used? */
		unlink(papfname);						/* yes, delete PAP data file */
	return WEXITSTATUS(stat);					/* return exit status */
}

void doBUpdate(FL_OBJECT *obj, long param)
{
	int br, stat;
	char buf[MAXBUF_CHILD+MAXLEN_IOTBUF+2], *p;
	static int dialWait = DIAL_WAIT/BU_INTERVAL;
	static int gotIP = 0;

	if (dialerON) {								/* if dial in progress */
		br = readT(dialerFD, buf,MAXBUF_CHILD);	/* read a buffer full */
		if (br > 0) {							/* if read OK */
			buf[br] = 0;						/* indicate string end */
			doutParse(buf);						/* parse dialer output */
			bprintf("%s", buf);					/* print it */
			fl_set_timer(btimer, BU_INTERVAL);	/* restart timer */
			if (dialWait) dialWait = 0;			/* no more waiting */
		}
		else if (br < 0) {						/* read failed */
			if (errno != EAGAIN)				/* pipe output unavailable? */
				doErr("doBUpdate: read");		/* no, abort with diagnostic */
			fl_set_timer(btimer, BU_INTERVAL);	/* yes, restart timer */
		}
		else {									/* EOF on pipe output */
			if (dialWait) {						/* wait for xispdial */
				if (! pppdStat)					/* only once */
					pppdCleanup();				/* collect pppd's status */
				--dialWait;						/* to open the pipe */
				fl_set_timer(btimer,			/* if it hasn't managed yet */
							 BU_INTERVAL);		/* due to machine load */
				return;
			}
			fl_set_idle_callback(0, NULL);		/* remove idle callback */
			dialWait = DIAL_WAIT/BU_INTERVAL;	/* reset for next time */
			dialerON = 0;						/* indicate dial done */
			dialerPID = 0;						/* and no dialer process */
			bflush();							/* flush browser output */
			if (connected) {					/* check connection status */
				if (p_xisprc->operOpts &		/* ip-up/-down support on? */
					IP_UPDOWN) {
					ipUDON =					/* yup, mark ip-up active */
						UD_WAIT/BU_INTERVAL;	/* with such max iterations */
					strcpy(ipWhich, "ip-up");	/* and set script name */
					fl_set_timer(btimer,		/* restart timer for ip-up */
								 BU_INTERVAL);
				}
			}
			else								/* if we disconnected */
				possibleActions(DISCONNECTED);	/* mark possible actions */
			if (linkPoll)						/* activate link polling */
				fl_set_timer(ltimer,			/* timer if we have been */
							 SP_INTERVAL);		/* cleared to do so */
		}
		if (! pppdStat)							/* only once */
			pppdCleanup();						/* collect pppd's status */
		return;
	}

	if (ipUDON) {								/* if ip-up/-down active */
		br = readT(dialerFD, buf,MAXBUF_CHILD);	/* read a buffer full */
		if (br > 0) {							/* if read OK */
			buf[br] = 0;						/* indicate string end */
			bprintf("%s", buf);					/* print it */
			fl_set_timer(btimer, BU_INTERVAL);	/* and restart timer */
		}
		else if (br < 0) {						/* read failed */
			if (errno != EAGAIN)				/* pipe output unavailable? */
				doErr("doBUpdate: read");		/* no, abort with diagnostic */
			fl_set_timer(btimer, BU_INTERVAL);	/* yes, restart timer */
		}
		else {									/* EOF on pipe output */
			if (! ipUDPID)						/* unless we know it already */
					ipUDPID = getipUDPID();		/* get PID of ip-up/-down */
			if (ipUDPID &&						/* if found ip-up/-down PID */
			    waitpid(- ipUDPID, &stat,		/* are all children done? */
						WNOHANG) > 0)
				fl_set_timer(btimer,			/* nope, restart timer */
							 BU_INTERVAL);
			else {								/* if no ip-up/-down process */
				--ipUDON;						/* count down ip-up/-down */
				ipUDPID = 0;					/* input drain iterations */
				if (!gotIP &&					/* if we don't have it yet */
					(p=pppAddr("ppp0"))!=NULL)	/* get ppp0 address */
				{
					strcpy(connIP, p);			/* yup, copy the IP address */
					updateCONN();				/* update indicators */
					gotIP = 1;					/* and don't try it again */
				}
				if (ipUDON)						/* iterations still to go? */
					fl_set_timer(btimer,		/* yes, restart the timer */
								 BU_INTERVAL);
				else							/* nope -> we're done, so */
					gotIP = 0;					/* reset this for next time */
			}
		}
		return;
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Callback routines for topFrame                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doConnect(FL_OBJECT *obj, long param)
{
	int fs, i, ns;
	char sa[8][8] = {"1200","2400","4800","9600",
					"19200","38400","57600","115200"},
		 *speed = sa[5], level[2*(MAXDIG_BSDCOMP+1)],
		 lips[MAXLEN_IP+1] = {0}, rips[MAXLEN_IP+1] = {0},
		 nms[MAXLEN_IP+1] = {0}, lrs[2*MAXLEN_IP+2] = {0},
		 mtus[MAXDIG_MTRU+1] = {0}, mrus[MAXDIG_MTRU+1] = {0},
		 dnsips[MAXLEN_IP+1] = {0};
	static char *ipparam = NULL;
	unsigned char pwr2[8] = {1,2,4,8,16,32,64,128};

	fl_deactivate_object(fd_topFrame->conButton);	/* disable button */
	fs = writeISPenv(p_xisprc);						/* create the env file */
	if (fs < 0) {									/* script lines OK? */
		bprintf("Syntax error in the user\n"		/* nope, print message */
				"specified script lines!\n");
		fl_activate_object(fd_topFrame->conButton);	/* enable button */
		unlink(envfname);							/* delete the script */
		return;										/* and do nothing */
	}
	ns = (p_xisprc->operOpts & CALL_BACK)? 4:2;		/* 4 for CB, 2 for DI */
	if (fs > ns) {									/* more than <ns> "%s"'s? */
		bprintf("More than %d '%%s' in the\n"		/* yup, print message */
			"user specified script lines!\n", ns);
		fl_activate_object(fd_topFrame->conButton);	/* enable button */
		unlink(envfname);							/* delete the script */
		return;										/* and do nothing */
	}
	for (i=0; i<8; i++)								/* create the speed str */
		if (pwr2[i] == p_xisprc->modemSpeed)
			speed = sa[i];
	if (p_xisprc->operOpts & PPPD_COMPRESS)			/* the compression */
		sprintf(level, "%d,%d",						/* level string */
				p_xisprc->compLevel,
				p_xisprc->compLevel);
	else
		sprintf(level, "%d,%d", 0,0);
	if (p_xisprc->operOpts & ACCEPT_LOCALIP)		/* the local IP string */
		strcpy(lips, LOCAL_IPSTR);
	else
		IPToStr(p_xisprc->localIP, lips);
	if (p_xisprc->operOpts & ACCEPT_REMOTEIP)		/* the remote IP string */
		strcpy(rips, REMOTE_IPSTR);
	else
		IPToStr(p_xisprc->remoteIP, rips);
	strcpy(lrs, lips);								/* the local:remote */
	strcat(lrs, ":");								/* pppd parameter */
	strcat(lrs, rips);
	IPToStr(p_xisprc->netmask, nms);				/* the netmask */
	sprintf(mtus, "%d", p_xisprc->mtu);				/* MTU and MRU */
	sprintf(mrus, "%d", p_xisprc->mru);
	if (p_xisprc->operOpts & PAP_LOGIN)				/* using PAP for login? */
		writeISPPAP(p_xisprc);						/* yes, create data file */
	if (p_xisprc->operOpts & IP_UPDOWN) {			/* if ip-up/-down active */
		if (ipparam == NULL) {						/* only the first fime */
			ipparam = (char *)malloc(				/* allocate space for */
				sizeof(pipefname)+1+MAXLEN_DESCR+	/* the composite string */
				1+2*MAXLEN_IP+2);
			if (ipparam == NULL)
				outofMem();
		}
		strcpy(ipparam, pipefname);					/* create ipparam string */
		strcat(ipparam, ":");						/* with named pipe file */
		strcat(ipparam, p_xisprc->descr);			/* name, ISP name */
		strcat(ipparam, ":");
		IPToStr(p_xisprc->dns1, dnsips);			/* and DNS server IP's */
		if (strcmp(dnsips, DNS_IPSTR))
			strcat(ipparam, dnsips);
		strcat(ipparam, ":");
		IPToStr(p_xisprc->dns2, dnsips);
		if (strcmp(dnsips, DNS_IPSTR))
			strcat(ipparam, dnsips);
	}
	bprintf("Connection initiated: %s\n",
			strtime());
	pppdArg(PPPD, p_xisprc->modemDevice, speed,		/* build argument list */
			BSDCOMP, level, "noipdefault",
			"modem", "lock",
			"mru", mrus, "mtu", mtus,
			"asyncmap", p_xisprc->asyncmap,
			(char *)0);
	if (p_xisprc->operOpts & ESCAPE_ON)
		pppdArg("escape", p_xisprc->escape,
				(char *)0);
	if (p_xisprc->operOpts & HW_FLOWCTRL)
		pppdArg("crtscts", (char *)0);
	if (pppdOptsFileOK())
		pppdArg("file", PPPD_OPTIONS, (char *)0);
	if (p_xisprc->operOpts & ACCEPT_LOCALIP)
		pppdArg("ipcp-accept-local", (char *)0);
	if (p_xisprc->operOpts & ACCEPT_REMOTEIP)
		pppdArg("ipcp-accept-remote", (char *)0);
	pppdArg(lrs, "netmask", nms, (char *)0);
	if (p_xisprc->operOpts & DEFAULT_ROUTE)
		pppdArg("defaultroute", (char *)0);
	if (p_xisprc->operOpts & PAP_LOGIN)
		pppdArg("+ua", papfname, (char *)0);
	if (p_xisprc->operOpts & PAPS_LOGIN)
		pppdArg("user", p_xisprc->name,
				"renotename", p_xisprc->rname,
				(char *)0);
	if (p_xisprc->operOpts & CHAPS_LOGIN)
		pppdArg("name", p_xisprc->name,
				"renotename", p_xisprc->rname,
				(char *)0);
	if (p_xisprc->operOpts & IP_UPDOWN) {
		pppdArg("ipparam", ipparam, (char *)0);
	}
	pppdArg("connect", PPPD_CONNECT, (char *)0);
	waitPppd("ppp0");								/* wait for old pppd */
	pppd(pppd_arg);									/* exec new pppd */
	fl_set_idle_callback(IOIdleCallback, NULL);		/* and set idle callback */
	return;
}

void doInterrupt(FL_OBJECT *obj, long param)
{
	fl_deactivate_object(fd_topFrame->intButton);	/* disable button */
	fl_call_object_callback(btimer);				/* flush/check status */
	if (dialerPID) {								/* dialer still there? */
		bprintf("\nDialing interrupted: %s\n",
				strtime());
		if (kill(dialerPID, SIGTERM))				/* yup, terminate it */
			doErr("doInterrupt: kill");
	}
	connected = 0;									/* invalidate connection */
	linkPoll = 0;									/* no link polling */
	return;
}

void doDisconnect(FL_OBJECT *obj, long param)
{
	userDiscon = 1;									/* set indicator */
	fl_deactivate_object(fd_topFrame->disButton);	/* disable button */
	bprintf("Disconnection: %s\n",
			strtime());
	if (dialerPID)									/* dialer still there? */
		kill(dialerPID, SIGTERM);					/* yup, terminate it */
	pppdPID = getPppdPID("ppp0");					/* PID of spawned pppd */
	if (pppdPID)									/* interrupt pppd proc */
		kill(pppdPID, SIGINT);
	pppdPID = 0;									/* indicate no pppd proc */
	fl_call_object_callback(ltimer);				/* update link status */
	possibleActions(DISCONNECTED);					/* mark possible actions */
	if (p_xisprc->operOpts & IP_UPDOWN) {			/* ip-up/-down support? */
		ipUDON = 1;									/* yup, mark ip-down on */
		ipUDON = UD_WAIT/BU_INTERVAL;				/* yup, mark ip-up active */
		strcpy(ipWhich, "ip-down");					/* and set script name */
		fl_set_timer(btimer, BU_INTERVAL);			/* start browser timer */
	}
	return;
}

void doOptions(FL_OBJECT *obj, long param)
{
	int selection, i;
	char dflt[5], phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0},
		 *p, ppasswd[MAXLEN_PASSWD+1] = {0};
	unsigned char pwr2[8] = {1,2,4,8,16,32,64,128};
	char ips[MAXLEN_IP+1] = {0};

	fl_deactivate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_INACTIVE);
	selection = fl_get_menu(obj);
	switch (selection) {
		case OPTIONS_ISP: {
			fl_clear_browser(fd_ISPInfo->ISPBrowser);
			for (i=1; i<(MAXNUM_ISP+1); i++) {
				p = xispOptions[i].descr;
				fl_add_browser_line(fd_ISPInfo->ISPBrowser, (*p)?p:"~");
			}
			fl_show_form(fd_ISPInfo->ISPInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "ISP Selection");
			if (*(xispOptions[currentRC].descr)) {
				fl_select_browser_line(fd_ISPInfo->ISPBrowser, currentRC);
				fd_ISPInfo->ldata = currentRC;
			}
			else {
				fd_ISPInfo->ldata = 0;
				fl_deactivate_object(fd_ISPInfo->ISPNameEdit);
			}
			break;
		}

		case OPTIONS_ACCOUNT: {
			for (i=0; i<p_xisprc->numPhones; i++) {
				if (i) strcat(phone, ";");
				strcat(phone, p_xisprc->phone[i]);
			}
			fl_set_input(fd_accountInfo->telInput, phone);
			fl_set_input(fd_accountInfo->accInput, p_xisprc->account);
			if (p_xisprc->passwd[0])
				pdecode(ppasswd, p_xisprc->passwd);
			fl_set_input(fd_accountInfo->pswInput, ppasswd);
			fl_set_input(fd_accountInfo->UNInput, p_xisprc->name);
			fl_set_input(fd_accountInfo->remoteInput, p_xisprc->rname);
			if (p_xisprc->operOpts & PAP_LOGIN) {
				fl_set_button(fd_accountInfo->authPAPButton, 1);
				fl_set_button(fd_accountInfo->authPAPSButton, 0);
				fl_set_button(fd_accountInfo->authCHAPSButton, 0);
				fl_set_button(fd_accountInfo->authNoneButton, 0);
				fl_activate_object(fd_accountInfo->accInput);
				fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
				fl_activate_object(fd_accountInfo->pswInput);
				fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
				fl_deactivate_object(fd_accountInfo->UNInput);
				fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->remoteInput);
				fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
			}
			else if (p_xisprc->operOpts & PAPS_LOGIN) {
				fl_set_button(fd_accountInfo->authPAPButton, 0);
				fl_set_button(fd_accountInfo->authPAPSButton, 1);
				fl_set_button(fd_accountInfo->authCHAPSButton, 0);
				fl_set_button(fd_accountInfo->authNoneButton, 0);
				fl_deactivate_object(fd_accountInfo->accInput);
				fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->pswInput);
				fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
				fl_activate_object(fd_accountInfo->UNInput);
				fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
				fl_activate_object(fd_accountInfo->remoteInput);
				fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
			}
			else if (p_xisprc->operOpts & CHAPS_LOGIN) {
				fl_set_button(fd_accountInfo->authPAPButton, 0);
				fl_set_button(fd_accountInfo->authPAPSButton, 0);
				fl_set_button(fd_accountInfo->authCHAPSButton, 1);
				fl_set_button(fd_accountInfo->authNoneButton, 0);
				fl_deactivate_object(fd_accountInfo->accInput);
				fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->pswInput);
				fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
				fl_activate_object(fd_accountInfo->UNInput);
				fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
				fl_activate_object(fd_accountInfo->remoteInput);
				fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
			}
			else {
				fl_set_button(fd_accountInfo->authPAPButton, 0);
				fl_set_button(fd_accountInfo->authPAPSButton, 0);
				fl_set_button(fd_accountInfo->authCHAPSButton, 0);
				fl_set_button(fd_accountInfo->authNoneButton, 1);
				fl_activate_object(fd_accountInfo->accInput);
				fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
				fl_activate_object(fd_accountInfo->pswInput);
				fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
				fl_deactivate_object(fd_accountInfo->UNInput);
				fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
				fl_deactivate_object(fd_accountInfo->remoteInput);
				fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
			}
			fd_accountInfo->ldata = p_xisprc->operOpts;
			fl_show_form(fd_accountInfo->accountInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "Account Information");
			break;
		}

		case OPTIONS_DIALING: {
			sprintf(dflt, "%d", p_xisprc->maxAttempts);
			fl_set_input(fd_dialInfo->rtrInput, dflt);
			sprintf(dflt, "%d", p_xisprc->sleepDelay);
			fl_set_input(fd_dialInfo->dlyInput, dflt);
			sprintf(dflt, "%d", p_xisprc->connectWait);
			fl_set_input(fd_dialInfo->CNWaitInput, dflt);
			if (p_xisprc->operOpts & MANUAL_LOGIN) {
				fl_set_button(fd_dialInfo->TTYesButton, 1);
				fl_set_button(fd_dialInfo->TTNoButton, 0);
				fl_deactivate_object(fd_dialInfo->CBYesButton);
    			fl_set_object_lcol(fd_dialInfo->CBYesButton,FL_INACTIVE);
			}
			else {
				fl_set_button(fd_dialInfo->TTYesButton, 0);
				fl_set_button(fd_dialInfo->TTNoButton, 1);
				fl_activate_object(fd_dialInfo->CBYesButton);
    			fl_set_object_lcol(fd_dialInfo->CBYesButton,FL_WHITE);
			}
			if (p_xisprc->operOpts & CALL_BACK) {
				fl_activate_object(fd_dialInfo->CBOptions);
    			fl_set_object_lcol(fd_dialInfo->CBOptions,FL_LCOL);
				fl_set_button(fd_dialInfo->CBYesButton, 1);
			}
			else {
				fl_deactivate_object(fd_dialInfo->CBOptions);
    			fl_set_object_lcol(fd_dialInfo->CBOptions,FL_INACTIVE);
				fl_set_button(fd_dialInfo->CBYesButton, 0);
			}
			if (p_xisprc->operOpts & CONNECT_BELL) {
				fl_set_button(fd_dialInfo->RBYesButton, 1);
				fl_set_button(fd_dialInfo->RBNoButton, 0);
			}
			else {
				fl_set_button(fd_dialInfo->RBYesButton, 0);
				fl_set_button(fd_dialInfo->RBNoButton, 1);
			}
			for (i=0; i<MAXNUM_SLINES; i++) {
				fl_set_input(fd_dialInfo->EscriptLine[i],
							 p_xisprc->sline[i]);
				fl_set_input(fd_dialInfo->SscriptLine[i],
							 p_xisprc->sline[MAXNUM_SLINES+i]);
			}
			fd_dialInfo->ldata = p_xisprc->operOpts;
			fl_set_form_hotspot(fd_dialInfo->dialInfo, 120,120);
			fl_show_form(fd_dialInfo->dialInfo, FL_PLACE_HOTSPOT,
						 FL_TRANSIENT, "Dialing and Login");
			break;
		}

		case OPTIONS_COMM: {
			fl_set_input(fd_commInfo->deviceInput,
						 p_xisprc->modemDevice);
			for (i=0; i<8; i++) {
				if (pwr2[i] == p_xisprc->modemSpeed)
					fl_set_button(fd_commInfo->speedButton[i], 1);
				else
					fl_set_button(fd_commInfo->speedButton[i], 0);
			}
			fl_set_input(fd_commInfo->modemResetInput,
						 p_xisprc->modemReset);
			fl_set_input(fd_commInfo->modemInitInput,
						 p_xisprc->modemInit);
			if (p_xisprc->operOpts & HW_FLOWCTRL) {
				fl_set_button(fd_commInfo->HWButton, 1);
				fl_set_button(fd_commInfo->SWButton, 0);
			}
			else {
				fl_set_button(fd_commInfo->HWButton, 0);
				fl_set_button(fd_commInfo->SWButton, 1);
			}
			if (p_xisprc->operOpts & MODEM_TONEDIAL) {
				fl_set_button(fd_commInfo->toneButton, 1);
				fl_set_button(fd_commInfo->pulseButton, 0);
			}
			else {
				fl_set_button(fd_commInfo->toneButton, 0);
				fl_set_button(fd_commInfo->pulseButton, 1);
			}
			if (p_xisprc->operOpts & PPPD_COMPRESS) {
				fl_set_button(fd_commInfo->SWCOnButton, 1);
				fl_set_button(fd_commInfo->SWCOffButton, 0);
				fl_activate_object(fd_commInfo->SWCInput);
    			fl_set_object_lcol(fd_commInfo->SWCInput,FL_WHITE);
			}
			else {
				fl_set_button(fd_commInfo->SWCOnButton, 0);
				fl_set_button(fd_commInfo->SWCOffButton, 1);
				fl_deactivate_object(fd_commInfo->SWCInput);
    			fl_set_object_lcol(fd_commInfo->SWCInput,FL_INACTIVE);
			}
			if (p_xisprc->operOpts & ESCAPE_ON) {
				fl_set_button(fd_commInfo->escapeYesButton, 1);
				fl_set_button(fd_commInfo->escapeNoButton, 0);
				fl_activate_object(fd_commInfo->escapeInput);
    			fl_set_object_lcol(fd_commInfo->escapeInput,FL_WHITE);
			}
			else {
				fl_set_button(fd_commInfo->escapeYesButton, 0);
				fl_set_button(fd_commInfo->escapeNoButton, 1);
				fl_deactivate_object(fd_commInfo->escapeInput);
    			fl_set_object_lcol(fd_commInfo->escapeInput,FL_INACTIVE);
			}
			sprintf(dflt, "%d", p_xisprc->compLevel);
			fl_set_input(fd_commInfo->SWCInput, dflt);
			fd_commInfo->ldata = ((unsigned)p_xisprc->modemSpeed << 16) |
								 p_xisprc->operOpts;
			fl_set_input(fd_commInfo->asyncmapInput,
						 p_xisprc->asyncmap);
			fl_set_input(fd_commInfo->escapeInput,
						 p_xisprc->escape);
			fl_set_form_hotspot(fd_commInfo->commInfo, 100,75);
			fl_show_form(fd_commInfo->commInfo, FL_PLACE_HOTSPOT,
						 FL_TRANSIENT, "Communication Options");
			break;
		}

		case OPTIONS_TCPIP: {
			IPToStr(p_xisprc->localIP, ips);
			fl_set_input(fd_tcpipInfo->localIPInput, ips);
			IPToStr(p_xisprc->remoteIP, ips);
			fl_set_input(fd_tcpipInfo->remoteIPInput, ips);
			IPToStr(p_xisprc->netmask, ips);
			fl_set_input(fd_tcpipInfo->NMInput, ips);
			IPToStr(p_xisprc->dns1, ips);
			fl_set_input(fd_tcpipInfo->pDNSInput, ips);
			IPToStr(p_xisprc->dns2, ips);
			fl_set_input(fd_tcpipInfo->sDNSInput, ips);
			if (p_xisprc->operOpts & ACCEPT_LOCALIP) {
				fl_set_button(fd_tcpipInfo->ALYesButton, 1);
				fl_set_button(fd_tcpipInfo->ALNoButton, 0);
				fl_deactivate_object(fd_tcpipInfo->localIPInput);
				fl_set_object_lcol(fd_tcpipInfo->localIPInput, FL_INACTIVE);
			}
			else {
				fl_set_button(fd_tcpipInfo->ALYesButton, 0);
				fl_set_button(fd_tcpipInfo->ALNoButton, 1);
				fl_activate_object(fd_tcpipInfo->localIPInput);
				fl_set_object_lcol(fd_tcpipInfo->localIPInput, FL_WHITE);
			}
			if (p_xisprc->operOpts & ACCEPT_REMOTEIP) {
				fl_set_button(fd_tcpipInfo->ARYesButton, 1);
				fl_set_button(fd_tcpipInfo->ARNoButton, 0);
				fl_deactivate_object(fd_tcpipInfo->remoteIPInput);
				fl_set_object_lcol(fd_tcpipInfo->remoteIPInput, FL_INACTIVE);
			}
			else {
				fl_set_button(fd_tcpipInfo->ARYesButton, 0);
				fl_set_button(fd_tcpipInfo->ARNoButton, 1);
				fl_activate_object(fd_tcpipInfo->remoteIPInput);
				fl_set_object_lcol(fd_tcpipInfo->remoteIPInput, FL_WHITE);
			}
			if (p_xisprc->operOpts & DEFAULT_ROUTE) {
				fl_set_button(fd_tcpipInfo->DRYesButton, 1);
				fl_set_button(fd_tcpipInfo->DRNoButton, 0);
			}
			else {
				fl_set_button(fd_tcpipInfo->DRYesButton, 0);
				fl_set_button(fd_tcpipInfo->DRNoButton, 1);
			}
			if (p_xisprc->operOpts & IP_UPDOWN) {
				fl_set_button(fd_tcpipInfo->UDYesButton, 1);
				fl_set_button(fd_tcpipInfo->UDNoButton, 0);
				fl_activate_object(fd_tcpipInfo->pDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_WHITE);
				fl_activate_object(fd_tcpipInfo->sDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_WHITE);
			}
			else {
				fl_set_button(fd_tcpipInfo->UDYesButton, 0);
				fl_set_button(fd_tcpipInfo->UDNoButton, 1);
				fl_deactivate_object(fd_tcpipInfo->pDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
				fl_deactivate_object(fd_tcpipInfo->sDNSInput);
				fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
			}
			fd_tcpipInfo->ldata = p_xisprc->operOpts;
			sprintf(dflt, "%d", p_xisprc->mtu);
			fl_set_input(fd_tcpipInfo->MTUInput, dflt);
			sprintf(dflt, "%d", p_xisprc->mru);
			fl_set_input(fd_tcpipInfo->MRUInput, dflt);
			fl_set_form_hotspot(fd_tcpipInfo->tcpipInfo, 100,75);
			fl_show_form(fd_tcpipInfo->tcpipInfo, FL_PLACE_HOTSPOT,
						 FL_TRANSIENT, "TCP/IP Options");
			break;
		}

		default: break;
	}
	return;
}

void doHelp(FL_OBJECT *obj, long param)
{
	int selection, i;
	static int helpLoaded = 0;

	selection = fl_get_menu(obj);
	switch (selection) {
		case HELP_ABOUT: {
			fl_show_form(fd_aboutInfo->aboutInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "About xISP");
			break;
		}

		case HELP_GENERAL: {
			if (! helpLoaded) {
				for (i=0; i<MAXNUM_HELPLINES; i++)
					fl_add_browser_line(fd_helpInfo->helpBrowser,
										helpText[i]);
				helpLoaded = 1;
			}
			fl_show_form(fd_helpInfo->helpInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "General Information");
			break;
		}

		default: break;
	}
	return;
}

void doAutoRedial(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_topFrame->autoRedial))
		p_xisprc->operOpts |= AUTO_REDIAL;
	else
		p_xisprc->operOpts &= ~AUTO_REDIAL;
	writeXisprc(rcfname, xispOptions, currentRC);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Selection routines for topFrame                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int select_IPText(FL_OBJECT *obj, int event,
				  FL_Coord mx, FL_Coord my,
				  int key, void *raw_event)
{
	static unsigned char selected = 0;

	if (linkOK && event == FL_RELEASE) {
		selected ^= 1;
		if (selected) {
			fl_set_object_color(obj,FL_YELLOW,FL_MCOL);
			XStoreBytes(fl_get_display(), connIP, strlen(connIP)); 
		}
		else
			fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
	}
	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                    Callback routines for accountInfo                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doTelNoInput(FL_OBJECT *obj, long param)
{
	static char phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0};

	strncpy(phone, (char *)fl_get_input(obj), MAXNUM_TELS*(MAXLEN_PHONE+1));
	fd_accountInfo->vdata = phone;
	return;
}

void doAccountInput(FL_OBJECT *obj, long param)
{
	static char account[MAXLEN_ACCOUNT+1] = {0};

	strncpy(account, (char *)fl_get_input(obj), MAXLEN_ACCOUNT);
	fd_accountInfo->vdata = account;
	return;
}

void doPasswdInput(FL_OBJECT *obj, long param)
{
	static char epasswd[MAXLEN_PASSWD+1] = {0};
	char ppasswd[MAXLEN_PASSWD+1] = {0};
	int i;

	strncpy(ppasswd, (char *)fl_get_input(obj), MAXLEN_PASSWD);
	for (i=strlen(ppasswd); i<MAXLEN_PASSWD; i++) ppasswd[i] = 0;
	pencode(epasswd, ppasswd);
	fd_accountInfo->vdata = epasswd;
	return;
}

void doAuthPAPButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_accountInfo->authPAPButton)) {
		fd_accountInfo->ldata |= PAP_LOGIN;
		fd_accountInfo->ldata &= ~(PAPS_LOGIN | CHAPS_LOGIN);
		fl_activate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
		fl_deactivate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
	}
	return;
}

void doAuthPAPSButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_accountInfo->authPAPSButton)) {
		fd_accountInfo->ldata |= PAPS_LOGIN;
		fd_accountInfo->ldata &= ~(PAP_LOGIN | CHAPS_LOGIN);
		fl_deactivate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
		fl_activate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
	}
	return;
}

void doAuthCHAPSButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_accountInfo->authCHAPSButton)) {
		fd_accountInfo->ldata |= CHAPS_LOGIN;
		fd_accountInfo->ldata &= ~(PAP_LOGIN | PAPS_LOGIN);
		fl_deactivate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_INACTIVE);
		fl_activate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_WHITE);
	}
	return;
}

void doAuthNoneButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_accountInfo->authNoneButton)) {
		fd_accountInfo->ldata &= ~(PAP_LOGIN | PAPS_LOGIN | CHAPS_LOGIN);
		fl_activate_object(fd_accountInfo->accInput);
		fl_set_object_lcol(fd_accountInfo->accInput, FL_WHITE);
		fl_activate_object(fd_accountInfo->pswInput);
		fl_set_object_lcol(fd_accountInfo->pswInput, FL_WHITE);
		fl_deactivate_object(fd_accountInfo->UNInput);
		fl_set_object_lcol(fd_accountInfo->UNInput, FL_INACTIVE);
		fl_deactivate_object(fd_accountInfo->remoteInput);
		fl_set_object_lcol(fd_accountInfo->remoteInput, FL_INACTIVE);
	}
	return;
}

void doUNInput(FL_OBJECT *obj, long param)
{
	static char name[MAXLEN_UNR+1] = {0};

	strncpy(name, (char *)fl_get_input(obj), MAXLEN_UNR);
	fd_accountInfo->vdata = name;
	return;
}

void doRemoteInput(FL_OBJECT *obj, long param)
{
	static char rname[MAXLEN_UNR+1] = {0};

	strncpy(rname, (char *)fl_get_input(obj), MAXLEN_UNR);
	fd_accountInfo->vdata = rname;
	return;
}

void doAccountOK(FL_OBJECT *obj, long param)
{
	char *p1, *p2;
	int i, nc;

	fl_call_object_callback(fd_accountInfo->telInput);
	p1 = fd_accountInfo->vdata;
	for (i=0; i<MAXNUM_TELS && *p1; i++) {
		p2 = strchr(p1, ';');
		if (p2 != NULL)
			nc = p2 - p1;
		else {
			nc = strlen(p1);
			p2 = p1 + nc - 1;
		}
		strncpy(p_xisprc->phone[i], p1, nc);
		p_xisprc->phone[i][nc] = 0;
		p1 = p2 + 1;
	}
	p_xisprc->numPhones = i;
	fl_call_object_callback(fd_accountInfo->accInput);
	strcpy(p_xisprc->account, (char *)fd_accountInfo->vdata);
	fl_call_object_callback(fd_accountInfo->pswInput);
	strcpy(p_xisprc->passwd, (char *)fd_accountInfo->vdata);
	fl_call_object_callback(fd_accountInfo->UNInput);
	strcpy(p_xisprc->name, (char *)fd_accountInfo->vdata);
	fl_call_object_callback(fd_accountInfo->remoteInput);
	strcpy(p_xisprc->rname, (char *)fd_accountInfo->vdata);
	p_xisprc->operOpts = (unsigned)fd_accountInfo->ldata & 0xFFFF;
	writeXisprc(rcfname, xispOptions, currentRC);
	fl_hide_form(fd_accountInfo->accountInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doAccountCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_accountInfo->accountInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for dialInfo                      |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doRetryInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_RETRY+1] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_RETRY);
	num = atoi(ns);
	if (! num) {
		num = MAXNUM_RETRY;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doDelayInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_DELAY+1] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_DELAY);
	num = atoi(ns);
	if (! num) {
		num = MAXSEC_DELAY;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doCNWaitInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_CNWAIT] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_CNWAIT);
	num = atoi(ns);
	if (! num) {
		num = MAXSEC_CNWAIT;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
	}
	fd_dialInfo->vdata = &num;
	return;
}

void doScriptInput(FL_OBJECT *obj, long param)
{
	char *p;
	static char line[2*MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	if (param < MAXNUM_SLINES)
		p = (char *)fl_get_input(fd_dialInfo->EscriptLine[param]);
	else
		p = (char *)fl_get_input(fd_dialInfo->
								 SscriptLine[param % MAXNUM_SLINES]);
	strncpy(line[param], p, MAXLEN_SLINE);
	fd_dialInfo->vdata = line;
	return;
}

void doRBYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->RBYesButton))
		fd_dialInfo->ldata |= CONNECT_BELL;
	return;
}

void doRBNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->RBNoButton))
		fd_dialInfo->ldata &= ~CONNECT_BELL;
	return;
}

void doTTYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->TTYesButton)) {
		fd_dialInfo->ldata |= MANUAL_LOGIN;
		fl_deactivate_object(fd_dialInfo->CBYesButton);
    	fl_set_object_lcol(fd_dialInfo->CBYesButton,FL_INACTIVE);
	}
	return;
}

void doTTNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->TTNoButton)) {
		fd_dialInfo->ldata &= ~MANUAL_LOGIN;
		fl_activate_object(fd_dialInfo->CBYesButton);
    	fl_set_object_lcol(fd_dialInfo->CBYesButton,FL_WHITE);
	}
	return;
}

void doCBYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_dialInfo->CBYesButton)) {
		fd_dialInfo->ldata |= CALL_BACK;
		fl_activate_object(fd_dialInfo->CBOptions);
    	fl_set_object_lcol(fd_dialInfo->CBOptions,FL_LCOL);
	}
	else {
		fd_dialInfo->ldata &= ~CALL_BACK;
		fl_deactivate_object(fd_dialInfo->CBOptions);
    	fl_set_object_lcol(fd_dialInfo->CBOptions,FL_INACTIVE);
		p_xisprc->numCBSlns = 0;
	}
	return;
}

void doCBOptions(FL_OBJECT *obj, long param)
{
	int i;
	char delay[5];

	sprintf(delay, "%d", p_xisprc->CBDelay);
	fl_set_input(fd_CBInfo->CBdlyInput, delay);
	for (i=0; i<MAXNUM_SLINES; i++) {
		fl_set_input(fd_CBInfo->CBEscriptLine[i],
					 p_xisprc->CBsln[i]);
		fl_set_input(fd_CBInfo->CBSscriptLine[i],
					 p_xisprc->CBsln[MAXNUM_SLINES+i]);
	}
	fl_deactivate_form(fd_dialInfo->dialInfo);
	fl_show_form(fd_CBInfo->CBInfo, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Edit call-back script");
	return;
}

void doDialOK(FL_OBJECT *obj, long param)
{
	char *Ep = p_xisprc->sline[0],
		 *Sp = p_xisprc->sline[8];
	int i;

	fl_call_object_callback(fd_dialInfo->rtrInput);
	p_xisprc->maxAttempts = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->dlyInput);
	p_xisprc->sleepDelay = *(unsigned char *)(fd_dialInfo->vdata);
	fl_call_object_callback(fd_dialInfo->CNWaitInput);
	p_xisprc->connectWait = *(unsigned char *)(fd_dialInfo->vdata);
	p_xisprc->operOpts = (unsigned)fd_dialInfo->ldata & 0xFFFF;
	for (i=0; i<MAXNUM_SLINES; i++) {
		fl_call_object_callback(fd_dialInfo->EscriptLine[i]);
		fl_call_object_callback(fd_dialInfo->SscriptLine[i]);
	}
	memcpy(p_xisprc->sline, fd_dialInfo->vdata,
		   2*MAXNUM_SLINES*(MAXLEN_SLINE+1));
	for (i=0; i<MAXNUM_SLINES && (*Ep | *Sp); ) {
		Ep = p_xisprc->sline[++i];
		Sp = p_xisprc->sline[MAXNUM_SLINES+i];
	}
	p_xisprc->numSlines = i;
	writeXisprc(rcfname, xispOptions, currentRC);
	fl_hide_form(fd_dialInfo->dialInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doDialCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_dialInfo->dialInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doCBScriptInput(FL_OBJECT *obj, long param)
{
	char *p;
	static char line[2*MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	if (param < MAXNUM_SLINES)
		p = (char *)fl_get_input(fd_CBInfo->CBEscriptLine[param]);
	else
		p = (char *)fl_get_input(fd_CBInfo->
								 CBSscriptLine[param % MAXNUM_SLINES]);
	strncpy(line[param], p, MAXLEN_SLINE);
	fd_CBInfo->vdata = line;
	return;
}

void doCBDelayInput(FL_OBJECT *obj, long param)
{
	char ns[MAXDIG_CBDELAY] = {0};
	static unsigned char num;

	strncpy(ns, (char *)fl_get_input(obj), MAXDIG_CBDELAY);
	num = atoi(ns);
	if (! num) {
		num = MAXSEC_CBDELAY;
		sprintf(ns, "%d", num);
		fl_set_input(obj, ns);
	}
	fd_CBInfo->vdata = &num;
	return;
}

void doCBInfoOK(FL_OBJECT *obj, long param)
{
	char *Ep = p_xisprc->CBsln[0],
		 *Sp = p_xisprc->CBsln[8];
	int i;

	fl_call_object_callback(fd_CBInfo->CBdlyInput);
	p_xisprc->CBDelay = *(unsigned char *)(fd_CBInfo->vdata);
	for (i=0; i<MAXNUM_SLINES; i++) {
		fl_call_object_callback(fd_CBInfo->CBEscriptLine[i]);
		fl_call_object_callback(fd_CBInfo->CBSscriptLine[i]);
	}
	memcpy(p_xisprc->CBsln, fd_CBInfo->vdata,
		   2*MAXNUM_SLINES*(MAXLEN_SLINE+1));
	for (i=0; i<MAXNUM_SLINES && (*Ep | *Sp); ) {
		Ep = p_xisprc->CBsln[++i];
		Sp = p_xisprc->CBsln[MAXNUM_SLINES+i];
	}
	p_xisprc->numCBSlns = i;
	writeXisprc(rcfname, xispOptions, currentRC);
	fl_hide_form(fd_CBInfo->CBInfo);
	fl_activate_form(fd_dialInfo->dialInfo);
	return;
}

void doCBInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_CBInfo->CBInfo);
	fl_activate_form(fd_dialInfo->dialInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for ISPInfo and renameISP               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doISPPick(FL_OBJECT *obj, long param)
{
	if (! fd_ISPInfo->ldata)
		fl_activate_object(fd_ISPInfo->ISPNameEdit);
	fd_ISPInfo->ldata = fl_get_browser(obj);
	return;
}

void doISPPickNEdit(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_ISPInfo->ISPBrowser);
	fl_call_object_callback(fd_ISPInfo->ISPNameEdit);
	return;
}

void doISPOK(FL_OBJECT *obj, long param)
{
	if (fd_ISPInfo->ldata) {
		currentRC = fd_ISPInfo->ldata;
		p_xisprc = &xispOptions[currentRC];
		writeXisprc(rcfname, xispOptions, currentRC);
		fl_set_object_label(fd_topFrame->ISPText, p_xisprc->descr);
		fl_set_button(fd_topFrame->autoRedial,
			(p_xisprc->operOpts & AUTO_REDIAL)? 1:0);
	}
	fl_hide_form(fd_ISPInfo->ISPInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doISPCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_ISPInfo->ISPInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doISPNameEdit(FL_OBJECT *obj, long param)
{
	if (fd_ISPInfo->ldata) {
		writeXisprc(rcfname, xispOptions, currentRC);
		currentRC = fd_ISPInfo->ldata;
		p_xisprc = &xispOptions[currentRC];
		fl_set_input(fd_renameISP->ISPNameInput, p_xisprc->descr);
		fl_deactivate_form(fd_ISPInfo->ISPInfo);
		fl_show_form(fd_renameISP->renameISP, FL_PLACE_MOUSE,
					 FL_TRANSIENT, "Rename ISP record");
	}
	return;
}

void doISPNameInput(FL_OBJECT *obj, long param)
{
	static char name[MAXLEN_DESCR+1] = {0};

	strncpy(name, (char *)fl_get_input(obj), MAXLEN_DESCR);
	fd_renameISP->vdata = name;
	return;
}

void doISPNameEditOK(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_renameISP->ISPNameInput);
	strcpy(p_xisprc->descr, (char *)fd_renameISP->vdata);
	fl_replace_browser_line(fd_ISPInfo->ISPBrowser,
							currentRC, p_xisprc->descr);
	fl_hide_form(fd_renameISP->renameISP);
	fl_activate_form(fd_ISPInfo->ISPInfo);
	return;
}

void doISPNameEditCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_renameISP->renameISP);
	fl_activate_form(fd_ISPInfo->ISPInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for aboutInfo and helpInfo              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doAboutOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_aboutInfo->aboutInfo);
	return;
}

void doHelpInfoOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_helpInfo->helpInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for commInfo                      |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doModemInitInput(FL_OBJECT *obj, long param)
{
	static char init[MAXLEN_MODEM+1] = {0};

	strncpy(init, (char *)fl_get_input(obj), MAXLEN_MODEM);
	if (! strlen(init)) {
		strcpy(init, MODEM_INIT);
		fl_set_input(obj, init);
	}
	fd_commInfo->vdata = init;
	return;
}

void doModemResetInput(FL_OBJECT *obj, long param)
{
	static char reset[MAXLEN_MODEM+1] = {0};

	strncpy(reset, (char *)fl_get_input(obj), MAXLEN_MODEM);
	if (! strlen(reset)) {
		strcpy(reset, MODEM_RESET);
		fl_set_input(obj, reset);
	}
	fd_commInfo->vdata = reset;
	return;
}

void doToneButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_commInfo->toneButton))
		fd_commInfo->ldata |= MODEM_TONEDIAL;
	return;
}

void doPulseButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_commInfo->pulseButton))
		fd_commInfo->ldata &= ~MODEM_TONEDIAL;
	return;
}

void doSWCOnButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_commInfo->SWCOnButton)) {
		fd_commInfo->ldata |= PPPD_COMPRESS;
		fl_activate_object(fd_commInfo->SWCInput);
    	fl_set_object_lcol(fd_commInfo->SWCInput,FL_WHITE);
	}
	return;
}

void doSWCOffButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_commInfo->SWCOffButton)) {
		fd_commInfo->ldata &= ~PPPD_COMPRESS;
		fl_deactivate_object(fd_commInfo->SWCInput);
    	fl_set_object_lcol(fd_commInfo->SWCInput,FL_INACTIVE);
	}
	return;
}

void doSWCInput(FL_OBJECT *obj, long param)
{
	char level[MAXDIG_BSDCOMP+1] = {0};
	static unsigned char num = COMP_LEVEL;

	strncpy(level, (char *)fl_get_input(obj), MAXDIG_BSDCOMP);
	num = atoi(level);
	if ((fd_commInfo->ldata & PPPD_COMPRESS) && (!num || num<9)) {
		XBell(fl_get_display(), 50);
		num = COMP_LEVEL;
		sprintf(level, "%d", num);
		fl_set_input(obj, level);
	}
	fd_commInfo->vdata = &num;
	return;
}

int SWCInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int test;

	test = atoi(cur);
	if (c == '-' || test > 15)
		return FL_INVALID|FL_RINGBELL;
	else
		return FL_VALID;
}

void doDeviceInput(FL_OBJECT *obj, long param)
{
	static char device[MAXLEN_DEVICE+1] = {0};

	strncpy(device, (char *)fl_get_input(obj), MAXLEN_DEVICE);
	if (! strlen(device)) {
		strcpy(device, MODEM_DEVICE);
		fl_set_input(obj, device);
	}
	fd_commInfo->vdata = device;
	return;
}

void doSpeedButton(FL_OBJECT *obj, long param)
{
	unsigned char speed;

	speed = (0x01 << param);
	fd_commInfo->ldata &= 0xFFFF;
	fd_commInfo->ldata |= (speed << 16);
	return;
}

void doAsyncmapInput(FL_OBJECT *obj, long param)
{
	static char asyncmap[MAXDIG_ASYNCMAP+1] = {0};

	strncpy(asyncmap, (char *)fl_get_input(obj), MAXDIG_ASYNCMAP);
	if (! *asyncmap) {
		if (p_xisprc->operOpts & HW_FLOWCTRL)
			strcpy(asyncmap, PPPD_HASYNCMAP);
		else
			strcpy(asyncmap, PPPD_SASYNCMAP);
		fl_set_input(obj, asyncmap);
	}
	fd_commInfo->vdata = asyncmap;
	return;
}

int AMInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	if (strcmp(old, cur)) {
		if (!((c>='0' && c<='9') || (c>='A' && c<='F')))
			return FL_INVALID|FL_RINGBELL;
	}
	return FL_VALID;
}

void doEscapeYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_commInfo->escapeYesButton)) {
		fd_commInfo->ldata |= ESCAPE_ON;
		fl_activate_object(fd_commInfo->escapeInput);
    	fl_set_object_lcol(fd_commInfo->escapeInput,FL_WHITE);
	}
	return;
}

void doEscapeNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_commInfo->escapeNoButton)) {
		fd_commInfo->ldata &= ~ESCAPE_ON;
		fl_deactivate_object(fd_commInfo->escapeInput);
    	fl_set_object_lcol(fd_commInfo->escapeInput,FL_INACTIVE);
	}
	return;
}

void doEscapeInput(FL_OBJECT *obj, long param)
{
	static char escape[MAXLEN_ESCAPE+1] = {0};

	strncpy(escape, (char *)fl_get_input(obj), MAXLEN_ESCAPE);
	if (! strlen(escape)) {
		strcpy(escape, PPPD_ESCAPE);
		fl_set_input(obj, escape);
	}
	fd_commInfo->vdata = escape;
	return;
}

int EInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	if (strcmp(old, cur)) {
		if (!((c>='0' && c<='9') || (c>='A' && c<='F') || c==','))
			return FL_INVALID|FL_RINGBELL;
	}
	return FL_VALID;
}

void doHWButton(FL_OBJECT *obj, long param)
{
	static char asyncmap[MAXDIG_ASYNCMAP+1] = {0};

	if (fl_get_button(fd_commInfo->HWButton)) {
		fd_commInfo->ldata |= HW_FLOWCTRL;
		strcpy(asyncmap, PPPD_HASYNCMAP);
		fl_set_input(fd_commInfo->asyncmapInput, asyncmap);
	}
	return;
}

void doSWButton(FL_OBJECT *obj, long param)
{
	static char asyncmap[MAXDIG_ASYNCMAP+1] = {0};

	if (fl_get_button(fd_commInfo->SWButton)) {
		fd_commInfo->ldata &= ~HW_FLOWCTRL;
		strcpy(asyncmap, PPPD_SASYNCMAP);
		fl_set_input(fd_commInfo->asyncmapInput, asyncmap);
	}
	return;
}

void doCommInfoOK(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_commInfo->deviceInput);
	strcpy(p_xisprc->modemDevice, (char *)fd_commInfo->vdata);
	p_xisprc->modemSpeed = (unsigned)fd_commInfo->ldata >> 16;
	fl_call_object_callback(fd_commInfo->modemInitInput);
	strcpy(p_xisprc->modemInit, (char *)fd_commInfo->vdata);
	fl_call_object_callback(fd_commInfo->modemResetInput);
	strcpy(p_xisprc->modemReset, (char *)fd_commInfo->vdata);
	p_xisprc->operOpts = (unsigned)fd_commInfo->ldata & 0xFFFF;
	fl_call_object_callback(fd_commInfo->SWCInput);
	p_xisprc->compLevel = *(unsigned char *)(fd_commInfo->vdata);
	fl_call_object_callback(fd_commInfo->asyncmapInput);
	strncpy(p_xisprc->asyncmap, fd_commInfo->vdata, MAXDIG_ASYNCMAP);
	fl_call_object_callback(fd_commInfo->escapeInput);
	strncpy(p_xisprc->escape, fd_commInfo->vdata, MAXLEN_ESCAPE);
	writeXisprc(rcfname, xispOptions, currentRC);
	fl_hide_form(fd_commInfo->commInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doCommInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_commInfo->commInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for tcpipInfo                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doLocalIPInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = LOCAL_IP, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, LOCAL_IP, 4);
		fl_set_input(obj, LOCAL_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doALYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ALYesButton)) {
		fd_tcpipInfo->ldata |= ACCEPT_LOCALIP;
		fl_deactivate_object(fd_tcpipInfo->localIPInput);
    	fl_set_object_lcol(fd_tcpipInfo->localIPInput,FL_INACTIVE);
	}
	return;
}

void doALNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ALNoButton)) {
		fd_tcpipInfo->ldata &= ~ACCEPT_LOCALIP;
		fl_activate_object(fd_tcpipInfo->localIPInput);
    	fl_set_object_lcol(fd_tcpipInfo->localIPInput,FL_WHITE);
	}
	return;
}

void doRemoteIPInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = REMOTE_IP, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, REMOTE_IP, 4);
		fl_set_input(obj, REMOTE_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doARYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ARYesButton)) {
		fd_tcpipInfo->ldata |= ACCEPT_REMOTEIP;
		fl_deactivate_object(fd_tcpipInfo->remoteIPInput);
    	fl_set_object_lcol(fd_tcpipInfo->remoteIPInput,FL_INACTIVE);
	}
	return;
}

void doARNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->ARNoButton)) {
		fd_tcpipInfo->ldata &= ~ACCEPT_REMOTEIP;
		fl_activate_object(fd_tcpipInfo->remoteIPInput);
    	fl_set_object_lcol(fd_tcpipInfo->remoteIPInput,FL_WHITE);
	}
	return;
}

void doNMInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = NETMASK, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, NETMASK, 4);
		fl_set_input(obj, NETMASKSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doPDNSInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = DNS, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, DNS, 4);
		fl_set_input(obj, DNS_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doSDNSInput(FL_OBJECT *obj, long param)
{
	static unsigned char ip[4] = DNS, ips[MAXLEN_IP+1];

	strncpy(ips, (char *)fl_get_input(obj), MAXLEN_IP);
	if (StrToIP(ips, ip) != 4) {
		XBell(fl_get_display(), 50);
		memcpy(ip, DNS, 4);
		fl_set_input(obj, DNS_IPSTR);
		fl_redraw_object(obj);
	}
	fd_tcpipInfo->vdata = ip;
	return;
}

void doDRYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DRYesButton))
		fd_tcpipInfo->ldata |= DEFAULT_ROUTE;
	return;
}

void doDRNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->DRNoButton))
		fd_tcpipInfo->ldata &= ~DEFAULT_ROUTE;
	return;
}

void doUDYesButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->UDYesButton)) {
		fd_tcpipInfo->ldata |= IP_UPDOWN;
		fl_activate_object(fd_tcpipInfo->pDNSInput);
    	fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_WHITE);
		fl_activate_object(fd_tcpipInfo->sDNSInput);
    	fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_WHITE);
	}
	return;
}

void doUDNoButton(FL_OBJECT *obj, long param)
{
	if (fl_get_button(fd_tcpipInfo->UDNoButton)) {
		fd_tcpipInfo->ldata &= ~IP_UPDOWN;
		fl_deactivate_object(fd_tcpipInfo->pDNSInput);
    	fl_set_object_lcol(fd_tcpipInfo->pDNSInput,FL_INACTIVE);
		fl_deactivate_object(fd_tcpipInfo->sDNSInput);
    	fl_set_object_lcol(fd_tcpipInfo->sDNSInput,FL_INACTIVE);
	}
	return;
}

int IPInputFilter(FL_OBJECT *obj, const char *old, const char *cur, int c)
{
	int nd;

	if (strcmp(old, cur)) {
		if (!((c>='0' && c<='9') || c=='.'))
			return FL_INVALID|FL_RINGBELL;
		for (nd=0; *cur; ++cur) if (*cur == '.') ++nd;
		if (nd > 3)
			return FL_INVALID|FL_RINGBELL;
	}
	return FL_VALID;
}

void doMTUInput(FL_OBJECT *obj, long param)
{
	char mtu[MAXDIG_MTRU+1] = {0};
	static unsigned int num = MTU;

	strncpy(mtu, (char *)fl_get_input(obj), MAXDIG_MTRU);
	num = atoi(mtu);
	if (num < 128) {
		num = MTU;
		sprintf(mtu, "%d", num);
		fl_set_input(obj, mtu);
	}
	fd_tcpipInfo->vdata = &num;
	return;
}

void doMRUInput(FL_OBJECT *obj, long param)
{
	char mru[MAXDIG_MTRU+1] = {0};
	static unsigned int num = MRU;

	strncpy(mru, (char *)fl_get_input(obj), MAXDIG_MTRU);
	num = atoi(mru);
	if (num < 128) {
		num = MRU;
		sprintf(mru, "%d", num);
		fl_set_input(obj, mru);
	}
	fd_tcpipInfo->vdata = &num;
	return;
}

void doTcpipInfoOK(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_tcpipInfo->localIPInput);
	memcpy(p_xisprc->localIP, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->remoteIPInput);
	memcpy(p_xisprc->remoteIP, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->NMInput);
	memcpy(p_xisprc->netmask, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->pDNSInput);
	memcpy(p_xisprc->dns1, (char *)fd_tcpipInfo->vdata, 4);
	fl_call_object_callback(fd_tcpipInfo->sDNSInput);
	memcpy(p_xisprc->dns2, (char *)fd_tcpipInfo->vdata, 4);
	p_xisprc->operOpts = (unsigned)fd_tcpipInfo->ldata & 0xFFFF;
	fl_call_object_callback(fd_tcpipInfo->MTUInput);
	p_xisprc->mtu = *(unsigned int *)(fd_tcpipInfo->vdata);
	fl_call_object_callback(fd_tcpipInfo->MRUInput);
	p_xisprc->mru = *(unsigned int *)(fd_tcpipInfo->vdata);
	writeXisprc(rcfname, xispOptions, currentRC);
	fl_hide_form(fd_tcpipInfo->tcpipInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}

void doTcpipInfoCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_tcpipInfo->tcpipInfo);
	fl_activate_object(fd_topFrame->optMenu);
    fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |       Callback routines for exitDialog and window delete handlers       |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doExitOK(FL_OBJECT *obj, long param)
{
	if (dialerPID) {
		kill(dialerPID, SIGTERM);
		mSleep((int)BU_INTERVAL*1000);
	}
	if ((pppdPID = getPppdPID("ppp0")))
		kill(pppdPID, SIGINT);
	exit(0);
}

void doExitCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_exitDialog->exitDialog);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

int deleteHandler(FL_FORM *form, void *data)
{
	if (form == fd_topFrame->topFrame) {
		fl_deactivate_form(fd_topFrame->topFrame);
		fl_show_form(fd_exitDialog->exitDialog, FL_PLACE_MOUSE,
				 	FL_TRANSIENT, "Exit Dialog");
	}
	else if (form == fd_accountInfo->accountInfo) {
		fl_call_object_callback(fd_accountInfo->accCancel);
	}
	else if (form == fd_dialInfo->dialInfo) {
		fl_call_object_callback(fd_dialInfo->dialCancel);
	}
	else if (form == fd_ISPInfo->ISPInfo) {
		fl_call_object_callback(fd_ISPInfo->ISPCancel);
	}
	else if (form == fd_renameISP->renameISP) {
		fl_call_object_callback(fd_renameISP->ISPNameEditCancel);
	}
	else if (form == fd_aboutInfo->aboutInfo) {
		fl_call_object_callback(fd_aboutInfo->aboutOK);
	}
	else if (form == fd_exitDialog->exitDialog) {
		fl_call_object_callback(fd_exitDialog->exitCancel);
	}
	else if (form == fd_helpInfo->helpInfo) {
		fl_call_object_callback(fd_helpInfo->helpInfoOK);
	}
	else if (form == fd_commInfo->commInfo) {
		fl_call_object_callback(fd_commInfo->commInfoOK);
	}
	else if (form == fd_tcpipInfo->tcpipInfo) {
		fl_call_object_callback(fd_tcpipInfo->tcpipInfoOK);
	}
	return FL_IGNORE;
}

void doQuit(FL_OBJECT *obj, long param)
{
	fl_deactivate_form(fd_topFrame->topFrame);
	fl_show_form(fd_exitDialog->exitDialog, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Exit Dialog");
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Command line options & Resources - Routines               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Parse user-specified background color from string */

void bgColor(char *color)
{
	color_t bgcol_default = BGCOL_DEFAULT;
	int i;

	i = sscanf(color, "#%2X%2X%2X",					/* scan the hex color */
			   &bgcol.r, &bgcol.g, &bgcol.b);
	if (i != 3)										/* if scan unsuccessful */
		bgcol = bgcol_default;						/* use the default color */
	fl_mapcolor(FL_INDIANRED,						/* replace the one used */
				bgcol.r, bgcol.g, bgcol.b);
}

/* Process arguments parsed in resources */

void processArgs(void)
{
	int status;

	bgColor(bgcols);								/* parse user bg color */
	status = XParseGeometry(geoms,					/* parse geometry from */
							&winPosX, &winPosY,		/* the input string */
							&winWidth, &winHeight);
	if ((status & XValue) && (status && YValue)) {	/* position specified? */
		fl_set_form_position(fd_topFrame->topFrame,	/* yup, adjust top form */
							 winPosX, winPosY);		/* size and change */
		placementMethod = FL_PLACE_POSITION;		/* adjust placement */
	}
	if ((status & WidthValue) &&					/* size specified? */
		(status & HeightValue))
		fl_set_form_size(fd_topFrame->topFrame,		/* yes, adjust top form */
						 winWidth, winHeight);
	if (iconic)										/* startup as iconic? */
		placementMethod = FL_PLACE_ICONIC;			/* adjust placement */
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                                 Main                                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int main(int argc, char *argv[])
{
	Pixmap icon, mask;
	char *rcsp;

	rcsp = RCSid;
	umask(077);										/* owner read/write only */
	pcode_init();									/* init passwd coding */
	initFnames();									/* initialize file names */
	readAllXisprc(rcfname, xispOptions);			/* try reading RC file */
	dialerFD = namedPipe(pipefname);				/* create/open pipe */
	colorSqueeze();									/* reduce colormap usage */
	fl_initialize(&argc, argv, "xISP",options,3);	/* initialize forms GUI */
	fl_get_app_resources(resources, 3);				/* parse any resources */

	atexit(fl_finish);								/* register cleanup proc */
	fd_topFrame = create_form_topFrame();			/* create all prog forms */
	fd_accountInfo = create_form_accountInfo();
	fd_dialInfo = create_form_dialInfo();
	fd_CBInfo = create_form_CBInfo();
	fd_aboutInfo = create_form_aboutInfo();
	fd_exitDialog = create_form_exitDialog();
	fd_ISPInfo = create_form_ISPInfo();
	fd_renameISP = create_form_renameISP();
	fd_helpInfo = create_form_helpInfo();
	fd_commInfo = create_form_commInfo();
	fd_tcpipInfo = create_form_tcpipInfo();

	processArgs();									/* process arguments */
#ifdef XPMANIMATE
	prepAnimPixmaps();								/* prep animation pixmaps */
	fl_register_raw_callback(fd_topFrame->topFrame,	/* icon animation hook */
							 FL_ALL_EVENT,			/* to figure out when we */
							 mapquery);				/* are iconized */
#endif

	fl_set_atclose(deleteHandler, NULL);			/* fdelete() callback */
	fl_set_app_mainform(fd_topFrame->topFrame);		/* indicate master form */

	fl_show_form(fd_topFrame->topFrame,				/* realize it */
				 placementMethod, FL_FULLBORDER,
				 "X-ISP");
	fl_get_pixmap_pixmap(fd_topFrame->topIcon,		/* use as window icon */
						 &icon, &mask);				/* the same as that on */
	fl_winicon(FL_ObjWin(fd_topFrame->topIcon),		/* the main form */
			   icon, mask);

	possibleActions(DISCONNECTED);					/* mark possible actions */

	fl_do_forms();									/* and enter main loop */

	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |         Program forms created with fdesign and annotated by hand        |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

FD_topFrame *create_form_topFrame(void)
{
  FL_OBJECT *obj;
  FD_topFrame *fdui = (FD_topFrame *) fl_calloc(1, sizeof(*fdui));

  fdui->topFrame = fl_bgn_form(FL_NO_BOX, 372, 299);
  fl_set_form_minsize(fdui->topFrame, 372,299);
  obj = fl_add_box(FL_FLAT_BOX,0,0,372,299,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,13,12,90,40,"X-ISP");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_COL1);
    fl_set_object_lsize(obj,FL_HUGE_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_ENGRAVED_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
 	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthWest);
  fdui->conButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,138,80,30,
										"Connect");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doConnect,0);
	fl_set_object_gravity(obj, FL_West, FL_West);
  fdui->intButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,178,80,30,
										"Interrupt");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doInterrupt,0);
	fl_set_object_gravity(obj, FL_West, FL_West);
  fdui->disButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,218,80,30,
										"Disconnect");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDisconnect,0);
	fl_set_object_gravity(obj, FL_West, FL_West);
  fdui->quitButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,258,80,30,
										 "Quit");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doQuit,0);
	fl_set_object_gravity(obj, FL_West, FL_West);
  obj = fl_add_box(FL_FRAME_BOX,111,10,248,39,"");
	fl_set_object_resize(obj, FL_RESIZE_X);
	fl_set_object_gravity(obj, FL_NorthWest, FL_NorthEast);
  fdui->optMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,120,16,63,27,"Options");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doOptions,0);
	fl_set_object_gravity(obj, FL_North, FL_North);
	fl_set_menu(obj,"ISP Selection|Account Information|"
				"Dialing and Login|Communication Options|TCP/IP Options");
	fl_setpup_default_cursor(XC_arrow);
  fdui->hlpMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,311,16,35,27,"Help");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHelp,0);
	fl_set_object_gravity(obj, FL_North, FL_North);
	fl_set_menu(obj, "About xISP|General Info");
	fl_setpup_default_cursor(XC_arrow);
  fdui->lstBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,111,63,247,108,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_YELLOW);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
	fl_set_browser_hscrollbar(obj, FL_OFF);
	fl_set_browser_scrollbarsize(obj, 24, 24);
	fl_set_object_gravity(obj, FL_NorthWest, FL_SouthEast);
  obj = fl_add_text(FL_NORMAL_TEXT,172,181,37,28,"ISP:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->ISPText = obj = fl_add_text(FL_NORMAL_TEXT,209,183,149,28,
									p_xisprc->descr);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->autoRedial = obj = fl_add_checkbutton(FL_PUSH_BUTTON,145,183,29,25,
											  "ARD:");
    fl_set_object_color(obj,FL_COL1,FL_DARKCYAN);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAutoRedial,0);
	fl_set_object_gravity(obj, FL_South, FL_South);
	fl_set_button(obj, (p_xisprc->operOpts & AUTO_REDIAL)? 1:0);
  obj = fl_add_text(FL_NORMAL_TEXT,107,257,49,28,"Speed:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  obj = fl_add_text(FL_NORMAL_TEXT,206,257,93,28,"Connect Time:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->timeText = obj = fl_add_text(FL_NORMAL_TEXT,299,259,59,28,EMPTY_TIME);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->speedText = obj = fl_add_text(FL_NORMAL_TEXT,155,259,50,29,EMPTY_SPEED);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  obj = fl_add_text(FL_NORMAL_TEXT,107,219,25,28,"IP:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->IPText = obj = fl_add_text(FL_NORMAL_TEXT,132,221,105,28,EMPTY_IP);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
	fl_set_object_posthandler(obj, select_IPText);
  obj = fl_add_text(FL_NORMAL_TEXT,246,219,53,28,"Status:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->statusText = obj = fl_add_text(FL_NORMAL_TEXT,299,221,59,28,"DISCON");
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_South, FL_South);
  fdui->topIcon = obj = fl_add_pixmap(FL_NORMAL_PIXMAP,14,55,78,73,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_object_gravity(obj, FL_West, FL_West);
	fl_set_pixmap_data(obj, xisp_xpm);
#ifdef XPMANIMATE
	fl_set_object_dblbuffer(obj, TRUE);
#endif

	btimer = fl_add_timer(FL_HIDDEN_TIMER,			/* timer for browser */
						  0, 0, 0, 0, NULL);
	ctimer = fl_add_timer(FL_HIDDEN_TIMER,			/* connection time */
						  0, 0, 0, 0, NULL);
	ltimer = fl_add_timer(FL_HIDDEN_TIMER,			/* link status */
						  0, 0, 0, 0, NULL);
#ifdef XPMANIMATE
	xpmtimer = fl_add_timer(FL_HIDDEN_TIMER,		/* pixmap animation */
						  0, 0, 0, 0, NULL);
#endif
    fl_set_object_callback(btimer, doBUpdate, 0);	/* register callbacks */
    fl_set_object_callback(ctimer, doCTupdate, 0);
    fl_set_object_callback(ltimer, doLPupdate, 0);
#ifdef XPMANIMATE
	fl_set_object_callback(xpmtimer, doXPMUpdate, 0);
#endif
	fl_set_timer(btimer, 0.0);						/* initialize timers */
	fl_set_timer(ctimer, 0.0);
	fl_set_timer(ltimer, 0.0);
#ifdef XPMANIMATE
	fl_set_timer(xpmtimer, 0.0);
#endif

  fl_end_form();

  fdui->topFrame->fdui = fdui;

  return fdui;
}

FD_accountInfo *create_form_accountInfo(void)
{
  FL_OBJECT *obj;
  FD_accountInfo *fdui = (FD_accountInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->accountInfo = fl_bgn_form(FL_NO_BOX, 302, 326);
  obj = fl_add_box(FL_FLAT_BOX,0,0,302,326,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->telInput = obj = fl_add_input(FL_NORMAL_INPUT,103,9,186,31,
									  "Phone no(s):");
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTelNoInput,0);
	fl_set_input_maxchars(obj, MAXNUM_TELS*(MAXLEN_PHONE+1));
  fdui->accInput = obj = fl_add_input(FL_NORMAL_INPUT,103,52,186,31,
									  "Account name:");
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInput,0);
	fl_set_input_maxchars(obj, MAXLEN_ACCOUNT);
  fdui->pswInput = obj = fl_add_input(FL_SECRET_INPUT,103,95,186,31,
									  "Password:");
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPasswdInput,0);
	fl_set_input_maxchars(obj, MAXLEN_PASSWD);
  fdui->UNInput = obj = fl_add_input(FL_NORMAL_INPUT,103,138,186,31,
									 "User/Name:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doUNInput,0);
	fl_set_input_maxchars(obj, MAXLEN_UNR);
  fdui->remoteInput = obj = fl_add_input(FL_NORMAL_INPUT,103,181,186,31,
										 "Remote name:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRemoteInput,0);
	fl_set_input_maxchars(obj, MAXLEN_UNR);
  fdui->accOK = obj = fl_add_button(FL_RETURN_BUTTON,204,279,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountOK,0);
  fdui->accCancel = obj = fl_add_button(FL_NORMAL_BUTTON,204,237,81,29,
										"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountCancel,0);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,232,165,81,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,22,221,151,20,"Authentication protocol:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->authType = fl_bgn_group();
  fdui->authPAPButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,21,246,65,
												 25,"PAP");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthPAPButton,0);
  fdui->authNoneButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,92,246,60,
												  25,"None");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthNoneButton,0);
  fdui->authPAPSButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,21,275,72,
												  25,"PAP-S");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthPAPSButton,0);
  fdui->authCHAPSButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,92,275,77,
												   25,"CHAP-S");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAuthCHAPSButton,0);
  fl_end_group();

  fl_end_form();

  fdui->accountInfo->fdui = fdui;

  return fdui;
}

FD_dialInfo *create_form_dialInfo(void)
{
  FL_OBJECT *obj;
  FD_dialInfo *fdui = (FD_dialInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->dialInfo = fl_bgn_form(FL_NO_BOX, 430, 350);
  obj = fl_add_box(FL_FLAT_BOX,0,0,430,350,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,13,20,176,272,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,18,8,96,20,"Dialer options:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,205,20,176,64,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,210,8,92,20,"Manual Login:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->rtrInput = obj = fl_add_input(FL_INT_INPUT,134,33,40,27,
									  "Maximum tries:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRetryInput,0);
	fl_set_input_maxchars(obj, MAXDIG_RETRY);
  obj = fl_add_text(FL_NORMAL_TEXT,49,72,82,23,"Inter-dialing");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,22,89,108,19,"delay (seconds):");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->dlyInput = obj = fl_add_input(FL_INT_INPUT,134,79,40,27,"");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDelayInput,0);
	fl_set_input_maxchars(obj, MAXDIG_DELAY);
  obj = fl_add_text(FL_NORMAL_TEXT,15,121,117,17,"Maximum wait for");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,18,135,113,19,"connection (sec):");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->CNWaitInput = obj = fl_add_input(FL_INT_INPUT,134,125,40,27,"");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCNWaitInput,0);
	fl_set_input_maxchars(obj, MAXDIG_CNWAIT);
  obj = fl_add_text(FL_NORMAL_TEXT,18,166,164,19,"Ring bell when connected:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->RBGroup = fl_bgn_group();
  fdui->RBYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,36,190,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRBYesButton,0);
  fdui->RBNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,116,190,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRBNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,19,224,161,19,"ISP server will call back:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->CBYesButton = obj = fl_add_checkbutton(FL_PUSH_BUTTON,36,250,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBYesButton,0);
  fdui->CBOptions = obj = fl_add_button(FL_NORMAL_BUTTON,109,249,62,29,
										"Options");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBOptions,0);
  obj = fl_add_text(FL_NORMAL_TEXT,210,26,166,19,"Terminal after connection:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->TTGroup = fl_bgn_group();
  fdui->TTYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,228,48,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTTYesButton,0);
  fdui->TTNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,308,48,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTTNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,198,94,107,20,"Automatic Login:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_box(FL_DOWN_BOX,202,133,104,160,"");
    fl_set_object_color(obj,FL_MCOL,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,197,112,60,19,"Expect:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_box(FL_DOWN_BOX,314,133,104,160,"");
    fl_set_object_color(obj,FL_MCOL,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,309,112,46,19,"Send:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->EscriptLine[0] = obj = fl_add_input(FL_NORMAL_INPUT,205,136,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,0);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[0] = obj = fl_add_input(FL_NORMAL_INPUT,317,136,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,8);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[1] = obj = fl_add_input(FL_NORMAL_INPUT,205,155,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,1);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[1] = obj = fl_add_input(FL_NORMAL_INPUT,317,155,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,9);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[2] = obj = fl_add_input(FL_NORMAL_INPUT,205,174,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,2);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[2] = obj = fl_add_input(FL_NORMAL_INPUT,317,174,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,10);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[3] = obj = fl_add_input(FL_NORMAL_INPUT,205,193,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,3);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[3] = obj = fl_add_input(FL_NORMAL_INPUT,317,193,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,11);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[4] = obj = fl_add_input(FL_NORMAL_INPUT,205,212,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,4);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[4] = obj = fl_add_input(FL_NORMAL_INPUT,317,212,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,12);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[5] = obj = fl_add_input(FL_NORMAL_INPUT,205,231,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,5);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[5] = obj = fl_add_input(FL_NORMAL_INPUT,317,231,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,13);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[6] = obj = fl_add_input(FL_NORMAL_INPUT,205,250,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,6);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[6] = obj = fl_add_input(FL_NORMAL_INPUT,317,250,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,14);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->EscriptLine[7] = obj = fl_add_input(FL_NORMAL_INPUT,205,269,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,7);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->SscriptLine[7] = obj = fl_add_input(FL_NORMAL_INPUT,317,269,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,15);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->dialOK = obj = fl_add_button(FL_RETURN_BUTTON,97,307,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDialOK,0);
  fdui->dialCancel = obj = fl_add_button(FL_NORMAL_BUTTON,250,307,81,29,
										 "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDialCancel,0);
  fl_end_form();

  fdui->dialInfo->fdui = fdui;

  return fdui;
}

FD_CBInfo *create_form_CBInfo(void)
{
  FL_OBJECT *obj;
  FD_CBInfo *fdui = (FD_CBInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->CBInfo = fl_bgn_form(FL_NO_BOX, 247, 301);
  obj = fl_add_box(FL_FLAT_BOX,0,0,247,301,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->CBdlyInput = obj = fl_add_input(FL_NORMAL_INPUT,181,10,40,27,
										"Call-back delay (seconds):");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBDelayInput,0);
	fl_set_input_maxchars(obj, MAXDIG_CBDELAY);
  obj = fl_add_text(FL_NORMAL_TEXT,10,40,141,21,"Call-back script lines:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_box(FL_DOWN_BOX,15,85,104,160,"");
    fl_set_object_color(obj,FL_MCOL,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_box(FL_DOWN_BOX,127,85,104,160,"");
    fl_set_object_color(obj,FL_MCOL,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,10,64,60,19,"Expect:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,122,64,46,19,"Send:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->CBEscriptLine[0] = obj = fl_add_input(FL_NORMAL_INPUT,18,88,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,0);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[0] = obj = fl_add_input(FL_NORMAL_INPUT,130,88,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,8);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[1] = obj = fl_add_input(FL_NORMAL_INPUT,18,107,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,1);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[1] = obj = fl_add_input(FL_NORMAL_INPUT,130,107,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,9);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[2] = obj = fl_add_input(FL_NORMAL_INPUT,18,126,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,2);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[2] = obj = fl_add_input(FL_NORMAL_INPUT,130,126,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,10);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[3] = obj = fl_add_input(FL_NORMAL_INPUT,18,145,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,3);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[3] = obj = fl_add_input(FL_NORMAL_INPUT,130,145,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,11);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[4] = obj = fl_add_input(FL_NORMAL_INPUT,18,164,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,4);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[4] = obj = fl_add_input(FL_NORMAL_INPUT,130,164,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,12);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[5] = obj = fl_add_input(FL_NORMAL_INPUT,18,183,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,5);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[5] = obj = fl_add_input(FL_NORMAL_INPUT,130,183,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,13);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[6] = obj = fl_add_input(FL_NORMAL_INPUT,18,202,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,6);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[6] = obj = fl_add_input(FL_NORMAL_INPUT,130,202,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,14);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBEscriptLine[7] = obj = fl_add_input(FL_NORMAL_INPUT,18,221,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,7);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBSscriptLine[7] = obj = fl_add_input(FL_NORMAL_INPUT,130,221,98,21,"");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBScriptInput,15);
	fl_set_input_maxchars(obj, MAXLEN_SLINE);
  fdui->CBInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,26,259,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBInfoOK,0);
  fdui->CBInfoCancel = obj = fl_add_button(FL_NORMAL_BUTTON,139,259,81,29,
										   "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCBInfoCancel,0);
  fl_end_form();

  fdui->CBInfo->fdui = fdui;

  return fdui;
}

FD_aboutInfo *create_form_aboutInfo(void)
{
  FL_OBJECT *obj;
  FD_aboutInfo *fdui = (FD_aboutInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->aboutInfo = fl_bgn_form(FL_NO_BOX, 205, 177);
  obj = fl_add_box(FL_FLAT_BOX,0,0,205,177,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_pixmap(FL_NORMAL_PIXMAP,10,12,78,73,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_pixmap_data(obj, xisp_xpm);
  obj = fl_add_text(FL_NORMAL_TEXT,88,18,74,23,"X-ISP  V.");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,156,17,29,25,Version);
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,122,58,32,23,"by");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,30,80,148,23,"Dimitrios P. Bouras");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,106,37,60,23,"coded");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->aboutOK = obj = fl_add_button(FL_RETURN_BUTTON,64,131,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAboutOK,0);
  obj = fl_add_text(FL_NORMAL_TEXT,29,100,148,23,"dbouras@hol.gr");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fl_end_form();

  fdui->aboutInfo->fdui = fdui;

  return fdui;
}

FD_exitDialog *create_form_exitDialog(void)
{
  FL_OBJECT *obj;
  FD_exitDialog *fdui = (FD_exitDialog *) fl_calloc(1, sizeof(*fdui));

  fdui->exitDialog = fl_bgn_form(FL_NO_BOX, 230, 121);
  obj = fl_add_box(FL_FLAT_BOX,0,0,230,121,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->exitOK = obj = fl_add_button(FL_RETURN_BUTTON,19,75,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExitOK,0);
  fdui->exitCancel = obj = fl_add_button(FL_NORMAL_BUTTON,132,75,81,29,
										 "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExitCancel,0);
  obj = fl_add_text(FL_NORMAL_TEXT,27,16,183,22,"Are you sure you want to");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,28,40,179,20,"EXIT   X-ISP?");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fl_end_form();

  fdui->exitDialog->fdui = fdui;

  return fdui;
}

FD_ISPInfo *create_form_ISPInfo(void)
{
  FL_OBJECT *obj;
  FD_ISPInfo *fdui = (FD_ISPInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->ISPInfo = fl_bgn_form(FL_NO_BOX, 218, 222);
  obj = fl_add_box(FL_FLAT_BOX,0,0,218,222,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,9,20,67,20,"Pick ISP:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->ISPBrowser = obj = fl_add_browser(FL_HOLD_BROWSER,13,49,192,121,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_YELLOW);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPPick,0);
    fl_set_browser_dblclick_callback(obj,doISPPickNEdit,0);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
  fdui->ISPOK = obj = fl_add_button(FL_RETURN_BUTTON,13,181,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPOK,0);
  fdui->ISPCancel = obj = fl_add_button(FL_NORMAL_BUTTON,124,181,81,29,
										"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPCancel,0);
  fdui->ISPNameEdit = obj = fl_add_button(FL_NORMAL_BUTTON,124,10,81,29,
										  "Rename");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEdit,0);
  fl_end_form();

  fdui->ISPInfo->fdui = fdui;

  return fdui;
}

FD_renameISP *create_form_renameISP(void)
{
  FL_OBJECT *obj;
  FD_renameISP *fdui = (FD_renameISP *) fl_calloc(1, sizeof(*fdui));

  fdui->renameISP = fl_bgn_form(FL_NO_BOX, 219, 112);
  obj = fl_add_box(FL_FLAT_BOX,0,0,219,112,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->ISPNameInput = obj = fl_add_input(FL_NORMAL_INPUT,12,27,194,31,
										  "Enter ISP description:");
    fl_set_input_maxchars(obj, MAXLEN_DESCR);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameInput,0);
  fdui->ISPNameEditOK = obj = fl_add_button(FL_RETURN_BUTTON,12,70,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEditOK,0);
  fdui->ISPNameEditCancel = obj = fl_add_button(FL_NORMAL_BUTTON,125,70,81,29,
												"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEditCancel,0);
  fl_end_form();

  fdui->renameISP->fdui = fdui;

  return fdui;
}

FD_helpInfo *create_form_helpInfo(void)
{
  FL_OBJECT *obj;
  FD_helpInfo *fdui = (FD_helpInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->helpInfo = fl_bgn_form(FL_NO_BOX, 507, 476);
  obj = fl_add_box(FL_FLAT_BOX,0,0,507,476,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->helpBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,10,11,486,419,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_YELLOW);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
  fdui->helpInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,416,439,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHelpInfoOK,0);
  fl_end_form();

  fdui->helpInfo->fdui = fdui;

  return fdui;
}

FD_commInfo *create_form_commInfo(void)
{
  FL_OBJECT *obj;
  FD_commInfo *fdui = (FD_commInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->commInfo = fl_bgn_form(FL_NO_BOX, 409, 379);
  obj = fl_add_box(FL_FLAT_BOX,0,0,409,379,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,14,132,180,107,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,14,259,180,61,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,213,239,180,81,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,213,22,180,51,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,213,97,180,78,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  fdui->commInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,101,336,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCommInfoOK,0);
  fdui->commInfoCancel = obj = fl_add_button(FL_NORMAL_BUTTON,224,336,81,29,
											 "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doCommInfoCancel,0);
  fdui->deviceInput = obj = fl_add_input(FL_NORMAL_INPUT,62,15,133,28,
										 "Device:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDeviceInput,0);
	fl_set_input_maxchars(obj, MAXLEN_DEVICE);
  fdui->modemResetInput = obj = fl_add_input(FL_NORMAL_INPUT,56,50,139,28,
											 "Reset:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doModemResetInput,0);
  fdui->modemInitInput = obj = fl_add_input(FL_NORMAL_INPUT,41,85,154,28,
											"Init:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doModemInitInput,0);
	fl_set_input_maxchars(obj, MAXLEN_MODEM);
  obj = fl_add_text(FL_NORMAL_TEXT,21,121,78,20,"Baud Rate:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->modemSpeedType = fl_bgn_group();
  fdui->speedButton[6] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,108,185,65,
												  25,"57600");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,6);
  fdui->speedButton[7] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,108,207,65,
												  25,"115200");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,7);
  fdui->speedButton[4] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,108,141,65,
												  25,"19200");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,4);
  fdui->speedButton[5] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,108,163,65,
												  25,"38400");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,5);
  fdui->speedButton[2] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,27,185,65,
												  25,"4800");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,2);
  fdui->speedButton[3] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,27,207,65,
												  25,"9600");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,3);
  fdui->speedButton[1] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,27,163,65,
												  25,"2400");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,1);
  fdui->speedButton[0] = obj = fl_add_checkbutton(FL_RADIO_BUTTON,27,141,65,
												  25,"1200");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSpeedButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,21,248,89,20,"Flow control:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->flowCtrlType = fl_bgn_group();
  fdui->HWButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,27,267,160,25,
											"Hardware  RTS/CTS");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHWButton,0);
  fdui->SWButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,27,289,160,25,
											"Software  XON/XOFF");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,220,11,102,20,"Dialing method:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->modemDialType = fl_bgn_group();
  fdui->toneButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,230,35,65,25,
											  "Tone");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doToneButton,0);
  fdui->pulseButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,311,35,65,25,
											   "Pulse");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPulseButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,220,86,112,20,"SW Compression:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->SWCType = fl_bgn_group();
  fdui->SWCOnButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,230,108,65,25,
											   "On");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCOnButton,0);
  fdui->SWCOffButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,311,108,65,25,
												"Off");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCOffButton,0);
  fl_end_group();

  fdui->SWCInput = obj = fl_add_input(FL_INT_INPUT,328,136,34,26,
									  "Level (9-15): ");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSWCInput,0);
	fl_set_input_maxchars(obj, MAXDIG_BSDCOMP);
	fl_set_input_filter(obj, SWCInputFilter);
  fdui->asyncmapInput = obj = fl_add_input(FL_NORMAL_INPUT,285,192,108,28,
										   "Asyncmap:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAsyncmapInput,0);
	fl_set_input_maxchars(obj, MAXDIG_ASYNCMAP);
	fl_set_input_filter(obj, AMInputFilter);
  obj = fl_add_text(FL_NORMAL_TEXT,219,224,56,29,"Escape:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->escapeType = fl_bgn_group();
  fdui->escapeYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,229,250,65,
												   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doEscapeYesButton,0);
    fl_set_button(obj, 1);
  fdui->escapeNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,310,250,65,
												  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doEscapeNoButton,0);
  fl_end_group();

  fdui->escapeInput = obj = fl_add_input(FL_NORMAL_INPUT,291,278,85,28,
										 "Byte List:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doEscapeInput,0);
	fl_set_input_maxchars(obj, MAXLEN_ESCAPE);
	fl_set_input_filter(obj, EInputFilter);
  fl_end_form();

  fdui->commInfo->fdui = fdui;

  return fdui;
}

FD_tcpipInfo *create_form_tcpipInfo(void)
{
  FL_OBJECT *obj;
  FD_tcpipInfo *fdui = (FD_tcpipInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->tcpipInfo = fl_bgn_form(FL_NO_BOX, 390, 346);
  obj = fl_add_box(FL_FLAT_BOX,0,0,390,346,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,19,360,44,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,82,360,44,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,146,163,44,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_frame(FL_ENGRAVED_FRAME,15,210,360,80,"");
    fl_set_object_color(obj,FL_COL1,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,20,8,148,20,"Dynamic local address:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->localIPInput = obj = fl_add_input(FL_NORMAL_INPUT,253,27,113,28,
										  "Local IP:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doLocalIPInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);

  fdui->ALTypeGroup = fl_bgn_group();
  fdui->ALYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,32,29,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doALYesButton,0);
  fdui->ALNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,104,29,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doALNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,20,71,162,20,"Dynamic remote address:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
  fdui->remoteIPInput = obj = fl_add_input(FL_NORMAL_INPUT,252,90,113,28,
										   "Remote IP:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRemoteIPInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);

  fdui->ARTypeGroup = fl_bgn_group();
  fdui->ARYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,32,92,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doARYesButton,0);
  fdui->ARNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,104,92,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doARNoButton,0);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,20,135,117,20,"Add default route:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->DRTypeGroup = fl_bgn_group();
  fdui->DRYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,32,156,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDRYesButton,0);
  fdui->DRNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,104,156,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDRNoButton,0);
  fl_end_group();

  fdui->NMInput = obj = fl_add_input(FL_NORMAL_INPUT,263,136,113,28,
									 "Netmask:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doNMInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  fdui->MTUInput = obj = fl_add_input(FL_INT_INPUT,228,171,48,28,"MTU:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMTUInput,0);
	fl_set_input_maxchars(obj, MAXDIG_MTRU);
  fdui->MRUInput = obj = fl_add_input(FL_INT_INPUT,328,171,48,28,"MRU:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doMRUInput,0);
	fl_set_input_maxchars(obj, MAXDIG_MTRU);
  obj = fl_add_text(FL_NORMAL_TEXT,20,199,151,20,"Support ip-up/ip-down:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  fdui->UDTypeGroup = fl_bgn_group();
  fdui->UDYesButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,32,222,58,
											   25,"Yes");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doUDYesButton,0);
  fdui->UDNoButton = obj = fl_add_checkbutton(FL_RADIO_BUTTON,32,253,55,
											  25,"No");
    fl_set_object_color(obj,FL_MCOL,FL_DARKCYAN);
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doUDNoButton,0);
  fl_end_group();

  fdui->pDNSInput = obj = fl_add_input(FL_NORMAL_INPUT,252,219,113,28,
									   "Primary DNS Server:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPDNSInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  fdui->sDNSInput = obj = fl_add_input(FL_NORMAL_INPUT,252,253,113,28,
									   "Secondary DNS Server:");
    fl_set_object_lcolor(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doSDNSInput,0);
	fl_set_input_maxchars(obj, MAXLEN_IP);
	fl_set_input_filter(obj, IPInputFilter);
  fdui->tcpipInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,92,304,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTcpipInfoOK,0);
  fdui->tcpipInfoCancel = obj = fl_add_button(FL_NORMAL_BUTTON,215,304,81,29,
											  "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTcpipInfoCancel,0);
  fl_end_form();

  fdui->tcpipInfo->fdui = fdui;

  return fdui;
}

