// KPPPLoad - a PPP load monitor
// Copyright (C) 1998  Sean Vyain, svyain@mail.tds.net
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <qpainter.h>

#include "Graph.h"
#include "Options.h"

Graph::Graph( QWidget* parent, const char* name )
        : QWidget( parent, name ),
          _historySize( 0 ),
          _txHistory( 0 ),
          _rxHistory( 0 ),
          _txAverage( 0 ),
          _rxAverage( 0 ),
          _max( 1 )
{
    // Allocate data arrays.
    _dataSize = options->getStatsPeriod() / options->getStatsInterval();
    _txData = new int[ _dataSize ];
    _rxData = new int[ _dataSize ];
    for ( int i = 0; i < _dataSize; i++ ) {
        _txData[i] = 0;
        _rxData[i] = 0;
    }

    connect( options, SIGNAL( sigRateDivider() ), this, SLOT( redrawGraph() ) );
    connect( options, SIGNAL( sigRxColor() )    , this, SLOT( redrawGraph() ) );
    connect( options, SIGNAL( sigTxColor() )    , this, SLOT( redrawGraph() ) );
}

Graph::~Graph()
{
}

void Graph::updateStats( uint rxDelta, uint txDelta, uint, uint )
{
    int oldMax = _max;

    // Compute the new transmit and receive load averages
    _txAverage = _txAverage - _txData[_dataSize-1] + txDelta;
    _rxAverage = _rxAverage - _rxData[_dataSize-1] + rxDelta;
	
    // Shift the transmit and receive data.
    for ( int i = _dataSize-1; i > 0; i-- ) {
        _txData[i] = _txData[i-1];
        _rxData[i] = _rxData[i-1];
    }
    _txData[0] = txDelta;
    _rxData[0] = rxDelta;
	
    // Find a new maximum, if needed.
    if (( _txHistory[_historySize-1] == _max ) || ( _rxHistory[_historySize-1] == _max )) {
        _max = 1;
        for ( int i = 0; i < _historySize-1; i++ ) {
            if ( _txHistory[i] > _max ) _max = _txHistory[i];
            if ( _rxHistory[i] > _max ) _max = _rxHistory[i];
        }
    }

    // Shift the history by one.
    for ( int i = _historySize-1; i > 0; i-- ) {
        _txHistory[i] = _txHistory[i-1];
        _rxHistory[i] = _rxHistory[i-1];
    }
	
    // Store the new transfer rates.
    _txHistory[0] = _txAverage / _dataSize;
    _rxHistory[0] = _rxAverage / _dataSize;
	
    // Update the maximum rate (again).
    if ( _txHistory[0] > _max ) _max = _txHistory[0];
    if ( _rxHistory[0] > _max ) _max = _rxHistory[0];

    // Rescale, if needed.
    if ( oldMax != _max ) {
        redrawGraph();
    } else {
        updateGraph();
    }
    paintEvent( 0 );
}

void Graph::linkDown()
{
    for ( int i = 0; i < _dataSize; i++ ) {
        _txData[i] = 0;
        _rxData[i] = 0;
    }
	
    for ( int i = 0; i < _historySize; i++ ) {
        _txHistory[i] = 0;
        _rxHistory[i] = 0;
    }
	
    _max = 1;
    _txAverage = 0;
    _rxAverage = 0;

    redrawGraph();
    paintEvent( 0 );
}

void Graph::resizeEvent( QResizeEvent* )
{
    int* txNew = new int[ width() ];
    int* rxNew = new int[ width() ];
	
    for ( int i = 0; (i < _historySize) && (i < width()); i++ ) {
        txNew[i] = _txHistory[i];
        rxNew[i] = _rxHistory[i];
    }
    for ( int i = _historySize; i < width(); i++ ) {
        txNew[i] = 0;
        rxNew[i] = 0;
    }
	
    delete [] _txHistory;
    delete [] _rxHistory;

    _historySize = width();
    _txHistory = txNew;
    _rxHistory = rxNew;
	
    // Redraw the graph.
    _pixmap.resize( width(), height() );
    redrawGraph();
}

void Graph::paintEvent( QPaintEvent* )
{
    bitBlt( this, 0, 0, &_pixmap, 0, 0, width(), height(), CopyROP );
}

void Graph::mousePressEvent( QMouseEvent* e )
{
    if ( e->button() == RightButton ) {
        emit popupMenu( mapToGlobal( e->pos() ) );
    }
}

void Graph::redrawGraph()
{
    QPainter p;
    p.begin( &_pixmap );
    p.setBackgroundColor( backgroundColor() );
    p.eraseRect( 0, 0, width(), height() );

    // Draw dividing lines.
    p.setPen( black );
    for ( int i = options->getRateDivider() * options->getStatsInterval(); i < _max;
          i += options->getRateDivider() * options->getStatsInterval() ) {
        p.drawLine( 0, height() - i * height() / _max, width(), height() - i * height() / _max );
    }

    // Plot data.
    for ( int i = 1; i < _historySize; i++ ) {
        p.setPen( options->getRxColor() );
        p.drawLine( width() - i - 1, height() - _rxHistory[i-1] * height() / _max, width() - i - 2, height() - _rxHistory[i] * height() / _max );
        p.drawLine( width() - i - 1, height() - _rxHistory[i-1] * height() / _max + 1, width() - i - 2, height() - _rxHistory[i] * height() / _max + 1 );
        p.setPen( options->getTxColor() );
        p.drawLine( width() - i - 1, height() - _txHistory[i-1] * height() / _max, width() - i - 2, height() - _txHistory[i] * height() / _max );
        p.drawLine( width() - i - 1, height() - _txHistory[i-1] * height() / _max + 1, width() - i - 2, height() - _txHistory[i] * height() / _max + 1 );
    }

    p.end();
}

void Graph::updateGraph()
{
    // Move the existing data over one pixel.
    bitBlt( &_pixmap, 0, 0, &_pixmap, 1, 0, width() - 1, height(), CopyROP );
    
    QPainter p;
    p.begin( &_pixmap );
    p.setBackgroundColor( backgroundColor() );
    p.eraseRect( width() - 1, 0, width() - 1, height() );

    // Draw dividing lines.
    p.setPen( black );
    for ( int i = options->getRateDivider() * options->getStatsInterval(); i < _max;
          i += options->getRateDivider() * options->getStatsInterval() ) {
        p.drawPoint( width()-1, height() - i * height() / _max );
    }

    // Plot data.
    p.setPen( options->getRxColor() );
    p.drawLine( width() - 1, height() - _rxHistory[0] * height() / _max, width() - 2, height() - _rxHistory[1] * height() / _max );
    p.drawLine( width() - 1, height() - _rxHistory[0] * height() / _max + 1, width() - 2, height() - _rxHistory[1] * height() / _max + 1 );
    p.setPen( options->getTxColor() );
    p.drawLine( width() - 1, height() - _txHistory[0] * height() / _max, width() - 2, height() - _txHistory[1] * height() / _max );
    p.drawLine( width() - 1, height() - _txHistory[0] * height() / _max + 1, width() - 2, height() - _txHistory[1] * height() / _max + 1 );

    p.end();
}
