(***********************************************************************)
(*                                                                     *)
(*                           The V6 Engine                             *)
(*                                                                     *)
(*          Francois Rouaix, projet Cristal, INRIA Rocquencourt        *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: urlenc.ml,v 1.1 1996/10/22 13:13:13 rouaix Exp $ *)
open Mstring

let hexchar c = 
  let s = String.make 3 '%'
  and i = Char.code c in
  s.[1] <- dec_to_hex (i/16);
  s.[2] <- dec_to_hex (i mod 16);
  s

(* Decode escaped characters *)
(* Note: beware of order of splitting wrt '&' and decoding *)
let decode s =
  let l = String.length s in
  let target = Ebuffer.create l in
  let pos = ref 0 in
  while !pos < l do
    if s.[!pos] = '%' & !pos + 2 < l  then begin
      let c = 16 * hex_to_dec s.[!pos+1] + hex_to_dec s.[!pos+2] in
      	Ebuffer.output_char target (Char.chr c);
	pos := !pos + 3
	end
    else if s.[!pos] = '+' then begin
      Ebuffer.output_char target ' ';
      incr pos
      end
    else begin
      Ebuffer.output_char target s.[!pos];
      incr pos
      end
  done;
  Ebuffer.get target

let encode s =
  let target = Ebuffer.create (String.length s) in
  for pos = 0 to String.length s - 1 do
    match s.[pos] with
      ' ' -> Ebuffer.output_char target '+'
    | '0'..'9' | 'a'..'z' | 'A'..'Z' as c -> Ebuffer.output_char target c
    | '\n' -> Ebuffer.output_string target "%0D%0A"
    | c -> Ebuffer.output_string target (hexchar c)
    done;
  Ebuffer.get target

let form_encode = function 
  [] -> ""
 | (e,v)::l ->
  let b = Ebuffer.create 512 in
    Ebuffer.reset b;
    Ebuffer.output_string b (encode e);
    Ebuffer.output_char b '=';
    Ebuffer.output_string b (encode v);
    List.iter (fun (e,v) ->
		 Ebuffer.output_char b '&';
		 Ebuffer.output_string b (encode e);
		 Ebuffer.output_char b '=';
		 Ebuffer.output_string b (encode v))
             l;
    Ebuffer.get b

let form_decode =
  let ampersand c = c = '&' and equals c = c = '=' in
  (function  s ->
     List.map (fun encp ->
       match split_str equals encp with
	   [x;y] -> (decode x, decode y)
	 | [x] -> (decode x, "")
	 | _ -> invalid_arg "form_decode")
       (split_str ampersand s))
		 
