/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** termNodeList.c
**
** based on list_template.c
**
** where T has T_equal (or change this) and T_unparse
*/

# include "lclintMacros.nf"
# include "llbasic.h"

termNodeList termNodeList_new ()
{
  termNodeList s = (termNodeList) dmalloc (sizeof (*s));
  
  s->nelements = 0;
  s->nspacelow = termNodeListGROWLOW;
  s->nspacehigh = termNodeListGROWHI;
  s->elements = (termNode *) dmalloc (sizeof (*s->elements) * termNodeListBASESIZE);
  s->elements += termNodeListGROWLOW;
  s->current = 0;

  return (s);
}

static void
termNodeList_grow (termNodeList s)
{
  int i;
  termNode *newelements = (termNode *) dmalloc (sizeof (*newelements)
						* (s->nelements + termNodeListBASESIZE));

  newelements += termNodeListGROWLOW;
    
  for (i = 0; i < s->nelements; i++)
    {
      newelements[i] = s->elements[i];
    }
  
  sfree (s->elements - s->nspacelow); 
  s->elements = newelements;
  s->nspacelow = termNodeListGROWLOW;
  s->nspacehigh = termNodeListGROWHI; 

}

void 
termNodeList_addh (termNodeList s, termNode el)
{
  llassert (termNodeListGROWHI > 0);

  if (s->nspacehigh <= 0)
    termNodeList_grow (s);

  s->nspacehigh--;
  s->elements[s->nelements] = el;
  s->nelements++;
}

termNodeList 
termNodeList_push (termNodeList s, termNode el)
{
  termNodeList_addh (s, el);
  return s;
}

void 
termNodeList_addl (termNodeList s, termNode el)
{
  llassert (termNodeListGROWLOW > 0);

  if (s->nspacelow <= 0)
    termNodeList_grow (s);

  s->nspacelow--;
  s->elements--;
  s->elements[0] = el;
  s->current++;
  s->nelements++;
}

void 
termNodeList_reset (termNodeList s)
{
  s->current = 0;
}

void 
termNodeList_finish (termNodeList s)
{
  s->current = s->nelements - 1;
}

void 
termNodeList_advance (termNodeList s)
{
  s->current++;
  llassert (s->current < s->nelements);
}

/*@exposed@*/ termNode 
termNodeList_head (termNodeList s)
{
  llassert (s->nelements > 0);
  return (s->elements[0]);
}

/*@only@*/ termNodeList 
termNodeList_copy (termNodeList s)
{
  termNodeList r = termNodeList_new ();

  termNodeList_elements (s, x)
  {
    termNodeList_addh (r, termNode_copySafe (x));
  } end_termNodeList_elements;

  return r;
}

/*@exposed@*/ termNode 
termNodeList_current (termNodeList s)
{
  llassert (!(s->current >= s->nelements));
  return (s->elements[s->current]);
}

termNode 
termNodeList_getN (termNodeList s, int n)
{
  llassert (n >= 0 && n < s->nelements);

  return (s->elements[n]);
}

/*@only@*/ cstring
termNodeList_unparse (termNodeList s)
{
  bool first = TRUE;
  cstring st = cstring_undefined;

  termNodeList_elements (s, current)
  {
    if (first)
      {
	st = termNode_unparse (current);
	first = FALSE;
      }
    else
      st = message ("%q, %q", st, termNode_unparse (current));
  } end_termNodeList_elements;

  return st;
}

/*@only@*/ cstring
termNodeList_unparseTail (termNodeList s)
{
  bool head = TRUE;
  bool first = TRUE;
  cstring st = cstring_undefined;

  termNodeList_elements (s, current)
  {
    if (head)
      {
	head = FALSE;
      }
    else
      {
	if (first)
	  {
	    st = termNode_unparse (current);
	    first = FALSE;
	  }
	else
	  st = message ("%q, %q", st, termNode_unparse (current));
      }
  } end_termNodeList_elements;

  return st;
}

/*@only@*/ cstring
termNodeList_unparseToCurrent (termNodeList s)
{
  int i;
  cstring st = cstring_undefined;

  for (i = 0; i < s->current; i++)
    {
      termNode current = s->elements[i];

      if (i == 0)
	st = termNode_unparse (current);
      else
	st = message ("%q, %q", st, termNode_unparse (current));
    }

  return st;
}

/*@only@*/ cstring
termNodeList_unparseSecondToCurrent (termNodeList s)
{
  int i;
  cstring st = cstring_undefined;

  for (i = 1; i < s->current; i++)
    {
      termNode current = s->elements[i];

      if (i == 1)
	{
	  st = termNode_unparse (current);
	}
      else
	{
	  st = message ("%q, %q", st, termNode_unparse (current));
	}
    }

  return st;
}

void
termNodeList_free (termNodeList s)
{
  int i;
  for (i = 0; i < s->nelements; i++)
    {
      termNode_free (s->elements[i]); 
    }

  sfree (s->elements - s->nspacelow);
  sfree (s);
}
