/*
 * leap-frog point to allow dmalloc on/off via relink.
 *
 * Copyright 1995 by Gray Watson
 *
 * This file is part of the dmalloc package.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * NON-COMMERCIAL purpose and without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies, and that the name of Gray Watson not be used in
 * advertising or publicity pertaining to distribution of the document
 * or software without specific, written prior permission.
 *
 * Please see the PERMISSIONS file or contact the author for information
 * about commercial licenses.
 *
 * Gray Watson makes no representations about the suitability of the
 * software described herein for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * The author may be contacted at gray.watson@letters.com
 */

/*
 * If anyone can think of a better way to do this *please* let me
 * know.
 *
 * The goal of the routines in this file is to allow people to use the
 * debug library during development by including the dmalloc.h file in
 * their C source files, and then disable and return to the system (or
 * more efficient) malloc functions easily.
 *
 * The dmalloc.h file provides the library with file/line information
 * with each call.  Backtracing the stack-frame is neither portable or
 * necessarily easy so the cpp __FILE__ and __LINE__ directives are
 * used instead to do this.
 *
 * dmalloc.h contains macros which override the malloc, calloc, free,
 * etc. functions so that they call _malloc_leap passing the file and
 * line information to these routines.  But we need to provide a
 * "quick-release" functionality so that people will not have to
 * remove the dmalloc.h includes and recompile to use other malloc
 * libraries.
 *
 * I have decided on the leap-frog routines in this file that will have
 * to *always* be compiled in if you include dmalloc.h in your source.
 * When your source wants to call malloc, it will first make a call to
 * _malloc_leap which will in turn call the malloc of your choice.
 * This does mean that an extra function call per memory interaction
 * will occur, but on most systems this is reasonably cheap.  To fully
 * disable the library, you will need to remove the dmalloc.h include
 * and recompile your source files.
 *
 * Please mail me with any reasonable ideas.
 */

#define DMALLOC_DISABLE

#include "dmalloc.h"				/* for DMALLOC_SIZE... */
#include "conf.h"				/* for const */

#include "dmalloc_loc.h"			/* for EXPORT */
#include "dmalloc_lp.h"

#if INCLUDE_RCS_IDS
static	char	*rcs_id =
  "$Id: dmalloc_lp.c,v 1.30 1995/09/06 17:33:56 gray Exp $";
#endif

/*
 * exported variables
 */
/* to inform the dmalloc library from which file the call comes from */
EXPORT	char		*_dmalloc_file = DMALLOC_DEFAULT_FILE;

/* to inform the library from which line-number the call comes from */
EXPORT	unsigned int	_dmalloc_line = DMALLOC_DEFAULT_LINE;

/* pre-set dmalloc_debug() value before the library is setup */
EXPORT	int		_dmalloc_debug_preset = DEBUG_PRE_NONE;

/* pointers to shutdown function to allow calls without linked routine */
EXPORT	void		(*_dmalloc_shutdown_func)() = NULL;

/* pointers to log_heap_map function to allow calls without linked routine */
EXPORT	void		(*_dmalloc_log_heap_map_func)() = NULL;

/* pointers to log_stats function to allow calls without linked routine */
EXPORT	void		(*_dmalloc_log_stats_func)() = NULL;

/* pointers to log_unfreed function to allow calls without linked routine */
EXPORT	void		(*_dmalloc_log_unfreed_func)() = NULL;

/* pointers to verify function to allow calls without linked routine */
EXPORT	int		(*_dmalloc_verify_func)() = NULL;

/* pointers to verify function to allow calls without linked routine */
EXPORT	int		(*_malloc_verify_func)() = NULL;

/* pointers to debug function to allow calls without linked routine */
EXPORT	void		(*_dmalloc_debug_func)() = NULL;

/* pointers to debug_current function to allow calls without linked routine */
EXPORT	int		(*_dmalloc_debug_current_func)() = NULL;

/* pointers to examine function to allow calls without linked routine */
EXPORT	int		(*_dmalloc_examine_func)() = NULL;

/* pointers to strerror function to allow calls without linked routine */
EXPORT	char		*(*_dmalloc_strerror_func)() = NULL;

#undef malloc
/*
 * leap routine to malloc
 */
EXPORT	DMALLOC_PNT	_malloc_leap(const char * file, const int line,
				     DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = malloc(size);
  
  return ret;
}

#undef calloc
/*
 * leap routine to calloc
 */
EXPORT	DMALLOC_PNT	_calloc_leap(const char * file, const int line,
			      DMALLOC_SIZE elen, DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = calloc(elen, size);
  
  return ret;
}

#undef realloc
/*
 * leap routine to realloc
 */
EXPORT	DMALLOC_PNT	_realloc_leap(const char * file, const int line,
				      DMALLOC_PNT oldp, DMALLOC_SIZE new_size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = realloc(oldp, new_size);
  
  return ret;
}

#undef free
/*
 * leap routine to free
 */
EXPORT	DMALLOC_FREE_RET	_free_leap(const char * file, const int line,
					   DMALLOC_PNT pnt)
{
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
#if defined(__STDC__) && __STDC__ == 1
  free(pnt);
#else
  return free(pnt);
#endif
}

/***************************** xmalloc functions *****************************/

/*
 * leap routine to calloc with error checking
 */
EXPORT	DMALLOC_PNT	_xcalloc_leap(const char * file, const int line,
				      DMALLOC_SIZE elen, DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = calloc(elen, size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine to free
 */
EXPORT	DMALLOC_FREE_RET	_xfree_leap(const char * file, const int line,
					    DMALLOC_PNT pnt)
{
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
#if defined(__STDC__) && __STDC__ == 1
  free(pnt);
#else
  return free(pnt);
#endif
}

/*
 * leap routine to malloc with error checking
 */
EXPORT	DMALLOC_PNT	_xmalloc_leap(const char * file, const int line,
				      DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = malloc(size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine to realloc with error checking
 */
EXPORT	DMALLOC_PNT	_xrealloc_leap(const char * file, const int line,
				       DMALLOC_PNT oldp, DMALLOC_SIZE new_size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = realloc(oldp, new_size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  new_size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine for strdup with error checking
 */
EXPORT	char 	*_xstrdup_leap(const char * file, const int line,
			       const char * str)
{
  char 	*buf;
  int	len;
  
  /* len + NULLC */
  len = strlen(str) + 1;
  
  /* xmalloc checks for NULL */
  buf = (char *)_xmalloc_leap(file, line, len);
  (void)strcpy(buf, str);
  
  return buf;
}

/*********************** routines when running dmalloc ***********************/

/*
 * routine to call dmalloc_shutdown when linked in
 */
EXPORT	void	dmalloc_shutdown(void)
{
  if (_dmalloc_shutdown_func != NULL)
    _dmalloc_shutdown_func();
}

/*
 * routine to call dmalloc_log_unfreed when linked in
 */
EXPORT	void	dmalloc_log_unfreed(void)
{
  if (_dmalloc_log_unfreed_func != NULL)
    _dmalloc_log_unfreed_func();
}

/*
 * routine to call dmalloc_log_stats when linked in
 */
EXPORT	void	dmalloc_log_stats(void)
{
  if (_dmalloc_log_stats_func != NULL)
    _dmalloc_log_stats_func();
}

/*
 * routine to call dmalloc_log_heap_map when linked in
 */
EXPORT	void	dmalloc_log_heap_map(void)
{
  if (_dmalloc_log_heap_map_func != NULL)
    _dmalloc_log_heap_map_func();
}

/*
 * routine to call dmalloc_verify when linked in
 */
EXPORT	int	dmalloc_verify(const DMALLOC_PNT pnt)
{
  if (_dmalloc_verify_func != NULL)
    return _dmalloc_verify_func(pnt);
  else
    return DMALLOC_VERIFY_NOERROR;
}

/*
 * routine to call malloc_verify when linked in
 */
EXPORT	int	malloc_verify(const DMALLOC_PNT pnt)
{
  if (_malloc_verify_func != NULL)
    return _malloc_verify_func(pnt);
  else
    return DMALLOC_VERIFY_NOERROR;
}

/*
 * routine to call dmalloc_debug when linked in
 */
EXPORT	void	dmalloc_debug(const int debug)
{
  if (_dmalloc_debug_func == NULL)
    _dmalloc_debug_preset = debug;
  else
    _dmalloc_debug_func(debug);
}

/*
 * routine to call dmalloc_debug_current when linked in
 */
EXPORT	int	dmalloc_debug_current(void)
{
  if (_dmalloc_debug_current_func != NULL)
    return _dmalloc_debug_current_func();
  else
    return 0;
}

/*
 * routine to call dmalloc_examine when linked in
 */
EXPORT	int	dmalloc_examine(const DMALLOC_PNT pnt, DMALLOC_SIZE * size,
				char ** file, unsigned int * line,
				DMALLOC_PNT * ret_attr)
{
  if (_dmalloc_examine_func != NULL)
    return _dmalloc_examine_func(pnt, size, file, line, ret_attr);
  else
    return ERROR;
}

/*
 * routine to call dmalloc_strerror when linked in
 */
EXPORT	char	*dmalloc_strerror(const int errnum)
{
  if (_dmalloc_strerror_func != NULL)
    return _dmalloc_strerror_func(errnum);
  else
    return "unknown";
}
