/* code to open or fetch FITS files for skyview.
 * this includes the net connections to STScI and ESO.
 */

#include <stdio.h>
#include <math.h>

#if defined(__STDC__)
#include <stdlib.h>
#include <string.h>
#else
extern void *malloc();
#endif

#if defined(_POSIX_SOURCE)
#include <unistd.h>
#else
extern int read();
extern int write();
extern int close();
#endif

#if defined(VMS) && !defined(O_WRONLY)
#include <file.h>
#endif
#if !defined(O_WRONLY) || !defined(O_CREAT)
#include <fcntl.h>
#endif


#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/DrawingA.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include <Xm/TextF.h>
#include <Xm/Scale.h>
#include <Xm/FileSB.h>
#include <Xm/MessageB.h>
#include <Xm/PanedW.h>
#include <Xm/Text.h>


#include "P_.h"
#include "astro.h"
#include "fits.h"
#include "net.h"

#define	MAXDSSFOV	(20.)	/* max field size we retreive, arcmins*/
#define	MINDSSFOV	(5.)	/* min field size we retreive, arcmins*/

extern Widget toplevel_w;
extern XtAppContext xe_app;
#define XtD XtDisplay(toplevel_w)
extern Colormap xe_cm;

extern char *getXRes P_((char *name));
extern char *syserrstr P_((void));
extern int existsh P_((char *filename));
extern int get_color_resource P_((Widget w, char *cname, Pixel *p));
extern int gray_ramp P_((Display *dsp, Colormap cm, Pixel **pixp));
extern int openh P_((char *name, int flags, ...));
extern int xy2RADec P_((FImage *fip, double x, double y, double *rap,
    double *decp));
extern void fs_sexa P_((char *out, double a, int w, int fracbase));
extern void get_something P_((Widget w, char *resource, XtArgVal value));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void pm_set P_((int percentage));
extern void pm_up P_((void));
extern void prompt_map_cb P_((Widget w, XtPointer client, XtPointer call));
extern void query P_((Widget tw, char *msg, char *label1, char *label2,
    char *label3, void (*func1)(void), void (*func2)(void),
    void (*func3)(void)));
extern void set_something P_((Widget w, char *resource, XtArgVal value));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void solve_sphere P_((double A, double b, double cosc, double sinc,
    double *cosap, double *Bp));
extern void sv_getcenter P_((int *aamode, double *fov,
    double *altp, double *azp, double *rap, double *decp));
extern void sv_newfits P_((void));
extern void watch_cursor P_((int want));
extern void wtip P_((Widget w, char *tip));
extern void xe_msg P_((char *msg, int app_modal));

static Pixel gray_pixel P_((int fp));
static int sf_idxcolor P_((Pixel *p));
static int sf_readfile P_((char *name));
static void sf_create P_((void));
static void sf_lohi P_((void));
static void initFSB P_((Widget w));
static void sf_stsci_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_eso_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_save_cb P_((Widget w, XtPointer client, XtPointer call));
static void save_file P_((void));
static void sf_autoname_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_open_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_help_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_exp_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_motion_eh P_((Widget w, XtPointer client, XEvent *ev,
    Boolean *dispatch));
static void sf_gmreset_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_gmcontrast_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_gmbrighter_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_inv_cb P_((Widget w, XtPointer client, XtPointer call));
static void sf_setgraymap P_((void));
static void sf_showHeader P_((void));
static void sf_setsavename P_((void));

static void makeGlassImage P_((Display *dsp));
static void makeGlassGC P_((Display *dsp, Window win));
static void fillGlass P_((int lr, int tb, int wx, int wy));

static void eso_fits P_((void));
static void stsci_fits P_((void));
static void fits_read_cb P_((XtPointer client, int *fd, XtInputId *id));
static void stopd_cb P_((Widget w, XtPointer client, XtPointer call));
static int dsswcs P_((FImage *fip));
static int dssxy2RADec P_((FImage *fip, double X, double Y, double *rap,
    double *decp));
static int wcsok P_((FImage *fip));
static void newfim P_((FImage *fip));
static void stopd_up P_((void));
static void stopd_down P_((void));
static int stopd_check P_((void));

static Widget sf_w;		/* main dialog */
static Widget savefn_w;		/* TF for save filename */
static Widget fsb_w;		/* FSB for opening a file */
static Widget hdr_w;		/* ScrolledText for the FITS header */
static Widget fda_w;		/* Drawing area for the contrast map */
static Widget inv_w;		/* inverse video TB */
static Widget autoname_w;	/* TB for whether to auto set save filename */ 

static FImage fim;		/* current FITS image */
static Pixmap fpm;		/* current pixmap with image */
static int fimok;		/* 1 if fim is ok, else 0 */
static int lopix, hipix;	/* lo and hi image values */
static Pixel *gray;		/* gray-scale ramp for drawing image */
static int ngray;		/* number of pixels usable in gray[] */
static int fdepth;		/* depth of image, in bits */
static int fbpp;		/* bits per pixel in image: 1, 8, 16 or 32 */
static XtIntervalId readId;	/* set while working on reading from socket  */
static Widget stopd_w;		/* the Stop dialog */
static int sockfd;		/* network socket; -1 when not in use */

static char ixr[] = "FITSGrayIndexColor";	/* X resource name of idx colr*/
#define	GMH	30		/* height of gray map */
#define	IDH	(GMH/5)		/* height of grap map marker */
#define	IDW	(IDH/2)		/* width of grap map marker */

static XImage *glass_xim;       /* glass XImage -- 0 means new or can't */
static GC glassGC;              /* GC for glass border */

#define GLASSSZ         50      /* mag glass width and heigth, pixels */
#define GLASSMAG        2       /* mag glass factor (may be any integer > 0) */

/* called to manage the fits dialog.
 */
void
sf_manage()
{
	if (!sf_w)
	    sf_create();
	XtManageChild(sf_w);
}

/* called to unmanage the fits dialog.
 */
void
sf_unmanage()
{
	if (sf_w)
	    XtUnmanageChild (sf_w);
}

/* return 1 if dialog is up, else 0.
 */
int
sf_ismanaged()
{
	return (sf_w && XtIsManaged(sf_w));
}

/* return 1 if there is currently a valid FITS image available, else 0 */
int
sf_ison()
{
	return (fimok);
}

/* discard any current FITS image */
void
sf_off()
{
	if (fimok) {
	    resetFImage (&fim);
	    fimok = 0;
	}
	if (fpm) {
	    XFreePixmap (XtD, fpm);
	    fpm = (Pixmap)0;
	}
}

/* called to put up or remove the watch cursor.  */
void
sf_cursor (c)
Cursor c;
{
	Window win;

	if (sf_w && (win = XtWindow(sf_w)) != 0) {
	    Display *dsp = XtDisplay(sf_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}
}

/* return pointer to the current FImage if any, else NULL */
FImage *
sf_getFImage ()
{
	return (fimok ? &fim : NULL);
}

/* return the current Pixmap */
Pixmap
sf_getPixmap ()
{
	return (fpm);
}

/* build a new fpm from fim with the given flipping orientation.
 * w is just used to get the pixmap depth.
 */
void
sf_newPixmap (w, lr, tb)
Widget w;
int lr, tb;
{
	Display *dsp = XtDisplay (w);
	Window win = XtWindow (w);
	GC gc = DefaultGC(dsp, DefaultScreen(dsp));
	XImage *f_xim;		/* XImage: adjusted for flipping and lo/hi */
	char *data;		/* pixel data for f_xim */
	int nbytes;		/* total bytes in data[] */
	char buf[1024];		/* misc buffer */
	CamPixel *ip;		/* quick index into fim.image */
	int nrows, ncols;	/* size of image */
	int x, y;

	if (!fimok)
	    return;

	watch_cursor (1);

	ncols = fim.sw;
	nrows = fim.sh;

	/* get a grayscale ramp */
	ngray = gray_ramp (dsp, xe_cm, &gray);

	/* establish depth and bits per pixel */
	get_something (w, XmNdepth, (XtArgVal)&fdepth);
	fbpp = (fdepth == 1 || ngray == 2) ? 1 :
				(fdepth>=17 ? 32 : (fdepth >= 9 ? 16 : 8));

	/* get memory for image pixels */
	nbytes = (nrows+7)*(ncols+7)*fbpp/8;
	data = (char *) malloc (nbytes);
	if (!data) {
	    (void)sprintf(buf,"Can not get %d bytes for fits pixels", nbytes);
	    xe_msg (buf, 1);
	    watch_cursor (0);
	    return;
	}

	/* create the XImage */
	f_xim = XCreateImage (dsp, XDefaultVisual (dsp, DefaultScreen(dsp)),
	    /* depth */         fbpp == 1 ? 1 : fdepth,
	    /* format */        fbpp == 1 ? XYBitmap : ZPixmap,
	    /* offset */        0,
	    /* data */          data,
	    /* width */         ncols,
	    /* height */        nrows,
	    /* pad */           fbpp < 8 ? 8 : fbpp,
	    /* bpl */           0);
	if (!f_xim) {
	    xe_msg ("Can not create shadow XImage", 1);
	    free ((void *)data);
	    watch_cursor (0);
	    return;
	}

        f_xim->bitmap_bit_order = LSBFirst;
	f_xim->byte_order = LSBFirst;

	/* fill image via gray map from lo to hi and allow for flipping */
	ip = (CamPixel *) fim.image;
	for (y = 0; y < nrows; y++) {
	    int flipy = tb ? nrows-y-1 : y;
	    for (x = 0; x < ncols; x++) {
		int flipx = lr ? ncols-x-1 : x;
		int fp = (int)(*ip++);
		Pixel xp = gray_pixel (fp);;
		XPutPixel (f_xim, flipx, flipy, xp);
	    }
	}

	/* create the pixmap */
	if (fpm)
	    XFreePixmap (dsp, fpm);
	fpm = XCreatePixmap (dsp, win, ncols, nrows, fdepth);

	/* copy the image to the pixmap */
	XPutImage (dsp, fpm, gc, f_xim, 0, 0, 0, 0, ncols, nrows);

	/* finished with the image */
	free ((void *)f_xim->data);
	f_xim->data = NULL;
	XDestroyImage (f_xim);

	watch_cursor (0);
}

/* handle the operation of the magnifying glass.
 * this is called whenever there is left button activity over the image.
 */
void
sf_doGlass (dsp, win, b1p, m1, lr, tb, wx, wy)
Display *dsp;
Window win;
int b1p, m1;	/* button/motion state */
int lr, tb;	/* flipping flags */
int wx, wy;	/* window coords of cursor */
{
	static int lastwx, lastwy;
	int rx, ry, rw, rh;		/* region */

	if (!fimok)
	    return;

	/* check for first-time stuff */
	if (!glass_xim)
	    makeGlassImage (dsp);
	if (!glass_xim)
	    return; /* oh well */
	if (!glassGC)
	    makeGlassGC (dsp, win);

	if (m1) {

	    /* motion: put back old pixels that won't just be covered again */

	    /* first the vertical strip that is uncovered */

	    rh = GLASSSZ*GLASSMAG;
	    ry = lastwy - (GLASSSZ*GLASSMAG/2);
	    if (ry < 0) {
		rh += ry;
		ry = 0;
	    }
	    if (wx < lastwx) {
		rw = lastwx - wx;	/* cursor moved left */
		rx = wx + (GLASSSZ*GLASSMAG/2);
	    } else {
		rw = wx - lastwx;	/* cursor moved right */
		rx = lastwx - (GLASSSZ*GLASSMAG/2);
	    }
	    if (rx < 0) {
		rw += rx;
		rx = 0;
	    }

	    if (rw > 0 && rh > 0)
		XCopyArea (dsp, fpm, win, glassGC, rx, ry, rw, rh, rx, ry);

	    /* then the horizontal strip that is uncovered */

	    rw = GLASSSZ*GLASSMAG;
	    rx = lastwx - (GLASSSZ*GLASSMAG/2);
	    if (rx < 0) {
		rw += rx;
		rx = 0;
	    }
	    if (wy < lastwy) {
		rh = lastwy - wy;	/* cursor moved up */
		ry = wy + (GLASSSZ*GLASSMAG/2);
	    } else {
		rh = wy - lastwy;	/* cursor moved down */
		ry = lastwy - (GLASSSZ*GLASSMAG/2);
	    }
	    if (ry < 0) {
		rh += ry;
		ry = 0;
	    }

	    if (rw > 0 && rh > 0)
		XCopyArea (dsp, fpm, win, glassGC, rx, ry, rw, rh, rx, ry);
	}

	if (b1p || m1) {

	    /* start or new location: show glass and save new location */

	    fillGlass (lr, tb, wx, wy);
	    XPutImage (dsp, win, glassGC, glass_xim, 0, 0,
			wx-(GLASSSZ*GLASSMAG/2), wy-(GLASSSZ*GLASSMAG/2),
			GLASSSZ*GLASSMAG, GLASSSZ*GLASSMAG);
	    lastwx = wx;
	    lastwy = wy;

	    /* kinda hard to tell boundry of glass so draw a line around it */
	    XDrawRectangle (dsp, win, glassGC,
			wx-(GLASSSZ*GLASSMAG/2), wy-(GLASSSZ*GLASSMAG/2),
			GLASSSZ*GLASSMAG-1, GLASSSZ*GLASSMAG-1);
	}
}

/* given a FITS pixel, return an X pixel */
static Pixel
gray_pixel (fp)
int fp;
{
	int gp;

	/* get a grayscale ramp */
	ngray = gray_ramp (XtD, xe_cm, &gray);

	if (fp < lopix)
	    gp = 0;
	else if (fp > hipix)
	    gp = ngray-1;
	else
	    gp = (fp-lopix)*(ngray-1)/(hipix-lopix);

	if (XmToggleButtonGetState (inv_w))
	    gp = (ngray-1) - gp;

	return (gray[gp]);

}

/* attempt to open a FITS file. also insure it has WCS fields.
 * if all ok, fill in fim and return 0, else return -1.
 */
static int
sf_readfile (name)
char *name;
{
	char buf[1024];
	FImage sfim, *fip = &sfim;
	int fd;
	int s;

	/* init */
	initFImage (fip);

	/* open the fits file */
	fd = openh (name, O_RDONLY);
	if (fd < 0) {
	    (void) sprintf (buf, "%s: %s", name, syserrstr());
	    xe_msg (buf, 1);
	    return(-1);
	}

	/* read in */
	s = readFITS (fd, fip, buf);
	close (fd);
	if (s < 0) {
	    char buf2[1024];
	    (void) sprintf (buf2, "%s: %s", name, buf);
	    xe_msg (buf2, 1);
	    return(-1);
	}

	/* we only support 16 bit integer images */
	if (fip->bitpix != 16) {
	    (void) sprintf (buf, "%s: must be BITPIX 16", name);
	    resetFImage (fip);
	    return (-1);
	}

	/* make sure it has WCS fields */
	if (wcsok (fip) < 0) {
	    resetFImage (fip);
	    return (-1);
	}

	/* ok!*/
	newfim (fip);
	return (0);
}

/* set up an initial guess at the lo and hi settings */
static void
sf_lohi()
{
	CamPixel *ip = (CamPixel *) fim.image;
	int npix = fim.sw * fim.sh;
	double sum, sum2;
	double mean, sd;
	int i;

	sum = sum2 = 0.0;
	for (i = 0; i < npix; i++) {
	    double p = (double)(*ip++);
	    sum += p;
	    sum2 += p*p;
	}
	mean = sum/npix;
	sd = sqrt ((sum2 - sum*sum/npix)/(npix-1));
	lopix = (int)floor(mean - sd + 0.5);
	if (lopix < 0)
	    lopix = 0;
	hipix = (int)floor(mean + 2*sd + 0.5);
	if (hipix >= MAXCAMPIX)
	    hipix = MAXCAMPIX-1;
}

/* create, but do not manage, the fits dialog and all its supporting dialogs */
static void
sf_create()
{
	Widget tf_w, bf_w;
	Widget p_w;
	Widget l_w;
	Widget w;
	EventMask mask;
	Arg args[20];
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNallowResize, True); n++;
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	sf_w = XmCreateFormDialog (toplevel_w, "SkyFITS", args, n);
	set_something (sf_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (sf_w, XmNhelpCallback, sf_help_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Sky FITS"); n++;
	XtSetValues (XtParent(sf_w), args, n);

	/* top and bottom halves are in their own forms, then
	 * each form in a paned window
	 */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	p_w = XmCreatePanedWindow (sf_w, "FITSPW", args, n);
	XtManageChild (p_w);

	/* the top form */

	n = 0;
	XtSetArg (args[n], XmNverticalSpacing, 10); n++;
	tf_w = XmCreateForm (p_w, "TF", args, n);
	XtManageChild (tf_w);

	    /* buttons to fetch networked images */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    l_w = XmCreateLabel (tf_w, "Lab", args, n);
	    set_xmstring (l_w, XmNlabelString,
				"Retrieve Digitized Sky Survey Image from ..");
	    XtManageChild (l_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, l_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 15); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 45); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	    w = XmCreatePushButton (tf_w, "STScI", args, n);
	    XtAddCallback (w, XmNactivateCallback, sf_stsci_cb, NULL);
	    wtip (w, "Retrieve DSS image from STScI");
	    set_xmstring (w, XmNlabelString, "STScI");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, l_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 55); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 85); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	    w = XmCreatePushButton (tf_w, "ESO", args, n);
	    wtip (w, "Retrieve DSS image from ESO");
	    XtAddCallback (w, XmNactivateCallback, sf_eso_cb, NULL);
	    set_xmstring (w, XmNlabelString, "ESO");
	    XtManageChild (w);

	    /* scrolled text in which to display the header */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    w = XmCreateLabel (tf_w, "Lab", args, n);
	    set_xmstring (w, XmNlabelString, "FITS Header:");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNbottomOffset, 10); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNautoShowCursorPosition, False); n++;
	    XtSetArg (args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
	    XtSetArg (args[n], XmNeditable, False); n++;
	    XtSetArg (args[n], XmNcursorPositionVisible, False); n++;
	    hdr_w = XmCreateScrolledText (tf_w, "Header", args, n);
	    wtip (hdr_w, "Scrolled text area containing FITS File header");
	    XtManageChild (hdr_w);

	/* the bottom form */

	n = 0;
	XtSetArg (args[n], XmNverticalSpacing, 10); n++;
	bf_w = XmCreateForm (p_w, "BF", args, n);
	XtManageChild (bf_w);

	    /* label, inverse TB and reset PB for contrast map */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNtopOffset, 10); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    w = XmCreateLabel (bf_w, "GML", args, n);
	    set_xmstring (w, XmNlabelString, "Gray scale:");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNtopOffset, 10); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    inv_w = XmCreateToggleButton (bf_w, "GMI", args, n);
	    XtAddCallback (inv_w, XmNvalueChangedCallback, sf_inv_cb, 0);
	    set_xmstring (inv_w, XmNlabelString, "Inverse");
	    wtip (inv_w, "When pushed in, image is displayed black-on-white");
	    XtManageChild (inv_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, inv_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    w = XmCreatePushButton (bf_w, "GMC", args, n);
	    XtAddCallback (w, XmNactivateCallback, sf_gmcontrast_cb, 0);
	    set_xmstring (w, XmNlabelString, "More contrast");
	    wtip (w, "Increase image contrast by increasing the lower limit");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, inv_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 45); n++;
	    w = XmCreatePushButton (bf_w, "GMB", args, n);
	    XtAddCallback (w, XmNactivateCallback, sf_gmbrighter_cb, 0);
	    set_xmstring (w, XmNlabelString, "Brighter");
	    wtip (w, "Increase image brightness by decreasing the upper limit");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, inv_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    w = XmCreatePushButton (bf_w, "GMRP", args, n);
	    XtAddCallback (w, XmNactivateCallback, sf_gmreset_cb, 0);
	    set_xmstring (w, XmNlabelString, "Reset scale");
	    wtip (w, "Set gray range to [Median-StdDev .. Median+2*StdDev]");
	    XtManageChild (w);

	    /* drawing area for the contrast map */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNbackground,
					BlackPixel(XtD,DefaultScreen(XtD)));n++;
	    XtSetArg (args[n], XmNheight, GMH); n++;
	    fda_w = XmCreateDrawingArea (bf_w, "FDA", args, n);
	    XtAddCallback (fda_w, XmNexposeCallback, sf_exp_cb, NULL);
	    mask = Button1MotionMask | ButtonPressMask | ButtonReleaseMask
						       | PointerMotionHintMask;
	    XtAddEventHandler (fda_w, mask, False, sf_motion_eh, NULL);
	    XtManageChild (fda_w);
	    wtip(fda_w,"Gray values assigned to 0..65535. Slide points to change");

	    /* label, go PB, Auto name TB and TF for saving a file */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, fda_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    w = XmCreateLabel (bf_w, "Save", args, n);
	    set_xmstring (w, XmNlabelString, "Save as FITS file:");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, fda_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 45); n++;
	    w = XmCreatePushButton (bf_w, "Save", args, n);
	    XtAddCallback (w, XmNactivateCallback, sf_save_cb, NULL);
	    wtip (w, "Save the current image to the file named below");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, fda_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    autoname_w = XmCreateToggleButton (bf_w, "AutoName", args, n);
	    set_xmstring (autoname_w, XmNlabelString, "Auto name");
	    XtAddCallback (autoname_w,XmNvalueChangedCallback,sf_autoname_cb,0);
	    XtManageChild (autoname_w);
	    wtip (autoname_w, "When pushed in, automatically chooses a filename based on center RA and Dec");

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, autoname_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    savefn_w = XmCreateTextField (bf_w, "SaveFN", args, n);
	    XtAddCallback (savefn_w, XmNactivateCallback, sf_save_cb, NULL);
	    wtip (savefn_w, "Enter name of file to write, then press Enter");
	    XtManageChild (savefn_w);

	    /* the Open FSB */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, savefn_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    w = XmCreateLabel (bf_w, "Lab", args, n);
	    set_xmstring (w, XmNlabelString, "Open a FITS file:");
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    fsb_w = XmCreateFileSelectionBox (bf_w, "FSB", args, n);
	    wtip (fsb_w, "Select a FITS file to read and display");
	    XtManageChild (fsb_w);
	    initFSB(fsb_w);
}

/* init the directory and pattern resources of the given FileSelectionBox.
 * we try to pull these from the basic program resources.
 */
static void
initFSB (fsb_w)
Widget fsb_w;
{
	static char fitsd[] = "FITSdirectory";
	static char fitsp[] = "FITSpattern";
	Widget w;
	char msg[256];
	char *res;

	res = getXRes (fitsd);
	if (!res) {
	    res = "work";
	    (void)sprintf(msg,"Can not get %s -- using %s", fitsd, res);
	    xe_msg (msg, 0);
	}
	set_xmstring (fsb_w, XmNdirectory, res);
	sprintf (msg, "%s/", res);

	res = getXRes (fitsp);
	if (!res) {
	    res = "*.f*ts";
	    (void)sprintf(msg,"Can not get %s -- using %s", fitsp, res);
	    xe_msg (msg, 0);
	}
	set_xmstring (fsb_w, XmNpattern, res);

	/* change some button labels.
	 * N.B. can't add tips because these are really Gadgets.
	 */
	w = XmFileSelectionBoxGetChild (fsb_w, XmDIALOG_OK_BUTTON);
	set_xmstring (w, XmNlabelString, "Open");
	w = XmFileSelectionBoxGetChild (fsb_w, XmDIALOG_CANCEL_BUTTON);
	set_xmstring (w, XmNlabelString, "Close");

	/* connect an Open handler */
	XtAddCallback (fsb_w, XmNokCallback, sf_open_cb, NULL);

	/* connect a Close handler */
	XtAddCallback (fsb_w, XmNcancelCallback, sf_close_cb, NULL);

	/* connect a Help handler */
	XtAddCallback (fsb_w, XmNhelpCallback, sf_help_cb, NULL);
}

/* called when STScI is hit */
/* ARGSUSED */
static void
sf_stsci_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (readId) {
	    xe_msg ("Download already in progress..", 1);
	    return;
	}

	watch_cursor (1);
	stsci_fits();
	watch_cursor (0);
}

/* called when ESO is hit */
/* ARGSUSED */
static void
sf_eso_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (readId) {
	    xe_msg ("Download already in progress..", 1);
	    return;
	}

	watch_cursor (1);
	eso_fits();
	watch_cursor (0);
}

/* called when the auto name TB is hit */
/* ARGSUSED */
static void
sf_autoname_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmToggleButtonCallbackStruct *tp = (XmToggleButtonCallbackStruct *)call;

	if (tp->set)
	    sf_setsavename();
}

/* called when CR is hit in the Save text field or the Save PB is hit */
/* ARGSUSED */
static void
sf_save_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	char *fn;

	if (!fimok) {
	    xe_msg ("No current FITS file to save", 1);
	    return;
	}

	fn = XmTextFieldGetString (savefn_w);
	if (!fn || (int)strlen(fn) < 1) {
	    xe_msg ("Please specify a filename", 1);
	    XtFree (fn);
	    return;
	}

	if (existsh (fn) == 0) {
	    char buf[1024];
	    (void) sprintf (buf, "%s exists: Ok to Overwrite?", fn);
	    query (toplevel_w, buf, "Yes -- Overwrite", "No -- Cancel",
						NULL, save_file, NULL, NULL);
	} else
	    save_file();

	XtFree (fn);
}

/* save to file named in savefn_w.
 * we already know everything is ok to just do it now.
 */
static void
save_file()
{
	char buf[1024];
	char *fn;
	int fd;

	fn = XmTextFieldGetString (savefn_w);

	fd = openh (fn, O_CREAT|O_WRONLY, 0666);
	if (fd < 0) {
	    (void) sprintf (buf, "%s: %s", fn, syserrstr());
	    xe_msg (buf, 1);
	    XtFree (fn);
	    return;
	}

	if (writeFITS (fd, &fim, buf, 1) < 0) {
	    char buf2[1024];
	    (void) sprintf (buf2, "%s: %s", fn, buf);
	    xe_msg (buf2, 1);
	} else {
	    (void) sprintf (buf, "%s:\nwritten successfully", fn);
	    xe_msg (buf, 1);
	}

	(void) close (fd);
	XtFree (fn);
}

static void
/* ARGSUSED */
sf_open_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmFileSelectionBoxCallbackStruct *s =
				    (XmFileSelectionBoxCallbackStruct *)call;
	char *sp;

	if (s->reason != XmCR_OK) {
	    printf ("%s: Unknown reason = 0x%x\n", "sf_open_cb()", s->reason);
	    exit (1);
	}

	watch_cursor(1);

	XmStringGetLtoR (s->value, XmSTRING_DEFAULT_CHARSET, &sp);
	if (sf_readfile (sp) == 0) {
	    sf_lohi();
	    sf_setgraymap();
	    sf_showHeader();
	    sv_newfits();
	    sf_setsavename();
	}

	XtFree (sp);

	watch_cursor(0);
}

/* ARGSUSED */
static void
sf_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XtUnmanageChild (sf_w);
}

/* ARGSUSED */
static void
sf_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static char *msg[] = {
"Read in local FITS files or read from Network.",
"Resulting image will be displayed in Sky View."
};

	hlp_dialog ("Sky FITS", msg, sizeof(msg)/sizeof(msg[0]));
}

/* button motion and press/release event handler for the gray map */
/* ARGSUSED */
static void
sf_motion_eh (w, client, ev, continue_to_dispatch)
Widget w;
XtPointer client;
XEvent *ev;
Boolean *continue_to_dispatch;
{
	static int moving_lopix;
	Display *dsp = XtDisplay(w);
	Window win = XtWindow(w);
	Dimension wid, hei;
	Window root, child;
	int rx, ry, x, y;
	unsigned mask;
	int evt = ev->type;
	int b1p, b1r;
	int newpix;

	/* do nothing if no current image */
	if (!fimok)
	    return;

	/* what happened? */
	b1p = evt == ButtonPress   && ev->xbutton.button == Button1;
	b1r = evt == ButtonRelease && ev->xbutton.button == Button1;

	/* where are we? */
	XQueryPointer (dsp, win, &root, &child, &rx, &ry, &x, &y, &mask);
	get_something (w, XmNwidth, (XtArgVal)&wid);
	get_something (w, XmNheight, (XtArgVal)&hei);
	if (x < 0)         x = 0;
	if (x >= (int)wid) x = wid-1;
	if (y < 0)         y = 0;
	if (y >= (int)hei) y = hei-1;

	/* scale x to pixel value */
	newpix = x*MAXCAMPIX/(int)wid;

	/* if button just pressed, chose which end to track */
	if (b1p)
	    moving_lopix = newpix < (lopix + hipix)/2;

	/* track the current end -- but never cross over */
	if (moving_lopix && newpix < hipix)
	    lopix = newpix;
	else if (!moving_lopix && newpix > lopix)
	    hipix = newpix;

	/* new map */
	sf_setgraymap();

	/* display net result when release */
	if (b1r)
	    sv_newfits();
}

/* expose callback for the gray map */
/* ARGSUSED */
static void
sf_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	Display *dsp = XtDisplay(fda_w);
	Window win = XtWindow(fda_w);

	switch (c->reason) {
	case XmCR_EXPOSE: {
	    static before;
	    XExposeEvent *e = &c->event->xexpose;

	    if (!before) {
		XSetWindowAttributes swa;
		unsigned long mask = CWBitGravity;

		swa.bit_gravity = ForgetGravity;
		XChangeWindowAttributes (dsp, win, mask, &swa);
		before = 1;
	    }

	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected fda_w event. type=%d\n", c->reason);
	    exit(1);
	}

	sf_setgraymap();
}

/* callback for the gray map reset PB */
/* ARGSUSED */
static void
sf_gmreset_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!fimok)
	    return;

	watch_cursor(1);

	sf_lohi();
	sf_setgraymap();
	sv_newfits();

	watch_cursor(0);
}

/* callback for the gray map more contrast PB */
/* ARGSUSED */
static void
sf_gmcontrast_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!fimok)
	    return;

	watch_cursor(1);

	lopix += (hipix - lopix)/3;
	sf_setgraymap();
	sv_newfits();

	watch_cursor(0);
}

/* callback for the gray map brighter PB */
/* ARGSUSED */
static void
sf_gmbrighter_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!fimok)
	    return;

	watch_cursor(1);

	hipix -= (hipix - lopix)/3;
	sf_setgraymap();
	sv_newfits();

	watch_cursor(0);
}

/* callback for the gray map Inverse Vid TB */
/* ARGSUSED */
static void
sf_inv_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!fimok)
	    return;

	watch_cursor(1);

	sf_setgraymap();
	sv_newfits();

	watch_cursor(0);
}

/* fill the gray map drawing area with the current pixel colors */
static void
sf_setgraymap()
{
#define	MAXSEG	50
	Display *dsp = XtDisplay(fda_w);
	Window win = XtWindow(fda_w);
	GC gc = DefaultGC(dsp, DefaultScreen(dsp));
	XPoint id[4];
	Dimension wid, hei;
	Pixel p;
	int hix, lox;
	int i;

	if (!fimok)
	    return;

	get_something (fda_w, XmNwidth, (XtArgVal)&wid);
	get_something (fda_w, XmNheight, (XtArgVal)&hei);

	/* lo end */
	lox = lopix*(int)wid/MAXCAMPIX;
	if (lox > 0) {
	    id[0].x = 0;   id[0].y = 0;
	    id[1].x = lox; id[1].y = 0;
	    id[2].x = lox; id[2].y = hei;
	    id[3].x = 0;   id[3].y = hei;
	    XSetForeground (dsp, gc, gray_pixel(0));
	    XFillPolygon (dsp, win, gc, id, 4, Convex, CoordModeOrigin);
	}

	/* hi end */
	hix = hipix*(int)wid/MAXCAMPIX;
	if (hix < (int)wid) {
	    id[0].x = hix; id[0].y = 0;
	    id[1].x = wid; id[1].y = 0;
	    id[2].x = wid; id[2].y = hei;
	    id[3].x = hix; id[3].y = hei;
	    XSetForeground (dsp, gc, gray_pixel(MAXCAMPIX));
	    XFillPolygon (dsp, win, gc, id, 4, Convex, CoordModeOrigin);
	}

	/* in between */
	for (i = lox; i < hix; i++) {
	    int v = i*MAXCAMPIX/(int)wid;
	    p = gray_pixel(v);
	    XSetForeground (dsp, gc, p);
	    XDrawLine (dsp, win, gc, i, 0, i, hei);
	}

	/* set the indicators */
	i = lopix*(int)wid/MAXCAMPIX;
	if (sf_idxcolor (&p) < 0)
	    p = WhitePixel(XtD,DefaultScreen(XtD));
	XSetForeground (dsp, gc, p);
	id[0].x = i-IDW; id[0].y = hei;
	id[1].x = i;     id[1].y = hei - IDH;
	id[2].x = i+IDW; id[2].y = hei;
	XFillPolygon (dsp, win, gc, id, XtNumber(id), Convex, CoordModeOrigin);

	i = hipix*(int)wid/MAXCAMPIX;
	if (sf_idxcolor (&p) < 0)
	    p = BlackPixel(XtD,DefaultScreen(XtD));
	XSetForeground (dsp, gc, p);
	id[0].x = i-IDW; id[0].y = hei;
	id[1].x = i;     id[1].y = hei - IDH;
	id[2].x = i+IDW; id[2].y = hei;
	XFillPolygon (dsp, win, gc, id, XtNumber(id), Convex, CoordModeOrigin);
}

/* get the pixel to use for the indicators on the Gray map.
 * return 0 with *cp set if ok, else -1 if trouble.
 * TODO: cache the answer
 */
static int
sf_idxcolor(p)
Pixel *p;
{
	return (get_color_resource (fda_w, ixr, p));
}

/* fill the hdr_w scrolled text with the FITS header entries.
 * keep hdrl up to date.
 */
static void
sf_showHeader ()
{
	char *header;
	int i;

	/* room for each FITS line, with nl and a final \0 */
	header = malloc (fim.nvar*(FITS_HCOLS+1) + 1);
	if (!header) {
	    xe_msg ("No memory to display FITS header", 0);
	    return;
	}

	/* copy from fim.var to header, adding \n after each line */
	for (i = 0; i < fim.nvar; i++) {
	    memcpy(header + i*(FITS_HCOLS+1), fim.var[i], FITS_HCOLS);
	    header[(i+1)*(FITS_HCOLS+1)-1] = '\n';
	}
	header[fim.nvar*(FITS_HCOLS+1)] = '\0';  /* add final \0 */

	XmTextSetString (hdr_w, header);
	free (header);

	/* scroll to the top */
	XmTextShowPosition (hdr_w, (XmTextPosition)0);
}

/* if enabled, fill in savefn_w based on fim */
static void
sf_setsavename()
{
	char newfn[1024];
	double ra, dec;
	int rh, rm, dd, dm;
	int dneg;
	char *fn;
	char *s;
	char *f, *l;

	if (!XmToggleButtonGetState(autoname_w) || !fimok)
	    return;

	fn = XmTextFieldGetString (savefn_w);

	/* set f to where to begin */
	for (f = s = fn; *s != '\0'; s++)
	    if (*s == '/')
		f = s+1;

	/* come back and set l where to end */
	for (l = s; s >= f; --s)
	    if (*s == '.') {
		l = s;
		break;
	    }

	/* set name from central location */
	(void) xy2RADec (&fim, fim.sw/2.0, fim.sh/2.0, &ra, &dec);
	ra = radhr(ra);
	dec = raddeg(dec);
	if ((dneg = (dec < 0)))
	    dec = -dec;
	rh = (int)floor(ra);
	rm = (int)floor((ra - rh)*60.0 + 0.5);
	if (rm == 60) {
	    if (++rh == 24)
		rh = 0;
	    rm = 0;
	}
	dd = (int)floor(dec);
	dm = (int)floor((dec - dd)*60.0 + 0.5);
	if (dm == 60) {
	    dd++;
	    dm = 0;
	}
	sprintf (newfn, "%.*s%02d%02d%c%02d%02d%s", f-fn, fn, rh, rm,
			    dneg ? '-' : '+', dd, dm, *l == '.' ? l : ".fts");
	XtFree (fn);

	XmTextFieldSetString (savefn_w, newfn);
}

/* make glass_xim of size GLASSSZ*GLASSMAG and same genre as fim.
 * leave glass_xim NULL if trouble.
 */
static void
makeGlassImage (dsp)
Display *dsp;
{
	int nbytes = (GLASSSZ*GLASSMAG+7) * (GLASSSZ*GLASSMAG+7) * fbpp/8;
	char *glasspix = (char *) malloc (nbytes);

	if (!glasspix) {
	    char msg[1024];
	    (void) sprintf (msg, "Can not malloc %d for Glass pixels", nbytes);
	    xe_msg (msg, 0);
	    return;
	}

	glass_xim = XCreateImage (dsp, XDefaultVisual (dsp, DefaultScreen(dsp)),
	    /* depth */         fbpp == 1 ? 1 : fdepth,
	    /* format */        fbpp == 1 ? XYBitmap : ZPixmap,
	    /* offset */        0,
	    /* data */          glasspix,
	    /* width */         GLASSSZ*GLASSMAG,
	    /* height */        GLASSSZ*GLASSMAG,
	    /* pad */           fbpp < 8 ? 8 : fbpp,
	    /* bpl */           0);

	if (!glass_xim) {
	    free ((void *)glasspix);
	    xe_msg ("Can not make Glass XImage", 0);
	    return;
	}

        glass_xim->bitmap_bit_order = LSBFirst;
	glass_xim->byte_order = LSBFirst;
}

/* make glassGC */
static void
makeGlassGC (dsp, win)
Display *dsp;
Window win;
{
	XGCValues gcv;
	unsigned int gcm;
	Pixel p;

	if (get_color_resource (fda_w, "GlassBorderColor", &p) < 0) {
	    xe_msg ("Can not get GlassBorderColor -- using White", 0);
	    p = WhitePixel (dsp, 0);
	}
	gcm = GCForeground;
	gcv.foreground = p;
	glassGC = XCreateGC (dsp, win, gcm, &gcv);
}

/* fill glass_xim with GLASSSZ*GLASSMAG view of fim centered at coords
 * xc,yc. take care at the edges.
 */
static void
fillGlass (lr, tb, xc, yc)
int lr, tb;	/* flipping flags */
int xc, yc;	/* center of glass, in window coords */
{
	CamPixel *ip = (CamPixel *)(fim.image);
	int ncols = fim.sw;
	int nrows = fim.sh;
	int sx, sy;	/* coords in fim */
	int gx, gy;	/* coords in glass_xim */
	int gdx, gdy;	/* glass delta in each loop */

	if (lr) {
	    xc = ncols - xc;
	    gx = (GLASSSZ - 1) * GLASSMAG;
	    gdx = -GLASSMAG;
	} else {
	    gx = 0;
	    gdx = GLASSMAG;
	}
	if (tb) {
	    yc = nrows - yc;
	    gy = (GLASSSZ - 1) * GLASSMAG;
	    gdy = -GLASSMAG;
	} else {
	    gy = 0;
	    gdy = GLASSMAG;
	}
	    
	for (sy = yc-GLASSSZ/2; sy < yc+GLASSSZ/2; sy++) {
	    for (sx = xc-GLASSSZ/2; sx < xc+GLASSSZ/2; sx++) {
		int i, j;
		Pixel p;
		int v;

		if (sx < 0 || sx >= ncols || sy < 0 || sy >= nrows)
		    v = 0;
		else
		    v = ip[sy*ncols + sx];
		p = gray_pixel (v);

		for (i = 0; i < GLASSMAG; i++)
		    for (j = 0; j < GLASSMAG; j++)
			XPutPixel (glass_xim, gx+i, gy+j, p);

		gx += gdx;
	    }
	    gx = lr ? (GLASSSZ - 1) * GLASSMAG : 0;
	    gy += gdy;
	}
}

/* start an input stream reading a FITS image from ESO */
static void
eso_fits()
{
	static char host[] = "archive.eso.org";
	static char fitscontent[] = "Content-type: application/x-fits";
	static char textcontent[] = "Content-type: text/html";
	static FImage sfim, *fip = &sfim;
	double fov, alt, az, ra, dec;
	char rastr[32], *rap, decstr[32], *decp;
	char buf[1024];
	int aamode;
	int sawfits, sawtext;
	int n;

	/* first, see if we can even contact the ESO DSS server */
	sockfd = mkconnection (host, 80, buf);
	if (sockfd < 0) {
	    stopd_down();
	    xe_msg (buf, 1);
	    return;
	}

	/* put up the stop dialog, then check occasionally */
	stopd_up();

	/* init fip (not reset because we copy the malloc'd fields to fim) */
	initFImage (fip);

	/* find current skyview center and size, in degrees */
	sv_getcenter (&aamode, &fov, &alt, &az, &ra, &dec);
	fov = 60*raddeg(fov);
	ra = radhr (ra);
	dec = raddeg (dec);

	if (fov > MAXDSSFOV)
	    fov = MAXDSSFOV;
	if (fov < MINDSSFOV)
	    fov = MINDSSFOV;

	/* format and send the request */
	/* can't tolerate leading blanks in ra dec specs */
	fs_sexa (rastr, ra, 2, 3600);
	for (rap = rastr; *rap == ' '; rap++)
	    continue;
	fs_sexa (decstr, dec, 3, 3600);
	for (decp = decstr; *decp == ' '; decp++)
	    continue;
	(void) sprintf (buf, "GET /dss/dss?ra=%s&dec=%s&mime-type=application/x-fits&x=%.0f&y=%.0f HTTP/1.0\n\n",
							rap, decp, fov, fov);
	n = strlen (buf);
	n = sendbytes (sockfd, (unsigned char *)buf, n);
	if (n < 0) {
	    (void) sprintf (buf, "%s: send error: %s", host, syserrstr());
	    xe_msg (buf, 1);
	    stopd_down();
	    return;
	}

	if (stopd_check() < 0)
	    return;

	/* read back the header -- ends with a blank line */
	sawfits = sawtext = 0;
	while (recvline (sockfd, buf, sizeof(buf)) > 1) {
	    if (stopd_check() < 0)
		return;
	    if (strncmp (buf, fitscontent, sizeof(fitscontent)-1) == 0)
		sawfits = 1;
	    if (strncmp (buf, textcontent, sizeof(textcontent)-1) == 0)
		sawtext = 1;
	}

	/* if don't see a fits file, see if there's an error message */
	if (!sawfits) {
	    int errtext = 0;
	    buf[0] = '\0';
	    if (sawtext) {
		char buf2[1024];
		while (recvline (sockfd, buf2, sizeof(buf2)) > 0) {
		    if (strncmp (buf2, "<P><B>ERROR", 11) == 0) {
			(void) strcat (buf, buf2+11);
			errtext = 1;
			break;
		    }
		}
	    }
	    stopd_down();
	    if (errtext)
		xe_msg (buf, 1);
	    else
		xe_msg ("Unknown error asking ESO for image", 1);
	    return;
	}

	if (stopd_check() < 0)
	    return;

	/* read the FITS header portion */
	if (readFITSHeader (sockfd, fip, buf) < 0) {
	    stopd_down();
	    xe_msg (buf, 1);
	    resetFImage (fip);
	    return;
	}

	if (stopd_check() < 0) {
	    resetFImage (fip);
	    return;
	}

	/* see if WCS headers are ok (or can be derived) */
	if (wcsok (fip) < 0) {
	    resetFImage (fip);
	    stopd_down();
	    return;
	}

	if (stopd_check() < 0) {
	    resetFImage (fip);
	    return;
	}

	/* ok, start reading the pixels and give user a way to abort */
	pm_up();	/* for your viewing pleasure */
	readId = XtAppAddInput (xe_app, sockfd, (XtPointer)XtInputReadMask,
						fits_read_cb, (XtPointer)fip);
}

/* start an input stream reading a FITS image from STScI */
static void
stsci_fits()
{
	static char host[] = "archive.stsci.edu";
	static char fitscontent[] = "Content-type: image/x-fits";
	static char textcontent[] = "Content-type: text/html";
	static FImage sfim, *fip = &sfim;
	double fov, alt, az, ra, dec;
	char buf[1024];
	int aamode;
	int sawfits, sawtext;
	int n;

	/* first, see if we can even contact the STScI DSS server */
	sockfd = mkconnection (host, 80, buf);
	if (sockfd < 0) {
	    stopd_down();
	    xe_msg (buf, 1);
	    return;
	}

	/* put up the stop dialog, then check occasionally */
	stopd_up();

	/* init fip (not reset because we copy the malloc'd fields to fim) */
	initFImage (fip);

	/* find current skyview center and size, in degrees */
	sv_getcenter (&aamode, &fov, &alt, &az, &ra, &dec);
	fov = 60*raddeg(fov);
	ra = raddeg (ra);
	dec = raddeg (dec);

	if (fov > MAXDSSFOV)
	    fov = MAXDSSFOV;
	if (fov < MINDSSFOV)
	    fov = MINDSSFOV;

	/* format and send the request */
	(void) sprintf (buf, "GET /cgi-bin/dss_search?ra=%.5f&dec=%.5f&equinox=J2000&height=%.0f&width=%.0f&format=FITS&compression=NONE&version=3 HTTP/1.0\n\n",
							    ra, dec, fov, fov);
	n = strlen (buf);
	n = sendbytes (sockfd, (unsigned char *)buf, n);
	if (n < 0) {
	    (void) sprintf (buf, "%s: send error: %s", host, syserrstr());
	    xe_msg (buf, 1);
	    stopd_down();
	    return;
	}

	if (stopd_check() < 0)
	    return;

	/* read back the header -- ends with a blank line */
	sawfits = sawtext = 0;
	while (recvline (sockfd, buf, sizeof(buf)) > 1) {
	    if (stopd_check() < 0)
		return;
	    if (strncmp (buf, fitscontent, sizeof(fitscontent)-1) == 0)
		sawfits = 1;
	    if (strncmp (buf, textcontent, sizeof(textcontent)-1) == 0)
		sawtext = 1;
	}

	/* if don't see a fits file, see if there's an error message */
	if (!sawfits) {
	    int pretext = 0;
	    buf[0] = '\0';
	    if (sawtext) {
		char buf2[1024];
		while (recvline (sockfd, buf2, sizeof(buf2)) > 0) {
		    if (strncmp (buf2, "<PRE>", 5) == 0) {
			while (recvline (sockfd, buf2, sizeof(buf2)) > 0) {
			    if (strncmp (buf2, "</PRE>", 6) == 0) {
				break;
			    } else {
				(void) strcat (buf, buf2);
				pretext = 1;
			    }
			}
		    }
		}
	    }
	    stopd_down();
	    if (pretext)
		xe_msg (buf, 1);
	    else
		xe_msg ("Unknown error asking STScI for image", 1);
	    return;
	}

	if (stopd_check() < 0)
	    return;

	/* read the FITS header portion */
	if (readFITSHeader (sockfd, fip, buf) < 0) {
	    stopd_down();
	    xe_msg (buf, 1);
	    resetFImage (fip);
	    return;
	}

	if (stopd_check() < 0) {
	    resetFImage (fip);
	    return;
	}

	/* see if WCS headers are ok (or can be derived) */
	if (wcsok (fip) < 0) {
	    resetFImage (fip);
	    stopd_down();
	    return;
	}

	if (stopd_check() < 0) {
	    resetFImage (fip);
	    return;
	}

	/* ok, start reading the pixels and give user a way to abort */
	pm_up();	/* for your viewing pleasure */
	readId = XtAppAddInput (xe_app, sockfd, (XtPointer)XtInputReadMask,
						fits_read_cb, (XtPointer)fip);
}

/* called whenever there is more data available on the sockfd.
 */
static void
fits_read_cb (client, fd, id)
XtPointer client;
int *fd;
XtInputId *id;
{
	FImage *fip = (FImage *)client;
	char buf[1024];

	/* check for abort */
	if (stopd_check() < 0) {
	    resetFImage (fip);
	    stopd_down();
	    return;
	}

	/* read another chunk */
	if (readIncFITS (sockfd, fip, buf) < 0) {
	    xe_msg (buf, 1);
	    resetFImage (fip);
	    stopd_down();
	    return;
	}

	/* report progress */
	pm_set (fip->nbytes * 100 / fip->totbytes);
	XmUpdateDisplay (toplevel_w);

	/* keep going if expecting more */
	if (fip->nbytes < fip->totbytes)
	    return;

	/* finished reading, so turn off input */
	stopd_down();

	/* YES! */
	newfim (fip);
	sf_lohi();
	sf_setgraymap();
	sf_showHeader();
	sv_newfits();
	sf_setsavename();
}

/* install fip as the new fim.
 * N.B. malloced fields in fip are just re-used, so don't reset it on return.
 */
static void
newfim (fip)
FImage *fip;
{
	resetFImage (&fim);
	(void) memcpy ((void *)&fim, (void *)fip, sizeof(fim));
	fimok = 1;
}

/* check for the WCS headers.
 * if it looks like a DSS image, try to infer the fields.
 * return 0 if ok, else xe_msg() and return -1.
 */
static int
wcsok (fip)
FImage *fip;
{
	double ra, dec;
	char buf[128];

	/* now try it */
	if (xy2RADec (fip, 0.0, 0.0, &ra, &dec) == 0)
	    return (0);

	/* only other possibility is a DSS image from STScI -- if it has
	 * the PLTLABEL, at least give it a try.
	 */
	if (getStringFITS (fip, "PLTLABEL", buf) == 0 && dsswcs (fip) == 0)
	    return (0);

	/* oh well */
	xe_msg ("No WCS fields in header", 1);
	return (-1);
}

/* given a DSS image, build the WCS headers.
 * return 0 if ok, else issue xe_msg() and return -1.
 */
static int
dsswcs (fip)
FImage *fip;
{
	double a0, d0, a1, d1;
	double px, py;
	double pltscale;
	double rot;

	/* find RA and Dec at the center of our image */
	if (dssxy2RADec (fip, fip->sw/2.0, fip->sh/2.0, &a0, &d0) < 0)
	    return (-1);

	/* use center as reference */
	setRealFITS (fip, "CRPIX1", fip->sw/2.0, 10, NULL);
	setRealFITS (fip, "CRPIX2", fip->sh/2.0, 10, NULL);
	setRealFITS (fip, "CRVAL1", raddeg(a0), 10, NULL);
	setRealFITS (fip, "CRVAL2", raddeg(d0), 10, NULL);

	/* rotation is based on how center relates edge */
	if (dssxy2RADec (fip, (double)fip->sw, fip->sh/2.0, &a1, &d1) < 0)
	    return (-1);
	solve_sphere (a0-a1, PI/2-d1, sin(d0), cos(d0), NULL, &rot);
	rot = raddeg(rot-PI/2);
	setRealFITS (fip, "CROTA2", rot, 10, NULL);
	setRealFITS (fip, "CROTA1", rot, 10, NULL);

	/* set scale */
	if (getRealFITS (fip, "PLTSCALE", &pltscale) < 0
			    || getRealFITS (fip, "XPIXELSZ", &px) < 0
			    || getRealFITS (fip, "YPIXELSZ", &py) < 0) {
	    xe_msg ("No Plate scale fields", 1);
	    return(-1);
	}
	setRealFITS (fip, "CDELT1", -pltscale*px/3600000.0, 10, NULL);
	setRealFITS (fip, "CDELT2",  pltscale*py/3600000.0, 10, NULL);

	/* and finally, the easy ones */
	setStringFITS (fip, "CTYPE1", "RA---TAN", NULL);
	setStringFITS (fip, "CTYPE2", "DEC--TAN", NULL);

	return (0);
}

/* convert pixel coords to ra/dec, using DSS header fields.
 * return 0 if ok, else issue xe_msg() and return -1.
 * this is all based on the equations on pages 14-16 of The Digitized Sky
 * Survey release notes.
 */
static int
dssxy2RADec (fip, X, Y, rap, decp)
FImage *fip;
double X, Y;
double *rap, *decp;
{
	char buf[1024];
	double cnpix1, cnpix2;
	double a[14], b[14];
	double rh, rm, rs;
	double dd, dm, ds;
	double xc, px, yc, py;
	double x, y, x2y2;
	double pltscale;
	double rac, decc, ra;
	double xi, nu;
	double tandecc;
	int i;

	if (getRealFITS (fip, "CNPIX1", &cnpix1) < 0
			    || getRealFITS (fip, "CNPIX2", &cnpix2) < 0) {
	    xe_msg ("No CNPIX fields", 1);
	    return(-1);
	}

	if (getRealFITS (fip, "PLTSCALE", &pltscale) < 0
			    || getRealFITS (fip, "PPO3", &xc) < 0
			    || getRealFITS (fip, "PPO6", &yc) < 0
			    || getRealFITS (fip, "XPIXELSZ", &px) < 0
			    || getRealFITS (fip, "YPIXELSZ", &py) < 0) {
	    xe_msg ("No Plate scale fields", 1);
	    return(-1);
	}

	X += cnpix1 - 0.5;
	Y += cnpix2 - 0.5;
	x = (xc - px*X)/1000.0;
	y = (py*Y - yc)/1000.0;

	for (i = 1; i <= 13; i++) {
	    char ax[32], ay[32];
	    sprintf (ax, "AMDX%d", i);
	    sprintf (ay, "AMDY%d", i);
	    if (getRealFITS (fip, ax, &a[i]) < 0 ||
					    getRealFITS (fip, ay, &b[i]) < 0) {
		xe_msg ("No AMD fields", 1);
		return (-1);
	    }
	}

	x2y2 = x*x + y*y;
	xi = a[1]*x + a[2]*y + a[3] + a[4]*x*x + a[5]*x*y + a[6]*y*y
		+ a[7]*x2y2 + a[8]*x*x*x + a[9]*x*x*y + a[10]*x*y*y
		+ a[11]*y*y*y + a[12]*x*x2y2 + a[13]*x*x2y2*x2y2;
	xi = degrad(xi/3600.0);
	nu = b[1]*y + b[2]*x + b[3] + b[4]*y*y + b[5]*x*y + b[6]*x*x
		+ b[7]*x2y2 + b[8]*y*y*y + b[9]*x*y*y + b[10]*x*x*y
		+ b[11]*x*x*x + b[12]*y*x2y2 + b[13]*y*x2y2*x2y2;
	nu = degrad(nu/3600.0);


	if (getRealFITS (fip, "PLTRAH", &rh) < 0
			    || getRealFITS (fip, "PLTRAM", &rm) < 0
			    || getRealFITS (fip, "PLTRAS", &rs) < 0
			    || getRealFITS (fip, "PLTDECD", &dd) < 0
			    || getRealFITS (fip, "PLTDECM", &dm) < 0
			    || getRealFITS (fip, "PLTDECS", &ds) < 0) {
	    xe_msg ("No PLT* fields", 1);
	    return(-1);
	}
	rac = rs/3600.0 + rm/60.0 + rh;
	rac = hrrad(rac);
	decc = ds/3600.0 + dm/60.0 + dd;
	if (getStringFITS (fip, "PLTDECSN", buf) < 0) {
	    xe_msg ("No PLTDECSN field", 1);
	    return(-1);
	}
	if (buf[0] == '-')
	    decc = -decc;
	decc = degrad(decc);
	tandecc = tan(decc);

	ra = atan ((xi/cos(decc))/(1.0-nu*tandecc)) + rac;
	if (ra < 0)
	    ra += 2*PI;
	*rap = ra;
	*decp = atan (((nu + tandecc)*cos(ra-rac))/(1.0 - nu*tandecc));

	return (0);
}

/* pop up a modal dialog to allow aborting the pixel reading.
 */
static void
stopd_up()
{
	if (!stopd_w) {
	    Arg args[20];
	    int n;

	    n = 0;
	    XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	    XtSetArg(args[n], XmNdefaultPosition, False);  n++;
	    XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL);  n++;
	    XtSetArg(args[n], XmNtitle, "xephem Net Stop");  n++;
	    stopd_w = XmCreateInformationDialog(toplevel_w, "NetStop", args, n);
	    set_something (stopd_w, XmNcolormap, (XtArgVal)xe_cm);
	    XtAddCallback (stopd_w, XmNokCallback, stopd_cb, (XtPointer)0);
	    XtAddCallback (stopd_w, XmNmapCallback, prompt_map_cb, NULL);
	    XtUnmanageChild(XmMessageBoxGetChild(stopd_w,
						    XmDIALOG_CANCEL_BUTTON));
	    XtUnmanageChild(XmMessageBoxGetChild(stopd_w,XmDIALOG_HELP_BUTTON));
	    set_xmstring(stopd_w, XmNmessageString, "Press Stop to cancel the\nFITS network read ...");
	    set_xmstring(stopd_w, XmNokLabelString, "Stop");
	}

	XtManageChild (stopd_w);
	XFlush (XtDisplay(toplevel_w));
	XmUpdateDisplay (toplevel_w);
	XSync (XtDisplay(toplevel_w), False);
}

/* called when the user presses the Stop button */
/* ARGSUSED */
static void
stopd_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	stopd_down();
}

/* bring down the network connection.
 * called in several contexts so be very conservative.
 */
static void
stopd_down()
{
	if (sockfd >= 0) {
	    (void) close (sockfd);
	    sockfd = -1;
	}
	if (readId) {
	    XtRemoveInput (readId);
	    readId = (XtInputId)0;
	}
	if (stopd_w && XtIsManaged(stopd_w)) {
	    XtUnmanageChild (stopd_w);
	    XmUpdateDisplay(toplevel_w);
	}
}

/* poll whether the user wants to stop.
 * return -1 to stop, else 0.
 */
static int
stopd_check()
{
	XmUpdateDisplay (toplevel_w);
	XSync (XtDisplay(toplevel_w), False);

	/* if user stopped the connection, stopd_cb() closed the socket */
	return (sockfd >= 0 ? 0 : -1);
}
