/* code to manage the stuff on the mars display.
 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>

#if defined(__STDC__)
#include <stdlib.h>
#else
extern void *malloc(), *realloc();
#endif

#if defined(_POSIX_SOURCE)
#include <unistd.h>
#else
extern int close();
#endif

#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/DrawingA.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/ScrolledW.h>

#include "P_.h"
#include "astro.h"
#include "circum.h"
#include "preferences.h"
#include "fits.h"


extern Widget toplevel_w;
extern XtAppContext xe_app;
#define XtD XtDisplay(toplevel_w)
extern Colormap xe_cm;


extern FILE *fopenh P_((char *name, char *how));
extern Now *mm_get_now P_((void));
extern Obj *db_basic P_((int id));
extern Obj *db_scan P_((DBScan *sp));
extern char *getXRes P_((char *name));
extern double atod P_((char *buf));
extern double delra P_((double dra));
extern int alloc_ramp P_((Display *dsp, XColor *basep, Colormap cm,
    Pixel pix[], int maxn));
extern int any_ison P_((void));
extern int fs_fetch P_((Now *np, double ra, double dec, double fov,
    double mag, ObjF **opp));
extern int get_color_resource P_((Widget w, char *cname, Pixel *p));
extern int magdiam P_((int fmag, int magstp, double scale, double mag,
    double size));
extern int openh P_((char *name, int flags, ...));
extern void db_scaninit P_((DBScan *sp, int mask, ObjF *op, int nop));
extern void db_update P_((Obj *op));
extern void f_date P_((Widget w, double jd));
extern void f_double P_((Widget w, char *fmt, double f));
extern void f_mtime P_((Widget w, double t));
extern void f_pangle P_((Widget w, double a));
extern void f_showit P_((Widget w, char *s));
extern void f_dm_angle P_((Widget w, double a));
extern void fs_dm_angle P_((char out[], double a));
extern void fs_manage P_((void));
extern void get_something P_((Widget w, char *resource, XtArgVal value));
extern void get_xmstring P_((Widget w, char *resource, char **txtp));
extern void get_views_font P_((Display *dsp, XFontStruct **fspp));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void obj_pickgc P_((Obj *op, Widget w, GC *gcp));
extern void pm_set P_((int percentage));
extern void range P_((double *v, double r));
extern void register_selection P_((char *name));
extern void set_something P_((Widget w, char *resource, XtArgVal value));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void solve_sphere P_((double A, double b, double cosc, double sinc,
    double *cosap, double *Bp));
extern void sv_draw_obj P_((Display *dsp, Drawable win, GC gc, Obj *op, int x,
    int y, int diam, int dotsonly));
extern char *syserrstr P_((void));
extern void timestamp P_((Now *np, Widget w));
extern void watch_cursor P_((int want));
extern void wtip P_((Widget w, char *tip));
extern void xe_msg P_((char *msg, int app_modal));
extern void zero_mem P_((void *loc, unsigned len));

#define	POLE_RA		degrad(317.61)
#define	POLE_DEC	degrad(52.85)

typedef struct {
    Obj o;			/* copy of sky object info */
    int x, y;			/* X location on mda_w */
} SkyObj;

static SkyObj *skyobjs;		/* malloced list of objects in sky background */
static int nskyobjs;		/* number of objects in skyobjs[] list */

static void resetSkyObj P_((void));
static void addSkyObj P_((Obj *op, int x, int y));
static Obj *closeSkyObj P_((int x, int y));
static void m_popup P_((XEvent *ep));
static void m_create_skypopup P_((void));

static void m_create_form P_((void));
static void m_create_msform P_((void));
static void m_set_buttons P_((int whether));
static void m_set_a_button P_((Widget pbw, int whether));
static void m_mstats_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_selection_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_cml_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_slt_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_see_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_aim_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_unmap_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_init_gcs P_((void));
static void m_help_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_helpon_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_option_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_exp_cb P_((Widget w, XtPointer client, XtPointer call));
static void m_pointer_eh P_((Widget w, XtPointer client, XEvent *ev,
    Boolean *continue_to_dispatch));
static int m_fits P_((void));
static int xy2ll P_((int x, int y, double *ltp, double *lgp));
static int ll2xy P_((double l, double L, int *xp, int *yp));
static void m_redraw P_((void));
static void m_refresh P_((XExposeEvent *ep));
static void m_getsize P_((Drawable d, unsigned *wp, unsigned *hp,unsigned *dp));
static void m_stats P_((void));
static void m_draw P_((void));
static int mxim_create P_((void));
static void mxim_setup P_((void));
static void mBWdither P_((void));
static void m_orientation P_((void));
static void m_sizecal P_((void));
static int m_labels P_((void));
static void m_sky P_((void));
static void m_grid P_((void));
static void m_reportloc P_((int x, int y));
static void m_eqproject P_((Now *np, double ra, double dec, int *xp, int *yp));
static void mars_cml P_((Now *np, double *cmlp, double *sltp, double *pap));

/* the FITS map image, read into mimage, is a 180x360 8bit FITS file of albedo
 * in 1 degree steps starting at longitude 0 on the left, latitude -PI/2 on the
 * top. use it to build a 360x360 view of the visible face in m_xim.
 */
#define	XEVERS		3	/* required version in header */
#define	IMSZ		360	/* n rows/cols in X image, n cols in FITS map */
#define	IMR	(IMSZ/2)	/* n rows in FITS map, handy radius */
#define	MAXCOLORS	40	/* max colors to use for image */
#define	BORD		150	/* extra drawing area border for labels/stars */
#define	LGAP		20	/* gap between NSEW labels and image edge */
#define	FMAG		12	/* faintest mag of sky background object */
#define	MAXR		10	/* max gap when picking sky objects, pixels */	
#define	GSP	degrad(15.0)	/* grid spacing */
#define	FSP	(GSP/5.)	/* fine spacing */
#define	XRAD		5	/* radius of central mark, pixels */
#define	SMPLSZ		2	/* pixel sample size */
#define	MAXOBJR 	50	/* clamp obj symbols to this max rad, pixels */

static FImage marsfits;		/* the Mars FITS images */
static unsigned char *mimage;	/* malloced array of raw mars image */
static Pixel mcolors[MAXCOLORS]; /* color-scale ramp for drawing image */
static int nmcolors;		/* number of pixels usable in mcolors[] */
static Pixel mbg;		/* background color for image */
static char mfxr[] = "MarsFile";/* X resource naming mars FITS file */
static char mdbxr[] = "MarsDB"; /* X resource naming mars features DB file */
static int mdepth;		/* depth of image, in bits */
static int mbpp;		/* bits per pixel in image: 1, 8, 16 or 32 */
static XImage *m_xim;		/* XImage of mars now at current size */
static double m_cml;		/* current central meridian longitude, rads */
static double m_slt;		/* current subearth latitude, rads */
static double m_sslt, m_cslt;	/*   " handy sin/cos */
static double m_pa;		/* current N pole position angle, rads */
static double m_spa, m_cpa;	/*   " handy sin/cos */
static int m_seeing;		/* seeing, arc seconds */
static Obj *marsop;		/* current mars info */
static double cm_dec, sm_dec;	/* handy cos and sin of mars' dec */

/* main's widgets */
static Widget mform_w;		/* main mars form dialog */
static Widget mda_w;		/* image view DrawingArea */
static Pixmap m_pm;		/* image view staging pixmap */
static Widget dt_w;		/* main date/time stamp widget */

/* "More info" stats widgets */
static Widget msform_w;		/* statistics form dialog */
static Widget sdt_w;		/* statistics date/time stamp widget */
static Widget lat_w, lng_w;	/* lat/long under cursor */
static Widget cml_w;		/* central merdian longitude PB */
static Widget cmls_w;		/* central merdian longitude scale */
static Widget slt_w;		/* subearth latitude PB */
static Widget slts_w;		/* subearth latitude scale */
static Widget see_w;		/* seeing label */
static int mswasman;            /* whether "More info" form was managed */
static int fakepos;		/* set when cml/slt/pa etc are not true */

/* sky background object's widgets */
static Widget skypu_w;		/* sky popup */
static Widget skypu_name_w;	/* popup name label */
static Widget skypu_mag_w;	/* popup mag label */
static Widget skypu_aim_w;	/* popup aim PB */
static double skypu_l;		/* latitude if Point PB is activated */
static double skypu_L;		/* longitude if Point PB is activated */

/* field star support */
static ObjF *fstars;            /* malloced list of field stars, or NULL */
static int nfstars;             /* number of entries in fstars[] */
static double fsdec, fsra;      /* location when field stars were loaded */
#define FSFOV   degrad(0.5)     /* size of FOV to fetch, rads */
#define FSMAG   FMAG            /* limiting mag for fetch */
#define FSMOVE  degrad(.1)      /* reload when mars has moved this far, rads */
static void m_loadfs P_((Now *np, double ra, double dec));

static GC m_fgc, m_bgc, m_agc;	/* various GCs */
static XFontStruct *m_fsp;	/*label font */

static int m_selecting;		/* set while our fields are being selected */

/* options list */
typedef enum {
    GRID_OPT, LABELS_OPT, FLIPLR_OPT, FLIPTB_OPT, SKY_OPT, COURSE_OPT,
    N_OPT
} Option;
static int option[N_OPT];
static Widget option_w[N_OPT];

/* Image to X Windows coord converter macros.
 * image coords have 0 in the center +x/right +y/down of the m_xim,
 * X Windows coords have upper left +x/right +y/down of the mda_w.
 */
#define	IX2XX(x)	(BORD + IMR + (option[FLIPLR_OPT] ? -(x) : (x)))
#define	IY2XY(y)	(BORD + IMR + (option[FLIPTB_OPT] ? -(y) : (y)))
#define	XX2IX(x)	(((x) - (BORD + IMR)) * (option[FLIPLR_OPT] ? -1 : 1))
#define	XY2IY(y)	(((y) - (BORD + IMR)) * (option[FLIPTB_OPT] ? -1 : 1))

/* called when the mars view is activated via the main menu pulldown.
 * if first time, build everything, else just toggle whether we are mapped.
 * allow for retrying to read the image file each time until find it.
 */
void
mars_manage ()
{
	if (!mform_w) {
	    /* one-time-only work */

	    /* build dialogs */
	    m_create_form();
	    m_create_msform();
	    m_create_skypopup();

	    /* establish depth, colors and bits per pixel */
	    get_something (mda_w, XmNdepth, (XtArgVal)&mdepth);
	    m_init_gcs();
	    mbpp = (mdepth == 1 || nmcolors == 2) ? 1 :
				    (mdepth>=17 ? 32 : (mdepth >= 9 ? 16 : 8));

	    /* establish initial mars circumstances */
	    m_stats();
	}

	/* make sure we can find the FITS file. */
	if (!mimage)
	    if (m_fits() < 0)
		return;

	if (XtIsManaged(mform_w)) {
	    XtUnmanageChild (mform_w);
	} else {
	    XtManageChild (mform_w);
	    if (mswasman) {
		XtManageChild (msform_w);
		m_set_buttons(m_selecting);
	    }
	    /* the expose will do the drawing */
	}
}

/* commanded from main to update with a new set of circumstances */
void
mars_update (np, how_much)
Now *np;
int how_much;
{
	if (!mform_w)
	    return;
	if (!XtIsManaged(mform_w) && !any_ison() && !how_much)
	    return;

	/* new mars stats */
	m_stats();

	/* only if we're up */
	if (m_pm)
	    m_redraw();
}

int
mars_ison()
{
	return (mform_w && XtIsManaged(mform_w));
}

/* called by other menus as they want to hear from our buttons or not.
 * the "on"s and "off"s stack - only really redo the buttons if it's the
 * first on or the last off.
 */
void
mars_selection_mode (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	m_selecting += whether ? 1 : -1;

	if (mars_ison()) {
	    if ((whether && m_selecting == 1)     /* first one to want on */
		|| (!whether && m_selecting == 0) /* last one to want off */)
		m_set_buttons (whether);
	}
}

/* called to put up or remove the watch cursor.  */
void
mars_cursor (c)
Cursor c;
{
	Window win;

	if (mform_w && (win = XtWindow(mform_w)) != 0) {
	    Display *dsp = XtDisplay(mform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}

	if (msform_w && (win = XtWindow(msform_w)) != 0) {
	    Display *dsp = XtDisplay(msform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}
}

/* called when the database has changed.
 * if we are drawing background, we'd best redraw everything.
 */
void
mars_newdb (appended)
int appended;
{
	if (option[SKY_OPT] && mars_ison())
	    m_redraw();
}

static void
m_create_form()
{
	typedef struct {
	    Option o;		/* which option */
	    char *name;		/* name of TB */
	    char *title;	/* title string of option */
	    char *tip;		/* widget tip */
	} OpSetup;
	static OpSetup ops[] = {
	    {LABELS_OPT,	"Labels",	"Feature labels",
	    	"When on, major surface features will be labeled"},
	    {SKY_OPT,		"SkyBkg",	"Sky background",
		"When on, sky will include database objects and Field Stars"},
	    {FLIPTB_OPT,	"FlipTB",	"Flip T/B",
	    	"Flip the map top-to-bottom"},
	    {FLIPLR_OPT,	"FlipLR",	"Flip L/R",
	    	"Flip the map left-to-right"},
	    {GRID_OPT,		"Grid",		"Grid",
	    	"When on, overlay 15 degree grid and mark Sub-Earth location"},
	    {COURSE_OPT,	"LowPrec",	"Course+Faster",
		"Draw map with lower precision, but somewhat faster"},
	};
	typedef struct {
	    char *label;	/* what goes on the help label */
	    char *key;		/* string to call hlp_dialog() */
	} HelpOn;
	static HelpOn helpon[] = {
	    {"Intro...",	"Mars - intro"},
	    {"on Mouse...",	"Mars - mouse"},
	    {"on Control...",	"Mars - control"},
	    {"on View...",	"Mars - view"},
	};
	Widget mb_w, pd_w, cb_w;
	Widget msw_w;
	Widget w;
	unsigned long mask;
	XmString str;
	Arg args[20];
	int i;
	int n;

	/* create master form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNhorizontalSpacing, 5); n++;
	XtSetArg (args[n], XmNverticalSpacing, 5); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	mform_w = XmCreateFormDialog (toplevel_w, "Mars", args, n);
	set_something (mform_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (mform_w, XmNhelpCallback, m_help_cb, 0);
	XtAddCallback (mform_w, XmNunmapCallback, m_unmap_cb, 0);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Mars view"); n++;
	XtSetValues (XtParent(mform_w), args, n);

	/* create the menu bar across the top */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	mb_w = XmCreateMenuBar (mform_w, "MB", args, n);
	XtManageChild (mb_w);

	/* make the Control pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ControlPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'C'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "ControlCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "Control");
	    XtManageChild (cb_w);

	    /* the "Field stars" push button */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "FS", args, n);
	    set_xmstring (w, XmNlabelString, "Setup Field Stars...");
	    XtAddCallback (w, XmNactivateCallback, (XtCallbackProc)fs_manage,0);
	    wtip (w, "Define where GSC and PPM catalogs are to be found");
	    XtManageChild (w);

	    /* add a separator */
	    n = 0;
	    w = XmCreateSeparator (pd_w, "CtS", args, n);
	    XtManageChild (w);

	    /* add the close button */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Close", args, n);
	    XtAddCallback (w, XmNactivateCallback, m_close_cb, 0);
	    wtip (w, "Close this and all supporting dialogs");
	    XtManageChild (w);

	/* make the View pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ViewPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'V'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "ViewCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "View");
	    XtManageChild (cb_w);

	    /* add options */

	    for (i = 0; i < XtNumber(ops); i++) {
		OpSetup *osp = &ops[i];
		Option o = osp->o;

		n = 0;
		XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
		XtSetArg (args[n], XmNindicatorType, XmN_OF_MANY); n++;
		w = XmCreateToggleButton (pd_w, osp->name, args, n);
		XtAddCallback(w, XmNvalueChangedCallback, m_option_cb,
								(XtPointer)o);
		set_xmstring (w, XmNlabelString, osp->title);
		option[o] = XmToggleButtonGetState (w);
		option_w[o] = w;
		if (osp->tip)
		    wtip (w, osp->tip);
		XtManageChild (w);
	    }

	    /* add a separator */

	    n = 0;
	    w = XmCreateSeparator (pd_w, "Sep", args, n);
	    XtManageChild (w);

	    /* add the More Info control */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Stats", args, n);
	    set_xmstring (w, XmNlabelString, "More info...");
	    XtAddCallback (w, XmNactivateCallback, m_mstats_cb, NULL);
	    wtip (w, "Display additional information and controls");
	    XtManageChild (w);

	/* make the help pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "HelpPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'H'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "HelpCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "Help");
	    XtManageChild (cb_w);
	    set_something (mb_w, XmNmenuHelpWidget, (XtArgVal)cb_w);

	    for (i = 0; i < XtNumber(helpon); i++) {
		HelpOn *hp = &helpon[i];

		str = XmStringCreate (hp->label, XmSTRING_DEFAULT_CHARSET);
		n = 0;
		XtSetArg (args[n], XmNlabelString, str); n++;
		XtSetArg (args[n], XmNmarginHeight, 0); n++;
		w = XmCreatePushButton (pd_w, "Help", args, n);
		XtAddCallback (w, XmNactivateCallback, m_helpon_cb,
							(XtPointer)(hp->key));
		XtManageChild (w);
		XmStringFree(str);
	    }

	/* make a label for the date stamp */

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrecomputeSize, False); n++;
	dt_w = XmCreateLabel (mform_w, "DateStamp", args, n);
	timestamp (mm_get_now(), dt_w);	/* sets initial size */
	wtip (dt_w, "Date and Time for which map is computed");
	XtManageChild (dt_w);

	/* make a drawing area in a scrolled window for the image view */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNscrollingPolicy, XmAUTOMATIC); n++;
	XtSetArg (args[n], XmNvisualPolicy, XmVARIABLE); n++;
	msw_w = XmCreateScrolledWindow (mform_w, "MarsSW", args, n);
	XtManageChild (msw_w);

	    n = 0;
	    XtSetArg (args[n], XmNmarginWidth, 0); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNwidth, 2*BORD+IMSZ); n++;
	    XtSetArg (args[n], XmNheight, 2*BORD+IMSZ); n++;
	    mda_w = XmCreateDrawingArea (msw_w, "Map", args, n);
	    XtAddCallback (mda_w, XmNexposeCallback, m_exp_cb, NULL);
            mask = Button1MotionMask | ButtonPressMask | PointerMotionHintMask;
	    XtAddEventHandler (mda_w, mask, False, m_pointer_eh, 0);
	    XtManageChild (mda_w);

	    /* SW assumes work is its child but just to be tiddy about it .. */
	    set_something (msw_w, XmNworkWindow, (XtArgVal)mda_w);

	/* match SW background to DA */
	get_something (msw_w, XmNclipWindow, (XtArgVal)&w);
	if (w) {
	    Pixel p;
	    get_something (mda_w, XmNbackground, (XtArgVal)&p);
	    set_something (w, XmNbackground, (XtArgVal)p);
	}
}

/* create the "more info" stats dialog */
static void
m_create_msform()
{
	typedef struct {
	    char *label;
	    Widget *wp;
	    char *tip;
	} DItem;
	static DItem citems[] = {
	    {"Under Cursor:", NULL, NULL},
	    {"Latitude +N:",  &lat_w, "Martian Latitude under cursor"},
	    {"Longitude +E:", &lng_w, "Martian Longitude under cursor"},
	};
	Widget sees_w;
	Widget rc_w;
	Widget sep_w;
	Widget w;
	char str[32];
	Arg args[20];
	int n;
	int i;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNverticalSpacing, 10); n++;
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	msform_w = XmCreateFormDialog (toplevel_w, "MarsStats", args, n);
	set_something (msform_w, XmNcolormap, (XtArgVal)xe_cm);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Mars info"); n++;
	XtSetValues (XtParent(msform_w), args, n);

	/* make a rowcolumn to hold the cursor tracking info */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNtopOffset, 10); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNspacing, 5); n++;
	XtSetArg (args[n], XmNpacking, XmPACK_COLUMN); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNnumColumns, XtNumber(citems)); n++;
	XtSetArg (args[n], XmNisAligned, False); n++;
	rc_w = XmCreateRowColumn (msform_w, "SRC", args, n);
	XtManageChild (rc_w);

	    for (i = 0; i < XtNumber(citems); i++) {
		DItem *dp = &citems[i];

		n = 0;
		XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
		w = XmCreateLabel (rc_w, "CLbl", args, n);
		set_xmstring (w, XmNlabelString, dp->label);
		XtManageChild (w);

		n = 0;
		XtSetArg (args[n], XmNrecomputeSize, False); n++;
		XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
		w = XmCreateLabel (rc_w, "CVal", args, n);
		set_xmstring (w, XmNlabelString, " ");

		if (dp->wp)
		    *(dp->wp) = w;
		if (dp->tip)
		    wtip (w, dp->tip);
		XtManageChild (w);
	    }


	/* make a separator between the 2 data sets */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, rc_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	sep_w = XmCreateSeparator (msform_w, "Sep1", args, n);
	XtManageChild(sep_w);

	/* make the slt row */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sep_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (msform_w, "SLTL", args, n);
	set_xmstring (w, XmNlabelString, "Sub Earth Lat:");
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sep_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	XtSetArg (args[n], XmNuserData, "Mars.SubLat"); n++;
	slt_w = XmCreatePushButton (msform_w, "SLTVal", args, n);
	XtAddCallback (slt_w, XmNactivateCallback, m_selection_cb, NULL);
	wtip (slt_w, "Martian latitude at center of map");
	XtManageChild (slt_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, slt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNminimum, -90); n++;
	XtSetArg (args[n], XmNmaximum, 90); n++;
	XtSetArg (args[n], XmNscaleMultiple, 10); n++;
	XtSetArg (args[n], XmNshowValue, False); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	slts_w = XmCreateScale (msform_w, "SLTS", args, n);
	XtAddCallback (slts_w, XmNvalueChangedCallback, m_slt_cb, NULL);
	XtAddCallback (slts_w, XmNdragCallback, m_slt_cb, NULL);
	wtip (slts_w, "Set arbitrary Martian latitude for center of map");
	XtManageChild (slts_w);

	/* make the cml row */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, slts_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (msform_w, "CMLL", args, n);
	set_xmstring (w, XmNlabelString, "Central M Long:");
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, slts_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	XtSetArg (args[n], XmNuserData, "Mars.CML"); n++;
	cml_w = XmCreatePushButton (msform_w, "CMLVal", args, n);
	XtAddCallback (cml_w, XmNactivateCallback, m_selection_cb, NULL);
	wtip (cml_w, "Martian longitude at center of map");
	XtManageChild (cml_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, cml_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNminimum, 0); n++;
	XtSetArg (args[n], XmNmaximum, 359); n++;
	XtSetArg (args[n], XmNscaleMultiple, 10); n++;
	XtSetArg (args[n], XmNshowValue, False); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	cmls_w = XmCreateScale (msform_w, "CMLS", args, n);
	XtAddCallback (cmls_w, XmNvalueChangedCallback, m_cml_cb, NULL);
	XtAddCallback (cmls_w, XmNdragCallback, m_cml_cb, NULL);
	wtip (cmls_w, "Set arbitrary Martian longitude for center of map");
	XtManageChild (cmls_w);

	/* make the seeing row */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, cmls_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (msform_w, "SeeingL", args, n);
	set_xmstring (w, XmNlabelString, "Seeing (arc seconds):");
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, cmls_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
	see_w = XmCreateLabel (msform_w, "SeeingV", args, n);
	wtip (see_w, "Image is blurred to simulate this seeing value");
	XtManageChild (see_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, see_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNshowValue, False); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	sees_w = XmCreateScale (msform_w, "Seeing", args, n);
	XtAddCallback (sees_w, XmNvalueChangedCallback, m_see_cb, NULL);
	XtAddCallback (sees_w, XmNdragCallback, m_see_cb, NULL);
	wtip (sees_w, "Set desired seeing conditions");
	XtManageChild (sees_w);

	/* pick up initial value */
	XmScaleGetValue (sees_w, &m_seeing);
	(void) sprintf (str, "%2d", m_seeing);
	set_xmstring (see_w, XmNlabelString, str);

	/* add a label for the current date/time stamp */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sees_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	sdt_w = XmCreateLabel (msform_w, "SDTstamp", args, n);
	wtip (sdt_w, "Date and Time for which data are computed");
	XtManageChild (sdt_w);

	/* add a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sdt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	sep_w = XmCreateSeparator (msform_w, "Sep3", args, n);
	XtManageChild (sep_w);

	/* add a close button */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sep_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomOffset, 10); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 20); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 80); n++;
	w = XmCreatePushButton (msform_w, "Close", args, n);
	XtAddCallback (w, XmNactivateCallback, m_mstats_cb, NULL);
	wtip (w, "Close this dialog");
	XtManageChild (w);
}

/* go through all the buttons pickable for plotting and set whether they
 * should appear to look like buttons or just flat labels.
 */
static void
m_set_buttons (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	m_set_a_button (cml_w, whether);
	m_set_a_button (slt_w, whether);
}

/* set whether the given button looks like a label.
 */
static void
m_set_a_button(pbw, whether)
Widget pbw;
int whether;
{
	static Arg look_like_button[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) True},
	    {XmNtraversalOn, (XtArgVal) True},
	};
	static Arg look_like_label[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) False},
	    {XmNtraversalOn, (XtArgVal) False},
	};
	static int called;
	Arg *ap;
	int na;

	if (!called) {
	    /* get baseline label and shadow appearances.
	     */
            Pixel topshadcol, botshadcol, bgcol;
            Pixmap topshadpm, botshadpm;
	    Arg args[20];
	    Widget tmpw;
	    int n;

	    n = 0;
	    tmpw = XmCreatePushButton (mform_w, "tmp", args, n);

	    n = 0;
            XtSetArg (args[n], XmNtopShadowColor, &topshadcol); n++;
            XtSetArg (args[n], XmNbottomShadowColor, &botshadcol); n++;
            XtSetArg (args[n], XmNtopShadowPixmap, &topshadpm); n++;
            XtSetArg (args[n], XmNbottomShadowPixmap, &botshadpm); n++;
	    XtSetArg (args[n], XmNbackground, &bgcol); n++;
	    XtGetValues (tmpw, args, n);

            look_like_button[0].value = topshadcol;
            look_like_button[1].value = botshadcol;
            look_like_button[2].value = topshadpm;
            look_like_button[3].value = botshadpm;
            look_like_label[0].value = bgcol;
            look_like_label[1].value = bgcol;
            look_like_label[2].value = XmUNSPECIFIED_PIXMAP;
            look_like_label[3].value = XmUNSPECIFIED_PIXMAP;

	    XtDestroyWidget (tmpw);
	     
	    called = 1;
	}

	if (whether) {
	    ap = look_like_button;
	    na = XtNumber(look_like_button);
	} else {
	    ap = look_like_label;
	    na = XtNumber(look_like_label);
	}

	XtSetValues (pbw, ap, na);
}

/* callback from the Close button is activated on the stats menu or when
 * the More Info button is activated. they each do the same thing.
 */
/* ARGSUSED */
static void
m_mstats_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if ((mswasman = XtIsManaged(msform_w)) != 0)
	    XtUnmanageChild (msform_w);
	else {
	    XtManageChild (msform_w);
	    m_set_buttons(m_selecting);
	}
}

/* callback from CML or SLT button being activated.
 */
/* ARGSUSED */
static void
m_selection_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (m_selecting) {
	    char *name;
	    get_something (w, XmNuserData, (XtArgVal)&name);
	    register_selection (name);
	}
}

/* callback from the CML scale */
/* ARGSUSED */
static void
m_cml_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmScaleCallbackStruct *sp = (XmScaleCallbackStruct *)call;
	char str[32];
	int v;

	v = sp->value;
	(void) sprintf (str, "%3d.0", v);
	set_xmstring (cml_w, XmNlabelString, str);

	if (sp->reason == XmCR_VALUE_CHANGED) {
	    if (fabs(m_cml - degrad(v)) >= degrad(1)) {
		m_cml = degrad(v);
		fakepos = 1;
		m_redraw();
	    }
	}
}

/* callback from the SLT scale */
/* ARGSUSED */
static void
m_slt_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmScaleCallbackStruct *sp = (XmScaleCallbackStruct *)call;
	char str[32];
	int v;

	v = sp->value;
	(void) sprintf (str, "%3d.0", v);
	set_xmstring (slt_w, XmNlabelString, str);

	if (sp->reason == XmCR_VALUE_CHANGED) {
	    if (fabs(m_cml - degrad(v)) >= degrad(1)) {
		m_slt = degrad(v);
		m_sslt = sin(m_slt);
		m_cslt = cos(m_slt);
		fakepos = 1;
		m_redraw();
	    }
	}
}

/* callback from the Seeing scale */
/* ARGSUSED */
static void
m_see_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmScaleCallbackStruct *sp = (XmScaleCallbackStruct *)call;
	char str[32];
	int v;

	v = sp->value;
	(void) sprintf (str, "%2d", v);
	set_xmstring (see_w, XmNlabelString, str);

	if (sp->reason == XmCR_VALUE_CHANGED) {
	    m_seeing = v;
	    m_redraw();
	}
}

/* callback from the Point PB */
/* ARGSUSED */
static void
m_aim_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	/* fake looking at skypu_{l,L} with pa 0 */

	m_slt = skypu_l;
	m_sslt = sin(m_slt);
	m_cslt = cos(m_slt);
	f_double (slt_w, "%5.1f", raddeg(m_slt));
	XmScaleSetValue (slts_w, (int)(raddeg(m_slt)));

	m_cml = skypu_L;
	f_double (cml_w, "%5.1f", raddeg(m_cml));
	XmScaleSetValue (cmls_w, (int)(raddeg(m_cml)));

	m_pa = 0.0;
	m_spa = 0.0;
	m_cpa = 1.0;

	fakepos = 1;

	m_redraw();
}

/* callback from mform_w being unmapped.
 */
/* ARGSUSED */
static void
m_unmap_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if ((mswasman = (msform_w && XtIsManaged(msform_w))) != 0)
	    XtUnmanageChild (msform_w);

	if (m_pm) {
	    XFreePixmap (XtD, m_pm);
	    m_pm = 0;
	}

	if (m_xim) {
	    free ((void *)m_xim->data);
	    m_xim->data = NULL;
	    XDestroyImage (m_xim);
	    m_xim = NULL;
	}

#if !defined(VMS) || (__VMS_VER < 70000000)
	/* don't free mimage on VMS V7 due to X server bug related to exposes
	 * after unmap
	 */
	if (mimage) {
	    free (mimage);
	    mimage = NULL;
	}
#endif

	if (fstars) {
	    free ((void *)fstars);
	    fstars = NULL;
	    nfstars = 0;
	}
}

/* called from Close button */
static void
m_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XtUnmanageChild (mform_w);

	/* unmapping mform_w will do all the real work */
}

/* callback from the any of the option TBs.
 * Option enum is in client.
 */
/* ARGSUSED */
static void
m_option_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	Option opt = (Option)client;
	int set;

	watch_cursor (1);

	/* toggle the option */
	option[opt] = set = XmToggleButtonGetState (w);

	switch (opt) {

	case GRID_OPT:
	    if (set) {
		m_grid();
		m_refresh (NULL);
	    } else
		m_redraw();
	    break;

	case SKY_OPT:
	    if (set) {
		m_sky();
		m_refresh (NULL);
	    } else
		m_redraw();
	    break;

	case LABELS_OPT:
	    if (set) {
		if (m_labels() < 0) {
		    XmToggleButtonSetState (option_w[LABELS_OPT], False, False);
		    option[LABELS_OPT] = 0;
		} else
		    m_refresh (NULL);
	    } else
		m_redraw();
	    break;

	case FLIPTB_OPT:
	    m_redraw();
	    break;

	case FLIPLR_OPT:
	    m_redraw();
	    break;

	case COURSE_OPT:
	    m_redraw();
	    break;

	case N_OPT:
	    break;
	}


	watch_cursor (0);
}

/* callback from the Help all button
 */
/* ARGSUSED */
static void
m_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static char *msg[] = {
	    "This is a map of Mars.",
	};

	hlp_dialog ("Mars", msg, XtNumber(msg));
}

/* callback from a specific Help button.
 * client is a string to use with hlp_dialog().
 */
/* ARGSUSED */
static void
m_helpon_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	hlp_dialog ((char *)client, NULL, 0);
}

/* expose (or reconfig) of mars image view drawing area.
 * N.B. since we are in ScrolledWindow we will never see resize events.
 */
/* ARGSUSED */
static void
m_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	XExposeEvent *e = &c->event->xexpose;
	Display *dsp = e->display;
	Window win = e->window;

	watch_cursor (1);

	switch (c->reason) {
	case XmCR_EXPOSE: {
	    /* turn off backing store */
	    static before;

	    if (!before) {
		XSetWindowAttributes swa;
		unsigned long mask = CWBackingStore;

		swa.backing_store = NotUseful;
		XChangeWindowAttributes (e->display, e->window, mask, &swa);
		before = 1;
	    }
	    break;
	    }
	default:
	    printf ("Unexpected mars mda_w event. type=%d\n", c->reason);
	    exit(1);
	}


	if (!m_pm) {
	    unsigned wid, hei, d;

	    m_getsize (win, &wid, &hei, &d);
	    if (wid != IMSZ+2*BORD || hei != IMSZ+2*BORD) {
		printf ("Mars exp_cb: Bad size: wid=%d IMSZ=%d hei=%d\n",
								wid, IMSZ, hei);
		exit(1);
	    }

	    m_pm = XCreatePixmap (dsp, win, wid, hei, d);
	    m_draw();
	}

	/* update exposed area */
	m_refresh (e);

	watch_cursor (0);
}

/* event handler from all Button events on the mda_w */
static void
m_pointer_eh (w, client, ev, continue_to_dispatch)
Widget w;
XtPointer client;
XEvent *ev;
Boolean *continue_to_dispatch;
{
	Display *dsp = ev->xany.display;
	Window win = ev->xany.window;
        int evt = ev->type;
	Window root, child;
	int rx, ry, x, y;
	unsigned mask;
	int b3p;

	XQueryPointer (dsp, win, &root, &child, &rx, &ry, &x, &y, &mask);

        b3p = evt == ButtonPress   && ev->xbutton.button == Button3;

	m_reportloc (x, y);
	if (b3p)
	    m_popup (ev);
}

/* establish mimage and m_xim and return 0 else xe_msg() and return -1 */
static int
m_fits()
{
	char msg[1024];
	char *fn;
	int fd;
	int v;

	/* fetch mars file name and open it */
	fn = getXRes (mfxr);
	if (!fn) {
	    (void) sprintf (msg, "No %s resource so can't find mars image.",
									mfxr);
	    xe_msg (msg, 1);
	    return (-1);
	}
	fd = openh (fn, 0);
	if (fd < 0) {
	    (void) sprintf (msg, "Can not open %.150s: %.50s\n",fn,syserrstr());
	    xe_msg (msg, 1);
	    return (-1);
	}

	/* read mars file into marsfits */
	if (readFITS (fd, &marsfits, msg) < 0) {
	    char msg2[1024];
	    (void) sprintf (msg2, "%s: %s", fn, msg);
	    xe_msg (msg2, 1);
	    (void) close (fd);
	    return (-1);
	}
	(void) close (fd);

	/* make sure it's the new flipped version */
	if (getIntFITS (&marsfits, "XEVERS", &v) < 0 || v != XEVERS) {
	    (void) sprintf (msg, "%s: Incorrect version", fn);
	    xe_msg (msg, 1);
	    return (-1);
	}

	/* make some sanity checks */
	if (marsfits.bitpix != 8 || marsfits.sh != IMR || marsfits.sw != IMSZ) {
	    (void) sprintf (msg, "%s: Expected %d x %d but found %d x %d", fn,
					IMR, IMSZ, marsfits.sw, marsfits.sh);
	    xe_msg (msg, 1);
	    resetFImage (&marsfits);
	    return (-1);
	}
	mimage = (unsigned char *) marsfits.image;

	/* dither mimage if we only have 2 colors to work with */
	if (mbpp == 1) 
	    mBWdither();

	/* create m_xim */
	if (mxim_create () < 0) {
	    resetFImage (&marsfits);
	    mimage = NULL;
	    return (-1);
	}

	return(0);
}

/* create m_xim of size IMSZxIMSZ, depth mdepth and bit-per-pixel mbpp.
 * make a Bitmap if only have 1 bit per pixel, otherwise a Pixmap.
 * return 0 if ok else -1 and xe_msg().
 */
static int
mxim_create ()
{
	Display *dsp = XtDisplay (mda_w);
	int nbytes = IMSZ*IMSZ*mbpp/8;
	char *data;

	/* get memory for image pixels.  */
	data = (char *) malloc (nbytes);
	if (!data) {
	    char msg[1024];
	    (void)sprintf(msg,"Can not get %d bytes for shadow pixels", nbytes);
	    xe_msg (msg, 1);
	    return (-1);
	}

	/* create the XImage */
	m_xim = XCreateImage (dsp, DefaultVisual (dsp, DefaultScreen(dsp)),
	    /* depth */         mbpp == 1 ? 1 : mdepth,
	    /* format */        mbpp == 1 ? XYBitmap : ZPixmap,
	    /* offset */        0,
	    /* data */          data,
	    /* width */         IMSZ,
	    /* height */        IMSZ,
	    /* pad */           mbpp < 8 ? 8 : mbpp,
	    /* bpl */           0);
	if (!m_xim) {
	    xe_msg ("Can not create shadow XImage", 1);
	    free ((void *)data);
	    return (-1);
	}

        m_xim->bitmap_bit_order = LSBFirst;
	m_xim->byte_order = LSBFirst;

	/* ok */
	return (0);
}

/* redraw the current scene */
static void
m_redraw()
{
	watch_cursor (1);

	XmUpdateDisplay (toplevel_w);
	m_draw ();
	m_refresh(NULL);

	watch_cursor (0);
}

/* copy the m_pm pixmap to the drawing area mda_w.
 * if ep just copy that much, else copy all.
 */
static void
m_refresh(ep)
XExposeEvent *ep;
{
	Display *dsp = XtDisplay(mda_w);
	Window win = XtWindow (mda_w);
	Pixmap pm = m_pm;
	unsigned w, h;
	int x, y;

	/* ignore of no pixmap now */
	if (!pm)
	    return;

	if (ep) {
	    x = ep->x;
	    y = ep->y;
	    w = ep->width;
	    h = ep->height;
	} else {
	    w = IMSZ+2*BORD;
	    h = IMSZ+2*BORD;
	    x = y = 0;
	}

	XCopyArea (dsp, pm, win, m_fgc, x, y, w, h, x, y);
}

/* get the width, height and depth of the given drawable */
static void
m_getsize (d, wp, hp, dp)
Drawable d;
unsigned *wp, *hp, *dp;
{
	Window root;
	int x, y;
	unsigned int bw;

	XGetGeometry (XtD, d, &root, &x, &y, wp, hp, &bw, dp);
}

/* make the various gcs, handy pixel values and fill in mcolors[].
 * N.B. just call this once.
 */
static void
m_init_gcs()
{
	Display *dsp = XtD;
	Window win = XtWindow(toplevel_w);
	Colormap cm = xe_cm;
	XGCValues gcv;
	XColor c;
	unsigned int gcm;
	Pixel fg;
	Pixel p;

	/* make gcs from Map colors
	 */
	get_something (mda_w, XmNforeground, (XtArgVal)&fg);
	get_something (mda_w, XmNbackground, (XtArgVal)&mbg);

	gcm = GCForeground | GCBackground;
	gcv.foreground = fg;
	gcv.background = mbg;
	m_fgc = XCreateGC (dsp, win, gcm, &gcv);

	gcv.foreground = mbg;
	gcv.background = fg;
	m_bgc = XCreateGC (dsp, win, gcm, &gcv);

	/* make the label marker gc.
	 * beware of monochrome screens that claim ok for any ol' color.
	 */
	if (get_color_resource (mda_w, "MarsAnnotColor", &p) < 0
						    || p == mbg || p == fg) {
	    xe_msg("Can not get Mars annotation color -- using white.", 0);
	    p = WhitePixel (dsp, 0);
	}
	gcm = GCForeground | GCBackground;
	gcv.foreground = p;
	gcv.background = mbg;
	m_agc = XCreateGC (dsp, win, gcm, &gcv);

	get_views_font (dsp, &m_fsp);
	XSetFont (dsp, m_agc, m_fsp->fid);

	/* build color ramp for image.
	 * base the scale off the foreground color.
	 */
	c.pixel = fg;
	XQueryColor (dsp, cm, &c);
	nmcolors = alloc_ramp (dsp, &c, cm, mcolors, MAXCOLORS);

	/* unless we will be using a bitmap, force color 0 to background. */
	if (nmcolors > 2)
	    mcolors[0] = mbg;

	if (nmcolors < MAXCOLORS) {
	    char msg[1024];
	    (void) sprintf (msg, "Wanted %d but only found %d colors for Mars.",
							MAXCOLORS, nmcolors);
	    xe_msg (msg, 0);
	}
}

/* update mars info and draw the stat labels */
static void
m_stats ()
{
	Now *np = mm_get_now();

	/* get fresh mars info */
	marsop = db_basic (MARS);
	db_update (marsop);
	cm_dec = cos(marsop->s_gaedec);
	sm_dec = sin(marsop->s_gaedec);

	/* compute and display the CML and SLT and polar position angle */
	mars_cml (np, &m_cml, &m_slt, &m_pa);
	m_sslt = sin(m_slt);
	m_cslt = cos(m_slt);
	m_spa = sin(m_pa);
	m_cpa = cos(m_pa);

	f_double (cml_w, "%5.1f", raddeg(m_cml));
	XmScaleSetValue (cmls_w, (int)(raddeg(m_cml)));
	f_double (slt_w, "%5.1f", raddeg(m_slt));
	XmScaleSetValue (slts_w, (int)(raddeg(m_slt)));

	/* reset fake flag */
	fakepos = 0;

	/* update time stamps too */
	timestamp (np, dt_w);
	timestamp (np, sdt_w);
}

/* setup m_xim, copy to m_pm, then add all overlay info.
 * N.B. this just fills the pixmap; call m_refresh() to copy to the screen.
 */
static void
m_draw ()
{
	/* check assumptions */
	if (!mimage) {
	    printf ("No mars mimage!\n");
	    exit (1);
	}
	if (!m_xim) {
	    printf ("No mars m_xim!\n");
	    exit (1);
	}
	if (!m_pm) {
	    printf ("No mars m_pm Pixmap!\n");
	    exit(1);
	}

	/* fill in m_xim from mimage */
	mxim_setup ();

	/* clear m_pm */
	XFillRectangle (XtD, m_pm, m_bgc, 0, 0, IMSZ+2*BORD, IMSZ+2*BORD);

	/* copy m_xim to p_pm */
	XPutImage (XtD, m_pm, m_fgc, m_xim, 0, 0, BORD, BORD, IMSZ, IMSZ);

	/* add grid, if enabled */
	if (option[GRID_OPT])
	    m_grid();

	/* add labels, if enabled */
	if (option[LABELS_OPT])
	    if (m_labels() < 0) {
		XmToggleButtonSetState (option_w[LABELS_OPT], False, False);
		option[LABELS_OPT] = 0;
	    }

	/* add sky background objects, if enabled */
	resetSkyObj();
	if (option[SKY_OPT])
	    m_sky();

	/* always add orientation markings */
	m_orientation();

	/* and the size calibration */
	m_sizecal();
}

/* put up the mars_db labels */
static int
m_labels()
{
	Display *dsp = XtDisplay (mda_w);
	Window win = m_pm;
	char buf[1024];
	char *fn;
	FILE *fp;

	/* open the file */
	fn = getXRes (mdbxr);
	if (!fn) {
	    (void) sprintf (buf, "No %s resource so can't find Mars features.",
								    mdbxr);
	    xe_msg (buf, 1);
	    return (-1);
	}
	fp = fopenh (fn, "r");
	if (!fp) {
	    (void) sprintf (buf, "%s: %s", fn, syserrstr());
	    xe_msg (buf, 1);
	    return (-1);
	}

	/* read and display each feature */
	while (fgets (buf, sizeof(buf), fp)) {
	    char name[256];
	    char type[256];
	    double lt, lg;
	    int dia;
	    int dir, asc, des;
	    XCharStruct all;
	    int x, y;
	    int nf;
	    int l;

	    /* ignore all lines that do not follow the pattern */
	    nf = sscanf(buf,"%[^|]|%[^|]|%lf|%lf|%d", name, type, &lt,&lg,&dia);
	    if (nf != 5)
		continue;

	    /* find map location in X windows coords */
	    if (ll2xy (degrad(lt), degrad(lg), &x, &y) < 0)
		continue;
	    x = IX2XX(x);
	    y = IY2XY(y);

	    /* center and display the name -- discard trailing white space */
	    l = strlen (name);
	    while (l > 0 && (name[l-1] == ' ' || name[l-1] == '\t'))
		l--;
	    name[l] = '\0';
	    XTextExtents (m_fsp, name, l, &dir, &asc, &des, &all);
	    y += all.ascent/2;
	    x -= all.width/2;
	    XDrawString (dsp, win, m_agc, x, y, name, l);
	}

	fclose(fp);
	return (0);
}

/* add background sky objects to m_pm and to skyobjs[] list */
static void
m_sky()
{
	static before;
	Now *np = mm_get_now();
	DBScan dbs;
	double scale;		/* rads per pixel */
	double mrad;		/* mars radius, rads */
	double maxr;		/* max rad to show objects, rads */
	Obj *op;

	if (!before && pref_get(PREF_EQUATORIAL) == PREF_GEO) {
	    xe_msg ("Equatorial preference should probably be set to Topocentric", 1);
	    before = 1;
	}

	/* find size and scale */
	mrad = degrad((double)(marsop->s_size)/3600.0/2.0);
	scale = mrad/IMR;
	maxr = 2*(IMR+BORD)*scale;

	/* load field stars */
	m_loadfs (np, marsop->s_gaera, marsop->s_gaedec);

	/* scan the database and draw whatever is near */
	for (db_scaninit(&dbs, ALLM, fstars, nfstars);
					    (op = db_scan (&dbs)) != NULL; ) {
	    int diam;
	    int x, y;
	    GC gc;

	    /* get fresh circumstances */
	    if (is_planet (op, MARS)) {
		/* we draw it elsewhere :-) */
		continue;
	    }
	    db_update (op);

	    /* reject if too faint or obviously too far away */
	    if (get_mag(op) > FMAG)
		continue;
	    if (fabs(op->s_gaedec - marsop->s_gaedec) > maxr ||
			    cm_dec*delra(op->s_gaera - marsop->s_gaera) > maxr)
		continue;

	    /* find location relative to image center */
	    m_eqproject (np, op->s_gaera, op->s_gaedec, &x, &y);

	    /* reject if within mars circle */
	    if (x*x + y*y <= IMR*IMR)
		continue;

	    /* allow for flipping and shift to find window coords */
	    x = IX2XX(x);
	    y = IY2XY(y);

	    /* pick a gc */
	    obj_pickgc(op, toplevel_w, &gc);

	    /* draw it -- beware of huge sizes at this scale */
	    diam = magdiam (FMAG, 2, scale, get_mag(op),
					    degrad((double)op->s_size/3600.0));
	    if (diam > MAXOBJR)
		diam = MAXOBJR;
	    sv_draw_obj (XtD, m_pm, gc, op, x, y, diam, 1);

	    /* add to skyobjs[] list */
	    addSkyObj (op, x, y);
	}

        sv_draw_obj (XtD, m_pm, (GC)0, NULL, 0, 0, 0, 0);        /* flush */
}

/* load field stars around the given location, unless current set is
 * already close enough.
 */
static void
m_loadfs (np, ra, dec)
Now *np;
double ra, dec;
{

	if (fstars && fabs(dec-fsdec)<FSMOVE && cm_dec*delra(ra-fsra)<FSMOVE)
	    return;

	if (fstars) {
	    free ((void *)fstars);
	    fstars = NULL;
	    nfstars = 0;
	}

        nfstars = fs_fetch (np, ra, dec, FSFOV, (double)FSMAG, &fstars);

	if (nfstars > 0) {
	    char msg[128];
            (void) sprintf (msg, "Mars View added %d field stars", nfstars);
            xe_msg (msg, 0);
	    fsdec = dec;
	    fsra = ra;
	}
}

/* draw the N/S E/W labels on m_pm */
static void
m_orientation()
{
	Now *np = mm_get_now();
	double mr, mra, mdec;
	double ra, dec;
	int x, y;

	mr = degrad((double)marsop->s_size/3600.0)/2 * 1.1;
	mra = marsop->s_gaera;
	mdec = marsop->s_gaedec;

	ra = mra + mr/cm_dec;
	dec = mdec;
	m_eqproject (np, ra, dec, &x, &y);
	x = IX2XX(x);
	y = IY2XY(y);
	XDrawString (XtD, m_pm, m_agc, x, y, "E", 1);

	ra = mra - mr/cm_dec;
	m_eqproject (np, ra, dec, &x, &y);
	x = IX2XX(x);
	y = IY2XY(y);
	XDrawString (XtD, m_pm, m_agc, x, y, "W", 1);

	ra = mra;
	dec = mdec + mr;
	m_eqproject (np, ra, dec, &x, &y);
	x = IX2XX(x);
	y = IY2XY(y);
	XDrawString (XtD, m_pm, m_agc, x, y, "N", 1);

	dec = mdec - mr;
	m_eqproject (np, ra, dec, &x, &y);
	x = IX2XX(x);
	y = IY2XY(y);
	XDrawString (XtD, m_pm, m_agc, x, y, "S", 1);
}

/* draw the the size calibration */
static void
m_sizecal()
{
	XID fid = XGContextFromGC (m_agc);
	int dir, asc, des;
	XCharStruct xcs;
	char buf[64];
	int l;

	(void) sprintf (buf, "%d\"", marsop->s_size);
	l = strlen (buf);
	XQueryTextExtents (XtD, fid, buf, l, &dir, &asc, &des, &xcs);

	XDrawLine (XtD, m_pm, m_agc, BORD, IMSZ+3*BORD/2, IMSZ+BORD,
							    IMSZ+3*BORD/2);
	XDrawLine (XtD, m_pm, m_agc, BORD, IMSZ+3*BORD/2-3, BORD,
							    IMSZ+3*BORD/2+3);
	XDrawLine (XtD, m_pm, m_agc, BORD+IMSZ, IMSZ+3*BORD/2-3, BORD+IMSZ,
							    IMSZ+3*BORD/2+3);
	XDrawString (XtD, m_pm, m_agc, BORD+IMSZ/2-xcs.width/2,
					    IMSZ+3*BORD/2+xcs.ascent+6, buf, l);
}

/* draw a coordinate grid over the image already on m_pm */
static void
m_grid()
{
	Display *dsp = XtDisplay (mda_w);
	double fsp = option[COURSE_OPT] ? GSP : FSP;
	Window win = m_pm;
	double lt, lg;
	int x, y;

	/* lines of constant lat */
	for (lt = -PI/2 + GSP; lt < PI/2; lt += GSP) {
	    XPoint xpt[(int)(2*PI/FSP)+2];
	    int npts = 0;

	    for (lg = 0; lg <= 2*PI+fsp; lg += fsp) {
		if (ll2xy(lt, lg, &x, &y) < 0) {
		    if (npts > 0) {
			XDrawLines (dsp, win, m_agc, xpt, npts,CoordModeOrigin);
			npts = 0;
		    }
		    continue;
		}

		if (npts >= XtNumber(xpt)) {
		    printf ("Mars lat grid overflow\n");
		    exit (1);
		}
		xpt[npts].x = IX2XX(x);
		xpt[npts].y = IY2XY(y);
		npts++;
	    }

	    if (npts > 0)
		XDrawLines (dsp, win, m_agc, xpt, npts, CoordModeOrigin);
	}

	/* lines of constant longitude */
	for (lg = 0; lg < 2*PI; lg += GSP) {
	    XPoint xpt[(int)(2*PI/FSP)+1];
	    int npts = 0;

	    for (lt = -PI/2; lt <= PI/2; lt += fsp) {
		if (ll2xy(lt, lg, &x, &y) < 0) {
		    if (npts > 0) {
			XDrawLines (dsp, win, m_agc, xpt, npts,CoordModeOrigin);
			npts = 0;
		    }
		    continue;
		}

		if (npts >= XtNumber(xpt)) {
		    printf ("Mars lng grid overflow\n");
		    exit (1);
		}
		xpt[npts].x = IX2XX(x);
		xpt[npts].y = IY2XY(y);
		npts++;
	    }

	    if (npts > 0)
		XDrawLines (dsp, win, m_agc, xpt, npts, CoordModeOrigin);
	}

	/* X marks the center, unless rotated by hand */
	if (!fakepos) {
	    XDrawLine (dsp, win, m_agc, IX2XX(-XRAD), IY2XY(-XRAD),
						    IX2XX(XRAD), IY2XY(XRAD));
	    XDrawLine (dsp, win, m_agc, IX2XX(-XRAD), IY2XY(XRAD),
						    IX2XX(XRAD), IY2XY(-XRAD));
	}
}

/* fill in m_xim from mimage and current circumstances.
 * m_xim is IMSZxIMSZ, mimage is IMSZ wide x IMR high.
 */
static void
mxim_setup ()
{
#define	SQR(x)	((x)*(x))
	int tb = option[FLIPTB_OPT];
	int lr = option[FLIPLR_OPT];
	unsigned char pict[IMSZ/SMPLSZ][IMSZ/SMPLSZ];	/* sqsz copy */
	unsigned char see[IMSZ/SMPLSZ];			/* sqsz seeing map */
	int pixseeing = IMSZ*m_seeing/(double)marsop->s_size;
	int sqsz = option[COURSE_OPT] ? 5*SMPLSZ : SMPLSZ;
	double csh;
	int lsh;
	int x, y;

	/* init the copy -- background is 0 */
	zero_mem ((void *)pict, sizeof(pict));

	/* scan to build up the sampled albedo map, and allow for flipping */
	for (y = -IMR; y < IMR; y++) {
	    int iy = (tb ? -y-1 : y) + IMR;
	    unsigned char *prow = pict[iy/sqsz];

	    pm_set ((y+IMR)*50/IMSZ);

	    for (x = -IMR; x < IMR; x++) {
		int ix = (lr ? -x-1 : x) + IMR;
		unsigned char *p = &prow[ix/sqsz];

		if (x*x + y*y < IMR*IMR && !*p) {
		    double l, L;
		    int mx, my;

		    (void) xy2ll (x, y, &l, &L);

		    my = (int)((l+PI/2)/PI*IMR);
		    mx = ((int)(L/(2*PI)*IMSZ));
		    *p = mimage[my*IMSZ+mx];
		}
	    }
	}

	/* find cos of shadow foreshortening angle based on planar
	 * sun-mars-earth triangle and earth-sun == 1.
	 * if we are faking it, turn off the shadow.
	 */
	csh = fakepos ? 1.0 : (SQR(marsop->s_sdist) + SQR(marsop->s_edist) - 1)/
					(2*marsop->s_sdist*marsop->s_edist);

	/* shadow is on left if elongation is positive and flipped lr
	 * or elongation is negative and not flipped lr.
	 */
	lsh = (marsop->s_elong > 0.0 && lr) || (marsop->s_elong < 0.0 && !lr);

	/* scan again to blur, add shadow and place real pixels */
	for (y = -IMR; y < IMR; y++) {
	    int iy = y + IMR;
	    int lx, rx;

	    pm_set (50+(y+IMR)*50/IMSZ);

	    if (lsh) {
		lx = -csh*sqrt((double)(IMR*IMR - y*y));
		rx = IMR;
	    } else {
		lx = -IMR;
		rx = csh*sqrt((double)(IMR*IMR - y*y));
	    }

	    /* fill in seeing table if first time in this sqsz-hi row */
	    if (pixseeing > 0 && (iy%sqsz) == 0) {
		for (x = -IMR; x < IMR; x += sqsz) {
		    int ix = x + IMR;
		    int s = (pixseeing+1)/2;
		    int nsc = 0;
		    int sc = 0;
		    int sx, sy;
		    int step;


		    /* establish a fairly sparce sampling step size */
		    step = 2*s/10;
		    if (step < sqsz)
			step = sqsz;

		    /* average legs of a sparse cross.
		     * tried just the corners -- leaves artifacts but fast
		     * tried filled square -- beautiful but dreadfully slow
		     */
		    for (sx = x-s; sx <= x+s; sx += step) {
			if (sx*sx + y*y <= IMR*IMR) {
			    sc += pict[iy/sqsz][(sx+IMR)/sqsz];
			    nsc++;
			}
		    }
		    for (sy = y-s; sy <= y+s; sy += step) {
			if (x*x + sy*sy <= IMR*IMR) {
			    sc += pict[(sy+IMR)/sqsz][ix/sqsz];
			    nsc++;
			}
		    }
		    see[ix/sqsz] = nsc > 0 ? sc/nsc : 0;
		}
	    }

	    for (x = -IMR; x < IMR; x++) {
		int ix = x + IMR;
		unsigned long p;
		int c;

		if (x < lx || x > rx || x*x + y*y >= IMR*IMR) {
		    c = 0;
		} else if (pixseeing > 0) {
		    c = see[ix/sqsz];
		} else
		    c = pict[iy/sqsz][ix/sqsz];

		p = mcolors[c*nmcolors/256];
		XPutPixel (m_xim, ix, iy, p);
	    }
	}
}

/* convert [x,y] to true mars lat/long, in rads.
 *   x: centered, +right, -IMR .. x .. IMR
 *   y: centered, +down,  -IMR .. y .. IMR
 * return 0 if x,y are really over the planet, else -1.
 * caller can be assured -PI/2 .. l .. PI/2 and 0 .. L .. 2*PI.
 * N.B. it is up to the caller to deal wth flipping.
 */
static int
xy2ll (x, y, lp, Lp)
int x, y;
double *lp, *Lp;
{
	double R = sqrt ((double)(x*x + y*y));
	double a;
	double ca, B;

	if (R >= IMR)
	    return (-1);

	if (y == 0)
	    a = x < 0 ? -PI/2 : PI/2;
	else 
	    a = atan2((double)x,(double)y);
	solve_sphere (a, asin(R/IMR), m_sslt, m_cslt, &ca, &B);

	*lp = PI/2 - acos(ca);
	*Lp = m_cml + B;
	range (Lp, 2*PI);

	return (0);
}

/* convert true mars lat/long, in rads, to [x,y].
 *   x: centered, +right, -IMR .. x .. IMR
 *   y: centered, +down,  -IMR .. y .. IMR
 * return 0 if loc is on the front face, else -1 (but still compute x,y);
 * N.B. it is up to the caller to deal wth flipping of the resulting locs.
 */
static int
ll2xy (l, L, xp, yp)
double l, L;
int *xp, *yp;
{
	double sR, cR;
	double A, sA, cA;

	solve_sphere (L - m_cml, PI/2 - l, m_sslt, m_cslt, &cR, &A);
	sR = sqrt(1.0 - cR*cR);
	sA = sin(A);
	cA = cos(A);

	*xp = (int)floor(IMR*sR*sA + 0.5);
	*yp = (int)floor(IMR*sR*cA + 0.5);

	return (cR > 0 ? 0 : -1);
}

/* dither mimage into a 2-intensity image: 1 and 255.
 * form 2x2 tiles whose pattern depends on intensity peak and spacial layout.
 */
static void
mBWdither()
{
	int idx[4];
	int y;

	idx[0] = 0;
	idx[1] = 1;
	idx[2] = IMSZ;
	idx[3] = IMSZ+1;

	for (y = 0; y < IMR; y += 2) {
	    unsigned char *mp = &mimage[y*IMSZ];
	    unsigned char *lp;

	    for (lp = mp + IMSZ; mp < lp; mp += 2) {
		int sum, numon;
		int i;

		sum = 0;
		for (i = 0; i < 4; i++)
		    sum += (int)mp[idx[i]];
		numon = sum*5/1021;	/* 1021 is 255*4 + 1 */

		for (i = 0; i < 4; i++)
		    mp[idx[i]] = 0;

		switch (numon) {
		case 0:
		    break;
		case 1:
		case 2:
		    mp[idx[0]] = 255;
		    break;
		case 3:
		    mp[idx[0]] = 255;
		    mp[idx[1]] = 255;
		    mp[idx[3]] = 255;
		    break;
		case 4:
		    mp[idx[0]] = 255;
		    mp[idx[1]] = 255;
		    mp[idx[2]] = 255;
		    mp[idx[3]] = 255;
		    break;
		default:
		    printf ("Bad numon: %d\n", numon);
		    exit(1);
		}
	    }
	}
}

/* report the location of x,y, which are with respect to mda_w.
 * N.B. allow for flipping and the borders.
 */
static void
m_reportloc (x, y)
int x, y;
{
	double lt, lg;

	/* convert from mda_w X Windows coords to centered m_xim coords */
	x = XX2IX(x);
	y = XY2IY(y);

	if (xy2ll (x, y, &lt, &lg) == 0) {
	    f_dm_angle (lat_w, lt);
	    f_dm_angle (lng_w, lg);
	} else {
	    set_xmstring (lat_w, XmNlabelString, " ");
	    set_xmstring (lng_w, XmNlabelString, " ");
	}
}

/* free the list of sky objects, if any */
static void
resetSkyObj ()
{
	if (skyobjs) {
	    free ((void *) skyobjs);
	    skyobjs = NULL;
	}
	nskyobjs = 0;
}

/* add op at [x,y] to the list of background sky objects */
static void
addSkyObj (op, x, y)
Obj *op;
int x, y;
{
	char *newmem;

	if (skyobjs)
	    newmem = realloc ((void *)skyobjs, (nskyobjs+1) * sizeof(SkyObj));
	else
	    newmem = malloc (sizeof(SkyObj));
	if (!newmem) {
	    xe_msg ("No mem for more sky objects", 1);
	    return;
	}

	skyobjs = (SkyObj *) newmem;
	skyobjs[nskyobjs].o = *op;
	skyobjs[nskyobjs].x = x;
	skyobjs[nskyobjs].y = y;
	nskyobjs++;
}

/* find the object in skyobjs[] that is closest to [x,y].
 * return NULL if none within MAXR.
 */
static Obj *
closeSkyObj (x, y)
int x, y;
{
	SkyObj *closesp = NULL;
	int mind = 0;
	int i;

	for (i = 0; i < nskyobjs; i++) {
	    SkyObj *sp = &skyobjs[i];
	    int d = abs(sp->x - x) + abs(sp->y - y);

	    if (!closesp || d < mind) {
		mind = d;
		closesp = sp;
	    }
	}

	if (closesp && mind <= MAXR)
	    return (&closesp->o);
	return (NULL);
}

/* called when hit button 3 over image */
static void
m_popup (ep)
XEvent *ep;
{
	XButtonEvent *bep;
	int overmars;
	int x, y;
	Obj *op;

	bep = &ep->xbutton;
	x = bep->x;
	y = bep->y;

	op = closeSkyObj (x, y);
	overmars = xy2ll (XX2IX(x), XY2IY(y), &skypu_l, &skypu_L) == 0;

	if (op) {
	    char buf[32];

	    f_showit (skypu_name_w, op->o_name);
	    XtManageChild (skypu_name_w);

	    (void) sprintf (buf, "Mag: %.2f", get_mag(op));
	    f_showit (skypu_mag_w, buf);
	    XtManageChild (skypu_mag_w);
	} else {
	    XtUnmanageChild (skypu_name_w);
	    XtUnmanageChild (skypu_mag_w);
	}

	if (overmars) {
	    XtManageChild (skypu_aim_w);
	} else {
	    XtUnmanageChild (skypu_aim_w);
	}

	if (op || overmars) {
	    XmMenuPosition (skypu_w, (XButtonPressedEvent *)ep);
	    XtManageChild (skypu_w);
	}
}

/* create the sky background object popup menu */
static void
m_create_skypopup()
{
	Widget w;
	Arg args[20];
	int n;

	n = 0;
	XtSetArg (args[n], XmNisAligned, True); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	skypu_w = XmCreatePopupMenu (mda_w, "MSKYPU", args, n);

	n = 0;
	skypu_name_w = XmCreateLabel (skypu_w, "MSKYPN", args, n);
	wtip (skypu_name_w, "Object name");
	XtManageChild (skypu_name_w);

	n = 0;
	skypu_mag_w = XmCreateLabel (skypu_w, "MSKYPM", args, n);
	wtip (skypu_mag_w, "Nominal magnitude");
	XtManageChild (skypu_mag_w);

	n = 0;
	w = XmCreateSeparator (skypu_w, "MSKYPS", args, n);
	XtManageChild (w);

	n = 0;
	skypu_aim_w = XmCreatePushButton (skypu_w, "Point", args, n);
	XtAddCallback (skypu_aim_w, XmNactivateCallback, m_aim_cb, NULL);
	wtip (skypu_aim_w, "Center this location in the view");
	XtManageChild (skypu_aim_w);
}

/* given geocentric ra and dec find image [xy] on martian equitorial projection.
 * [0,0] is center, +x martian right/west (celestial east) +y down/north.
 * this matches the raw albedo image coordinates without any flipping.
 * N.B. we do *not* allow for flipping.
 * N.B. this only works when ra and dec are near mars.
 */
static void
m_eqproject (np, ra, dec, xp, yp)
Now *np;
double ra, dec;
int *xp, *yp;
{
	double scale = IMSZ/degrad((double)marsop->s_size/3600.0); /* pix/rad */
	double xr, yr;
	double x, y;

	/* find x and y so +x is celestial right/east and +y is down/north */
	x = scale*(ra - marsop->s_gaera)*cm_dec;
	y = scale*(dec - marsop->s_gaedec);

	/* rotate by position angle, m_pa.
	 */
	xr = x*m_cpa - y*m_spa;
	yr = x*m_spa + y*m_cpa;

	*xp = (int)floor(xr + 0.5);
	*yp = (int)floor(yr + 0.5);
}

/* return:
 *   *cmlp: Martian central meridian longitude;
 *   *sltp: subearth latitude
 *   *pap:  position angle of N pole (ie, rads E of N)
 * all angles in rads.
 */

#define M_CML0  degrad(325.845)         /* Mars' CML towards Aries at M_MJD0 */
#define M_MJD0  (2418322.0 - MJD0)      /* mjd date of M_CML0 */
#define M_PER   degrad(350.891962)      /* Mars' rotation period, rads/day */

static void
mars_cml(np, cmlp, sltp, pap)
Now *np;
double *cmlp;
double *sltp;
double *pap;
{
	Obj *sp;
	double a;	/* angle from Sun ccw to Earth seen from Mars, rads */
	double Ae;	/* planetocentric longitude of Earth from Mars, rads */
	double cml0;	/* Mar's CML towards Aries, rads */
	double lc;	/* Mars rotation correction for light travel, rads */
	double tmp;

	sp = db_basic (SUN);
	db_update (sp);

	a = asin (sp->s_edist/marsop->s_edist*sin(marsop->s_hlong-sp->s_hlong));
	Ae = marsop->s_hlong + PI + a;
	cml0 = M_CML0 + M_PER*(mjd-M_MJD0) + PI/2;
	range(&cml0, 2*PI);
	lc = LTAU * marsop->s_edist/SPD*M_PER;
	*cmlp = cml0 - Ae - lc;
	range (cmlp, 2*PI);

	solve_sphere (POLE_RA - marsop->s_gaera, PI/2-POLE_DEC, sm_dec, cm_dec,
								    &tmp, pap);

	/* from Green (1985) "Spherical Astronomy", Cambridge Univ. Press,
	 * p.428. Courtesy Jim Bell.
	 */
        *sltp = asin (-sin(POLE_DEC)*sin(marsop->s_gaedec) -
					cos(POLE_DEC)*cos(marsop->s_gaedec)
						* cos(marsop->s_gaera-POLE_RA));
}
