/*
   netrik -- The ANTRIK Internet Viewer
   Copyright (C) Olaf D. Buddenhagen AKA antrik, et al (see AUTHORS)
   Published under the GNU GPL; see LICENSE for details.
*/
/*
 * parse-struct.c -- this one takes the syntax parsing tree generated by
 * parse_syntax() and creates another structure called item_tree().
 *
 * (C) 2001, 2002, 2003 antrik
 *     2002 Patrice Neff
 *
 * This item tree containes items as they are visible on the screen: text
 * blocks, blank lines, boxes aligning several other items etc.
 */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "colors.h"
#include "forms.h"
#include "items.h"
#include "syntax.h"

/* all information specific for every parse tree depth */
struct State {
   int			visible;    /* content inside this element will be displayed in output page */

   enum Text_mode	text_mode;    /* content encountered now will be stored in this mode */
   int			high;    /* highlighted text (additional to "text_mode") */

   int			list_depth;    /* nesting depth of item lists */

   struct Item		*first_item;    /* first item at this tree depth */
   struct Item		*last_item;    /* presently last item at this tree depth */

   enum Form_control	link_type;    /* indicates type of link/form element (-1: no link) */
   char			*link_value;    /* initial value of form element */
   int			form_enabled;    /* form element may be submitted */
   char			*select_name;    /* name assigned to all options inside a <select> */
   enum Form_control	select_type;    /* type (FORM_OPTION or FORM_MULTIOPTION) for all <option> elements in current select */

   int			link_start;    /* starting positon of link/anchor (string end position at time of entering this depth) */
   struct Item		*link_item;    /* string item where link started */
   int			id_attr;    /* position of the "id" (or "name") attribute of an anchor (or form control) */
};

static int fingerprint(char *start, char *end);    /* create checksum */
static struct State *push_state(void);    /* save state when descending in parse tree */
static struct State *pop_state(void);    /* recall state when ascending in parse tree */
struct Item *add_item(int virtual);    /* create new item in structure tree */
void add_text(char *text, int color);    /* append text to current string */
int find_attr(struct Element *search_el, enum Attr_type attr);    /* find attribute of specified type in specified element */

static struct State	*state;    /* (dynamic) array (stack) of states for all parse tree depths */

static struct State	*cur_state;    /* pointer to currently active state in state stack */
static int		depth;    /* depth in parse tree/state stack (-1=uninitialized state stack) */

static struct Item	*cur_item;    /* most recently added item */
static struct Item	*string_item;    /* text item containing currently active string (any new text will be added to this one) */
static struct Item	*first_item;    /* first item in structure tree */

static int		para_blank;    /* 1: blank line necessary before next item; -1: already items in this box (blank line necessary, if next item creates paragraph) */
static int		blank_depth;    /* tree depth of last "para_blank" request */


/* create a simple checksum of the desired string part */
static int fingerprint(start, end)
char	*start, *end;    /* handled memory area (string part) */
{
   char	*chr;
   int	sum;

   sum=0;
   for(chr=start; chr<end; ++chr)
      sum^=(sum<<8)^*chr;

   return sum;
}

/* save state when descending in parse tree;
 * increment depth counter, resize state stack, copy old state, and return pointer to new state */
static struct State *push_state(void)
{
   /* resize state stack to (new) tree depth */
   state=realloc(state, (++depth+1)*sizeof(struct State));
   if(state==NULL) {
      fprintf(stderr, "memory allocation error while parsing structure (in function push_state)\n");
      exit(1);
   }

   /* init new state */
   if(depth) {    /* normal push -> copy old state */
      state[depth]=state[depth-1];
      state[depth].last_item=state[depth].first_item=NULL;    /* don't inherit item list */
      state[depth].id_attr=-1;    /* don't inherit anchors */
      state[depth].link_type=-1;    /* don't inherit links */
   } else {    /* initialization of first entry -> default values */
      state[0].visible=1;
      state[0].text_mode=TM_NORMAL;
      state[0].high=0;
      state[0].list_depth=0;
      state[0].first_item=state[0].last_item=NULL;
      state[0].link_type=-1;
      state[0].form_enabled=0;
      state[0].select_name=NULL;
      state[0].id_attr=-1;
   }

   return &state[depth];    /* pointer to new state */
}

/* restore state when ascending in parse tree */
static struct State *pop_state(void)
{
   return &state[--depth];    /* previous state */
}

/*
 * Create new item and insert into item list of current parse tree depth.
 *
 * If last item was a string item, it is invalidated. If pending "para_blank",
 * first add blank line; after adding item, "para_blank" is set to -1 (=may
 * need blank line before next item).
 */

struct Item *add_item(virtual)
int	virtual;    /* don't change "para_blank" status nor break text blocks */
{
   struct Item	*new_item;    /* newly created item */

   if(para_blank==1 && !virtual) {    /* pending blank line -> first create&insert blank line into structure tree */
      /* alloc item */
      new_item=malloc(sizeof(struct Item));
      if(new_item==NULL) {
	 fprintf(stderr, "memory allocation error while parsing structure (in function create_item)\n");
	 exit(1);
      }

      /* insert into tree */
      cur_item->list_next=new_item;
      cur_item->next=new_item;    /* same level as last item */
      new_item->next=NULL;    /* last item at this tree depth */
      new_item->first_child=NULL;    /* blank line never has children... */
      state[blank_depth].last_item=new_item;    /* insert into item list of tree depth at which request was generated */

      new_item->type=ITEM_BLANK;

      new_item->center=0;    /* not really neccesary, but looks nicer... */

      cur_item=new_item;    /* now this one is newest... */
   }

   /* alloc item */
   new_item=malloc(sizeof(struct Item));
   if(new_item==NULL) {
      fprintf(stderr, "memory allocation error while parsing structure (in function create_item)\n");
      exit(1);
   }

   /* insert into tree */
   if(cur_item!=NULL)    /* any items present yet -> insert as next */
      cur_item->list_next=new_item;
   else    /* no items yet -> first one */
      first_item=new_item;
   new_item->list_next=NULL;    /* last item for now */
   new_item->first_child=NULL;    /* no children yet */

   /* insert into item list of current tree depth */
   if(cur_state->first_item!=NULL) {    /* alredy items at this tree depth -> insert as last */
      cur_state->last_item->next=new_item;
      cur_state->last_item=new_item;
   } else {    /* no items at this tree depth yet -> insert as only one */
      cur_state->last_item=cur_state->first_item=new_item;
      cur_state->first_item->parent=NULL;    /* indicates that list not inserted into tree yet */
   }
   new_item->next=NULL;    /* last item at this tree depth */

   new_item->center=0;    /* centering not handled at creation time */

   if(!virtual) {
      string_item=NULL;    /* if last item was a text item, it is no longer valid to add text to it */

      para_blank=-1;    /* now have some item in this box (blank line may be necessary before next item) */
   }
   blank_depth=depth;    /* need to know in which box to add next blank line... */

   return new_item;
}

/*
 * Append text to current string.
 *
 * If currently not in a text block, create new text item and new string to add
 * text to.
 *
 * If called with text==NULL, don't append anything; force creation of new div on
 * next invocation instead, even if color is same as in last div.
 *
 * color==-1 means to keep the last one. (For adding newlines.)
 */

void add_text(text, color)
char	*text;    /* appended text */
int	color;    /* text color */
{
   int			old_len, new_len;    /* string len before and after concatenating the new text */
   static struct String	*string;    /* currently active string */
   static int		force_div=0;    /* force creation of new div for next text part, even if same color as previous one */

   if(text==NULL) {
      force_div=1;
      return;
   }

   if(string_item==NULL) {    /* no open text item -> create new one */

      /* add new string item */
      cur_item=string_item=add_item(0); string_item->type=ITEM_TEXT;

      /* alloc string structure */
      string=string_item->data.string=malloc(sizeof(struct String));
      if(string==NULL) {
	 fprintf(stderr, "memory allocation error while parsing structure (in function add_text)\n");
	 exit(1);
      }

      /* init */
      string->text=NULL;
      string->div_count=0;
      string->div=NULL;
      string->link_count=0;
      string->link=NULL;

      old_len=0;    /* no text yet */

      if(text[0]==' ')    /* skip blank at beginning of text block */
	 ++text;
   } else
      old_len=string->div[string->div_count-1].end;    /* text len is equal to end position of last div */

   if(string->div==NULL    /* no divs yet */
      || (color!=-1 && string->div[string->div_count-1].color!=color    /* new color */
          && *text)    /* any text to add */
      || force_div
     ) {    /* new div */

      if(color==-1) {    /* no color supplied, but div forced -> need to set some */
	 if(string->div_count)    /* already have divs -> keep color of last */
	    color=string->div[string->div_count-1].color;
	 else    /* no divs yet -> use default color */
	    color=7;
      }

      /* alloc new div */
      string->div=realloc(string->div, ++string->div_count*sizeof(struct Div));
      if(string->div==NULL) {
	 fprintf(stderr, "memory allocation error while parsing structure (in function add_text)\n");
	 exit(1);
      }

      string->div[string->div_count-1].color=color;    /* set color of new div */
   }

   new_len=old_len+strlen(text);    /* add len of new text */
   string->div[string->div_count-1].end=new_len;    /* set/adjust len of last div */

   /* adjust text block size */
   string->text=realloc(string->text, new_len+1);
   if(string->text==NULL) {
      fprintf(stderr, "memory allocation error while parsing structure (in function add_text)\n");
      exit(1);
   }

   strcpy(&string->text[old_len], text);    /* concatenate new text */

   force_div=0;    /* next text part will be added normally again */
}

/* find attribute of specified type in specified element */
int find_attr(search_el, attr)
struct Element	*search_el;    /* element in whose attribute list to search */
enum Attr_type	attr;    /* attribute type to search */
{
   int	cur_attr;

   for(cur_attr=0; cur_attr<search_el->attr_count; ++cur_attr)    /* all attributes of this tag */
      if(search_el->attr[cur_attr].name.type==attr)    /* found desired attribute type */
	 return cur_attr;    /* -> return attr num */

   return -1;    /* not found */
}

/* create a structure tree containing all the elements (boxes, text) of a document, and their dependencies */
struct Item *parse_struct(syntax_tree)
struct Element	*syntax_tree;    /* top of syntax parse tree */
{
   struct Element	*cur_el;    /* current element in parse tree */
   struct Element	*new_el;    /* current element while ascending */

   para_blank=0;    /* no blank line before global item */
   string_item=cur_item=NULL;    /* no items yet */

   /* init state stack */
   depth=-1; state=NULL;    /* no state stack yet */
   cur_state=push_state();

   /* all elements in syntax parse tree */
   cur_el=syntax_tree;
   do {    /* until cur_el->parent==NULL (wrapped back to global element) */

      if(cur_el->content!=NULL)    /* some content before element -> add to current text block */
	 if(cur_state->visible)    /* ignore content inside <head> and some other elements */
	    add_text(cur_el->content, color_map[cur_state->text_mode]^(cur_state->high<<3));    /* color from colormap, intensity bit toggled by "high" mode */
 
      if(element_table[cur_el->name.type].breaks) {    /* element breaks lines */

	 string_item=NULL;    /* start new text block */

	 if(para_blank==-1) {    /* already items in this box, but no blank line request yet -> look if need one */
	    if(element_table[cur_el->name.type].breaks==2)    /* element creates paragraph */
	       para_blank=1;    /* -> needs blank line to seperate from previous paragraphs (inserted later, before next item) */
	    else if(element_table[cur_el->name.type].force_box)    /* element creates no paragraph, but own box -> no blank line before next paragraph */
	       para_blank=0;    /* no items in new box yet */
	 }

      }

      cur_state=push_state();    /* descend (depth is below depth of cur_el afterwards) */

      if(!element_table[cur_el->name.type].visible)
	 cur_state->visible=0;

      /* anchors */
      {
	 int	id_attr=find_attr(cur_el, ATTR_NAME);

	 if(id_attr<0)    /* no "name" -> try "id" */
	    id_attr=find_attr(cur_el, ATTR_ID);

	 if(id_attr>=0)    /* has id (or name) -> create anchor */
	    cur_state->id_attr=id_attr;    /* store for use when leaving element; also indicates existance of anchor */
      }

      /* save current position in text string for links/form controls/inline anchors */
      cur_state->link_item=string_item;
      if(string_item!=NULL)
	 cur_state->link_start=string_item->data.string->div[string_item->data.string->div_count-1].end;

      switch(cur_el->name.type) {
	 case EL_H1:
	 case EL_H2:
	 case EL_H3:
	 case EL_H4:
	 case EL_H5:
	 case EL_H6:
	    cur_state->high=1;    /* headings simply highlighted for now */
	    break;
	 case EL_EM:
	 case EL_I:
	    if(cur_state->text_mode<TM_ITALIC)    /* not already in higher priority mode -> set mode */
	       cur_state->text_mode=TM_ITALIC;
	    break;
	 case EL_STRONG:
	 case EL_B:
	    cur_state->high=1;    /* highlight (color itself isn't changed) */
	    break;
	 case EL_A: {
	    const int	href_attr=find_attr(cur_el, ATTR_HREF);

	    if(href_attr>=0) {    /* has href => link */
	       cur_state->link_type=FORM_NO;    /* normal link, not form */
	       cur_state->link_value=cur_el->attr[href_attr].value.str;

	       add_text(NULL, 0);    /* force new div */

	       cur_state->text_mode=TM_LINK;
	       if(cur_el->attr[href_attr].value.str[0]=='#')    /* local anchor -> print marker */
		  add_text("->", color_map[TM_SYS]^(cur_state->high<<3));
	    }
	    break;
	 }
	 case EL_BR:
	    if(string_item!=NULL) {    /* can't insert line break outside text block... */
	       const struct String	*string=string_item->data.string;

	       if(*string->text && string->text[string->div[string->div_count-1].end-1]!='\n')    /* last char isn't newline (never insert more than one '\n') */
		  add_text("\n", -1);    /* add newline, don't change color */
	    }
	    break;
	 case EL_TR:
	    add_text("-", color_map[TM_SYS]);    /* put indicator in front of every table row */
	    break;
	 case EL_TH:
	    cur_state->high=1;    /* header cells highlighted */
	    /* fallthrough */
	 case EL_TD:
	    add_text("|", color_map[TM_SYS]);    /* put indicator in front of every table cell */
	    break;
	 case EL_UL:
	 case EL_OL:
	    ++cur_state->list_depth;
	    break;
	 case EL_LI: {
	    int	i;

	    /* indent for all nesting depths but present one */
	    for(i=0; i<cur_state->list_depth-1; ++i)
	       add_text("\xa0\xa0", color_map[TM_SYS]^(cur_state->high<<3));

	    add_text("*\xa0", color_map[TM_SYS]^(cur_state->high<<3));    /* put list item indicator */
	    break;
	 }
/* Patrice, antrik --> */
	 case EL_DL:
            ++cur_state->list_depth;
	    break;
	 case EL_DT: {
            int i;

	    /* indent for all nesting depths but present one */
	    for(i=0; i<cur_state->list_depth-1; ++i)
	       add_text("\xa0\xa0", color_map[TM_SYS]^(cur_state->high<<3));

	    add_text("*\xa0", color_map[TM_SYS]^(cur_state->high<<3));    /* put definition term indicator */
	    break;
         }
/* <-- Patrice, antrik */
	 case EL_HR:
	    add_text("---", color_map[TM_SYS]^(cur_state->high<<3));
	    break;
	 case EL_INS:
	    add_text("[+", color_map[TM_SYS]^(cur_state->high<<3));
	    break;
	 case EL_DEL:
	 case EL_S:
	 case EL_STRIKE:
	    add_text("[-", color_map[TM_SYS]^(cur_state->high<<3));
	    break;
	 case EL_U:
	    cur_state->high=1;
	    break;
	 case EL_FORM:
	    add_text("<", color_map[TM_FORM]^(cur_state->high<<3));
	    break;
/* Patrice, antrik --> */
	 case EL_INPUT: {
	    const int	type_attr=find_attr(cur_el, ATTR_TYPE);
            const char	*type=cur_el->attr[type_attr].value.str;

	    const int	value_attr=find_attr(cur_el, ATTR_VALUE);
	    cur_state->link_value= value_attr>=0 ? cur_el->attr[value_attr].value.str : "";

	    add_text(NULL, 0);    /* force new div */

            if(strcasecmp(type, "submit")==0) {
	       cur_state->link_type=FORM_SUBMIT;

               add_text("[", color_map[TM_FORM]^(cur_state->high<<3));
               if(value_attr>=0)
	          add_text(cur_el->attr[value_attr].value.str, color_map[TM_LINK]^(cur_state->high<<3));
               else
                  add_text("SUBMIT", color_map[TM_LINK]^(cur_state->high<<3));
	       add_text("]", color_map[TM_FORM]^(cur_state->high<<3));
            } else if(strcasecmp(type, "reset")==0) {
               add_text("(", color_map[TM_FORM]^(cur_state->high<<3));
               if(value_attr>=0)
	          add_text(cur_el->attr[value_attr].value.str, color_map[TM_FORM]^(cur_state->high<<3));
               else
                  add_text("RESET", color_map[TM_FORM]^(cur_state->high<<3));
	       add_text(")", color_map[TM_FORM]^(cur_state->high<<3));
            } else if(strcasecmp(type, "button")==0) {
               add_text("(", color_map[TM_FORM]^(cur_state->high<<3));
               if(value_attr>=0)
	          add_text(cur_el->attr[value_attr].value.str, color_map[TM_FORM]^(cur_state->high<<3));
               else
                  add_text("*", color_map[TM_FORM]^(cur_state->high<<3));
	       add_text(")", color_map[TM_FORM]^(cur_state->high<<3));
            } else if(strcasecmp(type, "image")==0) {
	       cur_state->link_type=FORM_SUBMIT;

               add_text("[", color_map[TM_FORM]^(cur_state->high<<3));
	       {    /* store image */
		  const int	alt_attr=find_attr(cur_el, ATTR_ALT);

		  if(alt_attr>=0) {
		     add_text("[", color_map[TM_IMG]^(cur_state->high<<3));
		     add_text(cur_el->attr[alt_attr].value.str, color_map[TM_LINK]^(cur_state->high<<3));
		     add_text("]", color_map[TM_IMG]^(cur_state->high<<3));
		  } else {
		     add_text("(", color_map[TM_IMG]^(cur_state->high<<3));
		     add_text("*", color_map[TM_LINK]^(cur_state->high<<3));
		     add_text(")", color_map[TM_IMG]^(cur_state->high<<3));
		  }
	       }
	       add_text("]", color_map[TM_FORM]^(cur_state->high<<3));
            } else if(strcasecmp(type, "checkbox")==0) {
               const int	checked_attr=find_attr(cur_el, ATTR_CHECKED);

	       cur_state->link_type=FORM_CHECKBOX;
	       if(checked_attr>=0)
		  cur_state->form_enabled=1;

               add_text("[", color_map[TM_FORM]^(cur_state->high<<3));
               add_text("?", color_map[TM_NORMAL]^(cur_state->high<<3));    /* (will be overwritten...) */
               add_text("]", color_map[TM_FORM]^(cur_state->high<<3));
            } else if(strcasecmp(type, "radio")==0) {
               const int	checked_attr=find_attr(cur_el, ATTR_CHECKED);

	       cur_state->link_type=FORM_RADIO;
	       if(checked_attr>=0)
		  cur_state->form_enabled=1;

               add_text("(", color_map[TM_FORM]^(cur_state->high<<3));
               add_text("?", color_map[TM_NORMAL]^(cur_state->high<<3));    /* (will be overwritten...) */
               add_text(")", color_map[TM_FORM]^(cur_state->high<<3));
            } else if(strcasecmp(type, "password")==0 || strcasecmp(type, "text")==0 || strcasecmp(type, "file")==0 || strcasecmp(type, "hidden")==0) {
               const int	size_attr=find_attr(cur_el, ATTR_SIZE);
               const int	size= (size_attr>=0 && cur_el->attr[size_attr].value.num>0 && type[0]!='h') ? cur_el->attr[size_attr].value.num : 10;

	       switch(tolower(type[0])) {
		  case 'p': cur_state->link_type=FORM_PASS; break;
		  case 't': cur_state->link_type=FORM_TEXT; break;
		  case 'f': cur_state->link_type=FORM_FILE; break;
		  case 'h': cur_state->link_type=FORM_HIDDEN; break;
	       }
	       cur_state->form_enabled=1;

	       if(cur_state->link_type==FORM_FILE)
		  cur_state->link_value=NULL;    /* don't allow default values for file upload! (prevent uploading files not explicitely requested by user) */

               add_text(cur_state->link_type==FORM_FILE ? "[/" : "[", color_map[TM_FORM]^(cur_state->high<<3));
	       {
		  char	line_buf[size+1];
		  memset(line_buf, '\xa0', size);
		  line_buf[size]=0;
		  add_text(line_buf, color_map[cur_state->link_type!=FORM_HIDDEN?TM_NORMAL:TM_DIM]^(cur_state->high<<3));
	       }
	       add_text("]", color_map[TM_FORM]^(cur_state->high<<3));
            }

	    break;
	 }
/* <-- Patrice, antrik */
	 case EL_SELECT: {
	    const int	multi_attr=find_attr(cur_el, ATTR_MULTIPLE);
	    cur_state->select_type= multi_attr>=0 ? FORM_MULTIOPTION : FORM_OPTION;

	    /* save name for all <option> subelements */
	    if(cur_state->id_attr >= 0)
	       cur_state->select_name=cur_el->attr[cur_state->id_attr].value.str;

	    if(cur_state->select_type==FORM_MULTIOPTION)
	       add_text("{", color_map[TM_FORM]^(cur_state->high<<3));
	    else
	       add_text("(", color_map[TM_FORM]^(cur_state->high<<3));

	    break;
	 }
	 case EL_OPTION: {
	    const int	selected_attr=find_attr(cur_el, ATTR_SELECTED);
	    const int	value_attr=find_attr(cur_el, ATTR_VALUE);

	    cur_state->link_type=cur_state->select_type;    /* inherit option type (FORM_OPTION or FORM_MULTIOPTION) from <select> element */

	    /* get "value" */
	    if(value_attr>=0)    /* has "value" attribute */
	       cur_state->link_value=cur_el->attr[value_attr].value.str;
	    else if(cur_el->list_next!=NULL && cur_el->list_next->content!=NULL) {    /* no "value" attribute, but content */
	       char	*content=cur_el->list_next->content;
	       if(*content==' ')    /* skip space at beginning (workaround incorrect space handling) */
		  ++content;
	       cur_state->link_value=content;
	    } else    /* no "value" nor content */
	       cur_state->link_value="";

	    if(selected_attr>=0)    /* initially on */
	       cur_state->form_enabled=1;

	    add_text(NULL, 0);    /* force new div */
	    add_text("?", color_map[TM_FORM]^(cur_state->high<<3));    /* (will be overwritten...) */
	    break;
	 }
	 case EL_BUTTON: {
	    const int	type_attr=find_attr(cur_el, ATTR_TYPE);
            const char	*type= type_attr>=0 ? cur_el->attr[type_attr].value.str : NULL;

	    if(type==NULL || !strcasecmp(type, "submit")) {    /* submit button (default) */
	       const int	value_attr=find_attr(cur_el, ATTR_VALUE);

	       cur_state->link_type=FORM_SUBMIT;
	       cur_state->link_value= value_attr>=0 ? cur_el->attr[value_attr].value.str :""; 

	       add_text(NULL, 0);    /* force new div */
	       add_text("[", color_map[TM_FORM]^(cur_state->high<<3));
	       cur_state->text_mode=TM_LINK;
	    } else {    /* reset or push button -> no action */
	       add_text("(", color_map[TM_FORM]^(cur_state->high<<3));
	       cur_state->text_mode=TM_FORM;
	    }
	    break;
	 }
         case EL_TEXTAREA: {
            const int	cols_attr=find_attr(cur_el, ATTR_COLS);
	    const int	size= (cols_attr >= 0 && cur_el->attr[cols_attr].value.num > 0) ? cur_el->attr[cols_attr].value.num : 50;

	    /* gather initial value form content (stored in subelements) */
	    {
	       char		*value=NULL;
	       int		value_len=0;

	       struct Element	*sub_el;

	       for(sub_el=cur_el->list_next; sub_el->parent==cur_el; sub_el=sub_el->list_next) {    /* all direct children */
		  const int	new_len=value_len+strlen(sub_el->content);

		  value=realloc(value, new_len+1);
		  if(value==NULL) {
		     fprintf(stderr, "Memory allocation error while parsing structure (in function parse_struct)\n");
		     exit(1);
		  }
		  strcpy(&value[value_len], sub_el->content);

		  value_len=new_len;
	       }    /* for all direct children */
	       cur_state->link_value=value;
	    }

	    cur_state->link_type=FORM_TEXTAREA;
	    cur_state->form_enabled=1;

	    add_text("[[", color_map[TM_FORM]^(cur_state->high<<3));
	    {
	       char	line_buf[size+1];
	       memset(line_buf, '\xa0', size);
	       line_buf[size]=0;
	       add_text(line_buf, color_map[TM_NORMAL]^(cur_state->high<<3));
	    }
	    add_text("]]", color_map[TM_FORM]^(cur_state->high<<3));

	    break;
	 }
         case EL_IMG: {
	    const int	alt_attr=find_attr(cur_el, ATTR_ALT);

	    if(alt_attr>=0) {
	       add_text("[", color_map[TM_IMG]^(cur_state->high<<3));
	       add_text(cur_el->attr[alt_attr].value.str, color_map[cur_state->text_mode==TM_LINK?TM_LINK:TM_IMG]^(cur_state->high<<3));
	       add_text("]", color_map[TM_IMG]^(cur_state->high<<3));
	    } else {
	       add_text("(", color_map[TM_IMG]^(cur_state->high<<3));
	       if(cur_state->text_mode==TM_LINK)
		  add_text("*", color_map[TM_LINK]^(cur_state->high<<3));
	       add_text(")", color_map[TM_IMG]^(cur_state->high<<3));
	    }
	    break;
	 }
	 default:    /* html, body, p, center, pre, table, dd, div, span, no, global */
	    break;
      }    /* switch element type */

      /* ascend to depth of next element (no ascend if next element is child of current (keep the descend), ascend once if at same level (undo the descend), ascend more than once if above current) */
      for(new_el=cur_el; new_el!=cur_el->list_next->parent; new_el=new_el->parent) {    /* ascend until parent of next element found */
	 struct State	*old_state;    /* state before ascending */

	 switch(new_el->name.type) {
	    case EL_TR:
	       add_text("-", color_map[TM_SYS]);    /* put indicator after every table row */
	       break;
	    case EL_TD:
	    case EL_TH: {
	       const int	span_attr=find_attr(new_el, ATTR_COLSPAN);
	       int		i;

	       if(span_attr>=0)    /* has colspan */
		  for(i=1; i<atoi(new_el->attr[span_attr].value.str); ++i)    /* all but first spanned cell */
		     add_text("(|)", color_map[TM_SYS]);    /* put indicator for cells left out due to the span */
	       break;
	    }
	    case EL_DL:
               --cur_state->list_depth;
	       break;
	    case EL_DT:
	       add_text(":", color_map[TM_SYS]^(cur_state->high<<3));    /* put definition indicator after definiton term */
	       break;
	    case EL_INS:
	       add_text("+]", color_map[TM_SYS]^(cur_state->high<<3));
	       break;
	    case EL_DEL:
	    case EL_S:
	    case EL_STRIKE:
	       add_text("-]", color_map[TM_SYS]^(cur_state->high<<3));
	       break;
	    case EL_FORM:
	       add_text(">", color_map[TM_FORM]^(cur_state->high<<3));
	       break;
	    case EL_SELECT:
	       if(cur_state->select_type==FORM_MULTIOPTION)
		  add_text("}", color_map[TM_FORM]^(cur_state->high<<3));
	       else
		  add_text(")", color_map[TM_FORM]^(cur_state->high<<3));
	       break;
	    case EL_BUTTON:
	       if((signed)cur_state->link_type >= 0)    /* active button (submit) */
		  add_text("]", color_map[TM_FORM]^(cur_state->high<<3));
	       else    /* reset or push button */
		  add_text(")", color_map[TM_FORM]^(cur_state->high<<3));
	       break;
	    default:    /* html, body, h1-h6, p, em, i, strong, b, center, a, br, pre, table, ul, li, ol, dl, dd, hr, form, input, option, div, span, no, global */
	       break;
	 }    /* switch element type */

	 /* links */
	 if((signed)cur_state->link_type>=0 && (string_item!=NULL || cur_state->link_item!=NULL)) {    /* has link, and valid (inside text block) -> add link */
	    struct String	*string;    /* string where link will be stored */
	    int			link_end;    /* position inside string where link ends */
	    int			new_link;    /* link number in string's link table */
	    
	    if(cur_state->link_item!=NULL)    /* link began inside some string -> store in this one */
	       string=cur_state->link_item->data.string;
	    else {    /* link began outside string -> store in current string (that's the best we can do -- normally, this is the one where link begins...) */
	       string=string_item->data.string;
	       cur_state->link_start=0;    /* start at string start */
	    }
	    link_end=string->div[string->div_count-1].end;    /* string end (normally that's the current string position; if link started in other string, truncates to end of that string) */

	    new_link=string->link_count;
	    /* nested links workaround */
	    if(string->link_count && cur_state->link_start <= string->link[string->link_count-1].start) {    /* nested links (link not first, and starts before last link) */
	       /* find position where link belongs (after last link starting before current) */
	       for(new_link=string->link_count; new_link>0; --new_link)
		  if(cur_state->link_start > string->link[new_link-1].start)    /* found */
		     break;

	       link_end=string->link[new_link].start;    /* truncate to beginning of next link */
	    }

	    /* workaround incorrect space handling so activated links will be highlighted (more or less) correctly */
	    if(string->text[cur_state->link_start]==' ') {    /* ' ' at link start -> move start behind the ' ' */
	       if(cur_state->link_start>0 && !(string->link_count && string->link[string->link_count-1].end==cur_state->link_start)) {    /* can't handle links starting at string start or immediately following other links */
		  int start_div;    /* first div of link */

		  /* also move div start (if not already moved by some other (nested) link...) */
		  for(start_div=1; start_div<string->div_count; ++start_div) {    /* find first div of link */
		     if(string->div[start_div-1].end==cur_state->link_start) {    /* found */
			++string->div[start_div-1].end;
			break;
		     }
		  }

		  ++cur_state->link_start;
	       }    /* possible */
	    }    /* move link start past ' ' */

	    if(link_end > cur_state->link_start) {    /* link not empty -> store */
	       add_text(NULL, 0);    /* force new div */

	       /* resize link table of string */
	       string->link=realloc(string->link, ++string->link_count*sizeof(struct Link));
	       if(string->link==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
		  exit(1);
	       }

	       /* nested links workaround */
	       if(new_link < string->link_count-1) {    /* not inserted as last -> move all links after new_link one position behind */
		  int	link;
		  for(link=string->link_count-1; link>new_link; --link)
		     string->link[link]=string->link[link-1];
	       }

	       /* store link */
	       string->link[new_link].start=cur_state->link_start;    /* starting position saved before */
	       string->link[new_link].end=link_end;
	       string->link[new_link].form=cur_state->link_type;
	       string->link[new_link].enabled=cur_state->form_enabled;
	       string->link[new_link].name=NULL;

	       if(cur_state->link_value==NULL)
		  cur_state->link_value="";
	       string->link[new_link].value.size=strlen(cur_state->link_value);
	       string->link[new_link].value.data=strdup(cur_state->link_value);
	       if(string->link[new_link].value.data==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
		  exit(1);
	       }

	       if(cur_state->link_type!=FORM_NO) {    /* form element */
		  set_form(string, &string->link[new_link]);    /* display initial value */

		  if(cur_state->id_attr >= 0) {    /* has "name"/"id" -> store that (currently used by form handling for simplicity) */
		     string->link[new_link].name=strdup(new_el->attr[cur_state->id_attr].value.str);
		     if(string->link[new_link].name==NULL) {
			fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
			exit(1);
		     }
		  } else if(cur_state->select_name!=NULL) {    /* no "name"/"id", but name inherited from <select> */
		     string->link[new_link].name=strdup(cur_state->select_name);
		     if(string->link[new_link].name==NULL) {
			fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
			exit(1);
		     }
		  } else    /* no name */
		     string->link[new_link].name=NULL;
	       }    /* form element */

	       /* fingerprint link (for retrival of correct link when revisiting a page that changed) */
	       string->link[new_link].text_fingerprint=fingerprint(&string->text[cur_state->link_start], &string->text[link_end]);
	       string->link[new_link].url_fingerprint=fingerprint(cur_state->link_value, strchr(cur_state->link_value, 0));
	    }    /* link not empty */
	 }    /* has link */

	 /* ascend ("depth" is equal to depth of "new_el" afterwards) */
	 old_state=cur_state;
	 cur_state=pop_state();

	 if(element_table[new_el->name.type].force_box) {    /* element creates own box */
	    struct Item	*item;    /* currently handled child */

	    if(depth<blank_depth)    /* last blank line request was generated inside this box -> discard it */
	       para_blank=0;

	    /* create new box item */
	    cur_item=add_item(0);
	    cur_item->type=ITEM_BOX;

	    /* form box */
	    if(new_el->name.type==EL_FORM) {    /* box created by "form" element */
	       const int	action_attr=find_attr(new_el, ATTR_ACTION);
	       const int	method_attr=find_attr(new_el, ATTR_METHOD);

	       cur_item->type=ITEM_FORM;

	       cur_item->data.form=malloc(sizeof(struct Form));
	       if(cur_item->data.form==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure\n");
		  exit(1);
	       }

	       if(action_attr>=0) {    /* has "action" -> store URL */
		  cur_item->data.form->url=strdup(new_el->attr[action_attr].value.str);
		  if(cur_item->data.form->url==NULL) {
		     fprintf(stderr, "memory allocation error while parsing structure\n");
		     exit(1);
		  }
	       } else    /* no "action" */
		  cur_item->data.form->url=NULL;

	       if(method_attr>=0 && strcasecmp(new_el->attr[method_attr].value.str, "POST")==0) {
		  const int	enctype_attr=find_attr(new_el, ATTR_ENCTYPE);

		  if(enctype_attr>=0 && strcasecmp(new_el->attr[enctype_attr].value.str, "multipart/form-data")==0)
		     cur_item->data.form->method=METHOD_POST_MIMEENC;
		  else
		     cur_item->data.form->method=METHOD_POST_URLENC;
	       } else
		  cur_item->data.form->method=METHOD_GET;
	    }    /* form */

	    /* set parent of all items inside box */
	    for(item=old_state->first_item; item!=NULL; item=item->next)
	       item->parent=cur_item;

	    cur_item->first_child=old_state->first_item;

	 } else {    /* no box necessary */

	    /* handle <center> element or attribute */
	    {
	       const int	center_attr=find_attr(new_el, ATTR_ALIGN);

	       if(new_el->name.type==EL_CENTER || (center_attr>=0 && strcasecmp(new_el->attr[center_attr].value.str, "center")==0)) {    /* leaving <center> element or any elment containing align="center" -> center all items created inside it */
		  struct Item	*item;    /* currently handled child */

		  for(item=old_state->first_item; item!=NULL; item=item->next)
		     item->center=1;
	       }
	    }

	    /* lift all waiting items to new depth */
	    if(old_state->first_item!=NULL) {    /* any items to lift */
	       if(cur_state->first_item!=NULL)    /* already items at new depth */
		  cur_state->last_item->next=old_state->first_item;    /* -> add to end of list */
	       else    /* no items in new depth yet */
		  cur_state->first_item=old_state->first_item;    /* -> copy list */
	       cur_state->last_item=old_state->last_item;    /* new depth's item list now ends with old depth's last item */
	    }

	    /* also lift pending blank line */
	    if(depth<blank_depth)    /* last blank line request was generated inside this element */
	       blank_depth=depth;

	 }    /* no box */

	 if(element_table[new_el->name.type].breaks) {    /* element breaks lines */
	    string_item=NULL;    /* start new text block */

	    if(element_table[new_el->name.type].breaks==2 && para_blank==-1)    /* element creates paragraph and there are already items in this box */
	       para_blank=1;    /* -> may need blank line after it to seperate from next paragraph (inserted later, before next item) */
	 }

	 /* anchors */
	 if(old_state->id_attr>=0) {    /* has anchor -> create anchor item */
	    cur_item=add_item(1);    /* add a virtual item */

	    if(string_item!=NULL) {    /* inside string => inline anchor */
	       struct Inline_anchor	*anchor;

	       cur_item->type=ITEM_INLINE_ANCHOR;

	       if(old_state->link_item!=string_item)    /* link began in other string */
		  old_state->link_start=0;    /* start at string start (this way at least the last part of the anchor will be stored correctly) */

	       /* create data block */
	       anchor=cur_item->data.inline_anchor=malloc(sizeof(struct Inline_anchor));
	       if(anchor==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
		  exit(1);
	       }

	       /* store anchor data */
	       anchor->id=strdup(new_el->attr[old_state->id_attr].value.str);
	       if(anchor->id==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
		  exit(1);
	       }
	       anchor->virtual_parent=string_item;
	       anchor->start=old_state->link_start;    /* starting position saved before */
	       anchor->end=string_item->data.string->div[string_item->data.string->div_count-1].end;    /* current string end */

	       /* workaround incorrect space handling so anchors will be set (more or less) correctly */
	       if(string_item->data.string->text[anchor->start]==' ')    /* blank at anchor start -> skip */
		  ++anchor->start;
	    } else {    /* not inside string => block anchor */
	       struct Block_anchor	*anchor;

	       cur_item->type=ITEM_BLOCK_ANCHOR;

	       /* create data block */
	       anchor=cur_item->data.block_anchor=malloc(sizeof(struct Block_anchor));
	       if(anchor==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
		  exit(1);
	       }

	       /* store anchor data */
	       anchor->id=strdup(new_el->attr[old_state->id_attr].value.str);
	       if(anchor->id==NULL) {
		  fprintf(stderr, "memory allocation error while parsing structure (in function parse_struct)\n");
		  exit(1);
	       }
	       if(old_state->first_item!=NULL) {    /* any items inside the anchor element */
		  if(old_state->first_item->parent!=NULL)    /* items inside the anchor element have gotten some parent -> store only the parent in virtual box */
		     anchor->virtual_child=old_state->first_item->parent;
		  else    /* items have no parent -> store them directly */
		     anchor->virtual_child=old_state->first_item;    /* first item in virtual box */
	       } else    /* no items inside anchor element -> point back to itself */
		  anchor->virtual_child=cur_item;
	    }    /* block anchor */
	 }    /* has anchor */
		     
      }    /* for new_el (ascend to depth of next element) */

      cur_el=cur_el->list_next;
   } while(cur_el->parent!=NULL);    /* until wrapped back to global element (all elements processed) */

   cur_item->parent=first_item;    /* save reference to first item */
   cur_item->next=cur_item;    /* top points back to itself */

   free(state);

   return cur_item;
}

/* unallocate whole item tree;
 * traverses whole tree (list), and frees every node after freeing its data */
void free_items(item_tree)
struct Item	*item_tree;
{
   struct Item	*cur_item;    /* item to be deleted */
   struct Item	*next_item;    /* need to save "list_next", as won't be available after deleting "cur_item" */

   /* all items */
   next_item=item_tree->parent;    /* start with first item */
   do {
      cur_item=next_item;
      next_item=cur_item->list_next;    /* save "list_next" for later */

      switch(cur_item->type) {

	 case ITEM_TEXT:
	    if(cur_item->data.string!=NULL) {    /* has "String" -> free it */
	       free(cur_item->data.string->text);
	       free(cur_item->data.string->div);
	       free(cur_item->data.string->line_table);
	       {
		  int	link;
		  for(link=0; link<cur_item->data.string->link_count; ++link) {
		     free(cur_item->data.string->link[link].value.data);
		     if(cur_item->data.string->link[link].name!=NULL)
			free(cur_item->data.string->link[link].name);
		  }
	       }
	       free(cur_item->data.string->link);
	       free(cur_item->data.string);
	    }
	    break;

	 case ITEM_BLOCK_ANCHOR:
	    free(cur_item->data.block_anchor->id);
	    free(cur_item->data.block_anchor);
	    break;

	 case ITEM_INLINE_ANCHOR:
	    free(cur_item->data.inline_anchor->id);
	    free(cur_item->data.inline_anchor);
	    break;

	 case ITEM_FORM:
	    free(cur_item->data.form->url);
	    free(cur_item->data.form);
	    break;
	 
	 case ITEM_BOX:
	 case ITEM_BLANK:
	    break;
      }    /* switch item type */

      free(cur_item);
   } while(next_item!=NULL);    /* for all items (until list end) */
}

/* create list of all links in the page;
 * traverses the whole item tree, and for every text item stores all its links
 * to the list */
struct Link_list *make_link_list(item_tree)
struct Item	*item_tree;
{
   struct Item		*cur_item;
   struct Link_list	*list;

   /* create list */
   list=malloc(sizeof(struct Link_list));
   if(list==NULL) {
      fprintf(stderr, "memory allocation error while creating link list\n");
      exit(1);
   }
   list->count=0;
   list->link=NULL;

   /* find and insert links */
   for(cur_item=item_tree->parent; cur_item!=NULL; cur_item=cur_item->list_next) {    /* all items (starting from bottom) */
      if(cur_item->type==ITEM_TEXT) {
	 int	link;

	 for(link=0; link<cur_item->data.string->link_count; ++link) {    /* all links in this text block */
	    if(cur_item->data.string->link[link].form!=FORM_HIDDEN) {    /* active link -> add to link list */
	       list->link=realloc(list->link, ++list->count*sizeof(struct Link_ptr));    /* new entry in link list */
	       if(list->link==NULL) {
		  fprintf(stderr, "memory allocation error while creating link list\n");
		  exit(1);
	       }

	       list->link[list->count-1].item=cur_item;
	       list->link[list->count-1].num=link;
	    }    /* active link */
	 }    /* for all links in text block */
      }    /* ITEM_TEXT */
   }    /* for all items */

   return list;
}

void free_links(list)
struct Link_list	*list;
{
   free(list->link);
   free(list);
}

/* create list of all anchors in the page;
 * traverses the whole item tree, and store every encountered anchor
 * to the list */
struct Anchor_list *make_anchor_list(item_tree)
struct Item	*item_tree;
{
   struct Item		*cur_item;
   struct Anchor_list	*list;

   /* create list */
   list=malloc(sizeof(struct Anchor_list));
   if(list==NULL) {
      fprintf(stderr, "memory allocation error while creating link list\n");
      exit(1);
   }
   list->count=0;
   list->anchor_item=NULL;

   /* find all anchors and add to list */
   for(cur_item=item_tree->parent; cur_item!=NULL; cur_item=cur_item->list_next) {    /* all items (starting from bottom) */
      if(cur_item->type==ITEM_BLOCK_ANCHOR || cur_item->type==ITEM_INLINE_ANCHOR) {
	 list->anchor_item=realloc(list->anchor_item, ++list->count*sizeof(struct Item *));    /* new entry in anchor list */
	 if(list->anchor_item==NULL) {
	    fprintf(stderr, "memory allocation error while creating link list\n");
	    exit(1);
	 }

	 list->anchor_item[list->count-1]=cur_item;
      }    /* anchor item */
   }    /* for all items */

   return list;
}

void free_anchors(list)
struct Anchor_list	*list;
{
   free(list->anchor_item);
   free(list);
}
