/*************************************************************************************************
 * Java binding of Hyper Estraier
 *                                                      Copyright (C) 2004-2005 Mikio Hirabayashi
 * This file is part of Hyper Estraier.
 * Hyper Estraier is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License or any later version.  Hyper Estraier is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with Hyper
 * Estraier; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA.
 *************************************************************************************************/


package estraier;

import java.util.*;
import java.io.*;
import java.net.*;



/**
 * Abstraction of search condition.
 */
public class Condition {
  //----------------------------------------------------------------
  // static initializer
  //----------------------------------------------------------------
  static {
    Utility.init();
  }
  //----------------------------------------------------------------
  // public constants
  //----------------------------------------------------------------
  /** option: check every N-gram key */
  public static final int SURE = 1 << 0;
  /** option: check N-gram keys skipping by one */
  public static final int USUAL = 1 << 1;
  /** option: check N-gram keys skipping by two */
  public static final int FAST = 1 << 2;
  /** option: check N-gram keys skipping by three */
  public static final int AGITO = 1 << 3;
  /** option: without TF-IDF tuning */
  public static final int NOIDF = 1 << 4;
  /** option: with the simplified phrase */
  public static final int SIMPLE = 1 << 10;
  //----------------------------------------------------------------
  // private fields
  //----------------------------------------------------------------
  /** pointer to the native object */
  private long coreptr;
  //----------------------------------------------------------------
  // constructors and finalizers
  //----------------------------------------------------------------
  /**
   * Create a search condition object.
   */
  public Condition(){
    initialize();
  }
  /**
   * Release resources.
   */
  protected void finalize(){
    destroy();
  }
  //----------------------------------------------------------------
  // public methods
  //----------------------------------------------------------------
  /**
   * Set the search phrase.
   * @param phrase a search phrase.
   */
  public native void set_phrase(String phrase);
  /**
   * Add an expression for an attribute.
   * @param expr an expression for an attribute.
   */
  public native void add_attr(String expr);
  /**
   * Set the order.
   * @param expr an expression for the order.  By default, the order is by score descending.
   */
  public native void set_order(String expr);
  /**
   * Set the maximum number of retrieval.
   * @param max the maximum number of retrieval.  By default, the number of retrieval is not
   * limited.
   */
  public native void set_max(int max);
  /**
   * Set options of retrieval.
   * @param options options: `Condition.SURE' specifies that it checks every N-gram key,
   * `Condition.USUAL', which is the default, specifies that it checks N-gram keys with skipping
   * one key, `Condition.FAST' skips two keys, `Condition.AGITO' skips three keys,
   * `Condition.NOIDF' specifies not to perform TF-IDF tuning, `Condition.SIMPLE' specifies to
   * use simplified phrase.  Each option can be specified at the same time by bitwise or.  If
   * keys are skipped, though search speed is improved, the relevance ratio grows less.
   */
  public native void set_options(int options);
  //----------------------------------------------------------------
  // private methods
  //----------------------------------------------------------------
  /**
   * Set the native object.
   */
  private native void initialize();
  /**
   * Delete the native object.
   */
  private native void destroy();
}



/* END OF FILE */
