/***************************************************************************/
/* 	This code is part of X-toolkit widget library called Nws 	   */
/*	Copyright (c) 1997,1998,1999 Ondrejicka Stefan			   */
/*	(ondrej@idata.sk)						   */
/*	Distributed under GPL 2 or later				   */
/***************************************************************************/

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xmu/Xmu.h>

#include	"EntryLineP.h"
#include	"Init.h"
#include 	"utils.h"
#include	"misc.h"

#define INITIAL_BUFFER_SIZE	200
#define BSIZE_INCREMENT		20

#define offset(field) XtOffsetOf(EntryLineRec, entryLine.field)

static XtResource resources[] = { 
	{
	 XtNecho ,
	 XtCEcho ,
	 XtRBoolean ,
	 sizeof(Boolean) ,
	 offset(echo) ,
	 XtRImmediate ,
	 (XtPointer) True
	},
	{
	 XtNeditable ,
	 XtCEditable ,
	 XtRBoolean ,
	 sizeof(Boolean) ,
	 offset(editable) ,
	 XtRImmediate ,
	 (XtPointer) True
	},
	{
	 XtNfont ,
	 XtCFont ,
	 XtRFontStruct ,
	 sizeof(XFontStruct *) ,
	 offset(font) ,
	 XtRString ,
	 (XtPointer) XtDefaultFont
	},
	{
	 XtNtext ,
	 XtCText ,
	 XtRString ,
	 sizeof(char *) ,
	 offset(text) ,
	 XtRImmediate ,
	 (XtPointer) NULL
	},
	{
	 XtNvalid ,
	 XtCValid ,
	 XtRString ,
	 sizeof(char *) ,
	 offset(valid) ,
	 XtRImmediate ,
	 (XtPointer) NULL
	},
	{
	 XtNspacing ,
	 XtCSpacing ,
	 XtRInt ,
	 sizeof(int) ,
	 offset(spacing) ,
	 XtRImmediate ,
	 (XtPointer) 2 
	},
	{
	 XtNmax_length ,
	 XtCMax_length ,
	 XtRInt ,
	 sizeof(int) ,
	 offset(max_length) ,
	 XtRImmediate ,
	 (XtPointer) 0
	},
	{
	 XtNcursor_color ,
	 XtCCursor_color ,
	 XtRPixel ,
	 sizeof(Pixel) ,
	 offset(cursor_color) ,
	 XtRString ,
	 (XtPointer) XtDefaultForeground
	},
	{
	 XtNactivate ,
	 XtCActivate ,
	 XtRCallback ,
	 sizeof(XtCallbackList) ,
	 offset(activate) ,
	 XtRCallback ,
	 (XtPointer) NULL
	},
	{
	 XtNchange ,
	 XtCChange ,
	 XtRCallback ,
	 sizeof(XtCallbackList) ,
	 offset(change) ,
	 XtRCallback ,
	 (XtPointer) NULL
	},
	{
	 XtNselection_fg ,
	 XtCSelection_fg ,
	 XtRPixel ,
	 sizeof(Pixel) ,
	 offset(selection_fg) ,
	 XtRString ,
	 (XtPointer) XtDefaultBackground
	},
	{
	 XtNselection_bg ,
	 XtCSelection_bg ,
	 XtRPixel ,
	 sizeof(Pixel) ,
	 offset(selection_bg) ,
	 XtRString ,
	 (XtPointer) XtDefaultForeground
	},
	{
	 XtNbox_type ,
	 XtCBox_type ,
	 XtRBox_type ,
	 sizeof(int) ,
	 XtOffsetOf(EntryLineRec , base.box_type) ,
	 XtRImmediate ,
	 (XtPointer) XtCframein_box,
	}
};

#undef offset

static void Initialize ();
static Boolean SetValues ();
static void Redisplay ();
static void Destroy ();

static void GetTextDimension ();

static void RedrawCursor ();
static void RedrawAfterCursor ();
static void RedrawSelection ();
static int PositionToChar ();
static void DeleteSelection ();
static void ComputeShownLength ();
static void insert_text ();
static void delete_text ();

static void InsertChar();
static void MoveLeft();
static void MoveRight();
static void DeleteNext();
static void DeletePrevious();
static void EndLine();
static void BeginLine();
static void SelectionStart();
static void DoSelection();
static void SelectionEnd();
static void InsertSelection();
static void Activate();
static void KillLine();
static void KillEnd();
static void DrawCursor();
static void ClearCursor();

static XtActionsRec action [] = {
	{"insert_char", InsertChar},
	{"move_left" , MoveLeft},
	{"move_right" , MoveRight},
	{"delete_next" , DeleteNext},
	{"delete_previous" , DeletePrevious},
	{"eol" , EndLine},
	{"bol" , BeginLine},
	{"selection_start" , SelectionStart},
	{"do_selection" , DoSelection},
	{"selection_end" , SelectionEnd},
	{"insert_selection" , InsertSelection},
	{"activate" , Activate },
	{"kill_line" , KillLine},
	{"kill_end" , KillEnd},
	{"draw_cursor" , DrawCursor},
	{"clear_cursor" , ClearCursor}
};

static char trans_tab [] = 
	"<FocusIn>: focusIn() draw_cursor()\n\
	 <FocusOut>: focusOut() clear_cursor()\n\
	 ~Shift<Key>Tab: traverseForward()\n\
	 Shift<Key>Tab: traverseBackward()\n\
	 <Key>Home: bol() \n\
	 Ctrl<Key>E: eol() \n\
	 <Key>End: eol() \n\
	 Ctrl<Key>A: bol() \n\
	 Ctrl<Key>Y: kill_line() \n\
	 Ctrl<Key>K: kill_end() \n\
	 <Key>Right: move_right() \n\
	 <Key>Left: move_left() \n\
	 <Key>BackSpace: delete_previous() \n\
	 <Key>Delete: delete_next() \n\
	 <Key>Return: activate() \n\
	 <Key>: insert_char() \n\
	 <Btn1Down>: focusCurrent() selection_start() \n\
	 <Btn1Motion>: do_selection() \n\
	 <Btn1Up>: selection_end() \n\
	 <Btn2Down>: focusCurrent() insert_selection(CUT_BUFFER0) \n\
	 <Btn3Down>: focusCurrent() \n\
	 <Enter>: highlight() show_help() \n\
         <Leave>: unhighlight() hide_help() \n\
         <BtnDown>: hide_help() \n\
         <KeyDown>: hide_help() \n\
	 ";


EntryLineClassRec entryLineClassRec = {
/* core */
   {
    /* superclass            */ (WidgetClass) &baseClassRec,
    /* class_name            */ "EntryLine",
    /* widget_size           */ sizeof(EntryLineRec),
    /* class_initialize      */ NULL,
    /* class_part_initialize */ NULL,
    /* class_inited          */ FALSE,
    /* initialize            */ (XtInitProc) Initialize,
    /* initialize_hook       */ NULL,
    /* realize               */ XtInheritRealize,
    /* actions               */ action,
    /* num_actions           */ XtNumber(action),
    /* resources             */ resources,
    /* num_resources         */ XtNumber(resources),
    /* xrm_class             */ NULLQUARK,
    /* compress_motion       */ False,
    /* compress_exposure     */ False,
    /* compress_enterleave   */ False,
    /* visible_interest      */ FALSE,
    /* destroy               */ Destroy,
    /* resize                */ XtInheritResize,
    /* expose                */ Redisplay,
    /* set_values            */ (XtSetValuesFunc) SetValues,
    /* set_values_hook       */ NULL,
    /* set_values_almost     */ XtInheritSetValuesAlmost,
    /* get_values_hook       */ NULL,
    /* accept_focus          */ XtInheritAcceptFocus,
    /* version               */ XtVersion,
    /* callback_private      */ NULL,
    /* tm_table              */ trans_tab,
    /* query_geometry        */ XtInheritQueryGeometry,
    /* display_accelerator   */ XtInheritDisplayAccelerator,
    /* extension             */ NULL
   },
/* base */
   {
    /* get_internal_dimension  */ XtInheritGetInternalDimension,
    /* set_internal_dimension  */ XtInheritSetInternalDimension,
    /* highlight	       */ XtInheritHighlight,
    /* unhighlight	       */ XtInheritUnhighlight,
    /* highlightBorder	       */ XtInheritHighlightBorder,
    /* unhighlightBorder       */ XtInheritUnhighlightBorder,
   },
/* entryLine */
   {
    /* empty		       */ GetTextDimension,
   }
};

WidgetClass entryLineWidgetClass = (WidgetClass) &entryLineClassRec;

static void Initialize(req_widget,new_widget,args,num_args)
Widget req_widget;
Widget new_widget;
ArgList args;
Cardinal *num_args;
{
	EntryLineWidget nw = (EntryLineWidget) new_widget;
	XGCValues gc_res;
	XtGCMask  gc_mask;
	char *text_buf;
	Display *dpy = XtDisplay(new_widget);
	Dimension height;

	gc_res.foreground = nw->base.foreground;
	gc_res.background = nw->core.background_pixel;
	gc_res.font = nw->entryLine.font->fid;

	gc_mask = GCForeground | GCBackground | GCFont;

	nw->entryLine.normalGC = XCreateGC(dpy , DefaultRootWindow(dpy) ,
					gc_mask , &gc_res);

	gc_res.foreground = nw->entryLine.selection_fg;
	gc_res.background = nw->entryLine.selection_bg;

	gc_mask = GCForeground | GCBackground | GCFont;

	nw->entryLine.reverseGC = XCreateGC(dpy , DefaultRootWindow(dpy) ,
					gc_mask , &gc_res);

	nw->entryLine.first = 0;
	nw->entryLine.crsr_pos = 0;

	if (nw->entryLine.max_length)
		nw->entryLine.buffer_length = 
			MIN((nw->entryLine.max_length + 1) , INITIAL_BUFFER_SIZE);
	else if (nw->entryLine.text)
	{
		nw->entryLine.buffer_length =
			MAX(INITIAL_BUFFER_SIZE , strlen(nw->entryLine.text));
	}
	else nw->entryLine.buffer_length = INITIAL_BUFFER_SIZE;

	text_buf = (char *) XtMalloc(nw->entryLine.buffer_length);
	memset(text_buf , 0 , nw->entryLine.buffer_length);

	if (nw->entryLine.text) 
		strncpy(text_buf , nw->entryLine.text , nw->entryLine.buffer_length - 1);

	nw->entryLine.text_length = nw->entryLine.text ? strlen(text_buf) : 0;

	nw->entryLine.text = text_buf;

	nw->entryLine.have_selection = False;

	if (!nw->core.height)
	{
		height = nw->entryLine.font->max_bounds.ascent +
			nw->entryLine.font->max_bounds.descent + 
			2 * nw->entryLine.spacing + 2;

		((EntryLineWidgetClass) XtClass(new_widget))->base_class.
			set_internal_dimension(new_widget , nw->core.width , height);
	}

	nw->entryLine.selecting = False;
}

static void Destroy(w)
Widget w;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	XFreeGC(XtDisplay(w),cw->entryLine.normalGC);
	XFreeGC(XtDisplay(w),cw->entryLine.reverseGC);

	XtDisownSelection(w , XA_PRIMARY , XtLastTimestampProcessed(XtDisplay(w)));
}

static void Redisplay(w,event,region)
Widget w;
XEvent * event;
Region  region;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Display *dpy = XtDisplay(w);
	Window win = XtWindow(w);
	Dimension width,height;
	Position x,y;
	int ty;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(w ,
		&x , &y , &width , &height);

	XClearArea(dpy , win , x , y , width + 1, height + 1, False);

	ty = y + (height - cw->entryLine.font->max_bounds.ascent -
		cw->entryLine.font->max_bounds.descent) / 2;

	ComputeShownLength(cw , width);

	if (cw->entryLine.echo)
		XDrawImageString(dpy , win , cw->entryLine.normalGC , x ,
			ty + cw->entryLine.font->max_bounds.ascent,
			cw->entryLine.text + cw->entryLine.first ,
			cw->entryLine.shown_length);

	if (cw->base.focused)
		RedrawCursor(w , cw->entryLine.cursor_color);

	baseClassRec.core_class.expose(w,event,region);
}

#define WidgetValuesDiffer(w1,w2,component) (w1 -> entryLine.component != \
                                             w2 -> entryLine.component)

static Boolean SetValues(current, request, new_widget, args, num_args)
Widget current;
Widget request;
Widget new_widget;
ArgList args;
Cardinal *num_args;
{
	EntryLineWidget cw = (EntryLineWidget) current;
	EntryLineWidget nw = (EntryLineWidget) new_widget;
	char *text_buf;
	Boolean redraw = False;

	if WidgetValuesDiffer( cw , nw , echo)
	{
		redraw = True;
	}

	if WidgetValuesDiffer( cw , nw , font)
	{
		XSetFont(XtDisplay(new_widget) , nw->entryLine.normalGC ,
			nw->entryLine.font->fid);

		XSetFont(XtDisplay(new_widget) , nw->entryLine.reverseGC ,
			nw->entryLine.font->fid);

		redraw = True;
	}

	if WidgetValuesDiffer(cw , nw , text)
	{

		nw->entryLine.first = 0;
		nw->entryLine.crsr_pos = 0;

		if (nw->entryLine.max_length)
			nw->entryLine.buffer_length = 
				MIN((nw->entryLine.max_length + 1) , INITIAL_BUFFER_SIZE);
		else if (nw->entryLine.text)
		{
			nw->entryLine.buffer_length = 
				MAX(INITIAL_BUFFER_SIZE , strlen(nw->entryLine.text));
		}
		else nw->entryLine.buffer_length = INITIAL_BUFFER_SIZE;

		text_buf = (char *) XtMalloc(nw->entryLine.buffer_length);
		memset(text_buf , 0 , nw->entryLine.buffer_length);

		if (nw->entryLine.text) 
			strncpy(text_buf , nw->entryLine.text ,
				nw->entryLine.buffer_length - 1);

		nw->entryLine.text_length = strlen(text_buf);

		nw->entryLine.text = text_buf;

		nw->entryLine.have_selection = False;	

		XtFree(cw->entryLine.text);
		redraw = True;

		XtCallCallbackList((Widget)nw , nw->entryLine.change , nw->entryLine.text);

	}

	if WidgetValuesDiffer( cw , nw , selection_fg)
	{
		XSetForeground(XtDisplay(new_widget) , nw->entryLine.reverseGC ,
			nw->entryLine.selection_fg);
	}

	if WidgetValuesDiffer( cw , nw , selection_bg)
	{
		XSetBackground(XtDisplay(new_widget) , nw->entryLine.reverseGC ,
			nw->entryLine.selection_bg);
	}

	return redraw;
}

static void GetTextDimension(w , x , y , width , height)
Widget w;
Position *x;
Position *y;
Dimension *width;
Dimension *height;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	((EntryLineWidgetClass) XtClass(w))->base_class.get_internal_dimension(w ,
		x , y , width , height);

	*x += cw->entryLine.spacing;
	*y += cw->entryLine.spacing;
	*width -=  2 * cw->entryLine.spacing;
	*height -= 2 * cw->entryLine.spacing;

}

static void Activate (w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	XtCallCallbackList(w , cw->entryLine.activate , cw->entryLine.text);
}

static void  InsertChar(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
#define BUFSIZE 20

	EntryLineWidget cw = (EntryLineWidget) w;
	char buffer[BUFSIZE];
	int len;

	len = XLookupString((XKeyEvent *) event, buffer, BUFSIZE - 1 , NULL, NULL);

	if (!cw->entryLine.editable)
	{
		if (len > 0)
			XBell(XtDisplay(w) , 50);
		return;
	}
	
	if (cw->entryLine.have_selection && len > 0)
	{
		cw->entryLine.have_selection = False;
		DeleteSelection(w);
	}

	if (len > 0) 
	{
		RedrawCursor(w , cw->core.background_pixel);
		*(buffer + len) = '\0';
		insert_text(w , buffer);
	}

	XtCallCallbackList((Widget)w , cw->entryLine.change , cw->entryLine.text);
}

static void  MoveLeft(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}

	RedrawCursor(w , cw->core.background_pixel);

	if (cw->entryLine.crsr_pos > 0)
		cw->entryLine.crsr_pos --;

	if (cw->entryLine.first > cw->entryLine.crsr_pos)
	{
		cw->entryLine.first --;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}
	else
		if (cw->base.focused)
			RedrawCursor(w , cw->entryLine.cursor_color);
}

static void MoveRight(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Dimension width,height;
	Position x,y;

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(w ,
		&x , &y , &width , &height);

	RedrawCursor(w , cw->core.background_pixel);

	if (cw->entryLine.crsr_pos < cw->entryLine.text_length)
		cw->entryLine.crsr_pos ++;

	if (XTextWidth(cw->entryLine.font , cw->entryLine.text + cw->entryLine.first ,
		cw->entryLine.crsr_pos - cw->entryLine.first) > width)
	{
		cw->entryLine.first ++;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}
	else
		if (cw->base.focused)
			RedrawCursor(w , cw->entryLine.cursor_color);
}

static void  DeleteNext(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (!cw->entryLine.editable)
	{
		XBell(XtDisplay(w) , 50);
		return;
	}

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		DeleteSelection(w);
		return;
	}

	delete_text(w , cw->entryLine.text + cw->entryLine.crsr_pos , 1);

	RedrawAfterCursor(w);

	if (cw->base.focused)
		RedrawCursor(w , cw->entryLine.cursor_color);

	XtCallCallbackList((Widget)w , cw->entryLine.change , cw->entryLine.text);

}

static void  DeletePrevious(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (!cw->entryLine.editable)
	{
		XBell(XtDisplay(w) , 50);
		return;
	}

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		DeleteSelection(w);
		return;
	}

	if  (cw->entryLine.crsr_pos == 0) return;

	RedrawCursor(w , cw->core.background_pixel);

	cw->entryLine.crsr_pos --;

	delete_text(w , cw->entryLine.text + cw->entryLine.crsr_pos , 1);

	if (cw->entryLine.crsr_pos == cw->entryLine.first)
	{
		if (cw->entryLine.first) cw->entryLine.first --;

		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}
	else
	{
		RedrawAfterCursor(w);

		if (cw->base.focused)
			RedrawCursor(w , cw->entryLine.cursor_color);	
	}
	XtCallCallbackList((Widget)w , cw->entryLine.change , cw->entryLine.text);
}

static void  EndLine(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Dimension width,height;
	Position x,y;

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(w ,
		&x , &y , &width , &height);

	RedrawCursor(w , cw->core.background_pixel);

	cw->entryLine.crsr_pos = cw->entryLine.text_length;

	if (XTextWidth(cw->entryLine.font , cw->entryLine.text + cw->entryLine.first ,
		cw->entryLine.crsr_pos - cw->entryLine.first) > width)
	{

		while (XTextWidth(cw->entryLine.font , cw->entryLine.text + 
			cw->entryLine.first , cw->entryLine.crsr_pos - 
			cw->entryLine.first) > width)
				cw->entryLine.first ++;

		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}
	else
		if (cw->base.focused)
			RedrawCursor(w , cw->entryLine.cursor_color);	
}

static void  BeginLine(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}

	RedrawCursor(w , cw->core.background_pixel);

	cw->entryLine.crsr_pos = 0;
	if (cw->entryLine.first != 0)
	{
		cw->entryLine.first = 0;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}
	else
		if (cw->base.focused)
			RedrawCursor(w , cw->entryLine.cursor_color);	

}

static void  KillLine(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (!cw->entryLine.editable)
	{
		XBell(XtDisplay(w) , 50);
		return;
	}

	RedrawCursor(w , cw->core.background_pixel);

	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
	}

	cw->entryLine.text_length = 0;
	*(cw->entryLine.text) = '\0';
	cw->entryLine.crsr_pos = 0;

	XtCallCallbackList(w , cw->entryLine.change , cw->entryLine.text);

	((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);	
}



static void  KillEnd(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (!cw->entryLine.editable)
	{
		XBell(XtDisplay(w) , 50);
		return;
	}

	cw->entryLine.text_length = cw->entryLine.crsr_pos;

	*(cw->entryLine.text + cw->entryLine.text_length) = '\0';

	XtCallCallbackList(w , cw->entryLine.change , cw->entryLine.text);

	((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);	
}

static void  ClearCursor(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	RedrawCursor(w , cw->core.background_pixel);
}

static void  DrawCursor(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	RedrawCursor(w , cw->entryLine.cursor_color);
}


static void  SelectionStart(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	int pos;
	Dimension width,height;
	Position x,y;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(w ,
		&x , &y , &width , &height);

	if (!(BETWEEN(event->xbutton.x , x , x+width) && 
		BETWEEN(event->xbutton.y , y , y+height))) return;
	
	if (cw->entryLine.have_selection)
	{
		cw->entryLine.have_selection = False;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}

	RedrawCursor(w , cw->core.background_pixel);

	pos = PositionToChar(w , event->xbutton.x);

	if (pos != -1)
	{
		cw->entryLine.have_selection = False;
		cw->entryLine.crsr_pos = pos + cw->entryLine.first;
		cw->entryLine.selection_start = cw->entryLine.crsr_pos;
		cw->entryLine.selection_end = cw->entryLine.crsr_pos;
	}

	if (cw->base.focused)	RedrawCursor(w , cw->entryLine.cursor_color);

	cw->entryLine.selecting = True;
}

static void  DoSelection(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	int old_end = cw->entryLine.selection_end;
	int pos;
	Dimension width,height;
	Position x,y;


	if (!cw->entryLine.selecting) return;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(w ,
		&x , &y , &width , &height);

	RedrawCursor(w , cw->core.background_pixel);

	pos = PositionToChar(w , event->xbutton.x);

	if (cw->entryLine.echo)
		cw->entryLine.have_selection = True;

	if (cw->entryLine.crsr_pos || (pos > 0)) 
		cw->entryLine.crsr_pos = pos + cw->entryLine.first;

	cw->entryLine.selection_end = cw->entryLine.crsr_pos;

	if (cw->entryLine.crsr_pos < cw->entryLine.first)
	{
		cw->entryLine.first = cw->entryLine.crsr_pos > 0 ? cw->entryLine.crsr_pos : 0;
		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}	

	if (XTextWidth(cw->entryLine.font , cw->entryLine.text + cw->entryLine.first ,
		cw->entryLine.crsr_pos - cw->entryLine.first) > width)
	{

		while (XTextWidth(cw->entryLine.font , cw->entryLine.text + 
			cw->entryLine.first , cw->entryLine.crsr_pos - 
			cw->entryLine.first) > width)
				cw->entryLine.first ++;

		((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	}
	
	RedrawSelection(w,old_end);

	if (cw->base.focused)
		RedrawCursor(w , cw->entryLine.cursor_color);

}

static void SelectionOut(w , selection)
Widget w;
Atom *selection;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	XtFree(cw->entryLine.selection);

	((EntryLineWidgetClass) XtClass(w))->core_class.expose(w , NULL , NULL);
	
}

static Boolean SendSelection(w , selection , target , type , value , length , format)
Widget w;
Atom *selection;
Atom *target;
Atom *type;
XtPointer *value;
unsigned long *length;
int *format;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if (*target == XA_STRING)
	{
		*type = XA_STRING;
		*value = XtNewString(cw->entryLine.selection);
		*length = (long) strlen(cw->entryLine.selection);
		*format = 8;
		return True;
	}
	return False;
}

static void  SelectionEnd(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	int s_end,s_start;

	if (!cw->entryLine.selecting) return;

	if ((cw->entryLine.selection_end - cw->entryLine.selection_end) ||
		(!cw->entryLine.have_selection)) return;

	if (cw->entryLine.selection_start > cw->entryLine.selection_end)
	{
		s_start = cw->entryLine.selection_end;
		s_end = cw->entryLine.selection_start;
	}
	else
	{
		s_end = cw->entryLine.selection_end;
		s_start = cw->entryLine.selection_start;
	}
	
	cw->entryLine.selection = XtMalloc(s_end - s_start + 1);

	strncpy(cw->entryLine.selection , cw->entryLine.text + s_start , s_end - s_start);

	*(cw->entryLine.selection + s_end - s_start) = '\0';

	XtOwnSelection(w , XA_PRIMARY , event->xbutton.time ,
		SendSelection , SelectionOut , NULL);

	cw->entryLine.selecting = False;
}

static void GetSelection(w , client_data , selection , type , value , length , format)
Widget w;
XtPointer client_data;
Atom *selection;
Atom *type;
XtPointer *value;
unsigned long *length;
int *format;
{
	EntryLineWidget cw = (EntryLineWidget) w;

	if ((value == NULL) || (*length == 0)) return;

	RedrawCursor(w , cw->core.background_pixel);

	insert_text(w , value);

	if (cw->base.focused)
		RedrawCursor(w , cw->entryLine.cursor_color);

	XtCallCallbackList((Widget)w , cw->entryLine.change , cw->entryLine.text);
}

static void  InsertSelection(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Atom selection;
	int buffer;

	if (!cw->entryLine.editable)
	{
		XBell(XtDisplay(w) , 50);
		return;
	}

	if (*num_params)
	{
		selection = XInternAtom(XtDisplay(w), *params, False);
		switch (selection) {
			case XA_CUT_BUFFER0: buffer = 0; break;
			case XA_CUT_BUFFER1: buffer = 1; break;
			case XA_CUT_BUFFER2: buffer = 2; break;
			case XA_CUT_BUFFER3: buffer = 3; break;
			case XA_CUT_BUFFER4: buffer = 4; break;
			case XA_CUT_BUFFER5: buffer = 5; break;
			case XA_CUT_BUFFER6: buffer = 6; break;
			case XA_CUT_BUFFER7: buffer = 7; break;
			default: buffer = -1;
		}
		if (buffer >= 0)
		{
			int nbytes;
			unsigned long length;
			int fmt8 = 8;
			Atom type = XA_STRING;
			char *line = XFetchBuffer(XtDisplay(w), &nbytes, buffer);

			if ((length = nbytes))
				GetSelection(w, NULL, &selection, &type, (caddr_t)line,
                        	       &length, &fmt8);
		}
		else
			XtGetSelectionValue(w, XA_PRIMARY , XA_STRING , GetSelection ,
				NULL , XtLastTimestampProcessed(XtDisplay(w)));
	}
	else
		XtGetSelectionValue(w, XA_PRIMARY , XA_STRING , GetSelection ,
			NULL , XtLastTimestampProcessed(XtDisplay(w)));

}

static void insert_text(w , text)
EntryLineWidget w;
char *text;
{
	int len;
	Dimension width,height;
	Position x,y;


	if (w->entryLine.valid)
	{
		char *p;

		text = XtNewString(text);
		p = text;
		while(*p)
		{
			if (!strchr(w->entryLine.valid , *p))
			{
				memmove(p , p + 1 , strlen(p));
				XBell(XtDisplay((Widget)w) , 50);
			} else p++;
		}
	}

	len = strlen(text);	
	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(
		(Widget) w , &x , &y , &width , &height);
	
	if (((w->entryLine.text_length + len) > w->entryLine.max_length) &&
		(w->entryLine.max_length > 0))
	{
		len = w->entryLine.max_length - w->entryLine.text_length;
		XBell(XtDisplay((Widget)w) , 50);
	}

	if ((w->entryLine.text_length + len) > w->entryLine.buffer_length)
	{
		w->entryLine.buffer_length += len + BSIZE_INCREMENT;
		w->entryLine.text = XtRealloc(w->entryLine.text ,
			w->entryLine.buffer_length);
	}

	memmove(w->entryLine.text + w->entryLine.crsr_pos + len , 
		w->entryLine.text + w->entryLine.crsr_pos , 
		strlen(w->entryLine.text + w->entryLine.crsr_pos) + 1);

	memcpy(w->entryLine.text + w->entryLine.crsr_pos , text , len);

	w->entryLine.crsr_pos += len;
	w->entryLine.text_length += len;
	
	while (XTextWidth(w->entryLine.font , w->entryLine.text + w->entryLine.first ,
		w->entryLine.crsr_pos - w->entryLine.first) > width)
		w->entryLine.first ++;

	if (XtIsRealized((Widget)w)) 
		((EntryLineWidgetClass) XtClass((Widget)w))->core_class.expose((Widget) w , 
					NULL , NULL);

	if (w->entryLine.valid) XtFree(text);

}

static void delete_text(w , pos , len)
EntryLineWidget w;
char *pos;
int len;
{

	int length;

	if (len == 0) return;

	length = w->entryLine.text ? strlen(w->entryLine.text) : 0;

	if ((w->entryLine.text + length) < (pos + len)) return;

	memmove(pos , pos + len , strlen(pos + len) + 1);

	w->entryLine.text_length -= len;

}

static void RedrawCursor(w , color)
Widget w;
Pixel color;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Display *dpy = XtDisplay(w);
	Window win = XtWindow(w);
	Dimension width,height;
	Position x,y;
	int crsr_pos ,ty,theight;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(w ,
		&x , &y , &width , &height);

	crsr_pos = x - 1 + XTextWidth(cw->entryLine.font , cw->entryLine.text +
		cw->entryLine.first ,
		cw->entryLine.crsr_pos - cw->entryLine.first);

	ty = y + (height - cw->entryLine.font->max_bounds.ascent -
		cw->entryLine.font->max_bounds.descent) / 2;

	theight = cw->entryLine.font->max_bounds.ascent + 
		cw->entryLine.font->max_bounds.descent;
	
	X_DrawTextCursor(dpy , win , crsr_pos , ty - 1, theight + 2, color);
}

static void RedrawAfterCursor(w)
EntryLineWidget w;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Display *dpy = XtDisplay(w);
	Window win = XtWindow(w);
	Dimension width,height;
	Position x,y;
	int crsr_pos , ty ;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(
		(Widget) w , &x , &y , &width , &height);

	crsr_pos = x + XTextWidth(cw->entryLine.font , cw->entryLine.text +
		cw->entryLine.first ,
		cw->entryLine.crsr_pos - cw->entryLine.first);

	ty = y + (height - cw->entryLine.font->max_bounds.ascent -
		cw->entryLine.font->max_bounds.descent) / 2;

	ComputeShownLength(cw , width);

	XClearArea(dpy , win , crsr_pos , ty , width - crsr_pos + 2, height , False);
	
	if (cw->entryLine.echo)
		XDrawImageString(dpy , win , cw->entryLine.normalGC , crsr_pos ,
			ty + cw->entryLine.font->max_bounds.ascent,
			cw->entryLine.text + cw->entryLine.crsr_pos ,
			cw->entryLine.shown_length - (cw->entryLine.crsr_pos -
			cw->entryLine.first));

}

static int PositionToChar(w , pos)
Widget w;
int pos;
{
	EntryLineWidget cw = (EntryLineWidget) w;
	Dimension width,height;
	Position x,y;
	char *start;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(
		(Widget) w , &x , &y , &width , &height);

	if (pos > x + width) pos = x + width + XTextWidth(cw->entryLine.font , "m" , 1);
	if (pos < x) 
	{
		pos = x - 1;
	}
	
	start = cw->entryLine.text + cw->entryLine.first - 1;

	while (start <= (cw->entryLine.text + cw->entryLine.text_length))
	{
		if (pos < (x + XTextWidth(cw->entryLine.font , 
			cw->entryLine.text + cw->entryLine.first , start - 
			cw->entryLine.text - cw->entryLine.first)))
				return (start - cw->entryLine.text - 
					cw->entryLine.first - 1);

		start ++;
	}

	return cw->entryLine.text_length - cw->entryLine.first;
}

static void RedrawSelection(w , old_end)
EntryLineWidget w;
int old_end;
{
	Display *dpy = XtDisplay(w);
	Window win = XtWindow(w);
	Dimension width,height;
	Position x,y;
	int start_pos , ty ;
	int s_start , s_end;
	int r_len;

	if (! w->entryLine.have_selection) return;

	((EntryLineWidgetClass) XtClass(w))->entryLine_class.get_text_dimension(
		(Widget) w , &x , &y , &width , &height);

	if ((r_len = old_end - w->entryLine.selection_end) > 0)
	{
		s_start = w->entryLine.selection_end;
		s_end = old_end;
	}
	else
	{
		s_end = w->entryLine.selection_end;
		s_start = old_end;
	}

	start_pos = x + XTextWidth(w->entryLine.font , w->entryLine.text +
		w->entryLine.first ,
		s_start - w->entryLine.first);

	ty = y + (height - w->entryLine.font->max_bounds.ascent -
		w->entryLine.font->max_bounds.descent) / 2;
	
	XDrawImageString(dpy , win , w->entryLine.normalGC , start_pos ,
		ty + w->entryLine.font->max_bounds.ascent,
		w->entryLine.text + s_start ,
		s_end - s_start);

	if (w->entryLine.selection_start > w->entryLine.selection_end)
	{
		s_start = w->entryLine.selection_end;
		s_end = w->entryLine.selection_start;
	}
	else
	{
		s_end = w->entryLine.selection_end;
		s_start = w->entryLine.selection_start;

	}
	
	start_pos = x + XTextWidth(w->entryLine.font , w->entryLine.text +
		w->entryLine.first ,
		s_start - w->entryLine.first);

	ty = y + (height - w->entryLine.font->max_bounds.ascent -
		w->entryLine.font->max_bounds.descent) / 2;
	
	XDrawImageString(dpy , win , w->entryLine.reverseGC , start_pos ,
		ty + w->entryLine.font->max_bounds.ascent,
		w->entryLine.text + s_start ,
		s_end - s_start);

}

static void DeleteSelection(w)
EntryLineWidget w;
{
	int s_start , s_end;

	if (w->entryLine.selection_start > w->entryLine.selection_end)
	{
		s_start = w->entryLine.selection_end;
		s_end = w->entryLine.selection_start;
	}
	else
	{
		s_end = w->entryLine.selection_end;
		s_start = w->entryLine.selection_start;

	}
	w->entryLine.crsr_pos = s_start;
	delete_text(w , w->entryLine.text + s_start , s_end - s_start);

	XtCallCallbackList((Widget)w , w->entryLine.change , w->entryLine.text);

	((EntryLineWidgetClass) XtClass(w))->core_class.expose((Widget) w , NULL , NULL);
}

static void ComputeShownLength(w , width)
EntryLineWidget w;
{

	w->entryLine.shown_length = 0;

	for (w->entryLine.shown_length = 0 ; 
		(XTextWidth(w->entryLine.font , w->entryLine.text + 
			w->entryLine.first , w->entryLine.shown_length ) < width ) &&
			((w->entryLine.shown_length + w->entryLine.first) < 
			w->entryLine.text_length);
		w->entryLine.shown_length ++);


}

void EntryLineSetText(w , text)
Widget w;
char *text;
{
	EntryLineWidget cw = (EntryLineWidget) w;	

	delete_text(w , cw->entryLine.text , cw->entryLine.text_length);
	cw->entryLine.crsr_pos = 0;
	cw->entryLine.text_length = 0;
	cw->entryLine.first = 0;

	if (text) insert_text(w , text);

	XtCallCallbackList((Widget)w , cw->entryLine.change , cw->entryLine.text);

}

char * EntryLineGetText(w)
Widget w;
{
	return ((EntryLineWidget)w)->entryLine.text;
}
