/*
	**
	** traffic-tops.c
	**
	** Main module for report postscript generation for traffic-vis
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: traffic-tops.c,v 1.5 1999/04/07 23:07:09 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <time.h>
#include <sys/types.h>

#include "util.h"
#include "report.h"
#include "read-report.h"
#include "sort-report.h"
#include "psfe.h"

static char			rcsid[] = "$Id: traffic-tops.c,v 1.5 1999/04/07 23:07:09 dmiller Exp $";

#ifdef HAVE_GETOPT_LONG
/* Commandline options */
static struct option long_options[] =
{
	{"input", 1, NULL, 'i'},
	{"output", 1, NULL, 'o'},
	{"no-randomise", 0, NULL, 'n'},
	{"suppress-title", 0, NULL, 's'},
	{"suppress-times", 0, NULL, 't'},
	{"title", 1, NULL, 'T'},
	{"base-line-width", 1, NULL, 'b'},
	{"max-line-width", 1, NULL, 'm'},
	{"version", 0, NULL, 'V'},
	{"help", 0, NULL, 'h'},
	{NULL, 0, NULL, 0}
};
#endif /* HAVE_GETOPT_LONG */

/* Prototypes */
void usage(void);
void version(void);
int main(int argc, char **argv);

void usage(void)
{
	fprintf(stderr, "Usage: traffic-tops [OPTION]\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Filter which convers a network traffic summary created by traffic-collector\n");
	fprintf(stderr, "into a Postscript(tm) graph.\n");

	fprintf(stderr, "\n");

#ifdef HAVE_GETOPT_LONG
	fprintf(stderr, "  -i, --input  file         Read input from 'file' instead of stdin.\n");
	fprintf(stderr, "  -o, --output file         Write output to 'file' instead of stdout.\n");
	fprintf(stderr, "  -n, --no-randomise        Do not randomise order of hosts on graph.\n");
	fprintf(stderr, "  -s, --suppress-title      Suppress printing title of chart.\n");
	fprintf(stderr, "  -t, --suppress-times      Suppress printing times on chart.\n");
	fprintf(stderr, "  -T, --title [title]       Specify custom title.\n");
	fprintf(stderr, "  -b, --base-line-width [n] Specify minimum inter-host line width.\n");
	fprintf(stderr, "  -m, --max-line-width  [n] Specify maximum inter-host line width.\n");
	fprintf(stderr, "  -V, --version             Print program version.\n");
	fprintf(stderr, "  -h, --help                Display this help text.\n");
#else /* HAVE_GETOPT_LONG */
	fprintf(stderr, "  -i file    Read input from 'file' instead of stdin.\n");
	fprintf(stderr, "  -o file    Write output to 'file' instead of stdout.\n");
	fprintf(stderr, "  -n         Do not randomise order of hosts on graph.\n");
	fprintf(stderr, "  -s         Suppress printing title of chart.\n");
	fprintf(stderr, "  -t         Suppress printing times on chart.\n");
	fprintf(stderr, "  -T [title] Specify custom title.\n");
	fprintf(stderr, "  -b [n]     Specify minimum inter-host line width.\n");
	fprintf(stderr, "  -m [n]     Specify maximum inter-host line width.\n");
	fprintf(stderr, "  -V         Print program version.\n");
	fprintf(stderr, "  -h         Display this help text.\n");
#endif /* HAVE_GETOPT_LONG */

	fprintf(stderr, "\n");

	fprintf(stderr, "Please report bugs to dmiller@ilogic.com.au\n");
}

void version(void)
{
	fprintf(stderr, "traffic-tops %s\n", VERSION);
}

int main(int argc, char **argv)
{
	report_t			*r;

	int 				c;
	extern char 	*optarg;
	
	/* Commandline options */
	static int		randomise_order = 1;
	static int		suppress_title = 0;
	static int		suppress_times = 0;
	static char		*title = NULL;
	static float	base_line_width = -1;
	static float	max_line_width = -1;
	static char		*input_file = NULL;
	static char		*output_file = NULL;
	
	/* Fetch commandline options */
	while (1)
	{
#ifdef HAVE_GETOPT_LONG
		c = getopt_long (argc, argv, "i:o:b:m:T:tsnhV?", long_options, NULL);
#else /* HAVE_GETOPT_LONG */
		c = getopt (argc, argv, "i:o:b:m:T:tsnhV?");
#endif /* HAVE_GETOPT_LONG */
		if (c == -1)
			break;
		
		switch (c)
		{
			case 'i':
				input_file = util_strdup(optarg);
				break;
				
			case 'o':
				output_file = util_strdup(optarg);
				break;
				
			case 'n':
				randomise_order = 0;
				break;
				
			case 's':
				suppress_title = 1;
				break;
				
			case 't':
				suppress_times = 1;
				break;
				
			case 'T':
				title = strdup(optarg);
				break;
				
			case 'b':
				base_line_width = atof(optarg);
				break;
				
			case 'm':
				max_line_width = atof(optarg);
				break;
				
			case 'V':
				version();
				exit(0);
				
			case 'h':
			case '?':
				usage();
				exit(0);
				
			default:
				fprintf(stderr, "Invalid commandline options.\n\n");
				usage();
				exit(-1);
		}
	}

	if (base_line_width == -1)
	{
		base_line_width = 0;
		max_line_width = 0;
	} else if (max_line_width == -1)
	{
		max_line_width = base_line_width;
	}

	if (randomise_order)
		util_seed_rng();
	
	r = parse_report(input_file);
	
	if (randomise_order)
		r->hosts = sort_hosts(r->hosts, SORT_RANDOM, 0);

	ps_display_summary(r, output_file, base_line_width, max_line_width, 
							 suppress_title, title, suppress_times);

	return(0);
}
