/*
	**
	** summary-output.c
	**
	** Outputs summary to specified file
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: summary-output.c,v 1.3 1999/02/15 12:15:07 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <syslog.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "summary-output.h"
#include "util.h"

static char rcsid[] = "$Id: summary-output.c,v 1.3 1999/02/15 12:15:07 dmiller Exp $";

/* Structure passed to print_host callback */
struct output_info
{
	FILE *out;
	GHashTable *peers;
};

/* Prototypes */
void print_host(u_int8_t *dummy, host_t *host, struct output_info *oinfo);
void print_peers(GHashTable *peers, u_int64_t *peer_keys, unsigned int num_peers, FILE *out);

/* Displays summary of network traffic */
void output_summary(const char *summary_file, summary_t *p)
{
	struct output_info oinfo;
	
	oinfo.out = fopen(summary_file, "w");
	if (oinfo.out == NULL)
	{
		syslog(LOG_CRIT, "Unable to open output file '%s': %m");
		return;
	}
	oinfo.peers = p->peers;
	
	fprintf(oinfo.out, "# Report created at: %s\n", ctime(&(p->summary_finish)));
	
	fprintf(oinfo.out, "START_TIME\t%lu\n", p->summary_start);
	fprintf(oinfo.out, "FINISH_TIME\t%lu\n", p->summary_finish);
	fprintf(oinfo.out, "TOTAL_HOSTS\t%u\n", p->active_hosts);
	fprintf(oinfo.out, "TOTAL_BYTES\t%llu\n", p->total_bytes);
	fprintf(oinfo.out, "TOTAL_PACKETS\t%llu\n", p->total_packets);
	fprintf(oinfo.out, "TOTAL_CONREQS\t%llu\n", p->total_connections);
	fprintf(oinfo.out, "\n");

	g_hash_table_foreach(p->hosts, (GHFunc)print_host, (gpointer)&oinfo);

	fclose(oinfo.out);
}

void print_host(u_int8_t *dummy, host_t *host, struct output_info *oinfo)
{
	struct in_addr i;

	i.s_addr = htonl(host->ip_addr);
	
	fprintf(oinfo->out, "NEW_HOST\n");
	fprintf(oinfo->out, "\tHOST_ADDR\t%s\n", inet_ntoa(i));
	fprintf(oinfo->out, "\tHOST_BYTES_SENT\t%llu\n", host->bytes_sent);
	fprintf(oinfo->out, "\tHOST_BYTES_RECEIVED\t%llu\n", host->bytes_received);
	fprintf(oinfo->out, "\tHOST_PACKETS_SENT\t%llu\n", host->packets_sent);
	fprintf(oinfo->out, "\tHOST_PACKETS_RECEIVED\t%llu\n", host->packets_received);
	fprintf(oinfo->out, "\tHOST_CONREQS_SENT\t%llu\n", host->connections_sent);
	fprintf(oinfo->out, "\tHOST_CONREQS_RECEIVED\t%llu\n", host->connections_received);
	fprintf(oinfo->out, "\tHOST_FIRST_TRAFFIC_TIME\t%lu\n", host->first_seen);
	fprintf(oinfo->out, "\tHOST_LAST_TRAFFIC_TIME\t%lu\n", host->last_seen);

	if (host->n_peers != 0)
		print_peers(oinfo->peers, host->peer_keys, host->n_peers, oinfo->out);
	
	fprintf(oinfo->out, "END_HOST\n");
	fprintf(oinfo->out, "\n");
}

void print_peers(GHashTable *peers, u_int64_t *peer_keys, unsigned int num_peers, FILE *out)
{
	struct 			in_addr i;
	char				host1_addr[20];
	char				host2_addr[20];
	unsigned long	c;
	peer_t			*peer;
	
	for(c = 0; c < num_peers; c++)
	{
		peer = g_hash_table_lookup(peers, (gpointer)&(peer_keys[c]));
		if (peer == NULL)
		{
			syslog(LOG_CRIT, "Missing peer record for key %lli", peer_keys[c]);
			raise(SIGSEGV);
		}
	
		i.s_addr = htonl(peer->host1->ip_addr);
		snprintf(host1_addr, sizeof(host1_addr), "%s", inet_ntoa(i));
		i.s_addr = htonl(peer->host2->ip_addr);
		snprintf(host2_addr, sizeof(host2_addr), "%s", inet_ntoa(i));
		
		if (peer->bytes_sent_1_to_2)
		{
			fprintf(out, "\tNEW_PEER\n");
			fprintf(out, "\t\tPEER_SRC_ADDR\t%s\n", host1_addr);
			fprintf(out, "\t\tPEER_DST_ADDR\t%s\n", host2_addr);
			fprintf(out, "\t\tPEER_BYTES\t%llu\n", peer->bytes_sent_1_to_2);
			fprintf(out, "\t\tPEER_PACKETS\t%llu\n", peer->packets_sent_1_to_2);
			fprintf(out, "\t\tPEER_CONREQS\t%llu\n", peer->connections_sent_1_to_2);
			fprintf(out, "\tEND_PEER\n");
		}

		if (peer->bytes_sent_2_to_1)
		{
			fprintf(out, "\tNEW_PEER\n");
			fprintf(out, "\t\tPEER_SRC_ADDR\t%s\n", host2_addr);
			fprintf(out, "\t\tPEER_DST_ADDR\t%s\n", host1_addr);
			fprintf(out, "\t\tPEER_BYTES\t%llu\n", peer->bytes_sent_2_to_1);
			fprintf(out, "\t\tPEER_PACKETS\t%llu\n", peer->packets_sent_2_to_1);
			fprintf(out, "\t\tPEER_CONREQS\t%llu\n", peer->connections_sent_2_to_1);
			fprintf(out, "\tEND_PEER\n");
		}
	}
}
