/* -*- mode: c; c-file-style: "gnu" -*-
 * src/methods/pam.c -- PAM authentication routines
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy-Auth.
 *
 * Thy-Auth is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Thy-Auth is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file pam.c
 * PAM authentication routines. This is used by all Authorisers that
 * wish to do the authentication with PAM.
 */

#include "compat/compat.h"
#include "methods/pam.h"
#include "misc.h"

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <security/pam_appl.h>

/** @internal Free up some PAM stuff.
 */
static void
_auth_freeresp (int numresp, struct pam_response *resp)
{
  int respcount;

  for (respcount = 0; respcount < numresp; respcount++)
    if (resp[respcount].resp != NULL)
      free (resp[respcount].resp);
  free (resp);
}

/** @internal PAM conversation routine.
 * See the PAM docs for details on what this does.
 */
static int
_auth_pconv (int num_msg, const struct pam_message **msg, struct
	     pam_response **result, void *appdata_ptr)
{
  int account;
  struct pam_response *resp;

  resp = (struct pam_response *)bhc_malloc
    (num_msg * sizeof (struct pam_response));

  if (resp == NULL)
    return PAM_CONV_ERR;

  /* Do the conversation... */
  for (account = 0; account < num_msg; account++)
    switch (msg[account]->msg_style)
      {
      case PAM_PROMPT_ECHO_OFF:
	resp[account].resp =
	  bhc_strdup (((char **) appdata_ptr)[1]);
	if (resp[account].resp == NULL)
	  {
	    _auth_freeresp (account, resp);
	    return PAM_CONV_ERR;
	  }
	resp[account].resp_retcode = PAM_SUCCESS;
	break;
      case PAM_PROMPT_ECHO_ON:
	resp[account].resp =
	  bhc_strdup (((char **) appdata_ptr)[0]);
	if (resp[account].resp == NULL)
	  {
	    _auth_freeresp (account, resp);
	    return PAM_CONV_ERR;
	  }
	resp[account].resp_retcode = PAM_SUCCESS;
	break;
      case PAM_TEXT_INFO:
      case PAM_ERROR_MSG:
	resp[account].resp_retcode = PAM_SUCCESS;
	resp[account].resp = NULL;
	break;
      default:
	_auth_freeresp (account, resp);
	return PAM_CONV_ERR;
      }
  *result = resp;
  return PAM_SUCCESS;
}

/** Authenticate a user via PAM.
 * @param user is the name of the user to authenticate.
 * @param pw is his password.
 * @param realm is the realm we're authenticating in.
 * @param service is the service name, or NULL if we should use the
 * default.
 *
 * @returns Zero on sucess, -1 if authentication failed.
 */
int
thy_auth_pam_auth (const char *user, const char *pw,
		   const char *realm, const char *service)
{
  char *app_data[2];
  struct pam_conv pamconv;
  static pam_handle_t *pamhandle;
  int result;

  app_data[0] = NULL;
  asprintf (&app_data[0], "%s@%s", user, realm);
  app_data[1] = bhc_strdup (pw);

  pamconv.conv = &_auth_pconv;
  pamconv.appdata_ptr = app_data;

  result = pam_start ((service) ? service : THY_AUTH_PAM_DEFAULT_SERVICE,
		      app_data[0], &pamconv, &pamhandle);

  if (result != PAM_SUCCESS)
    {
      free (app_data[0]);
      free (app_data[1]);
      pam_end (pamhandle, pam_close_session (pamhandle, 0));
      return -1;
    }
  if (pam_authenticate (pamhandle, 0) != PAM_SUCCESS)
    {
      free (app_data[0]);
      free (app_data[1]);
      pam_end (pamhandle, pam_close_session (pamhandle, 0));
      return -1;
    }
  if (pam_acct_mgmt (pamhandle, 0) != PAM_SUCCESS)
    {
      free (app_data[0]);
      free (app_data[1]);
      pam_end (pamhandle, pam_close_session (pamhandle, 0));
      return -1;
    }
  if (pam_setcred (pamhandle, PAM_ESTABLISH_CRED) != PAM_SUCCESS)
    {
      free (app_data[0]);
      free (app_data[1]);
      pam_end (pamhandle, pam_close_session (pamhandle, 0));
      return -1;
    }
  if (pam_open_session (pamhandle, 0) != PAM_SUCCESS)
    {
      free (app_data[0]);
      free (app_data[1]);
      pam_end (pamhandle, pam_close_session (pamhandle, 0));
      return -1;
    }

  free (app_data[0]);
  free (app_data[1]);
  pam_end (pamhandle, pam_close_session (pamhandle, 0));

  return 0;
}
