/*
 * tcpreen.cc - TCP connection reverse engineering tool.
 * $Id: tcpreen.cc,v 1.8 2003/01/26 09:09:08 rdenisc Exp $
 */

/***********************************************************************
 *  Copyright (C) 2002-2003 Rmi Denis-Courmont.                       *
 *  This program is free software; you can redistribute and/or modify  *
 *  it under the terms of the GNU General Public License as published  *
 *  by the Free Software Foundation; version 2 of the license.         *
 *                                                                     *
 *  This program is distributed in the hope that it will be useful,    *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.               *
 *  See the GNU General Public License for more details.               *
 *                                                                     *
 *  You should have received a copy of the GNU General Pulic License   *
 *  along with this program; if not, you can get it from:              *
 *  http://www.gnu.org/copyleft/gpl.html                               *
 ***********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h> /* strncpy(), memset(), memcmp() */
#include <stdlib.h> /* exit(), calloc(), free(), abort() */
#include <stddef.h> /* size_t */
#ifdef HAVE_LIMITS_H
# include <limits.h> /* ULONG_MAX */
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h> /* uid_t */
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h> /* close(), fork(), select() */
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#if defined(HAVE_SYS_SOCKET_H)
# include <sys/socket.h>
#endif
#include "getaddrinfo.h" /* NI_* */
#include "systlog.h" /* syslog() */
#include "tcpreen.h"
#include "bridge.h"
#include "log.h"
#include "socketinfo.h"
#include "solve.h"

#ifndef HAVE_WORKING_FORK
# define fork( ) -1
# define wait( retval_ptr ) -1
#endif


static int use_syslog = 0;

static void
bridge_cleanup (int *in = NULL, int *out = NULL)
{
	if (use_syslog)
		closelog ();
	if (in != NULL)
		fdarray_close (in);
	if (out != NULL)
		fdarray_close (out);
}


static void
bridge_perror (const char *str, int level = LOG_ERR)
{
	const char *msg = _(str);

	if (use_syslog)
		syslog (level, "%s: %m\n", msg);
	else
		perror (msg);
}



void
bridge_hosterror (int errval, const struct sockhostinfo *host,
			int level = LOG_ERR)
{
	if (use_syslog)
	{
		const char *name = (host->hostname[0])
			? host->hostname
			: "localhost", *serv = host->service;

		if (errval == EAI_SYSTEM)
			syslog (level, "[%s]:%s: %m\n", name, serv);
		else
			syslog (level, "[%s]:%s: %s\n", name, serv,
					gai_strerror (errval));
	}
	else
		sockhostinfo_perror (errval, host);
}


/*
 * Establishes a bridge between a TCP listener and a TCP active
 * connection.
 * The listening socket is dropped as soon as the first incoming
 * connection is established.
 *
 * If <conf->maxclients> is nul, bridge_main will not fork, and will
 * treat each client connection one by one (this is the only possible
 * mode if you don't have fork()). Otherwise, it defines the maximum
 * number of child processes that can run at a time.
 *
 * Returns 0 on success, true on error.
 */
int
bridge_main (const struct bridgeconf *conf)
{
	int *in = NULL, *out = NULL, numclients = 0;

	/* explicitly bufferized settings */
	int mode = conf->mode;
	long maxclients = conf->maxclients;
	int logging = !(!*(conf->logmaker));

	/* Hostname lookup flags for getsockaddr(ie. getnameinfo()) */
	int addrflags = (mode & tcpreen_numeric)
		? NI_NUMERICHOST | NI_NUMERICSERV
		: 0;

	/* Number of connections to be accepted left */
	int countdown = conf->totalclients;

	// Opens system log
	if (mode & tcpreen_syslog)
	{
		openlog ("tcpreen", LOG_PID, LOG_DAEMON);
		syslog (LOG_NOTICE, _("starting\n"));
		use_syslog = 1;
	}

	/* Server socket(s) */
	if (!(mode & tcpreen_connect_both))
	{
		if (seteuid (conf->priv))
		{
			bridge_perror ("seteuid", LOG_CRIT);
			bridge_cleanup ();
			return -1;
		}
		
		int check = sockhostinfo_listen (&in, &conf->bridge);
		if (check)
		{
			bridge_hosterror (check, &conf->bridge, LOG_CRIT);
			bridge_cleanup ();
			return -1;
		}
		
		if (seteuid (conf->unpriv))
		{
			bridge_perror ("seteuid", LOG_CRIT);
			bridge_cleanup (in);
			return -1;
		}

		/* Displays listening addresses */
		if (mode & tcpreen_speak)
		for (int *fdp = in; *fdp != -1; fdp++)
		{
			char addr[NI_MAXADDRESS];

			getsockaddr (*fdp, addrflags, 0, addr, sizeof(addr));
			if (use_syslog)
				syslog (LOG_NOTICE, _("Listening on: %s\n"),
						addr);
			if (mode & tcpreen_verbose)
				printf (_("Listening on: %s\n"), addr);
		}
	}

	if (mode & tcpreen_listen_both)
	{
		if (seteuid (conf->priv))
		{
			bridge_perror ("seteuid", LOG_CRIT);
			bridge_cleanup (in);
			return -1;
		}

		int check = sockhostinfo_listen (&out, &conf->server);
		if (check)
		{
			bridge_hosterror (check, &conf->server, LOG_CRIT);
			bridge_cleanup (in);
			return -1;
		}
		if (seteuid (conf->unpriv))
			abort (); /* impossible error */

		if (mode & tcpreen_speak)
		for (int *fdp = out; *fdp != -1; fdp++)
		{
			char addr[NI_MAXADDRESS];

			getsockaddr(*fdp, addrflags, 0, addr, sizeof(addr));
			if (use_syslog)
				syslog (LOG_INFO,
					_("Listening for server on: %s\n"),
					addr);
			if (mode & tcpreen_verbose)
				printf (_("Listening for server on: %s\n"),
					addr);
		}
	}

	/* Definitely drops priviledges */
	if (seteuid (conf->priv) || setuid (conf->unpriv))
	{
		bridge_perror ("setuid", LOG_CRIT);
		bridge_cleanup (in, out);
		return -1;
	}

	// Goes in the background
	if (mode & tcpreen_daemon)
	{
		if (daemon (0, 0))
		{
			perror ("daemon");
			return -1;
		}
	}

/*
 * Server loop (one client processed at a time).
 */
	do
	{
		int fd[4];
		char svr[NI_MAXADDRESS], clt[NI_MAXADDRESS];

		if (countdown > 0)
			countdown--; /* one connection attempt */
		
		/* settinp up client socket... */
		if (mode & tcpreen_connect_both)
		{
			int check = sockhostinfo_connect (fd, &conf->bridge);
			if (check)
			{
				bridge_hosterror (check, &conf->bridge);
				continue;
			}
		}
		else
		{
			if ((fd[0] = fdarray_accept(in)) == -1)
			{
				bridge_perror ("accept");
				continue;
			}
			
			if (countdown == 0)
				fdarray_close (in); // no longer needed
		}

		/* Computes client address */
		if (logging)
			getsockaddr (fd[0], addrflags, 1, clt, sizeof(clt));
		if (use_syslog)
		{
			char addr[NI_MAXADDRESS];

			getsockaddr (fd[0], NI_NUMERICHOST, 1, addr,
					sizeof(addr));
			syslog (LOG_INFO, _("Connection from: %s\n"), addr);
		}
		fd[3] = fd[0];

		/* setting up server socket... */
		if (mode & tcpreen_listen_both)
		{
			if ((fd[1] = fdarray_accept (out)) == -1)
			{
				bridge_perror (N_("server-side socket error"));
				close (fd[0]);
				continue;
			}

			if (countdown == 0)
				fdarray_close (out); // no longer needed
		}
		else
		{
			int val = sockhostinfo_connect (fd + 1, &conf->server);
			if (val)
			{
				bridge_hosterror (val, &conf->server);
				close (fd[0]);
				continue;
			}
		}

		/* Computes server address */
		if (logging)
			getsockaddr (fd[1], addrflags, 1, svr, sizeof(svr));
		if (use_syslog)
		{
			char addr[NI_MAXADDRESS];

			getsockaddr (fd[1], NI_NUMERICHOST, 1, addr,
					sizeof(addr));
			syslog (LOG_INFO, _("Connection to: %s\n"), addr);
		}
		fd[2] = fd[1];

		/* Both sides of the bridge are now properly connected!	*/
		pid_t pid = (maxclients) ? fork () : 0;
		switch (pid)
		{
			case -1:
				bridge_perror ("fork");
				close (fd[0]);
				close (fd[1]);
				break;

			case 0: // child process
			{	/* NOTE: a good reason to avoid using -c and
				 * -F at the same time... */
				int retval = 0;
				DataLogList *logs = conf->logmaker->MakeLogList ((maxclients) ? clt : NULL);
				if (logs == NULL)
				{
					bridge_perror (_("Fatal error"));
					retval = 1;
				}
				else
				{
					if (logging)
						logs->Connect (svr, clt);
					retval = monitor_bridge (fd, logs,
							conf->bytelimit);
					delete logs;
				}

				if (fd[0] != -1)
					close (fd[0]);
				else if (fd[2] != -1)
					close (fd[2]);

				if (fd[1] != -1)
					close (fd[1]);
				else if (fd[3] != -1)
					close (fd[3]);

				if (maxclients)
					exit (retval);
			}

			default: // parent process
				if (maxclients)
				{
					close (fd[0]);
					close (fd[1]);
					numclients++;
				}
		}

		// Makes room for future clients
		if (maxclients)
			while (numclients >= maxclients)
			{
				int val;

				if (wait (&val) != -1)
					numclients--;
			}
	}
	while (countdown);

	if (use_syslog)
	{
		syslog (LOG_NOTICE, _("stopping\n"));
		closelog ();
	}

	/*
	 * Graceful termination for pending clients
	 */
	while (numclients)
	{
		int val;

		if (wait(&val) != -1)
			numclients--;
	}
	return 0;
}

