// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "pyramid.h"
#include "exceptions.h"
#include GL_INCLUDE

namespace visual {


void
pyramid::set_width( const double& w)
{
	write_lock L(mtx);
	width = w;
}

void
pyramid::set_height( const double& h)
{
	write_lock L(mtx);
	height = h;
}


void
pyramid::set_length( const double& len)
{
	if (len == 0.0)
		throw python::ZeroDivisionError("Degenerate primitive");
	axis *= (len / axis.mag());
}


void
pyramid::set_size( const vector& v)
{
	axis = axis.norm() * v.x;
	write_lock L(mtx);
	height = v.y;
	width = v.z;
}

vector
pyramid::getScale()
{
	return vector( axis.mag(), height, width);
}

void
pyramid::glRender( rView& view)
{
	if (degenerate)
		return;

	static double brect[] = { 1.0,  1.0,  1.0,
	                          0.0, -1.0, -1.0};
	view.ext_brect(mwt, brect);

	tmatrix mct(mwt,view.wct);

	// Calculate the lighting on each of the faces of the pyramid.  
	float L[5];
	float length = axis.mag();
	float nrfb = 1.0/sqrt(length*length+height*height/4.0); // front and back faces
	float ncfb = width/2 * nrfb;
	float nsfb = length * nrfb;
	float nrul = 1.0/sqrt(length*length+width*width/4.0); // upper and lower faces
	float ncul = height/2 * nrul;
	float nsul = length * nrul;
	float xfb, yfb, zfb, xul, yul, zul; // Defaulted to zero

	for(int i=0; i<5; i++) 
		L[i] = view.lights.ambient;
	for(int l=0; l<view.lights.n_lights; l++) {
		vector lt = wlt * view.lights.L[l];
		if (lt.x < 0)
			L[0]-=lt.x; // Lighting the base
		xfb = ncfb*lt.x; // for front and back faces
		yfb = nsfb*lt.y;
		zfb = nsfb*lt.z;
		xul = ncul*lt.x; // for upper and lower faces
		yul = nsul*lt.y;
		zul = nsul*lt.z;

		if (xfb > zfb)
			L[1] += xfb - zfb; // back face (in default position)
		if (xul > yul)
			L[2] += xul - yul; // bottom face
		if (xul > -yul)
			L[3] += xul + yul; // top face
		if (xfb > -zfb)
			L[4] += xfb + zfb; // front face
	} 

	// The following static arrays contain the vertex and
	// face data for a pyramid.  For symmetry with sphere and
	// cylinder, this could be wrapped in a shape_model
	// object.
	static vertex projected[5];
	static int s[5][4] = { {0,1,2,3}, // pyramid bas
	                       {0,1,4,-1},
	                       {0,2,4,-1},
	                       {1,3,4,-1},
	                       {2,3,4,-1} };
	static float v[5][3] = { {0.0,-0.5,-0.5},
	                         {0.0,0.5,-0.5},
	                         {0.0,-0.5,0.5},
	                         {0.0,0.5,0.5},
	                         {1.0,0.0,0.0} };

	// Projection and rendering
	for(int corner=0; corner<5; corner++)
		mct.project(v[corner], projected[corner]);

	glEnableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glVertexPointer(4, GL_DOUBLE, sizeof(vertex), &projected[0].x);
	glShadeModel(GL_FLAT);

	for(int side=0; side<5; side++) {
		float c = L[side];
		glColor3f(c*color.r,c*color.g,c*color.b);
		glDrawElements(GL_TRIANGLE_STRIP, s[side][3]<0 ? 3 : 5, GL_UNSIGNED_INT, s[side]);
	}
}

} // !namespace visual
