/* ptal-mfpdtf -- PTAL-based MFPDTF (HP Multi-Function Peripheral Data
 * Transfer Format) filter. */

/* Copyright (C) 2000-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: David Paschal */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include "ptal.h"


enum ptalMfpdtfImageRecordID_e {
	PTAL_MFPDTF_ID_START_PAGE=0,
	PTAL_MFPDTF_ID_RASTER_DATA=1,
	PTAL_MFPDTF_ID_END_PAGE=2
};

struct ptalMfpdtfFixedHeader_s {
	unsigned char blockLength[4];	/* Includes header(s). */
	unsigned char headerLength[2];	/* Just header(s). */
	unsigned char dataType;
	unsigned char pageFlags;
} __attribute__((packed));

struct ptalMfpdtfImageRasterDataHeader_s {
	unsigned char traits;			/* Unused. */
	unsigned char byteCount[2];
} __attribute__((packed));

struct ptalMfpdtf_s {
	ptalChannel_t chan;
	int fdLog;				/* <0 means not (yet) open. */
	int logOffset;

	struct {
		struct timeval timeout;
		int simulateImageHeaders;

		int lastServiceResult;
		int dataType;			/* <0 means not (yet) valid. */
		int arrayRecordCount;
		int arrayRecordSize;
		int fixedBlockBytesRemaining;	/* Also generic data. */
		int innerBlockBytesRemaining;	/* Image or array data. */
		int dontDecrementInnerBlock;

		struct ptalMfpdtfFixedHeader_s fixedHeader;
		int lenVariantHeader;
		union ptalMfpdtfVariantHeader_u *pVariantHeader;
		struct ptalMfpdtfImageStartPageRecord_s imageStartPageRecord;
		struct ptalMfpdtfImageRasterDataHeader_s imageRasterDataHeader;
		struct ptalMfpdtfImageEndPageRecord_s imageEndPageRecord;

	} read;

	struct {
		/* TODO: Implement write. */


	} write;
};


union ptalMfpdtfVariantHeader_u *ptalMfpdtfReadAllocateVariantHeader(
    ptalMfpdtf_t mfpdtf,int datalen) {
	if (mfpdtf->read.pVariantHeader) {
		free(mfpdtf->read.pVariantHeader);
		mfpdtf->read.pVariantHeader=0;
	}
	mfpdtf->read.lenVariantHeader=datalen;
	if (datalen) {
		mfpdtf->read.pVariantHeader=malloc(datalen);
	}
	return mfpdtf->read.pVariantHeader;
}

ptalMfpdtf_t ptalMfpdtfAllocate(ptalChannel_t chan) {
	int size=sizeof(struct ptalMfpdtf_s);
	ptalMfpdtf_t mfpdtf=malloc(size);
	if (mfpdtf) {
		memset(mfpdtf,0,size);
		mfpdtf->chan=chan;
		mfpdtf->fdLog=PTAL_ERROR;
		ptalMfpdtfReadSetTimeout(mfpdtf,30);
		ptalMfpdtfReadStart(mfpdtf);
	}
	return mfpdtf;
}

int ptalMfpdtfDeallocate(ptalMfpdtf_t mfpdtf) {
	if (!mfpdtf) {
		return PTAL_ERROR;
	}
	ptalMfpdtfLogToFile(mfpdtf,0);
	ptalMfpdtfReadAllocateVariantHeader(mfpdtf,0);
	free(mfpdtf);
	return PTAL_OK;
}

int ptalMfpdtfSetChannel(ptalMfpdtf_t mfpdtf,ptalChannel_t chan) {
	mfpdtf->chan=chan;
	/* If necessary, we can query the device ID string using the
	 * channel's device pointer. */
	return PTAL_OK;
}

int ptalMfpdtfLogToFile(ptalMfpdtf_t mfpdtf,char *filename) {
	if (mfpdtf->fdLog!=PTAL_ERROR) {
		close(mfpdtf->fdLog);
		mfpdtf->fdLog=PTAL_ERROR;
	}
	mfpdtf->logOffset=0;
	if (filename) {
		int fd=creat(filename,0600);
		if (fd<0) {
			return PTAL_ERROR;
		}
		mfpdtf->fdLog=fd;
	}
	return PTAL_OK;
}

int ptalMfpdtfReadGetTimeout(ptalMfpdtf_t mfpdtf) {
	return mfpdtf->read.timeout.tv_sec;
}

int ptalMfpdtfReadSetTimeout(ptalMfpdtf_t mfpdtf,int seconds) {
	mfpdtf->read.timeout.tv_sec=seconds;
	mfpdtf->read.timeout.tv_usec=0;

	return seconds;
}

int ptalMfpdtfReadGetSimulateImageHeaders(ptalMfpdtf_t mfpdtf) {
	return mfpdtf->read.simulateImageHeaders;
}

int ptalMfpdtfReadSetSimulateImageHeaders(ptalMfpdtf_t mfpdtf,
    int simulateImageHeaders) {
	mfpdtf->read.simulateImageHeaders=simulateImageHeaders;
	return simulateImageHeaders;
}

int ptalMfpdtfReadStart(ptalMfpdtf_t mfpdtf) {
	mfpdtf->read.lastServiceResult=0;
	mfpdtf->read.dataType=PTAL_ERROR;
	mfpdtf->read.arrayRecordCount=mfpdtf->read.arrayRecordSize;
	mfpdtf->read.fixedBlockBytesRemaining=0;
	mfpdtf->read.innerBlockBytesRemaining=0;
	mfpdtf->read.dontDecrementInnerBlock=0;
	ptalMfpdtfReadAllocateVariantHeader(mfpdtf,0);

	return PTAL_OK;
}

#define READ(buffer,datalen) \
	do { \
		int r=ptalMfpdtfReadGeneric(mfpdtf, \
			(unsigned char *)(buffer),datalen); \
		if (r!=datalen) { \
			if (r<0) return PTAL_MFPDTF_RESULT_READ_ERROR; \
			return PTAL_MFPDTF_RESULT_READ_TIMEOUT; \
		} \
	} while(0)

#define RETURN(_result) \
	return (mfpdtf->read.lastServiceResult=(_result));

int ptalMfpdtfReadService(ptalMfpdtf_t mfpdtf) {
	int result=0;
	int datalen,blockLength,headerLength;

    if (mfpdtf->read.fixedBlockBytesRemaining<=0) {
	/* Read fixed header. */
	datalen=sizeof(mfpdtf->read.fixedHeader);
	mfpdtf->read.fixedBlockBytesRemaining=datalen;
	PTAL_LOG_DEBUG("Reading fixed header.\n");
	mfpdtf->read.dontDecrementInnerBlock=1;
	READ(&mfpdtf->read.fixedHeader,datalen);

	/* Parse fixed header. */
	blockLength=LEND_GET_LONG(mfpdtf->read.fixedHeader.blockLength);
	mfpdtf->read.fixedBlockBytesRemaining=blockLength-datalen;
	headerLength=LEND_GET_SHORT(mfpdtf->read.fixedHeader.headerLength);

	/* Is this a new data type? */
	if (mfpdtf->read.dataType!=mfpdtf->read.fixedHeader.dataType) {
		mfpdtf->read.dataType=mfpdtf->read.fixedHeader.dataType;
		result|=PTAL_MFPDTF_RESULT_NEW_DATA_TYPE;
	}

	/* Read variant header (if any). */
	datalen=headerLength-sizeof(mfpdtf->read.fixedHeader);
	if (datalen>0) {
		if (!ptalMfpdtfReadAllocateVariantHeader(mfpdtf,datalen)) {
			RETURN(PTAL_MFPDTF_RESULT_OTHER_ERROR);
		}
		PTAL_LOG_DEBUG("Reading variant header (%d bytes).\n",datalen);
		mfpdtf->read.dontDecrementInnerBlock=1;
		READ(mfpdtf->read.pVariantHeader,datalen);
		result|=PTAL_MFPDTF_RESULT_NEW_VARIANT_HEADER;

		/* Is this a valid array variant header? */
		mfpdtf->read.arrayRecordSize=0;
		mfpdtf->read.arrayRecordCount=0;
		mfpdtf->read.innerBlockBytesRemaining=0;
		if (ptalMfpdtfReadIsArrayData(mfpdtf) &&
		    mfpdtf->read.lenVariantHeader>=sizeof(mfpdtf->read.pVariantHeader->array)) {
			mfpdtf->read.arrayRecordCount=LEND_GET_SHORT(
				mfpdtf->read.pVariantHeader->array.recordCount);
			mfpdtf->read.arrayRecordSize=LEND_GET_SHORT(
				mfpdtf->read.pVariantHeader->array.recordSize);
			mfpdtf->read.innerBlockBytesRemaining=
				mfpdtf->read.arrayRecordCount*
				mfpdtf->read.arrayRecordSize;
		}
	}

    } else if (ptalMfpdtfReadIsImageData(mfpdtf)) {
	if (mfpdtf->read.innerBlockBytesRemaining>0) {
		result|=PTAL_MFPDTF_RESULT_IMAGE_DATA_PENDING;

	} else if (mfpdtf->read.simulateImageHeaders) {
		mfpdtf->read.innerBlockBytesRemaining=
			mfpdtf->read.fixedBlockBytesRemaining;
		if (mfpdtf->read.innerBlockBytesRemaining>0) {
			result|=PTAL_MFPDTF_RESULT_IMAGE_DATA_PENDING;
		}

	} else {
		unsigned char id;
		datalen=1;
		READ(&id,datalen);

		if (id==PTAL_MFPDTF_ID_RASTER_DATA) {
			datalen=sizeof(mfpdtf->read.imageRasterDataHeader);
			PTAL_LOG_DEBUG("Reading raster data header.\n");
			READ(&mfpdtf->read.imageRasterDataHeader,datalen);
			mfpdtf->read.innerBlockBytesRemaining=LEND_GET_SHORT(
				mfpdtf->read.imageRasterDataHeader.byteCount);
			result|=PTAL_MFPDTF_RESULT_IMAGE_DATA_PENDING;

		} else if (id==PTAL_MFPDTF_ID_START_PAGE) {
			datalen=sizeof(mfpdtf->read.imageStartPageRecord);
			PTAL_LOG_DEBUG("Reading start of page record.\n");
			READ(&mfpdtf->read.imageStartPageRecord,datalen);
			result|=PTAL_MFPDTF_RESULT_NEW_START_OF_PAGE_RECORD;

		} else if (id==PTAL_MFPDTF_ID_END_PAGE) {
			datalen=sizeof(mfpdtf->read.imageEndPageRecord);
			PTAL_LOG_DEBUG("Reading end of page record.\n");
			READ(&mfpdtf->read.imageEndPageRecord,datalen);
			result|=PTAL_MFPDTF_RESULT_NEW_END_OF_PAGE_RECORD;

		} else {
			RETURN(PTAL_MFPDTF_RESULT_OTHER_ERROR);
		}
	}

    } else if (ptalMfpdtfReadIsArrayData(mfpdtf)) {
	if (mfpdtf->read.innerBlockBytesRemaining>0) {
		result|=PTAL_MFPDTF_RESULT_ARRAY_DATA_PENDING;
	}
    }

	if (mfpdtf->read.fixedBlockBytesRemaining>0) {
		result|=PTAL_MFPDTF_RESULT_GENERIC_DATA_PENDING;
	}

	RETURN((result|mfpdtf->read.fixedHeader.pageFlags));
}

int ptalMfpdtfReadGetDataType(ptalMfpdtf_t mfpdtf) {
	return mfpdtf->read.dataType;
}

int ptalMfpdtfReadIsImageData(ptalMfpdtf_t mfpdtf) {
	return ((PTAL_MFPDTF_DT_MASK_IMAGE&(1<<mfpdtf->read.dataType))!=0);
}

int ptalMfpdtfReadIsArrayData(ptalMfpdtf_t mfpdtf) {
	return (!ptalMfpdtfReadIsImageData(mfpdtf));
}

int ptalMfpdtfReadGetArrayRecordCountSize(ptalMfpdtf_t mfpdtf,
    int *pCount,int *pSize) {
	if (pCount) *pCount=mfpdtf->read.arrayRecordCount;
	if (pSize) *pSize=mfpdtf->read.arrayRecordSize;
	return mfpdtf->read.innerBlockBytesRemaining;
}

int ptalMfpdtfReadGetFixedBlockBytesRemaining(ptalMfpdtf_t mfpdtf) {
	return mfpdtf->read.fixedBlockBytesRemaining;
}

int ptalMfpdtfReadGetInnerBlockBytesRemaining(ptalMfpdtf_t mfpdtf) {
	return ptalMfpdtfReadGetArrayRecordCountSize(mfpdtf,0,0);
}

int ptalMfpdtfReadGetLastServiceResult(ptalMfpdtf_t mfpdtf) {
	return mfpdtf->read.lastServiceResult;
}

int ptalMfpdtfReadGetVariantHeader(ptalMfpdtf_t mfpdtf,
    union ptalMfpdtfVariantHeader_u *buffer,int maxlen) {
	if (!mfpdtf->read.pVariantHeader) return 0;
	if (!buffer) return mfpdtf->read.lenVariantHeader;
	if (maxlen>mfpdtf->read.lenVariantHeader) {
		maxlen=mfpdtf->read.lenVariantHeader;
	}
	memcpy(buffer,mfpdtf->read.pVariantHeader,maxlen);
	return maxlen;
}

int ptalMfpdtfReadGetStartPageRecord(ptalMfpdtf_t mfpdtf,
    struct ptalMfpdtfImageStartPageRecord_s *buffer,int maxlen) {
	int len=sizeof(struct ptalMfpdtfImageStartPageRecord_s);
	if (maxlen>len) maxlen=len;
	memcpy(buffer,&mfpdtf->read.imageStartPageRecord,maxlen);
	return maxlen;
}

int ptalMfpdtfReadGetEndPageRecord(ptalMfpdtf_t mfpdtf,
    struct ptalMfpdtfImageEndPageRecord_s *buffer,int maxlen) {
	int len=sizeof(struct ptalMfpdtfImageEndPageRecord_s);
	if (maxlen>len) maxlen=len;
	memcpy(buffer,&mfpdtf->read.imageEndPageRecord,maxlen);
	return maxlen;
}

int ptalMfpdtfReadGeneric(ptalMfpdtf_t mfpdtf,
    unsigned char *buffer,int datalen) {
	int r=0;

	/* Don't read past the currently-defined fixed block. */
	PTAL_LOG_DEBUG("read.fixedBlockBytesRemaining=%d.\n",
		mfpdtf->read.fixedBlockBytesRemaining);
	if (datalen>mfpdtf->read.fixedBlockBytesRemaining) {
		datalen=mfpdtf->read.fixedBlockBytesRemaining;
	}

	/* Read the data. */
	if (datalen>0) {
		PTAL_LOG_DEBUG("Reading %d bytes at offset=0x%8.8X.\n",
			datalen,mfpdtf->logOffset);
		r=ptalChannelReadTimeout(mfpdtf->chan,buffer,datalen,
			&mfpdtf->read.timeout,&mfpdtf->read.timeout);

		if (r>0) {
			/* Account for and log what was read. */
			mfpdtf->read.fixedBlockBytesRemaining-=r;
			if (!mfpdtf->read.dontDecrementInnerBlock) {
				mfpdtf->read.innerBlockBytesRemaining-=r;
			}
			mfpdtf->read.dontDecrementInnerBlock=0;
			mfpdtf->logOffset+=r;
			if (mfpdtf->fdLog>=0) {
				write(mfpdtf->fdLog,buffer,datalen);
			}
		}
		if (r!=datalen) {
			mfpdtf->read.lastServiceResult=r<0?
				PTAL_MFPDTF_RESULT_READ_ERROR:
				PTAL_MFPDTF_RESULT_READ_TIMEOUT;
		}
	}

	return r;
}

int ptalMfpdtfReadInnerBlock(ptalMfpdtf_t mfpdtf,
    unsigned char *buffer,int countdown) {
	int r,countup=0;

	while (42) {
		PTAL_LOG_DEBUG("read.innerBlockBytesRemaining=%d.\n",
			mfpdtf->read.innerBlockBytesRemaining);
		if (countdown>mfpdtf->read.innerBlockBytesRemaining) {
			countdown=mfpdtf->read.innerBlockBytesRemaining;
		}
		if (countdown<=0) break;
		r=ptalMfpdtfReadGeneric(mfpdtf,buffer,countdown);
		if (ptalMfpdtfReadGetLastServiceResult(mfpdtf)&
		     PTAL_MFPDTF_RESULT_ERROR_MASK) {
			break;
		}
		buffer+=r;
		countdown-=r;
		countup+=r;
		if (countdown<=0) break;
		r=ptalMfpdtfReadService(mfpdtf);
		if (r&(PTAL_MFPDTF_RESULT_ERROR_MASK|
		       PTAL_MFPDTF_RESULT_NEW_DATA_TYPE|
		       PTAL_MFPDTF_RESULT_NEW_VARIANT_HEADER)) {
			break;
		}
	}

	return countup;
}

/* TODO: Implement write. */
