/* some sort of xml parser. Used to read/write compilations.
 *
 * This xml parser is optimized for it's designated purpose and does by
 * no means adhere to the standard. It will process all data on-the-fly
 * without unnecessarily wasting memory.
 * Also note that this parser will perform all of it's actions asynchronously
 * that is: all functions will return immediately and call a callback
 * once done */
#ifndef XML_H
#define XML_H

#include <stdio.h>

#include <glib.h>

typedef struct
{
   const char *argument;
   const char *value;
} xml_argument_t;

typedef struct
{
   void *handlers; // handlers for items found within this context (xml_taghandler_t*)
   FILE *source;
   char *currenttag;   // contains the currently open tag
   char *parenttag;    // contains the parent tag
   
   void *donecallback; // xml_donecallback_t
   void *data;         // data passed to the callback
} xml_context_t;

/* Call this from xml_itemhandler_t to tell parser we're done
 * processing the current item.
 * 
 * xml_itemhandler is responsible for leaving FILE* aimed right behind
 * the closing tag (</whatever>) so's we can continue parsing right after
 * that */
void xml_doneitem(int status,xml_context_t *context);

/* Callback for xml handling.
 * 
 * May return immediately without performing any action.
 * itemhandler has to call xml_doneitem(status,context) to tell parser
 * it is done parsing the current item.
 * The parser will then and only then continue it's work
 * according to the status specified to xml_doneitem() */
typedef void(*xml_itemhandler_t)(xml_context_t *context, /* current context */
				 const char *tag,        /* tag that caused this handler to be called */
				 int argc,               /* number of arguments */
				 const xml_argument_t *args, /* arguments, see xml_argument_t */
				 void *data);            /* user data */

typedef struct
{
   const char *tag;
   xml_itemhandler_t handler;
   void *data;
}
xml_taghandler_t;

/* called once xml_parse() is done */
typedef void(*xml_donecallback_t)(int status,
				  xml_context_t *context,
				  void *data);

/* This function will read from the FILE specified and
 * will process any xml compatible data found according to the
 * rules defined by passing a list of handlers. 
 * Will call specified callback upon completion */
void xml_parse(FILE *source,
	       const xml_taghandler_t *initial_handlers,
	       xml_donecallback_t,void*data);

/* Same as above, only FILE will be taken from specified context.
 * Specified context will be left untouched otherwise */
void xml_parse_context(xml_context_t *source,
		       const xml_taghandler_t *initial_handlers,
		       xml_donecallback_t,void*data);

/* returns pointer to the contents of the current xml tag.
 * if length != NULL the length of the content area will be stored
 * in the int addressed by length.
 * Any content will be preprocessed to resolve any escape sequences
 * before it is returned (e.g. &amp; will be replaced by &) 
 * 
 * Returned buffer is caller-owned */
char *xml_getcontent(xml_context_t *context,int *status,int *length);

#endif // XML_H
