/* this file contains a primitive filesystem abstraction layer.
 * It provides functions to read and manipulate directories.
 * Thus, it isn't a proper filesystem, it's merely an abstraction of the
 * directory management functions of a filesystem.
 * It's purpose is to provide a common interface for both the virtualdir.c
 * constructs and the real filesystem functions as defined in dirlow.c
 * We'll stick to Posix Path rules here */

#include <stdlib.h>
#include <glib.h>

#include "vfs.h"
#include "helpings.h"
#include "datacopydlg.h"
#include "main.h"
#include "pseudorecurse.h"

#include "int.h"

/* uncomment for debugging */
//#define DEBUG
//
/* create a vfs filesystem structure to be used for consequent accesses */
vfs_filesystem *vfs_create(createdir_t    		createdir,
			   opendir_t      		opendir,
			   readdirentry_t 		readdirentry,
			   closedir_t     		closedir,
			   filestat_t     		filestat,
			   getlinkdestination_t 	getlinkdestination,
			   symlink_t      		symlink,
			   copy_t         		copy,
			   createfile_getrealname_t     createfile_getrealname,
			   getrealname_t  		getrealname,
			   openfile_t     		openfile,
			   del_t          		del,
			   ren_t          		ren,
			   destroy_t      		destroy,
			   gpointer data)
{
   vfs_filesystem *result=(vfs_filesystem*)malloc(sizeof(vfs_filesystem));

   result->createdir=createdir;
   result->opendir=opendir;
   result->readdirentry=readdirentry;
   result->closedir=closedir;
   result->filestat=filestat;
   result->getlinkdestination=getlinkdestination;
   result->symlink=symlink;
   result->copy=copy;
   result->createfile_getrealname=createfile_getrealname;
   result->getrealname=getrealname;
   result->openfile=openfile;
   result->del=del;
   result->ren=ren;
   result->destroy=destroy;
   result->data=data;

   /* No errors so far :-) */
   result->errorstrings=NULL;
   result->unprocessed_errors=0;

   return result;
};

/* High-level filesystem access */

/* Get latest error message, returns NULL if none.
 * String is owned by the filesystem and will be freed together with it */
const char *vfs_getlasterrorstring(vfs_filesystem*fs)
{
   const char *result=NULL;
   if ((fs->errorstrings)&&(fs->unprocessed_errors))
     result=(const char*)fs->errorstrings->data;
   if (fs->unprocessed_errors)
     --fs->unprocessed_errors;
   return result;
};

void vfs_adderrorstring(vfs_filesystem *fs,const char *msg)
{
   char *newerror=strdup(msg);
   ++fs->unprocessed_errors;
   fs->errorstrings=g_list_prepend(fs->errorstrings,(gpointer)newerror);
};

/* Create a directory, return 0 on success */
int vfs_createdir (vfs_filesystem *fs,char* newdir)
{
   if (fs->createdir)
     return fs->createdir((gpointer)fs,newdir);
   else
     return 1;
};
/* Open a directory, returns directory handle */
gtoaster_handle_t vfs_opendir (vfs_filesystem *fs,char* dir)
{
#ifdef DEBUG
   printf("vfs_opendir: %s\n",dir);
#endif
   if (fs->opendir)
     return fs->opendir((gpointer)fs,dir);
   else
     return 0;
};
/* will return the full path of the next directory item or
 * NULL if the end of the directory has been reached.
 * caller frees string */
char *vfs_readdirentry(vfs_filesystem *fs,gtoaster_handle_t handle)
{
   char *result=NULL;
   if ((fs->readdirentry)&&(handle))
     result=fs->readdirentry((gpointer)fs,handle);
#ifdef DEBUG
   printf("vfs_readdirentry: result=%s\n",result);
#endif
   return result;
};
void vfs_closedir (vfs_filesystem *fs,gtoaster_handle_t handle)
{
   if ((fs->closedir)&&(handle))
     fs->closedir((gpointer)fs,handle);
};
/* this will retrieve the file statistics of a specific file.
 * caller frees structure */
vfs_direntry *vfs_filestat(vfs_filesystem *fs,const char *path,int getfilelinks)
{
   if (fs->filestat)
     return fs->filestat((gpointer)fs,path,getfilelinks);
   else
     return NULL;
};

char *vfs_getlinkdestination(vfs_filesystem *fs,char *path)
{
   if ((fs->getlinkdestination)&&(path))
     return fs->getlinkdestination((gpointer)fs,path);
   else
     return NULL;
};

int vfs_symlink(vfs_filesystem *fs,char *linkdest,char *linkname)
{
   if ((fs->symlink)&&linkname&&linkdest)
     return fs->symlink((gpointer)fs,linkdest,linkname);
   else
     return 1;
};

typedef struct
{
   vfs_filesystem *removefs;
   char *removefile;
   vfs_operationcomplete callback;
   gpointer data;
}
vfs_copy_move_info;

void vfs_copy_move_callback(int status,gpointer data)
{
   vfs_copy_move_info *info=(vfs_copy_move_info*)data;
   /* success ? */
   if (!status)
     status=vfs_deletefile(info->removefs,
			   info->removefile);
   free(info->removefile);
   info->callback(status,info->data);
   free(info);
};

void vfs_install_autodelete(vfs_filesystem *fs,
			    const char *path,
			    vfs_operationcomplete *callback,
			    gpointer *data)
{
   vfs_copy_move_info *info=(vfs_copy_move_info*)malloc(sizeof(vfs_copy_move_info));

   info->removefs=fs;
   info->removefile=strdup(path);

   info->callback=*callback;
   info->data=*data;

   *data=(gpointer)info;
   *callback=vfs_copy_move_callback;

};

void vfs_copy(vfs_filesystem*destinationfs,
	      char *destinationpath,
	      vfs_filesystem*sourcefs,
	      char *sourcepath,
	      datacopydlg_dlginfo *progress,
	      vfs_operationcomplete callback,
	      gpointer data,
	      int link_requested,
	      int copy_move)
{
#ifdef DEBUG
   printf("vfs_copy: copying from %p,%s to %p,%s\n",
	  sourcefs,sourcepath,
	  destinationfs,destinationpath);
#endif
   if (destinationfs->copy)
     {
	destinationfs->copy((gpointer)destinationfs,
			    destinationpath,
			    (gpointer)sourcefs,
			    sourcepath,
			    progress,
			    callback,
			    data,
			    link_requested,copy_move);
     }
   else
     // oops, not implemented. Tell our callback about it
     callback(1,data);
};

char *vfs_createfile_getrealname(vfs_filesystem *fs,
				 char *path)
{
   if (fs->createfile_getrealname)
     return fs->createfile_getrealname((gpointer)fs,path);
   else
     return NULL;
};

char *vfs_getrealname(vfs_filesystem*fs,
		      char *path)
{
   if (fs->getrealname)
     return fs->getrealname((gpointer)fs,path);
   else
     return NULL;
};

/* we have to get our data from somewhere, unless we're using
 * the "normal" filesystem of the operating system as mkisofs does in
 * link mode.
 * Note that the return value is a normal file descriptor which can be
 * used to read up to filesize bytes from the file.
 * Never try to do anything unusual with it. It's for reading.
 * No seeking etc. is guaranteed to work */
int vfs_openfile(vfs_filesystem *fs,char *path)
{
   if (fs->openfile)
     return fs->openfile((gpointer)fs,path);
   else
     return -1;
};

int vfs_deletefile(vfs_filesystem *fs,char *path)
{
#ifdef DEBUG
   printf("vfs_deletefile: deleting file '%s' in %p\n",path,fs);
#endif
   if (fs->del)
     return fs->del((gpointer)fs,path);
   else
     return 1;
};
int vfs_renamefile(vfs_filesystem *fs,char *path,char *newname)
{
   if (fs->ren)
     return fs->ren((gpointer)fs,path,newname);
   return 1;
};

/* End of story. free our stuff */
void vfs_destroy(vfs_filesystem *fs)
{
   fs->destroy((gpointer)fs);
   free(fs);
};

/* is the object a directory ? */
int vfs_isdirectory(vfs_filesystem *fs,char *path)
{
   int result=0;
   vfs_direntry *info=vfs_filestat(fs,path,0);
   if (info)
     {
	result=info->isdir;
	free(info);
     };
   return result;
};

/* is the object a link ? */
int vfs_islink(vfs_filesystem *fs,char *path)
{
   int result=0;
   vfs_direntry *info=vfs_filestat(fs,path,0);
   if (info)
     {
	result=info->islink;
	free(info);
     };
   return result;
};

/* is the object a reference ? */
int vfs_isreference(vfs_filesystem *fs,const char *path)
{
   int result=0;
   vfs_direntry *info=vfs_filestat(fs,path,0);
   if (info)
     {
	result=info->is_reference;
	free(info);
     };
   return result;
};

/* filesize of a file */
unsigned long int vfs_filesize(vfs_filesystem *fs,char *path)
{
   unsigned long int result=0;
   vfs_direntry *info=vfs_filestat(fs,path,0);
   if (info)
     {
	result=info->filesize;
	free(info);
     };
   return result;
};

int vfs_numberofsubdirs(vfs_filesystem*vfs,char*path)
{
   int result=0;
   vfs_direntry *info=vfs_filestat(vfs,path,1);
   if (info)
     {
	result=info->filelinks;
	free(info);
     };
   return result;
};

char *vfs_buildvfsuri(vfs_filesystem*fs,char *path)
{
   char *result=(char*)malloc(strlen(path)+65);

   if (fs==unixtree)
     strcpy(result,"file:");
   else
     {
	strcpy(result,"gtoastervfs:");
	sprintf(&result[strlen(result)],
		"%p",fs);
	strcat(result,",");
     };
   strcat(result,path);
   return result;
};

vfs_filesystem *vfs_parseuri(char *uri,char *path)
{
   vfs_filesystem *result=NULL;
#ifdef DEBUG
   printf("vfs_parseuri: parsing '%s'\n",uri);
#endif
   if (!strncmp(uri,"gtoastervfs:",12))
     {
	sscanf(strchr(uri,':')+1,"%p",&result);
	if (strchr(uri,','))
	  strcpy(path,strchr(uri,',')+1);
	else
	  result=NULL;
     };
   if (!strncmp(uri,"file:",5))
     {
	if (strchr(uri,':'))
	  {
	     strcpy(path,strchr(uri,':')+1);
	     result=unixtree;
	  };
     };
   return result;
};

typedef struct
{
   /* path of the current destination directory */
   char *destinationpath;
   /* directory handle for current source directory */
   gtoaster_handle_t  sourcedir;
   /* path of current source directory.
    * used for deleting it if the move option was set */
   char *sourcepath;
}
vfs_dirstack;

typedef struct
{
   /* input parameters of the initial function call (unmodified throughout
    * the recursion) */
   int link_requested;
   int copy_move;
   vfs_filesystem *destinationfs;
   vfs_filesystem *sourcefs;

   /* a GList of vfs_dirstack.
    * The first entry of the list is always the current one */
   GList *currentstack;

   /* Communicating status */

   /* 0,initially. Counts the number of errors encountered while
    * doing the recursive copying */
   int errors;
   datacopydlg_dlginfo *progress;
   int progress_thread;
   vfs_operationcomplete callback;
   gpointer data;

   /* handle of the current copy process */
   gtoaster_handle_t copy_process_handle;
   /* will be set true by the cancel function.
    * If set to true, function will exit immediately */
   int exit_request;
}
vfs_recurse_copy_info;

void vfs_fixdatacopydisplay(datacopydlg_dlginfo *progress,
			    int progress_thread,
			    const char *newname,
			    unsigned long newsize)
{
			    /* Get style for the info field.
			     * FixMe: maybe define an interface for that stuff in datacopydlg.h */
   int max_x,max_y;
   char *cfn=NULL;
   gdk_window_get_size(progress->threads[progress_thread]->infofield->window,
		       &max_x,&max_y);
   cfn=helpings_cutstring(max_x,newname,
			  gtk_widget_get_style(progress->threads[progress_thread]->infofield)->font
			  );
   datacopydlg_restartthread(progress,
			     progress_thread,
			     newsize);
   datacopydlg_configurethread(progress,
			       progress_thread,
			       0,
			       NULL,
			       cfn,
			       -1);
   free(cfn);
};

void vfs_copy_recurse_callback(int result,gpointer data)
{
   vfs_recurse_copy_info *info=(vfs_recurse_copy_info*)data;

   /* something went wrong while copying a file */
   if (result)
     ++info->errors;

   /* something left to copy ? */
   if (info->currentstack)
     {
	vfs_dirstack *stack=(vfs_dirstack*)info->currentstack->data;
	char *current=vfs_readdirentry(info->sourcefs,
				       stack->sourcedir);
	if ((current)&&(!info->exit_request))
	  {
	     vfs_direntry *sourceinfo=vfs_filestat(info->sourcefs,
						   current,
						   0);
	     if (sourceinfo)
	       {
		  char *destination=helpings_fileinpath(stack->destinationpath,
							sourceinfo->name);
		  if (sourceinfo->isdir)
		    {
		       /* may or may not fail */
		       vfs_createdir(info->destinationfs,
				     destination);

		       stack=(vfs_dirstack*)malloc(sizeof(vfs_dirstack));
		       stack->destinationpath=destination;
		       stack->sourcepath=strdup(current);
		       stack->sourcedir=vfs_opendir(info->sourcefs,
						    current);
			    /* Go ahead into the next recursion level */
		       info->currentstack=g_list_prepend(info->currentstack,
							 (gpointer)stack);
		       vfs_copy_recurse_callback(0,data);
		    }
		  else
		    {
		       /* reinitialize progress */
		       if ((info->progress)&&(info->progress_thread!=-1))
			 vfs_fixdatacopydisplay(info->progress,
						info->progress_thread,
						current,
						sourceinfo->filesize);

		       /* It's just a file */
		       /*info->copy_process_handle=*/vfs_copy(info->destinationfs,
							      destination,
							      info->sourcefs,
							      current,
							      info->progress,
							      vfs_copy_recurse_callback,
							      info,
							      info->link_requested,
							      info->copy_move);
		       free(destination);

		    };
		  free(sourceinfo);
	       }
	     else
	       {
		  /* Couldn't stat source file.
		   * This is a definite problem */
		  ++info->errors;
		  vfs_adderrorstring(info->sourcefs,_("Couldn't get sourcefile informations"));
		  vfs_copy_recurse_callback(0,data);
	       };
	  }
	else
	  {
	     /* done with current directory */
	     vfs_closedir(info->sourcefs,
			  stack->sourcedir);
	     free(stack->destinationpath);
	     if (info->copy_move)
	       vfs_deletefile(info->sourcefs,stack->sourcepath);
	     free(stack->sourcepath);
	     info->currentstack=g_list_remove(info->currentstack,stack);
	     free(stack);

	     /* process rest of upper dir */
	     vfs_copy_recurse_callback(0,data);
	  };
	if (current)
	  free(current);
     }
   else
     {
	/* we're done, inform initial caller */
	info->callback(info->errors,info->data);
	free(info);
     };
};

gtoaster_handle_t vfs_copy_recursively(vfs_filesystem*destinationfs,
			 char *destinationpath,
			 vfs_filesystem*sourcefs,
			 char *sourcepath,
			 datacopydlg_dlginfo *progress,
			 int progress_thread,
			 vfs_operationcomplete callback,
			 gpointer data,
			 int link_requested,
			 /* move instead of copy */
			 int copy_move)
{
   vfs_recurse_copy_info *info=(vfs_recurse_copy_info*)malloc(sizeof(vfs_recurse_copy_info));

   /* Ok, setup the "usual" stuff */
   info->link_requested=link_requested;
   info->copy_move=copy_move;
   info->destinationfs=destinationfs;
   info->sourcefs=sourcefs;

   info->exit_request=0;
   info->copy_process_handle=0;
   info->callback=callback;
   info->data=data;
   info->errors=0;
   info->currentstack=NULL;

   /* do we have to apply our pseudo-recursion stuff at all ? */
   if (vfs_isdirectory(sourcefs,sourcepath))
     {
	/* avoid endless recursions when
	 * copying within the same filesystem */
	if ((destinationfs!=sourcefs)||
	    (!helpings_isinpath(destinationpath,sourcepath)))
	  {

	     /* Yes, we do. So let's rock :-) */
	     /* initial stack entry */
	     vfs_dirstack *stack=(vfs_dirstack*)malloc(sizeof(vfs_dirstack));

	     /* Try to create directory.
	      * We don't check for errors here because the directory might
	      * already exist. Errors will happen soon enough as soon as
	      * we start copying */
	     vfs_createdir(destinationfs,destinationpath);

	     info->progress=progress;
	     info->progress_thread=progress_thread;

	     /* initialize stack */
	     stack->destinationpath=strdup(destinationpath);
	     stack->sourcepath=strdup(sourcepath);
	     stack->sourcedir=vfs_opendir(sourcefs,sourcepath);
	     info->currentstack=g_list_prepend(NULL,(gpointer)stack);

	  }
	else
	  {
	     ++info->errors;
	     vfs_adderrorstring(destinationfs,
				_("Cannot copy/move directory into itself"));
	  };

	/* ... and off we go */
	vfs_copy_recurse_callback(0,info);
     }
   else
     {
		       /* reinitialize progress */
	if ((progress)&&(progress_thread!=-1))
	  vfs_fixdatacopydisplay(progress,
				 progress_thread,
				 sourcepath,
				 vfs_filesize(sourcefs,
					      sourcepath));
     /*  No, obviously not :-) */
     /*info->copy_process_handle=*/vfs_copy(destinationfs,
					    destinationpath,
					    sourcefs,
					    sourcepath,
					    progress,
					    vfs_copy_recurse_callback,
					    (gpointer)info,
					    link_requested,copy_move);
     };
   return (gtoaster_handle_t)info;
};

void vfs_copy_recursively_cancel(gtoaster_handle_t handle)
{
   vfs_recurse_copy_info *info=(vfs_recurse_copy_info*)handle;

   /* Ask recursion handler to exit */
   info->exit_request=1;

   /* FixMe:
    * Make copy process exit as well */
   if (info->copy_process_handle);
};

/* Deleting file recursively */

typedef struct
{
   /* path of the current directory
    * (used to finally delete it)
    * May be NULL if directory shouldn't be deleted after it's been emptied */
   char *path;
   /* Directory entries of current directory */
   GList*entries;
}
vfs_delete_dirstack;

typedef struct
{
   /* input parameters of the initial function call (unmodified throughout
    * the recursion) */
   vfs_filesystem *fs;

   /* A Glist of vfs_delete_dirstack. The first element is the current one */
   GList *currentstack;

   /* Communicating status */

   /* 0,initially. Counts the number of errors encountered while
    * doing the recursive copying */
   int errors;
   vfs_operationcomplete callback;
   gpointer data;
}
vfs_recurse_delete_info;

/* read the directory given in path into a GList and return it */
GList *vfs_readdirectory(vfs_filesystem *fs,char *path)
{
   GList *list=NULL;
   char *ci;

   gtoaster_handle_t dir=vfs_opendir(fs,path);
   while ((ci=vfs_readdirentry(fs,dir)))
     list=g_list_prepend(list,ci);
   vfs_closedir(fs,dir);

   return list;
};

void vfs_delete_recurse_callback(int result,gpointer data);
typedef struct
{
   int result;
   gpointer data;
}
vfs_delete_callback_proxy_info;

gint vfs_delete_callback_proxy(gpointer data)
{
   vfs_delete_callback_proxy_info *info=(vfs_delete_callback_proxy_info*)data;
   vfs_delete_recurse_callback(info->result,info->data);
   free(info);
   return 0;
};

void vfs_delete_callback_proxy_call(int result,gpointer data)
{
   vfs_delete_callback_proxy_info *info=(vfs_delete_callback_proxy_info*)malloc(sizeof(vfs_delete_callback_proxy_info));
   info->result=result;
   info->data=data;
   gtk_timeout_add(0,
		   vfs_delete_callback_proxy,
		   info);
};

void vfs_delete_recurse_callback(int result,gpointer data)
{
   vfs_recurse_delete_info *info=(vfs_recurse_delete_info*)data;

   /* something went wrong while deleting a file */
   if (result)
     ++info->errors;

   /* something left to remove ? */
   if (info->currentstack)
     {
	vfs_delete_dirstack *stack=(vfs_delete_dirstack*)info->currentstack->data;
	char *current=((stack->entries)?(char*)stack->entries->data:NULL);
	if (current)
	  {
	     vfs_direntry *sourceinfo=vfs_filestat(info->fs,
						   current,
						   0);
	     /* remove current fileitem of current directory */
	     stack->entries=g_list_remove(stack->entries,
					  current);
	     if ((sourceinfo)&&(sourceinfo->isdir))
	       {
		  stack=(vfs_delete_dirstack*)malloc(sizeof(vfs_delete_dirstack));
		  stack->entries=vfs_readdirectory(info->fs,current);
		  stack->path=strdup(current);
		       /* Go ahead into the next recursion level */
		  info->currentstack=g_list_prepend(info->currentstack,
						    (gpointer)stack);
	       }
	     else
	       /* normal file or whatever.
		* Anyway, it's worth a try ;-) */
	       vfs_deletefile(info->fs,current);
	     if (sourceinfo)
	       free(sourceinfo);
	     else
	       {
		  ++info->errors;
		  vfs_adderrorstring(info->fs,
				     _("Couldn't get file informations"));
	       };
	     vfs_delete_callback_proxy_call(0,data);
	     free(current);
	  }
	else
	  {
	     if (stack->path)
	       {
	       /* delete directory if no more files left therein
		* and upper recursion level requested us to do so */
		  vfs_deletefile(info->fs,stack->path);

		  free(stack->path);
	       }
	     free(stack);

	     /* done with current directory */
	     info->currentstack=g_list_remove(info->currentstack,stack);
	     /* process rest of upper dir */
	     vfs_delete_callback_proxy_call(0,data);
	  };
     }
   else
     {
	/* we're done, inform initial caller */
	info->callback(info->errors,info->data);
	free(info);
     };
};

gtoaster_handle_t vfs_remove_recursively(vfs_filesystem *fs,
			   char *path,
			   vfs_operationcomplete callback,
			   gpointer data,
			   int content_only)
{
   vfs_recurse_delete_info *info=(vfs_recurse_delete_info*)malloc(sizeof(vfs_recurse_delete_info));

	/* Ok, setup the "usual" stuff */
   info->fs=fs;

   info->errors=0;
   info->callback=callback;
   info->data=data;

   info->currentstack=NULL;

   /* do we have to apply our pseudo-recursion stuff at all ? */
   if (vfs_isdirectory(fs,path))
     {
	/* Yes, we do. So let's rock :-) */
	/* initial stack entry */
	vfs_delete_dirstack *stack=(vfs_delete_dirstack*)malloc(sizeof(vfs_delete_dirstack));

	/* initialize stack */
	stack->path=((content_only)?NULL:strdup(path));
	stack->entries=vfs_readdirectory(fs,path);
	info->currentstack=g_list_prepend(NULL,(gpointer)stack);

	/* ... and off we go */
	vfs_delete_callback_proxy_call(0,info);
     }
   else
     {
     /*  No, obviously not :-) */
	vfs_deletefile(fs,path);
	callback(0,data);

	/* if only a normal file is to be removed, that's it.
	 * Handle is invalid once callback's been called */
	free(info);
	info=NULL;
     };
   return (gtoaster_handle_t)info;
};

int vfs_remove_recursively_blocking(vfs_filesystem *fs,
				    char *path,
				    int content_only)
{
   int result=0;
   gtoaster_handle_t dir=vfs_opendir(fs,path);
   char *entry;
   while ((entry=vfs_readdirentry(fs,dir)))
     {
	if (vfs_isdirectory(fs,entry))
	  result+=vfs_remove_recursively_blocking(fs,entry,0);
	else
	  result+=vfs_deletefile(fs,entry);
	free(entry);
     };
   vfs_closedir(fs,dir);
   if (!content_only)
     result+=vfs_deletefile(fs,path);
   return result;
};

