/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#ifndef SND_H
#define SND_H

#include <audiofile.h>
#include <config.h>
#include "track.h"
#include "mixer.h"
#include "rwlock.h"

#define MAP_ALL 4294967295U

#define RESULT_ERROR_NOMEM NULL
#define RESULT_IS_ERROR(s) (!(s) || ((s) && ((snd *)s)->error_str))

typedef enum {
    CLONE_STRUCTURE,
    CLONE_TRACK_STRUCTURE,
    CLONE_TRACK_DATA
} snd_clone_method;

typedef struct _snd {

    /* Error condition on this snd (see RESULT_IS_ERROR macro). */

    char *error_str;

    /* Name/identifier (may be NULL). */

    char *name;

    /* The format as returned by libaudiofile. */

    int fmt;

    /* Endianness. */

    int endian;

    /* Number of channels, i.e. number of valid tracks in tracks array
       below. */

    int channels;

    /* Rate in frames per second. */

    double rate;

    /* Frame width in bytes. */

    int frame_width;

    /* Duration in seconds. */

    double time;

    /* Read/write lock. Read lock must be acquired before traversing
       the tracks array below. */

    rwlock rwl;

    /* Array for the tracks. The channels field above
       specifies how many tracks there are. */

    struct _track *tracks[MAX_TRACKS];

    /* Array for the markers. */

    struct marker_list *markers[MAX_TRACKS];

} snd;

snd *
snd_new();

snd *
snd_clone(snd *sr,
          snd_clone_method method);

snd *
snd_copy(snd *sr,
         track_map_t channel_map,
         AFframecount frame_offset,
         AFframecount frame_count);

void 
snd_destroy(snd *sr);

int
snd_insert(snd *sr,
           snd *src,
           track_map_t channel_map,
           AFframecount frame_offset);

snd *
snd_erase(snd *sr,
          track_map_t channel_map,
          AFframecount frame_offset,
          AFframecount frame_count);

snd *
snd_delete(snd *sr,
           track_map_t channel_map,
           AFframecount frame_offset,
           AFframecount frame_count);

void
snd_tracks_insert(snd *sr,
                  snd *src,
                  track_map_t channel_map);

snd *
snd_tracks_delete(snd *sr,
                  track_map_t channel_map);

int
snd_demux(snd *sr,
          frame_bits_t fb_target,
          frame_bits_t fb_source,
          track_map_t channel_map,
          int channel_count,
          AFframecount frame_offset,
          AFframecount frame_count);

AFframecount
snd_mux(snd *sr,
        frame_bits_t fb_target,
        frame_bits_t *fb_sources,
        mixer *mixer,
        track_map_t channel_map,
        AFframecount frame_offset,
        AFframecount frame_count);

void
snd_name_set(snd *sr,
             const char *filename);

int
snd_track_count(snd *sr);

AFframecount
snd_frame_count(snd *sr);

double 
snd_frames_to_time(const snd *sr, 
                   AFframecount c);

void
snd_convert(snd *sr,
            int sample_width,
            float sample_rate);

void
snd_info_dump(snd *sr);

void
snd_dump(snd *sr);

void 
snd_frames_buffer_to_graph_buffer(graph_bits_unit_t *graph_bits_low,
                                  graph_bits_unit_t *graph_bits_high,
                                  frame_bits_t frame_bits,
                                  AFframecount frame_count,
                                  int frame_width,
                                  float hres);

void 
snd_scale(graph_bits_unit_t *graph_bits_low,
          graph_bits_unit_t *graph_bits_high,
          void *source_bits_low,
          void *source_bits_high,
          AFframecount frame_count,
          int frame_width,
          float skip_amount,
          AFframecount src_off,
          AFframecount dst_off,
          int source_isgraph_bits_t);

void 
snd_error_free(snd *s);

void
snd_error_set(snd *s, 
              const char *format,
              ...);

const char *
snd_error_get(snd *s);

int
snd_verify(snd *sr);

#endif /* ! SND_H */
