/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <errno.h>
#include <stdio.h>
#include <dlfcn.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <stdarg.h>
#include <glib.h>
#include <audiofile.h>
#include <config.h>
#include "pref.h"
#include "mem.h"
#include "snd.h"
#include "track.h"
#include "gui.h"
#include "undo.h"
#include "action.h"
#include "play.h"
#include "file.h"
#include "shell.h"
#include "module.h"
#include "modutils.h"

int buffers_being_saved = 0;
int quit_requested = 0;
shell *clipboard_shell = NULL;
snd *clipboard = NULL;

extern struct marker_list *cuepoints_clipboard[];
extern struct marker_list *envelopes_clipboard[];

char *path = NULL;
extern int mem_fail_allocation_on_zero;
extern int module_count;
extern GList *shells;
extern module modules[];
static action_result last_action_result;
static action_result false_action_result = { -1, 0, NULL, NULL };

action_group *
action_marker_list_insert(action *a) {
    int t, i = 0;
    action *a_undo;
    shell *shl = a->shl;
    for(t = 0; t < shl->sr->channels; t++) {
        if((1 << t) & a->channel_map) {
            if(a->m_time)
                marker_list_insert_time(a->m_target[t],
                                        a->offset,
                                        a->count,
                                        a->m_type);
            if(a->m_source && a->m_source[i++]) 
                marker_list_insert_list(a->m_target[t],
                                        a->m_source[i-1],
                                        a->offset,
                                        a->count,
                                        a->m_type);
        }
    }
    
    a->shl->has_changed = 1;
    if(!a->undo)
        return NULL;

    a_undo = ACTION_MARKER_LIST_DELETE_NEW(DONT_UNDO,
                                           a->shl,
                                           a->m_target, 
                                           a->channel_map,
                                           a->offset,
                                           a->count,
                                           a->m_type);
    a_undo->m_time = a->m_time;
    return action_group_new(2, a_undo,
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->shl->sr,
                                              a->channel_map,
                                              a->offset,
                                              a->count));
}

action_group *
action_marker_list_delete(action *a) {
    shell *shl = a->shl;
    int t, i = 0;
    action *a_undo;
    struct marker_list **m_deleted = 
        marker_list_array_new();
    DEBUG("a->channel_map: %d\n", a->channel_map);
    for(t = 0; t < shl->sr->channels; t++) {
        if((1 << t) & a->channel_map) {
            DEBUG("deleting on %d, %ld-%ld, type: %d\n", t, a->offset, a->offset+a->count, a->m_type);
            if(a->m_time) 
                m_deleted[i++] = marker_list_delete_time(a->m_target[t],
                                                         a->offset,
                                                         a->count,
                                                         a->m_type);
            else
                m_deleted[i++] = marker_list_delete(a->m_target[t],
                                                    a->offset,
                                                    a->count,
                                                    a->m_type);
            DEBUG("dumping deleted markers:\n");
            marker_list_dump(m_deleted[i-1]);
        }
    }

    a->shl->has_changed = 1;
    if(!a->undo) {
        marker_list_array_destroy(m_deleted);
        return NULL;
    }
    a_undo = ACTION_MARKER_LIST_INSERT_NEW(DONT_UNDO,
                                           a->shl,
                                           a->m_target,
                                           m_deleted,
                                           a->channel_map,
                                           a->offset,
                                           a->count,
                                           a->m_type);
    a_undo->m_time = a->m_time;
    
    return action_group_new(2, a_undo,
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->shl->sr,
                                              a->channel_map,
                                              a->offset,
                                              a->count));
    
}

action_group *
action_player_play(action *a) {
    shell *shl = a->shl;

    /* This is racey but that's ok because player_start()
       and player_stop() are supposed to be robust. */

    if(!shl->player.player_running) {
        /* Wrap. */
        if(!shl->player.record_mode &&
           shl->select_start == shl->select_end &&
           shl->select_start == snd_frame_count(shl->sr)) {
            shl->select_start = shl->select_end = 0;
            shl->hadjust->value = 0;
        }
        player_start(shl); 
    }
    return NULL;
}

action_group *
action_player_stop(action *a) {
    shell *shl = a->shl;

    /* This is racey but that's ok because player_start()
       and player_stop() are supposed to be robust. */

    if(shl->player.player_running) {
        player_stop(shl);
        if(shl->select_start == shl->select_end)
            shl->select_start = shl->select_end = shl->player.player_pos;
    }
    return NULL;
}

action_group *
action_file_new(action *a) {
    shell *shl = shell_new();
    if(!shl) 
        gui_alert("Could not create interface.\nProbably out of memory.");    
    else 
        gtk_widget_show(GTK_WIDGET(shl->appwindow));
    ACTION_RESULT_RETURN_PTR(a->id, shl);
    return NULL;
}

action_group *
action_file_init(action *a) {
    if(test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ?
                  "load" : "save");
        return NULL;
    }
    file_init(a->shl);
    /* FIXME: actually this does need an undo. */
    return NULL;
}


action_group *
action_file_open(action *a) {
    if(test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ?
                  "load" : "save");
        return NULL;
    }
    if(path)
        mem_free(path);
    path = strdup(a->str);
    if(a->shl->file_load_cancel_requested)
        gui_alert("Not now please.");
    else
        file_load(a->shl, a->str);
    return NULL;
}

void 
gui_open_fileselection_ok_clicked(GtkWidget *w, 
                                  gpointer fs) {
    char *filename = strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
    action_result *ar;
    shell *shl;
    gtk_widget_destroy(GTK_WIDGET(fs));
    ar = action_do(ACTION_FILE_NEW_NEW());
    shl = action_result_as_ptr(ar);
    action_do(ACTION_FILE_OPEN_NEW(shl, filename));
    mem_free(filename);    
}

action_group *
action_file_select(action *a) {
    GtkWidget *fs;

    fs = gui_file_selection_new("Select an audio file");
    
    if(path) {
        DEBUG("setting path: %s\n", path);
        gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), path);
    }

    g_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                     "clicked", 
                     G_CALLBACK(gui_open_fileselection_ok_clicked), 
                     fs);
    
    g_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
                            "clicked", 
                            G_CALLBACK(gtk_widget_destroy),
                            (gpointer) fs,
                            0);
    
    gtk_widget_show(fs);
    return NULL;
}

int
file_write_verify(const char *filename) {
    int button;
    struct stat buf;
    if(stat(filename, &buf) == -1) {
        switch(errno) {
        case ENOENT:
            /* A component of the path file_name does not exist, or the path is an empty string. */
            break;
        default:
            gui_alert("The filename '%s' is invalid: %s", filename, strerror(errno));
            return -1;
        }

    } else {

        /* File exists. */

        button = gui_yes_no("Replace file?", "The file %s already exists. Replace?", 
                            filename);
        
        if(button == GUI_YES) {
            if(unlink(filename) == -1) {
                gui_alert("Cannot remove file %s: %s", filename, strerror(errno));
                return -1;
            }
        } else {
            /* User does not want to replace. */
            return -1;
        }
    }
    return 0;
}

/* FIXME: saveas_file_selected() and mixdown_file_selected()
   are virtually identical. */

enum output_method {
    OUTPUT_SAVE = 0,
    OUTPUT_MIXDOWN = 1
};

struct output_request_info {
    enum output_method method;
    shell *shl;
};

void 
output_file_selected(GtkWidget *w, 
                     gpointer fs) {
    struct output_request_info *oi = g_object_get_data(G_OBJECT(fs), "user_data");
    shell *shl = oi->shl;
    enum output_method method = oi->method;
    char *filename = strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
    gtk_widget_destroy(GTK_WIDGET(fs));
    
    if(!filename) {
        FAIL("ridiculously low on memory, cannot copy filename string\n");
        return;
    }

    if(file_write_verify(filename)) {
        free(filename);
        return;
    }

    if(path)
        free(path);
    path = strdup(filename);

    if(method == OUTPUT_SAVE) 
        file_save(shl, filename, FALSE);
    else 
        file_mixdown(shl, filename, FALSE);
    
    free(filename);
}

void
output_fileselector_destroy(gpointer fs) {
    struct output_request_info *oi = g_object_get_data(fs, "user_data");
    if(oi) {
        DEBUG("destroying output_request_info\n");
        free(oi);
        g_object_set_data(fs, "user_data", NULL);
    }
    gtk_widget_destroy(GTK_WIDGET(fs));
}

void
select_output_filename(shell *shl,
                       const char *action,
                       int output_method) {
    struct output_request_info *oi = mem_alloc(sizeof(struct output_request_info));
    GtkWidget *fs;
                                                                                                                       
    if(!oi) {
        FAIL("could not allocate save_request_info, ridiculously low memory\n");
        return;
    }
    DEBUG("allocated output request info\n");
    oi->shl = shl;
    oi->method = output_method;
    fs = gtk_file_selection_new("Save under what name?");
                                                                                                                       
    if(path) {
        DEBUG("setting path: %s\n", path);
        gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), path);
    }
                                                                                                                       
    g_object_set_data(G_OBJECT(fs), "user_data", oi);
    g_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                     "clicked",
                     G_CALLBACK(output_file_selected),
                     fs);
    g_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
                            "clicked",
                            G_CALLBACK(output_fileselector_destroy),
                            (gpointer)fs,
                            0);
    g_signal_connect_object(GTK_OBJECT(fs),
                            "destroy",
                            G_CALLBACK(output_fileselector_destroy),
                            (gpointer)fs,
                            0);

    gtk_widget_show(fs);
    return;
}

action_group *
action_file_save_as(action *a) {
    if(test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ?
                  "load" : "save");
        return NULL;
    }
    select_output_filename(a->shl, "Save", OUTPUT_SAVE);
    return NULL;
}

action_group *
action_file_save(action *a) {
    if(test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ?
                  "load" : "save");
        return NULL;
    }

    if(!a->shl->has_name) 
        action_file_save_as(a);
    else
        file_save(a->shl, a->shl->sr->name, TRUE);

    return NULL;
}

action_group *
action_file_mixdown(action *a) {
    if(test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS) ?
                  "load" : "save");
        return NULL;
    }
    select_output_filename(a->shl, "Mixdown", OUTPUT_MIXDOWN);
    return NULL;
}

action_group *
action_file_close(action *a) {
    int button, allow = 0;

    DEBUG("buffers_being_saved: %d\n", buffers_being_saved);
    if(test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS))  {
        gui_alert("Please wait for file save to complete.");
        ACTION_RESULT_RETURN_BOOL(a->id, !allow);
        return NULL;
    }

    if(!a->shl->has_changed) {
        allow = 1;
    } else {
        if(a->shl == clipboard_shell) {
            allow = 1;
        } else if(a->shl->sr) {
            button = gui_yes_no("Close file?", "Close file %s without saving changes?", 
                                a->shl->sr->name);
            if(button == GUI_NO || button == GUI_CANCEL) 
                allow = 0;
            else
                allow = 1;
        } else {
            allow = 1;
        }
    }

    if(allow) {
        a->shl->close_requested = 1;
        a->shl->module_cancel_requested = 1;
        a->shl->file_load_cancel_requested = 1;
    }

    ACTION_RESULT_RETURN_BOOL(a->id, !allow);
    return NULL;
}

action_group *
action_selection_to_loop(action *a) {
    if(a->shl->select_start == a->shl->select_end)
        return NULL;

    a->shl->loop = 1;
    a->shl->loop_start = a->shl->select_start;
    a->shl->loop_end = a->shl->select_end;
    return NULL;
}

action_group *
action_loop_to_selection(action *a) {
    if(a->shl->loop_start == a->shl->loop_end)
        return NULL;

    a->shl->select_start = a->shl->loop_start;
    a->shl->select_end = a->shl->loop_end;
    return NULL;
}

action_group *
action_selection_fit(action *a) {
    AFframecount size = a->shl->select_end - a->shl->select_start;
    float hres;
    int canvas_size = GTK_WIDGET(a->shl->canvas)->allocation.width;
    for(hres = HRES_MIN; 
        hres <= HRES_MAX && (size / hres) > canvas_size; 
        hres += (hres < GRAPH_BITS_HRES ? hres : GRAPH_BITS_HRES));
    a->shl->hres = hres;
    shell_viewport_center(a->shl, a->shl->select_start, a->shl->select_end);
    
    DEBUG("hres: %f, size: %ld, allocation.width: %d\n", hres, size, 
          (GTK_WIDGET(a->shl->canvas)->allocation.width));
    return NULL;
}

action_group *
action_tracks_insert(action *a) {

    if(test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS) ? 
                  "save" : "load");
        return NULL;
    }

    snd_tracks_insert(a->sr_target, 
                      a->sr_source,
                      a->channel_map);
    if(RESULT_IS_ERROR(a->sr_target)) 
        return NULL;

    mixer_configure(a->shl->mixer, 
                    pref_get_as_int("playback_channels"), //OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, a->sr_target->channels), 
                    a->sr_target->channels);

    shell_status_default_set(a->shl);

    gtk_widget_queue_draw(GTK_WIDGET(a->shl->mixercanvas));

    if(!a->undo) 
        return NULL;
    
    a->shl->has_changed = 1;
    return action_group_new(1,
                            ACTION_TRACKS_DELETE_NEW(DONT_UNDO,
                                                     a->shl,
                                                     a->sr_target, 
                                                     a->channel_map));
    
}

action_group *
action_tracks_delete(action *a) {
    snd *del_sr;
    
    if(test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS) ||
       test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS)) {
        gui_alert("Please wait for file %s to complete.",
                  test_flag(a->shl->action_state, ACTION_SAVE_IN_PROGRESS) ? 
                  "save" : "load");
        return NULL;
    }

    a->shl->select_channel_map = 0;
    del_sr = snd_tracks_delete(a->sr_target,
                               a->channel_map);
    
    if(RESULT_IS_ERROR(a->sr_target)) 
        return NULL;
    
    mixer_configure(a->shl->mixer, 
                    pref_get_as_int("playback_channels"), //OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, a->sr_target->channels), 
                    a->sr_target->channels);
    
    shell_status_default_set(a->shl);

    gtk_widget_queue_draw(GTK_WIDGET(a->shl->mixercanvas));

    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }
    
    a->shl->has_changed = 1;
    return action_group_new(1,
                            ACTION_TRACKS_INSERT_NEW(DONT_UNDO,
                                                     a->shl,
                                                     a->sr_target, 
                                                     del_sr, 
                                                     a->channel_map));

}

action_group *
action_copy(action *a) {
    action_group *ag;
    
    if(!a->count) {
        FAIL("cannot copy empty selection\n");
        return NULL;
    }

    ag = action_group_new(3,
                          ACTION_CUT_NEW(DONT_UNDO,
                                         a->shl,
                                         a->sr_target,
                                         a->channel_map,
                                         a->offset,
                                         a->count),
                          ACTION_PASTE_NEW(DONT_UNDO,
                                           a->shl,
                                           a->sr_target,
                                           a->channel_map,
                                           a->offset,
                                           0),
                          ACTION_SELECT_NEW(DONT_UNDO,
                                            a->shl,
                                            a->sr_target,
                                            a->channel_map,
                                            a->offset,
                                            a->count));
    
    rwlock_wlock(&a->sr_target->rwl);
    action_group_do(ag);
    rwlock_wunlock(&a->sr_target->rwl);
    return NULL;
}

action_group *
action_select(action *a) {
    AFframecount old_ss = a->shl->select_start,
        old_se = a->shl->select_end,
        old_sc = a->shl->select_channel_map;

    a->shl->select_channel_map = a->channel_map;
    
    if(a->shl->select_channel_map == 0) 
        a->shl->select_channel_map = 1;
    
    a->shl->select_start = a->offset;
    a->shl->select_end = a->offset + a->count;

    if(!a->undo) 
        return NULL;

    /* Don't save undo steps for simply moving the cursor (cursor
       appears when start == end). */

    if(old_ss == old_se)
        return NULL;

    return action_group_new(1,
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->shl->sr,
                                              old_sc,
                                              old_ss,
                                              old_se - old_ss));
        
}

action_group *
action_cut(action *a) {
    snd *del_sr;

    if(a->count < 1) {
        FAIL("cannot cut empty selection.\n");
        return NULL;
    }

    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset, 
                        a->count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_insert(a->sr_target,
                   del_sr,
                   a->channel_map,
                   a->offset);
        snd_destroy(del_sr);
        return NULL;
    }

    snd_destroy(clipboard);

    clipboard = snd_clone(del_sr, 
                          CLONE_STRUCTURE |
                          CLONE_TRACK_STRUCTURE | 
                          CLONE_TRACK_DATA);    
    if(RESULT_IS_ERROR(del_sr)) {
        snd_error_set(a->sr_target,
                      "sorry, clipboard lost (%s)", 
                      snd_error_get(del_sr));
        snd_error_free(del_sr);
        snd_destroy(clipboard);
        clipboard = (snd *)snd_new();
        snd_name_set(clipboard, "clipboard");
        if(clipboard_shell) {
            clipboard_shell->sr = clipboard;
            mixer_configure(clipboard_shell->mixer, 
                            pref_get_as_int("playback_channels"), //OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, clipboard_shell->sr->channels), 
                            clipboard_shell->sr->channels);
            shell_status_default_set(clipboard_shell);
            shell_redraw(clipboard_shell);
        }
        return NULL;
    }

    snd_name_set(clipboard, "clipboard");
    if(clipboard_shell) {
        clipboard_shell->sr = clipboard;
        mixer_configure(clipboard_shell->mixer, 
                        pref_get_as_int("playback_channels"), //OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, clipboard_shell->sr->channels), 
                        clipboard_shell->sr->channels);
        shell_status_default_set(clipboard_shell);
        shell_redraw(clipboard_shell);
    }

    a->shl->select_start = a->offset;
    a->shl->select_end = a->offset;
    a->shl->select_channel_map = a->channel_map;

    shell_status_default_set(a->shl);

    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    a->shl->has_changed = 1;
    return action_group_new(2,
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map,
                                              a->offset),
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target,
                                              a->channel_map,
                                              a->offset,
                                              a->count));
    
}

action_group *
action_paste_mix(action *a) {
    int t, t2 = 0;
    snd *copy;
    action_group *undo = NULL;

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_destroy(copy);
        return NULL;
    }

    if(a->undo) 
        undo = action_group_undo_create(a->shl,
                                        a->channel_map, 
                                        a->offset,
                                        a->count,
                                        a->offset,
                                        a->count);

    rwlock_rlock(&a->shl->sr->rwl);
    rwlock_rlock(&copy->rwl);

    /* Find first non-empty track on clipboard. */

    for(t2 = 0; t2 < copy->channels; t2++)
        if(track_frame_count(copy->tracks[t2]))
            break;

    for(t = 0; t < a->shl->sr->channels && t2 < copy->channels; t++)
        if((1 << t) & a->channel_map)
            mix(a->shl,
                t,
                a->offset,
                copy,
                t2++,
                0,
                MIN(snd_frame_count(copy), a->count));
    rwlock_runlock(&a->shl->sr->rwl);
    rwlock_runlock(&copy->rwl);
    
    snd_destroy(copy);

    shell_status_default_set(a->shl);
    a->shl->has_changed = 1;

    return undo;

}

action_group *
action_paste_fit(action *a) {
    int t;
    snd *copy, *del_sr;
    action_group *undo = NULL;
    int ins_map;
    AFframecount frame_count_before_operation, frame_count_after_operation;

    if(!a->count) 
        return NULL;

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_destroy(copy);
        return NULL;
    }

    /* FIXME: error checking. Again this is ugly because we need to
       store the sample inside a shell before we can resample (same as
       paste_mix above). */

    if(a->undo) 
        undo = action_group_undo_create(a->shl,
                                        a->channel_map, 
                                        a->offset,
                                        a->count,
                                        a->offset,
                                        a->count);

    frame_count_before_operation = snd_frame_count(a->shl->sr);
    del_sr = snd_delete(a->shl->sr,
                        a->channel_map,
                        a->offset,
                        a->count);
    ins_map = snd_insert(a->shl->sr,
                         copy,
                         a->channel_map,
                         a->offset);
    if(RESULT_IS_ERROR(a->shl->sr)) {
        if(del_sr) 
            snd_insert(a->shl->sr,
                       del_sr,
                       a->channel_map,
                       a->offset);
        snd_destroy(del_sr);
        return NULL;
    }

    shell_status_push(a->shl, "Paste Fit ...");
    rwlock_rlock(&a->shl->sr->rwl);
    for(t = 0; t < a->shl->sr->channels; t++) {
        if((1 << t) & ins_map) {            
            resample(a->shl,
                     t,
                     a->offset,
                     a->offset + snd_frame_count(copy),
                     a->count,
                     RESAMPLE_BEST_ALGORITHM, 
                     !HONOR_ENVELOPES,
                     GUARANTEE_NEW_FRAME_COUNT);
        }
    }
    rwlock_runlock(&a->shl->sr->rwl);
    shell_status_default_set(a->shl);
    
    snd_destroy(copy);
    snd_destroy(del_sr);

    shell_status_default_set(a->shl);
    a->shl->has_changed = 1;
    frame_count_after_operation = snd_frame_count(a->shl->sr);
    
    if(frame_count_after_operation != frame_count_before_operation) {
        FAIL("resampling introduced inaccuracy: "
             "frame_count_before_operation: %ld, frame_count_after_operation: %ld\n",
             frame_count_before_operation, frame_count_after_operation);
    }
            
    return undo;

}

action_group *
action_paste_over(action *a) {
    int i_map;
    snd *copy, *del_sr;
    AFframecount del_frame_count, pasted_frame_count;

    if(a->shl == clipboard_shell)
        return NULL;

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");
    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_error_free(copy);
    }

    del_frame_count = MIN(snd_frame_count(copy), a->count);
    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset,
                        del_frame_count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(copy);
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }

    DEBUG("snd_frame_count(copy): %ld, del_frame_count: %ld\n",
          snd_frame_count(copy), del_frame_count);
    
    snd_destroy(snd_delete(copy,
                           MAP_ALL,
                           MIN(snd_frame_count(copy), del_frame_count),
                           snd_frame_count(copy) -
                           MIN(snd_frame_count(copy), del_frame_count)));
    i_map = snd_insert(a->sr_target,
                       copy,
                       a->channel_map,
                       a->offset);
    pasted_frame_count = snd_frame_count(copy);
    
    a->shl->select_start = a->offset;
    a->shl->select_end = a->offset + pasted_frame_count;
    a->shl->select_channel_map = a->channel_map;
    
    snd_destroy(copy);

    shell_status_default_set(a->shl);

    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    a->shl->has_changed = 1;
    return action_group_new(3,
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              i_map, 
                                              a->offset, 
                                              del_frame_count),
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map,
                                              a->offset),
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              a->channel_map, 
                                              a->offset,
                                              a->count));
    
}

action_group *
action_paste(action *a) {
    int i_map;
    snd *copy, *del_sr;

    if(a->shl == clipboard_shell) {
        FAIL("cannot paste in clipboard\n");
        abort();
    }

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_error_free(copy);
    }

    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset,
                        a->count);    
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(copy);
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }

    i_map = snd_insert(a->sr_target,
                       copy,
                       a->channel_map,
                       a->offset);
    
    a->shl->select_start = a->offset + snd_frame_count(copy);
    a->shl->select_end = a->shl->select_start;
    a->shl->select_channel_map = a->channel_map;
    
    snd_destroy(copy);
    
    shell_status_default_set(a->shl);
    
    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }
    DEBUG("*** snd_frame_count(clipboard): %ld\n", snd_frame_count(clipboard));
    
    a->shl->has_changed = 1;
    return action_group_new(3,
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              i_map, 
                                              a->offset, 
                                              snd_frame_count(clipboard)),
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map,
                                              a->offset),
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              a->channel_map, 
                                              a->offset,
                                              a->count));
    
}

action_group *
action_insert(action *a) {
    snd_insert(a->sr_target,
               a->sr_source,
               a->channel_map,
               a->offset);
    
    if(!a->undo) 
        return NULL;

    return action_group_new(1, 
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target,
                                              a->channel_map,
                                              a->offset,
                                              snd_frame_count(a->sr_source)));
}

action_group *
action_erase(action *a) {
    snd *del_sr;

    del_sr = snd_erase(a->sr_target,
                       a->channel_map,
                       a->offset,
                       a->count);
    
    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    return action_group_new(2, 
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target,
                                              a->channel_map,
                                              a->offset,
                                              a->count),
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target,
                                              del_sr,
                                              a->channel_map,
                                              a->offset));
}

action_group *
action_delete(action *a) {
    snd *del_sr;

    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset,
                        a->count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }
    
    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    a->shl->has_changed = 1;
    return action_group_new(1,
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map, 
                                              a->offset));

}

/*
 * FIXME: action_delete_track is a hack that is needed so that we can
 * construct undo's for actions that may change the size of each track
 * individually. For example, resampling can be controlled by using
 * markers. Because of this, when resampling 2 tracks of equal length,
 * you may end up with 2 tracks of different size. To undo this
 * action, we must delete a different number of frames from the two
 * tracks. Real (well, complex) solution is a selection model that can
 * describe this situation, i.e. something more powerful than just
 * "select start", "select end" and "select map".
 */
action_group *
action_delete_on_track(action *a) {
    snd *del_sr;

    del_sr = snd_delete(a->sr_target,
                        1 << a->channel_map,
                        a->offset,
                        a->count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }
    
    snd_destroy(del_sr);
    
    return NULL;
}

action_group * 
action_module_execute(action *a) {
    module_id id = a->channel_map;
    action_group *(*module_exec)(shell *, int);
    char *error;
    char canceltext[50];
    action_group *undo;

    if(a->shl->active_module_id > -1) {
        gui_alert("%s needs to complete first.",
                  modules[a->shl->active_module_id].name);
        return NULL;
    }

    DEBUG("executing module %s\n", modules[id].name);
    module_exec = dlsym(modules[id].handle, "module_execute");
    if((error = dlerror()) != NULL) {
        gui_alert("Could not execute %s: %s\n", modules[id].name, error);
        return NULL;
    }
    shell_status_push(a->shl, "%s ...", modules[id].name);
    snprintf(canceltext, 50, "Cancel '%s'", modules[id].name);
    gtk_label_set_text(GTK_LABEL(GTK_BIN(a->shl->cancelmodule)->child), 
                       canceltext);
    gtk_widget_set_sensitive(GTK_WIDGET(a->shl->cancelmodule), TRUE);

    set_flag(a->shl->action_state, ACTION_MODULE_EXECUTE_IN_PROGRESS);
    a->shl->active_module_id = id;
    a->shl->module_cancel_requested = 0;
    a->shl->has_changed = 1;
    
    undo = (*module_exec)(a->shl, a->undo);

    clear_flag(a->shl->action_state, ACTION_MODULE_EXECUTE_IN_PROGRESS);
    a->shl->active_module_id = -1;
    DEBUG("module finished, got undo: %p\n", undo);

    /* We were quit or closed, so don't update UI. FIXME: tail close
       should make these kinds of things unnecessary... */

    if(quit_requested || a->shl->close_requested) 
        return undo;

    shell_status_pop(a->shl);
    shell_status_default_set(a->shl);
    gtk_label_set_text(GTK_LABEL(GTK_BIN(a->shl->cancelmodule)->child), 
                       "Cancel");
    gtk_widget_set_sensitive(GTK_WIDGET(a->shl->cancelmodule), FALSE);

    return undo;
}

action_group *
action_module_open(action *a) {
    int id = a->channel_map;
    void *(*module_open)(module_id, shell *, int);
    char *error;

    if(a->shl->module_state[id].is_open) {
        DEBUG("module %s already open\n", modules[id].name);
        return NULL;
    }

    DEBUG("opening module %s\n", modules[id].name);

    module_open = dlsym(modules[id].handle, "module_open");
    if((error = dlerror()) != NULL) {
        DEBUG("module %s doesn't have module_open(), trying module_execute...\n",
              modules[id].name);
        return action_module_execute(a);
    } else {
        (*module_open)(id, a->shl, a->undo);
    }
    return NULL;
}

action_group *
action_exit(action *a) {
    GList *l;
    shell *shl;
    action_result *ar;

    DEBUG("buffers_being_saved: %d\n", buffers_being_saved);
    if(buffers_being_saved) {
        gui_alert("Please wait for file saving to complete.");
        return NULL;
    }

    DEBUG("shells: %p\n", shells);

    /* Try to close each shell window. */

    for(l = shells; l && l->data; ) {
        shl = l->data;
        l = l->next;
        ar = action_do(ACTION_FILE_CLOSE_NEW(shl));
        if(action_result_as_bool(ar)) {
            ACTION_RESULT_RETURN_BOOL(a->id, 0);
            return NULL;
        }
    }
    DEBUG("calling gtk_main_quit\n");
    gtk_main_quit();
    quit_requested = 1;
    return NULL;
}

static action_desc action_descriptions[] = {
    { "new file", action_file_new, 0 },
    { "init file", action_file_init, HAS_SHELL },
    { "select file", action_file_select, 0 },
    { "open file", action_file_open, HAS_SHELL | HAS_STRING },
    { "save file", action_file_save, HAS_SHELL },
    { "save file as", action_file_save_as, HAS_SHELL },
    { "mixdown", action_file_mixdown, HAS_SHELL },
    { "close file", action_file_close, HAS_SHELL },
    { "insert frames", action_insert, HAS_TARGET | HAS_SOURCE | HAS_MAP | HAS_OFFSET },
    { "erase frames", action_erase, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "delete frames", action_delete, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "delete frames on track", action_delete_on_track,
      HAS_SHELL | HAS_TARGET | HAS_TRACK_INDEX | HAS_OFFSET | HAS_COUNT },
    { "cut frames", action_cut, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste frames", action_paste, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste frames over", action_paste_over, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste mix", action_paste_mix, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste fit", action_paste_fit, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "select frames", action_select, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "copy frames", action_copy, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "insert tracks", action_tracks_insert, HAS_SHELL | HAS_TARGET | HAS_SOURCE | NULL_SOURCE | HAS_MAP },
    { "delete tracks", action_tracks_delete, HAS_SHELL | HAS_TARGET | HAS_MAP },
    { "fit selection", action_selection_fit, HAS_SHELL },
    { "selection to loop", action_selection_to_loop, HAS_SHELL },
    { "loop to selection", action_loop_to_selection, HAS_SHELL },
    { "play", action_player_play, HAS_SHELL },
    { "stop", action_player_stop, HAS_SHELL },
    { "module_open", action_module_open, HAS_SHELL | HAS_MODULE_INDEX },
    { "module_execute", action_module_execute, HAS_SHELL | HAS_MODULE_INDEX },
    { "marker list insert", action_marker_list_insert, HAS_SHELL | HAS_MARKER_TARGET | HAS_SEL_SPEC },
    { "marker list delete", action_marker_list_delete, HAS_SHELL | HAS_MARKER_TARGET | HAS_SEL_SPEC },
    { "exit", action_exit, 0 },
    { "LAST", NULL }
};

/*
 * It is legal to pass NULL as the action. In that case this function
 * returns a FALSE value. This way the result from action_new() can be
 * immediately fed to action_do(). The user is alerted of any errors
 * in action_new(). It is very unlikely for action_new() to fail
 * anyway since it mainly involves allocation a few bytes for the
 * action struct. 
 * Otherwise this function executes and destroys the action, passing 
 * any errors along to the user. 
 */

action_result *
action_do(action *a) {
    action_group *undo;
    action *a_tail;

    /* Check consistency. */

    if(strcmp(action_descriptions[ACTION_LAST].name, "LAST") ||
       action_descriptions[ACTION_LAST].func != NULL) {
        FAIL("action_description list length does not match ACTION_LAST. abort.\n");
        abort();
    }

    if(!a) 
        return &false_action_result;

    if(a->id < 0 || a->id >= ACTION_LAST) {
        FAIL("invalid action id: %d\n", a->id);
        action_destroy(a);
        gui_alert("Internal error: attempt to execute unknown action %d.\n", a->id);
        return &false_action_result;
    }

    /* We can stack file load commands so that we can cancel a file
       load in progress. */

    if(a->id == ACTION_FILE_OPEN &&
       test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS)) {
        a->shl->file_load_cancel_requested = 1;
        DEBUG("canceling file load in progress...\n");
        a->shl->tail_action = a;
        return NULL;
    }

    if(a->shl)
        a->shl->use++;

    ACTION_RESULT_INIT(a->id);
    DEBUG("performing action %s\n", action_descriptions[a->id].name);
    undo = (*action_descriptions[a->id].func)(a);

    if(!a->suppress_redraw &&
       a->shl &&
       a->id != ACTION_FILE_CLOSE &&
       a->id != ACTION_EXIT &&
       !quit_requested && 
       !a->shl->close_requested)
        shell_redraw(a->shl);

    //    DEBUG("undo: %p, a->in_group: %d\n",
    //          undo, a->in_group);
    if(undo) {
        if(!a->in_group)
            a->shl->undo_stack = 
                undo_push(a->shl->undo_stack, undo);
        else
            ACTION_RESULT_UNDO_SET(undo);
    }

    if(a->shl) 
        a->shl->use--;

    /* If a close was requested, we wait for all outstanding
       recursions on the shell to wind down, and have the last one
       turn out the lights by invoking gtk_object_destroy() to emit a
       destroy signal on the shell window. */

    if(a->shl && a->shl->close_requested && a->shl->use == 0) 
        gtk_object_destroy(GTK_OBJECT(a->shl->appwindow));
    
    /* Loading was cancelled and we have a queued action, so execute
       it. */

    if(a->shl && 
       a->shl->tail_action && 
       !test_flag(a->shl->action_state, ACTION_LOAD_IN_PROGRESS)) {
        a->shl->file_load_cancel_requested = 0;
        a_tail = a->shl->tail_action;
        a->shl->tail_action = NULL;
        action_do(a_tail);
    }

    action_destroy(a);
    return ACTION_RESULT;
}

/*
 * This function makes of copy of the frames frame_offset to
 * frame_count with the given channel_map and then pushes a
 * "delete-insert-select" action sequence onto the undo stack. It's an
 * easy way to preserve sound region before it gets modified. FIXME:
 * error return, failure modes?
 */
action_group *
action_group_undo_create(shell *shl,
                         track_map_t channel_map,
                         AFframecount frame_offset,
                         AFframecount frame_count,
                         AFframecount delete_frame_offset,
                         AFframecount delete_frame_count) {
    action_group *ag;
    snd *del_sr;

    del_sr = snd_copy(shl->sr, 
                      channel_map,
                      frame_offset,
                      frame_count);
    if(RESULT_IS_ERROR(shl->sr)) 
        return NULL;
    
    ag = action_group_new(3,
                          ACTION_DELETE_NEW(DONT_UNDO,
                                            shl,
                                            shl->sr,
                                            channel_map,
                                            delete_frame_offset,
                                            delete_frame_count),
                          ACTION_INSERT_NEW(DONT_UNDO,
                                            shl->sr, 
                                            del_sr, 
                                            channel_map,
                                            frame_offset),
                          ACTION_SELECT_NEW(DONT_UNDO,
                                            shl,
                                            shl->sr,
                                            channel_map,
                                            frame_offset,
                                            snd_frame_count(del_sr)));
    return ag;
}

void
action_group_dump(action_group *ag) {
    int i;
    DEBUG("action_group: %p, count: %d", 
          ag, ag->count);
    INFO(" (");
    for(i = 0; i < ag->count; i++) {
        INFO("%p", ag->a[i]);
        if(i < ag->count - 1)
            INFO(", ");
    }
    INFO(")\n");
    //    DEBUG("\n");
}

/* FIXME: failure here clutters up code everywhere, maybe promise
   that it never fails by eliminating mem allocation. */

action_group *
action_group_append(action_group *ag,
                    action *a) {
    action_group *ag_new;
    //    DEBUG("before append:\n");
    //    action_group_dump(ag);

    ag_new = mem_realloc(ag, sizeof(action_group) + 
                         (sizeof(action *) * (ag->count + 1)));
    if(!ag_new)
        return NULL;

    ag_new->a[ag_new->count] = a;
    ag_new->count++;
    //    DEBUG("after append:\n");
    //    action_group_dump(ag_new);
    return ag_new;
}

action_group *
action_group_prepend(action_group *ag,
                     action_group *ag_ins) {
    int i;
    action_group *ag_new;
    //    DEBUG("before prepend:\n");
    //    action_group_dump(ag);
    //    action_group_dump(ag_ins);

    ag_new = mem_alloc(sizeof(action_group) + 
                       (sizeof(action *) * (ag->count + ag_ins->count)));
    if(!ag_new)
        return NULL;
    ag_new->count = ag->count + ag_ins->count;
    for(i = 0; i < ag_ins->count; i++) 
        ag_new->a[i] = ag_ins->a[i];
    
    for(i = ag_ins->count; i < ag_new->count; i++)
        ag_new->a[i] = ag->a[i - ag_ins->count];

    mem_free(ag);
    mem_free(ag_ins);

    //    DEBUG("after prepend:\n");
    //    action_group_dump(ag_new);
    return ag_new;
}

void
action_group_do(action_group *ag) {
    int i;
    action *a;
    action_result *ar;
    action_group *undo = NULL, *ag_new;
    shell *shl = NULL;

    //    DEBUG("ag: %p\n", ag);
    /* Find shell. */

    for(i = 0; i < ag->count; i++)
        if(ag->a[i]->shl)
            shl = ag->a[i]->shl;

    if(shl)
        shell_cursor_set(shl, GDK_WATCH);
    for(i = 0; i < ag->count; i++) {
        a = ag->a[i];

        if(i < ag->count - 1)
            a->suppress_redraw = 1;
        a->in_group = 1;
        
        //        DEBUG("a[%d]: %p...\n", i, a);
        ar = action_do(a);
        //        DEBUG("a[%d]: ...returns %p (undo: %p)\n", i, ar, ar->undo);
        if(ar->undo) {
            if(!undo) {
                undo = ar->undo;
                continue;
            }
            ag_new = action_group_prepend(undo, ar->undo);
            if(!ag_new) {
                FAIL("could not prepend action_groups for undo, " \
                     "undo may be lost or broken\n");
                continue;
            }
            undo = ag_new;
        }
    }
    if(shl)
        shell_cursor_set(shl, GDK_LEFT_PTR);
    ag->count = 0;
    action_group_destroy(ag);

    //    DEBUG("a->shl->undo_stack: %p, undo: %p\n", a->shl->undo_stack, undo);
    //    action_group_dump(undo);

    if(undo) {
        if(!shl) {
            FAIL("undo but no shell in any of the action in group");
            return;
        }
        shl->undo_stack = 
            undo_push(shl->undo_stack, undo);
    }

}

action_group *
action_group_new_empty(int count) {
    action_group *ag;

    ag = mem_alloc(sizeof(action_group) + (sizeof(action *) * count));
    if(!ag) {
        FAIL("cannot allocate memory for action group (%d items)\n", count);
        return NULL;
    }

    ag->count = count;
    return ag;
}

action_group *
action_group_new(int count, ...) {
    int i;
    va_list ap;
    action_group *ag;

    ag = action_group_new_empty(count);
    if(!ag) 
        return NULL;

    va_start(ap, count);

    for(i = 0; count--; i++) 
        ag->a[i] = va_arg(ap, action *);
    
    va_end(ap);

    return ag;
}

action *
action_new(action_id id,
           shell *shl,
           struct marker_list **m_target,
           struct marker_list **m_source,
           enum marker_type m_type,
           snd *sr_target,
           snd *sr_source,
           track_map_t channel_map,
           AFframecount offset,
           AFframecount count,
           const char *s,
           int undo) {
    action *a;
    action_desc *ad;

    if(id < 0 && id > ACTION_LAST) {
        gui_alert("Internal error: attempt to create unknown action %d.\n", id);
        return NULL;
    }
    ad = &action_descriptions[id];

    if((ad->signature & HAS_SHELL) && !shl) {
        /* This might happen because we have no memory. */
        FAIL("attempt to create action '%s' but shell is undefined\n",
             ad->name);
        return NULL; 
        //        abort();
    }
    if((ad->signature & HAS_TARGET) && !sr_target) {
        FAIL("attempt to create action '%s' but target is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_SOURCE) && !sr_source && !(ad->signature & NULL_SOURCE)) {
        FAIL("attempt to create action '%s' but source is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MAP) && channel_map == -1) {
        FAIL("attempt to create action '%s' but channel_map is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_OFFSET) && offset == -1) {
        FAIL("attempt to create action '%s' but offset is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_COUNT) && count == -1) {
        FAIL("attempt to create action '%s' but count is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MODULE_INDEX) && channel_map == -1) {
        FAIL("attempt to create action '%s' but module index is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_TRACK_INDEX) && channel_map == -1) {
        FAIL("attempt to create action '%s' but track index is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MARKER_TARGET) && m_target == NULL) {
        FAIL("attempt to create action '%s' but marker target is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MARKER_SOURCE) && m_source == NULL) {
        FAIL("attempt to create action '%s' but marker source is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MARKER_TYPE) && m_type == -1) {
        FAIL("attempt to create action '%s' but marker type is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_STRING) && s == NULL) {
        FAIL("attempt to create action '%s' but string is undefined\n",
             ad->name);
        abort();
    }
    a = mem_alloc(sizeof(action));
    if(!a) {
        FAIL("could not allocate new action (%s)\n", action_descriptions[id].name);
        gui_alert("Not enough memory to create action for %s.", action_descriptions[id].name);
        return NULL;
    }
    a->id = id;
    a->shl = shl;
    a->m_type = m_type;
    a->m_target = m_target;
    a->m_source = m_source;
    a->sr_target = sr_target;
    a->sr_source = sr_source;
    a->channel_map = channel_map;
    a->offset = offset;
    a->count = count;
    a->undo = undo;
    a->suppress_redraw = 0;
    a->in_group = 0;
    a->m_time = 0;
    a->str = s ? strdup(s) : NULL;
    return a;
}

void
action_destroy(action *a) {
    if(a->str)
        free(a->str);
    if(a->m_source &&
       a->m_source != cuepoints_clipboard && 
       a->m_source != envelopes_clipboard)
        marker_list_array_destroy(a->m_source);
    if(a->sr_source) 
        snd_destroy(a->sr_source);
    mem_free(a);
}

void
action_group_destroy(action_group *ag) {
    int i;
    for(i = 0; i < ag->count; i++) 
        if(ag->a[i]) 
            action_destroy(ag->a[i]);
    mem_free(ag);
}

gboolean
action_result_as_bool(action_result *ar) {
    if(!ar->defined)
        return FALSE;
    return ar->val ? TRUE : FALSE;
}

void *
action_result_as_ptr(action_result *ar) {
    if(!ar->defined)
        return NULL;
    return ar->val;
}

void
action_result_undo_set(action_result *ar,
                       action_group *undo) {
    ar->undo = undo;
}

void
action_result_bool_set(action_result *ar, 
                       action_id id,
                       gboolean b) {
    ar->id = id;
    ar->defined = 1;
    ar->val = b ? (void *)TRUE : NULL;
}

void
action_result_ptr_set(action_result *ar, 
                      action_id id,
                      void *ptr) {
    ar->id = id;
    ar->defined = 1;
    ar->val = ptr;
}

void
action_result_init(action_result *ar, 
                   action_id id) {
    ar->id = id;
    ar->defined = 0;
    ar->undo = NULL;
}
