/* vi:set ts=8 sts=0 sw=8:
 * $Id: toolbar.c,v 1.35 2000/02/28 17:23:52 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#ifdef USE_TOOLBARS
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif
#include "doc.h"
#include "toolbar.h"
#include "msgbar.h"
#include "file.h"
#include "menu.h"
#include "prefs.h"
#include "htmltags.h"
#include "misc.h"

#include "gnpintl.h"

/*** external declarations ***/
extern toolbar_data_t main_tbdata[];
#ifdef USE_HTMLTAGS
extern toolbar_data_t html_tbdata[];
#endif	/* USE_HTMLTAGS */
#ifdef USE_DOCLIST
extern toolbar_data_t dlw_tbdata[];
#endif	/* USE_DOCLIST */


#if defined(ENABLE_NLS) && defined(GTK_HAVE_FEATURES_1_1_0)
static char G_GNUC_UNUSED *dummyMsg[] = {
        N_("Toolbar with icons and text"),
        N_("Toolbar with icons only"),
        N_("Toolbar with text only"),
        N_("Toolbar ON"),
        N_("Toolbar OFF"),
        N_("Tooltips ON"),
        N_("Tooltips OFF"),
        N_("RAISED toolbar buttons"),
        N_("FLAT toolbar buttons"),
        N_("HTML Toolbar ON"),
        N_("HTML Toolbar OFF") };
#endif

/*** local definitions ***/
#define PIC_TEXT		gettext("Toolbar with icons and text")
#define PIC_ONLY		gettext("Toolbar with icons only")
#define TEXT_ONLY		gettext("Toolbar with text only")
#define TOOLBAR_ON		gettext("Toolbar ON")
#define TOOLBAR_OFF		gettext("Toolbar OFF")
#define TOOLTIPS_ON		gettext("Tooltips ON")
#define TOOLTIPS_OFF		gettext("Tooltips OFF")
#define RAISED_ON		gettext("RAISED toolbar buttons")
#define RAISED_OFF		gettext("FLAT toolbar buttons")
#define HTML_TOOLBAR_ON		gettext("HTML Toolbar ON")
#define HTML_TOOLBAR_OFF	gettext("HTML Toolbar OFF")


/*** local function prototypes ***/
static GtkWidget *toolbar_create_common(toolbar_data_t *tbdata, win_t *w);
static void tb_on_off(win_t *w, GtkWidget *wgt, long showit);
static void tb_ttip_on_off(GtkWidget *wgt);
static void tb_pic_text_style(GtkWidget *tb);
#ifdef GTK_HAVE_FEATURES_1_1_0
static void tb_raised_on_off(GtkWidget *tb);
#else
# define tb_raised_on_off(tb)
#endif
static void tb_hide_unhide_items(GtkWidget *tb, toolbar_data_t *tbdata);


/*** global function definitions ***/
/*
 * PUBLIC: tb_item_enable
 *
 * enables/disables a toolbar item (using GTK terminology, by setting its
 * "sensivity").
 */
void
tb_item_enable(GtkWidget *tbwidget, char *itemname, bool_t enable)
{
	GtkToolbar *tb;
	GtkContainer *container;
	GList *childl, *tipl;
	GtkToolbarChild *gtbc;

	if (!tbwidget)
		return;

	tb = GTK_TOOLBAR(tbwidget);
	g_return_if_fail(GTK_IS_TOOLBAR(tb));
	container = GTK_CONTAINER(&tb->container);

	for (childl = tb->children, tipl = tb->tooltips->tips_data_list;
	     childl && tipl;
	     childl = childl->next) {

		GtkTooltipsData *tooltipsdata = (GtkTooltipsData *)tipl->data;

		gtbc = (GtkToolbarChild *)childl->data;
		if (gtbc->type != GTK_TOOLBAR_CHILD_SPACE) {
			if (strcmp(itemname, tooltipsdata->tip_private) == 0) {
				gtk_widget_set_sensitive(gtbc->widget, enable);
				break;
			}
			tipl = tipl->next;
		}
	}
} /* tb_item_enable */


/*
 * PUBLIC: tb_redraw
 *
 * redraws all toolbars for a specified window
 */
void
tb_redraw(void *data)
{
	win_t *w = (win_t *)data;
	GSList *tblp;
#ifdef USE_HTMLTAGS
	GtkWidget *tmp;
	tblist_data_t *tbdp;
	GSList *mtblp, *htblp, *next;
	gboolean found = FALSE;
	GtkWidget *tb;
	int len;

	/* show/hide the (main) toolbar handles first */
	for (tblp = w->tbhlist; tblp; tblp = tblp->next) {
		if (!tblp->data)
			continue;

		tb_on_off(w, GTK_WIDGET(tblp->data), IS_SHOW_TOOLBAR());
	}

	/*
	 * if there is a change between advanced<->nonadvanced HTML toolbars,
	 * need to first scan the html_tblist, then for each element in that
	 * list, check if their exists a pointer to the same toolbar widget in
	 * the main_tblist.  If there is, remove it from the main_tblist.
	 * After the main_tblist has been purged of all entries that exist in
	 * the html_tblist, we can blow away the HTML toolbar and recreate it.
	 * Note that we keep the HTML toolbar *handle* around since we can
	 * reuse it.  It's only the toolbar widget itself that gets destroyed
	 * and recreated.
	 */
	len = g_slist_length(w->html_tblist);
	if ((len == 1 &&  IS_ADVANCED_HTML_TB()) ||
	    (len > 1  && !IS_ADVANCED_HTML_TB())) {
		GNPDBG_TOOLBAR(("tb_redraw: transition detected\n"));
	    	/*
		 * There's been a transition from Advanced to nonAdvanced, or
		 * vice versa, so hide the HTML toolbar (handle) before blowing
		 * away the toolbar widget itself and recreating it.
		 */
		tb_on_off(w, w->html_tb_h, FALSE);

		for (htblp = w->html_tblist; htblp; htblp = next) {
			tbdp = (tblist_data_t *)htblp->data;
			for (mtblp = w->tblist; mtblp; mtblp = mtblp->next) {
				if (GTK_WIDGET(mtblp->data) == tbdp->tb) {
					GNPDBG_TOOLBAR((
						"tb_redraw: found match with "
						"toolbar '%s'\n",
						tbdp->private_name));
					w->tblist = g_slist_remove_link(
							w->tblist, mtblp);
					g_slist_free_1(mtblp);
					if (!found)
						found = TRUE;
					break;
				}
			}
			g_assert(found);
			next = htblp->next;
			w->html_tblist = g_slist_remove_link(w->html_tblist,
							     htblp);
			g_free(tbdp->name);
			g_free(tbdp->private_name);
			gtk_widget_destroy(tbdp->tb);
			g_slist_free_1(htblp);
		}

		tb = html_tb_create(w);
#ifdef USE_GNOME
		tmp = GNOME_DOCK_ITEM(w->html_tb_h)->bin.child;
#else
		tmp = GTK_HANDLE_BOX(w->html_tb_h)->bin.child;
#endif
		if (tmp) {
			GNPDBG_TOOLBAR(("tb_redraw: destroying child widget in "
					"bin container\n"));
			gtk_widget_destroy(tmp);
		}
		gtk_container_add(GTK_CONTAINER(w->html_tb_h), tb);
		tb_on_off(w, w->html_tb_h, IS_SHOW_HTML_TOOLBAR());
	} else {
		GNPDBG_TOOLBAR(("tb_redraw: NO transition\n"));
		/*
		 * No transition, so just update the existing HTML toolbar if
		 * it has been changed.
		 */
		tb_on_off(w, w->html_tb_h, IS_SHOW_HTML_TOOLBAR());
		if (IS_SHOW_HTML_TOOLBAR()) {
			for (tblp = w->html_tblist; tblp; tblp = tblp->next) {
				tblist_data_t *tbldp =
						(tblist_data_t *)tblp->data;

				tb_hide_unhide_items(tbldp->tb, tbldp->tbdata);
			}
		}
	}
#endif	/* USE_HTMLTAGS */

	/* now set the characteristics of the toolbars themselves */
	for (tblp = w->tblist; tblp; tblp = tblp->next) {
		tb_raised_on_off(GTK_WIDGET(tblp->data));
		tb_pic_text_style(GTK_WIDGET(tblp->data));
	}

	/*
	 * Eventually, the "main" toolbar might be a list as well...  If that
	 * happens, we'll have to change this.
	 */
	if (IS_SHOW_TOOLBAR())
		tb_hide_unhide_items(w->main_tb, main_tbdata);

#ifdef USE_GNOME
	gtk_widget_queue_resize(GNOME_APP(w->toplev)->dock);
#endif
	menu_main_state_set(w);
} /* tb_redraw */


/*
 * PUBLIC: tb_pic_only
 *
 * updates toolbar to show buttons with icons only
 */
void
tb_pic_only(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_PIC_ONLY_TOOLBAR();
	CLEAR_PIC_TEXT_TOOLBAR();
	CLEAR_TEXT_ONLY_TOOLBAR();
	win_foreach(tb_redraw);
	msgbar_printf(w, PIC_ONLY);
} /* tb_pic_only */


/*
 * PUBLIC: tb_pic_text
 *
 * updates toolbar to show buttons with icons and text
 */
void
tb_pic_text(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_PIC_TEXT_TOOLBAR();
	CLEAR_TEXT_ONLY_TOOLBAR();
	CLEAR_PIC_ONLY_TOOLBAR();
	win_foreach(tb_redraw);
	msgbar_printf(w, PIC_TEXT);
} /* tb_pic_text */


/*
 * PUBLIC: tb_text_only
 *
 * updates toolbar to show buttons with text only
 */
void
tb_text_only(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_TEXT_ONLY_TOOLBAR();
	CLEAR_PIC_TEXT_TOOLBAR();
	CLEAR_PIC_ONLY_TOOLBAR();
	win_foreach(tb_redraw);
	msgbar_printf(w, TEXT_ONLY);
} /* tb_text_only */


/*
 * PUBLIC: tb_toggle
 *
 * toggles main toolbar and doclist toolbar
 */
void
tb_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (IS_SHOW_TOOLBAR())
		CLEAR_SHOW_TOOLBAR();
	else
		SET_SHOW_TOOLBAR();

	tb_on_off(w, w->main_tb->parent, IS_SHOW_TOOLBAR());

	msgbar_printf(w, IS_SHOW_TOOLBAR() ?  TOOLBAR_ON : TOOLBAR_OFF);
} /* tb_toggle */


#if defined(APP_GNP) && defined(USE_HTMLTAGS)
/*
 * PUBLIC: tb_html_toggle
 *
 * toggles html toolbar
 */
void
tb_html_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (IS_SHOW_HTML_TOOLBAR())
		CLEAR_SHOW_HTML_TOOLBAR();
	else
		SET_SHOW_HTML_TOOLBAR();

	tb_on_off(w, w->html_tb_h, IS_SHOW_HTML_TOOLBAR());

	msgbar_printf(w, IS_SHOW_HTML_TOOLBAR() ?
				HTML_TOOLBAR_ON : HTML_TOOLBAR_OFF);
} /* tb_html_toggle */
#endif


#ifdef GTK_HAVE_FEATURES_1_1_0
/*
 * PUBLIC: tb_raised_toggle
 *
 * toggles toolbar button relief
 */
void
tb_raised_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (IS_TOOLBAR_RAISED())
		CLEAR_TOOLBAR_RAISED();
	else
		SET_TOOLBAR_RAISED();

	win_foreach(tb_redraw);

	msgbar_printf(w, IS_TOOLBAR_RAISED() ? RAISED_ON : RAISED_OFF);
} /* tb_raised_toggle */
#endif


/*
 * PUBLIC: tb_ttip_toggle
 *
 * toggles toolbar tooltips on/off
 */
void
tb_ttip_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	GSList *tblp;

	g_assert(w != NULL);

	if (IS_SHOW_TOOLTIPS())
		CLEAR_SHOW_TOOLTIPS();
	else
		SET_SHOW_TOOLTIPS();

	msgbar_printf(w, IS_SHOW_TOOLTIPS() ? TOOLTIPS_ON : TOOLTIPS_OFF);

	/* now set the characteristics of the toolbars themselves */
	for (tblp = w->tblist; tblp; tblp = tblp->next)
		tb_ttip_on_off(GTK_WIDGET(tblp->data));

#ifdef GTK_HAVE_FEATURES_1_1_0
	win_foreach(menu_main_state_set);
#endif
} /* tb_ttip_toggle */


/*
 * PUBLIC: toolbar_create
 *
 * in:
 *	w: current window
 *	parent: parent widget (a box type) to hold toolbar (or handle)
 *	tbdata: toolbar data table
 *	tbname: Name of toolbar
 *
 * out:
 *	tb: the toolbar
 *	tb_h: toolbar handle
 */
void
toolbar_create(win_t *w, GtkWidget *parent, toolbar_data_t tbdata[],
	       char *tbname)
{
	GtkWidget *tb;

	tb = toolbar_create_start(w, tbdata);
	(void)toolbar_create_finish(w, parent, tb, tbname);
} /* toolbar_create */

/*
 * Simply creates the toolbar, sets some aesthetics, and adds the toolbar to
 * the window's toolbar list.  Returns pointer to the toolbar widget.
 */
GtkWidget *
toolbar_create_start(win_t *w, toolbar_data_t tbdata[])
{
	GtkWidget *tb;

	tb = toolbar_create_common(tbdata, w);
	GTK_WIDGET_UNSET_FLAGS(tb, GTK_CAN_FOCUS);
	gtk_toolbar_set_button_relief(GTK_TOOLBAR(tb),
		(IS_TOOLBAR_RAISED()) ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
	gtk_container_border_width(GTK_CONTAINER(tb), 2);
	w->tblist = g_slist_prepend(w->tblist, tb);
	return tb;
} /* toolbar_create_start */


/*
 * Finishes up toolbar creation:
 *	- creates a handlebox
 *	- adds toolbar to it
 *	- adds the handlebox to the parent
 *	- shows the handlebox
 *	- adds handlebox to the main window's (toolbar) handle list
 *	- returns toolbar handle with toolbar inside of it
 */
GtkWidget *
toolbar_create_finish(win_t *w, GtkWidget *parent, GtkWidget *tb, char *tbname)
{
	GtkWidget *tb_h;
#ifdef USE_GNOME
	GtkWidget *dockitem;
	GnomeDockItemBehavior behavior;
	GnomeApp *app = GNOME_APP(w->toplev);

	if (tbname) {
		behavior = GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
			   GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL;
		dockitem = gnome_dock_item_new(tbname, behavior);
		gtk_container_add(GTK_CONTAINER(dockitem), GTK_WIDGET(tb));
		tb_h = dockitem;
		gnome_dock_add_item(GNOME_DOCK(app->dock),
				    GNOME_DOCK_ITEM(dockitem),
				    GNOME_DOCK_TOP, /* placement, */
				    1, /* band_num, */
				    0, /* band_position, */
				    0, /* offset, */
				    TRUE);
	} else {
#endif
	tb_h = gtk_handle_box_new();
	gtk_container_add(GTK_CONTAINER(tb_h), tb);
	if (parent)
		gtk_box_pack_start(GTK_BOX(parent), tb_h, FALSE, TRUE, 0);
#ifdef USE_GNOME
	}
#endif
	if (IS_SHOW_TOOLBAR())
		gtk_widget_show(tb_h);

	w->tbhlist = g_slist_prepend(w->tbhlist, tb_h);

	return tb_h;
} /* toolbar_create_finish */


/*** local function definitions */
/*
 * PRIVATE: tb_on_off
 *
 * hides/unhides the toolbar
 */
static void
tb_on_off(win_t *w, GtkWidget *tbh, long showit)
{
	if (!tbh)
		return;

	GNPDBG_TOOLBAR(("tb_on_off: showit = %ld\n", showit));
	if (showit && !GTK_WIDGET_VISIBLE(tbh))
		gtk_widget_show(tbh);
	else if (!showit && GTK_WIDGET_VISIBLE(tbh))
		gtk_widget_hide(tbh);
#ifdef USE_GNOME
	gtk_widget_queue_resize(GNOME_APP(w->toplev)->dock);
#endif
} /* tb_on_off */


#ifdef GTK_HAVE_FEATURES_1_1_0
/*
 * PRIVATE: tb_raised_on_off
 *
 * sets toolbar button relief according to toolbar prefs.
 */
static void
tb_raised_on_off(GtkWidget *tb)
{
	if (tb && GTK_IS_TOOLBAR(tb)) {
		if (IS_TOOLBAR_RAISED())
			gtk_toolbar_set_button_relief(GTK_TOOLBAR(tb),
						      GTK_RELIEF_NORMAL);
		else
			gtk_toolbar_set_button_relief(GTK_TOOLBAR(tb),
						      GTK_RELIEF_NONE);
	}
} /* tb_raised_on_off */
#endif


/*
 * PRIVATE: tb_ttip_on_off
 *
 * turns OFF tooltips
 */
static void
tb_ttip_on_off(GtkWidget *tb)
{
	if (tb && GTK_IS_TOOLBAR(tb))
		gtk_toolbar_set_tooltips(GTK_TOOLBAR(tb), IS_SHOW_TOOLTIPS());
} /* tb_ttip_on_off */


/*
 * PRIVATE: toolbar_create_common
 *
 * common routine to create a toolbar.
 *
 * in: toolbar_data_t and pointer to callback data of gE_data type
 * out: GtkWidget *toolbar
 */
static GtkWidget *
toolbar_create_common(toolbar_data_t *tbdata, win_t *w)
{
	GtkWidget *tb;
	toolbar_data_t *tbdp = tbdata;

	g_assert(tbdp != NULL);

	if (IS_TEXT_TOOLBAR()) {
		tb = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				     GTK_TOOLBAR_TEXT);
	} else if (IS_PIC_TOOLBAR()) {
		tb = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				     GTK_TOOLBAR_ICONS);
	} else {
		SET_PIC_TEXT_TOOLBAR();
		tb = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				     GTK_TOOLBAR_BOTH);
	}

	while (tbdp->text != NULL) {
		gtk_toolbar_append_item(
			GTK_TOOLBAR(tb),
			tbdp->text,
			gettext(tbdp->tooltip_text),
			tbdp->tooltip_private_text,
			misc_new_pixmap(tbdp->icon, w->toplev, NULL, NULL),
			GTK_SIGNAL_FUNC(tbdp->callback),
			(tbdp->cbdata) ? tbdp->cbdata : (gpointer)w);

		tbdp++;

		if (tbdp->text != NULL && strcmp(tbdp->text, " SPACE ") == 0) {
			gtk_toolbar_append_space(GTK_TOOLBAR(tb));
			tbdp++;
		}
	}

	/* set according to preferences */
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(tb), IS_SHOW_TOOLTIPS());

	/*
	 * this is OK to show even if prefs says don't show, because it's the
	 * toolbar handle that gets shown.  after we show the overall toolbar,
	 * we selectively hide the ones that have preferences set as hidden.
	 */
	gtk_widget_show(tb);
	tb_hide_unhide_items(tb, tbdata);

#if 0
	/* tooltips: black on white is more noticeable than black on grey.
	 * XXX - why does this not work?  is GTK fubar?
	 */
	gtk_tooltips_set_colors(GTK_TOOLBAR(tb)->tooltips,
                                &tb->style->white, &tb->style->black);
#endif

	return tb;
} /* toolbar_create_common */


/*
 * PRIVATE: tb_hide_unhide_items
 *
 * given a toolbar, hide the toolbar items that are hidden, and show the items
 * that are not.  used when redrawing the toolbar after (possible) changes in
 * the preferences, where the user may select to show/hide certain buttons on
 * the toolbar(s).
 */
static void
tb_hide_unhide_items(GtkWidget *tb, toolbar_data_t *tbdata)
{
	toolbar_data_t *tbdp;
	GtkToolbarChild *child;
	GList *clp;

	if (!tb || !GTK_IS_TOOLBAR(tb))
		return;

	for (clp = GTK_TOOLBAR(tb)->children, tbdp = tbdata;
	     clp && tbdp->text;
	     clp = clp->next, tbdp++) {

		if (strcmp(tbdp->text, " SPACE ") == 0)
			continue;

		child = (GtkToolbarChild *)(clp->data);
		if (!child->widget || !child->icon || !child->label)
			continue;

		if (prefs_bool_by_name(tbdp->prefname)) {
			if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(child->widget)))
				gtk_widget_show(GTK_WIDGET(child->widget));
		} else {
			if (GTK_WIDGET_VISIBLE(GTK_WIDGET(child->widget)))
				gtk_widget_hide(GTK_WIDGET(child->widget));
		}
	}
} /* tb_hide_unhide_items */


static void
tb_pic_text_style(GtkWidget *tb)
{
	GtkToolbarStyle tbstyle;

	if (!tb || !GTK_IS_TOOLBAR(tb))
		return;

	if (IS_TEXT_PIC_TOOLBAR())
		tbstyle = GTK_TOOLBAR_BOTH;
	else if (IS_TEXT_TOOLBAR())
		tbstyle = GTK_TOOLBAR_TEXT;
	else /* if (IS_PIC_TOOLBAR()) */
		tbstyle = GTK_TOOLBAR_ICONS;

	switch (tbstyle) {
	case GTK_TOOLBAR_ICONS:
		if (GTK_TOOLBAR(tb)->style != tbstyle) {
			gtk_toolbar_set_style(GTK_TOOLBAR(tb), tbstyle);

			SET_PIC_ONLY_TOOLBAR();
			CLEAR_PIC_TEXT_TOOLBAR();
			CLEAR_TEXT_ONLY_TOOLBAR();
		}
		break;
	case GTK_TOOLBAR_TEXT:
		if (GTK_TOOLBAR(tb)->style != tbstyle) {
			gtk_toolbar_set_style(GTK_TOOLBAR(tb), tbstyle);

			SET_TEXT_ONLY_TOOLBAR();
			CLEAR_PIC_TEXT_TOOLBAR();
			CLEAR_PIC_ONLY_TOOLBAR();
		}
		break;
	case GTK_TOOLBAR_BOTH:
		if (GTK_TOOLBAR(tb)->style != tbstyle) {
			gtk_toolbar_set_style(GTK_TOOLBAR(tb), tbstyle);

			SET_PIC_TEXT_TOOLBAR();
			CLEAR_TEXT_ONLY_TOOLBAR();
			CLEAR_PIC_ONLY_TOOLBAR();
		}
		break;
	default:
		return;
	}

	if (GTK_WIDGET_VISIBLE(tb)) {
		gtk_widget_hide(tb);
		gtk_widget_show(tb);
	}

} /* tb_pic_text_style */


#endif	/* USE_TOOLBARS */

/* the end */
