/* gnobog_bookmarks.c
 *
 * Copyright (C) 2000 Frdric LESPEZ & Renaud CHAILLAT
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnobog_bookmarks.h"
#include "gnobog_bookmarks-private.h"
#include "gnobog_mozilla_backend.h"
#include "glib_addon.h"

enum
{
  LIST_CREATED,
  LIST_MODIFIED,
  LIST_DELETED,
  BOOKMARKS_CLEARED,
  LAST_SIGNAL
};

enum
{
  ARG_0,
  ARG_FILENAME,
  ARG_TYPE,
  ARG_NODE,
  ARG_READ_ONLY,
  ARG_MODIFIED,
  ARG_INSERT_DEFAULT_MODE,
  ARG_FOLDER_INSERT_DEFAULT_MODE
};

/* The following contains GCCism. What is more, GCC on RH 6.2 don't like at all */
/* #define gnobog_debug(format, ...)   g_log (G_LOG_DOMAIN,  G_LOG_LEVEL_DEBUG, format, ##__VA_ARGS__) */
#define gnobog_debug   g_message

/* Command lines for sending URL to programs */
#define NETSCAPE_IN_EXISTING_WINDOW_CL      "gnome-moz-remote --raise %s"
#define NETSCAPE_IN_NEW_WINDOW_CL           "gnome-moz-remote --newwin %s"
#define MOZILLA_IN_EXISTING_WINDOW_CL       "gnome-moz-remote --raise %s"
#define MOZILLA_IN_NEW_WINDOW_CL            "gnome-moz-remote --newwin %s"
#define GALEON_IN_EXISTING_WINDOW_CL        "galeon --existing %s"
#define GALEON_IN_NEW_WINDOW_CL             "galeon %s"
#define NAUTILUS_IN_EXISTING_WINDOW_CL      "nautilus %s"
#define NAUTILUS_IN_NEW_WINDOW_CL           "nautilus %s"
#define KONQUEROR_IN_NEW_WINDOW_CL          "konqueror %s"
#define OPERA_IN_EXISTING_WINDOW_CL         "opera %s"
#define OPERA_IN_NEW_WINDOW_CL              "opera -newwindow %s"
#define LYNX_IN_NEW_WINDOW_CL               "gnome-terminal -e 'lynx %s'"
#define LINKS_IN_NEW_WINDOW_CL              "gnome-terminal -e 'links %s'"
#define GFTP_IN_NEW_WINDOW_CL               "gftp %s"
#define NCFTP_IN_NEW_WINDOW_CL              "gnome-terminal -e 'ncftp %s'"
#define LFTP_IN_NEW_WINDOW_CL               "gnome-terminal -e 'lftp %s'"

static GnobogBookmarksClass *parent_class = NULL;
static guint gnobog_bookmarks_signals[LAST_SIGNAL] = {0};

/* Forward declaration of static function */
/* Forward declarations of GTK+ object system specific methods */
static void     gnobog_bookmarks_init (GnobogBookmarks* bookmarks);
static void     gnobog_bookmarks_class_init (GnobogBookmarksClass* klass);
static void     gnobog_bookmarks_set_arg (GtkObject* object, GtkArg* arg, guint arg_id);
static void     gnobog_bookmarks_get_arg (GtkObject* object, GtkArg* arg, guint arg_id);
static gboolean gnobog_bookmarks_destroy_name_hash_table (gpointer key, gpointer value, gpointer user_data);
static gboolean gnobog_bookmarks_destroy_alias_hash_table (gpointer key, gpointer value, gpointer user_data);
static void     gnobog_execute_command_line (const gchar* commandline);
static void     gnobog_url_browse_in_netscape (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_mozilla (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_galeon (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_gnome_handler (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_nautilus (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_konqueror (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_opera (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_lynx (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_links (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_gftp (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_ncftp (gchar* location, gboolean open_new_window);
static void     gnobog_url_browse_in_lftp (gchar* location, gboolean open_new_window);
static void     gnobog_bookmarks_node_unalias (GnobogBookmarks* bookmarks, GnobogBookmarksNode node);
static gboolean gnobog_bookmarks_node_auto_alias_entry (GnobogBookmarksNode node, gpointer data);
static GNode*   gnobog_bookmarks_node_duplicate_entry (GnobogBookmarksNode node, gpointer data);
static gboolean gnobog_bookmarks_node_destroy_entry (GnobogBookmarksNode node, gpointer data);
static void     gnobog_bookmarks_list_auto_alias (GnobogBookmarks* bookmarks,  GList* list);
static void     gnobog_bookmarks_node_auto_alias (GnobogBookmarks* bookmarks, GnobogBookmarksNode node);

/*
** GTK+ object system specific methods
*/

static void
gnobog_bookmarks_init (GnobogBookmarks* bookmarks)
{
  bookmarks->title       = NULL;
  bookmarks->description = NULL;

  bookmarks->filename    = NULL;
  bookmarks->format      = UNKNOWN_FILEFORMAT;
  bookmarks->readonly    = FALSE;
  bookmarks->modified    = FALSE;

  bookmarks->node  = gnobog_bookmarks_node_import /* our own root node  */
                                     (TITLE,
                                      "Gnobog Root Folder",
                                      "None",
                                      "Internal root folder",
                                      0,
                                      0,
                                      0,
                                      TRUE);

  bookmarks->alias_hash_table = g_hash_table_new (g_str_hash, g_str_equal);

  bookmarks->insert_default_mode        = INSERT_AFTER;
  bookmarks->folder_insert_default_mode = INSERT_INTO;

  /* Initialize alias management */
  gnobog_bookmarks_node_auto_alias (bookmarks, bookmarks->node);

  return;
}

static void
gnobog_bookmarks_class_init (GnobogBookmarksClass* klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass *) klass;

  gtk_object_add_arg_type ("GnobogBookmarks::filename",
                           GTK_TYPE_STRING,
                           GTK_ARG_READABLE,
                           ARG_FILENAME);
  gtk_object_add_arg_type ("GnobogBookmarks::type",
                           GTK_TYPE_ENUM,
                           GTK_ARG_READABLE,
                           ARG_TYPE);
  gtk_object_add_arg_type ("GnobogBookmarks::node",
                           GTK_TYPE_POINTER,
                           GTK_ARG_READABLE,
                           ARG_NODE);
  gtk_object_add_arg_type ("GnobogBookmarks::read_only",
                           GTK_TYPE_BOOL,
                           GTK_ARG_READWRITE,
                           ARG_READ_ONLY);
  gtk_object_add_arg_type ("GnobogBookmarks::modified",
                           GTK_TYPE_BOOL,
                           GTK_ARG_READABLE,
                           ARG_MODIFIED);
  gtk_object_add_arg_type ("GnobogBookmarks::insert_default_mode",
                           GTK_TYPE_ENUM,
                           GTK_ARG_READWRITE,
                           ARG_INSERT_DEFAULT_MODE);
  gtk_object_add_arg_type ("GnobogBookmarks::folder_insert_default_mode",
                           GTK_TYPE_ENUM,
                           GTK_ARG_READWRITE,
                           ARG_FOLDER_INSERT_DEFAULT_MODE);
                
  gnobog_bookmarks_signals[LIST_CREATED] =
    gtk_signal_new ("list_created",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnobogBookmarksClass, list_created),
                    gtk_marshal_NONE__POINTER_INT_POINTER,
                    //                          gtk_marshal_NONE__POINTER_ENUM_POINTER,
                    GTK_TYPE_NONE,
                    3,
                    GTK_TYPE_POINTER,
                    GTK_TYPE_ENUM,
                    GTK_TYPE_POINTER);
  gnobog_bookmarks_signals[LIST_MODIFIED] =
    gtk_signal_new ("list_modified",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnobogBookmarksClass, list_modified),
                    gtk_marshal_NONE__POINTER,
                    GTK_TYPE_NONE,
                    1,
                    GTK_TYPE_POINTER);
  gnobog_bookmarks_signals[LIST_DELETED] =
    gtk_signal_new ("list_deleted",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnobogBookmarksClass, list_deleted),
                    gtk_marshal_NONE__POINTER,
                    GTK_TYPE_NONE,
                    1,
                    GTK_TYPE_POINTER);
  gnobog_bookmarks_signals[BOOKMARKS_CLEARED] =
    gtk_signal_new ("bookmarks_cleared",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnobogBookmarksClass, bookmarks_cleared),
                    gtk_marshal_NONE__NONE,
                    GTK_TYPE_NONE,
                    0 /*,
                        GTK_TYPE_POINTER */);

  gtk_object_class_add_signals (object_class, gnobog_bookmarks_signals, LAST_SIGNAL);

  object_class->set_arg = gnobog_bookmarks_set_arg;
  object_class->get_arg = gnobog_bookmarks_get_arg;
  object_class->destroy = gnobog_bookmarks_destroy;
  parent_class = gtk_type_class (gtk_object_get_type ());

  klass->list_created = NULL;
  klass->list_deleted = NULL;
  klass->list_modified = NULL;
        
  return;
}

GtkType
gnobog_bookmarks_get_type (void)
{
  static GtkType bookmarks_type = 0;

  if (!bookmarks_type) {
    GtkTypeInfo bookmarks_info = {
      "GnobogBookmarks",
      sizeof (GnobogBookmarks),
      sizeof (GnobogBookmarksClass),
      (GtkClassInitFunc) gnobog_bookmarks_class_init,
      (GtkObjectInitFunc) gnobog_bookmarks_init,
      (GtkArgSetFunc) NULL,
      (GtkArgGetFunc) NULL,
    };

    bookmarks_type = gtk_type_unique (gtk_object_get_type(), &bookmarks_info);
  }
  return bookmarks_type;
}

static void     
gnobog_bookmarks_set_arg (GtkObject* object, GtkArg* arg, guint arg_id)
{

}

static void     
gnobog_bookmarks_get_arg (GtkObject* object, GtkArg* arg, guint arg_id)
{

}

/*
** Public Methods
*/

GnobogBookmarks*
gnobog_bookmarks_new (void)
{
  GnobogBookmarks*  new_bookmarks;

  /* Log a little message for debugging */
  gnobog_debug ("Creating New Bookmarks Object");

  new_bookmarks = gtk_type_new (gnobog_bookmarks_get_type ());

  new_bookmarks->title       = g_strdup (_("New Bookmarks Document")); 
  new_bookmarks->description = g_strdup ("");

  return new_bookmarks;
}

GnobogBookmarks*
gnobog_bookmarks_new_with_title (gchar* title, gchar* description)
{
  GnobogBookmarks*  new_bookmarks;

  /* Log a little message for debugging */
  gnobog_debug ("Creating New Bookmarks Object");
      
  new_bookmarks = gtk_type_new (gnobog_bookmarks_get_type ());

  new_bookmarks->title       = g_strdup (title); 
  new_bookmarks->description = g_strdup (description);

  return new_bookmarks;
}

GnobogBookmarks*
gnobog_bookmarks_new_from_bookmarks (GnobogBookmarks* bookmarks)
{
  GnobogBookmarks*  new_bookmarks;
  GList*            list;

  g_return_val_if_fail (bookmarks  != NULL, NULL);  

  /* Log a little message for debugging */
  gnobog_debug ("Creating New Bookmarks Object From Bookmarks Object %s", bookmarks->title);
      
  new_bookmarks = gtk_type_new (gnobog_bookmarks_get_type ());

  new_bookmarks->title       = g_strdup (bookmarks->title); 
  new_bookmarks->description = g_strdup (bookmarks->description);
  new_bookmarks->modified    = TRUE;

  /* Destroy default root node */
  gnobog_bookmarks_node_destroy (new_bookmarks, new_bookmarks->node);

  new_bookmarks->node = gnobog_bookmarks_node_duplicate
                                        (bookmarks,
                                         bookmarks->node);

  new_bookmarks->insert_default_mode        = bookmarks->insert_default_mode;
  new_bookmarks->folder_insert_default_mode = bookmarks->folder_insert_default_mode;

  /* Auto alias */
  gnobog_bookmarks_node_auto_alias (new_bookmarks, new_bookmarks->node);

  /* Tell everybody that the work has been done */
  /* FIXME : Seems Unuseful */
  list = gnobog_bookmarks_get_list (new_bookmarks);
  gtk_signal_emit (GTK_OBJECT (new_bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   NULL,
                   INSERT_INTO,
                   list);
  g_list_free (list);

  return new_bookmarks;
}

GnobogBookmarks*
gnobog_bookmarks_new_from_file (gchar* filename)
{
  GnobogBookmarks*     new_bookmarks;
  GList*               list;
        
  g_return_val_if_fail (filename  != NULL, NULL);  

  /* FIXME : Must work with other format */
  /* FIXME : This function share a lot of code
   * with gnobog_bookmarsk_file_load(). Remove one of them
   * or create a common function */

  /* Log a little message for debugging */
  gnobog_debug ("Creating New Bookmarks Object From File %s ...", filename);
  
  if (!g_file_exists (filename)) {
    g_message("File %s does not exist", filename);
    return NULL;
  }

  new_bookmarks = gtk_type_new (gnobog_bookmarks_get_type ());
  
  /* Destroy default root node */
  gnobog_bookmarks_node_destroy (new_bookmarks, new_bookmarks->node);
  new_bookmarks->node = NULL; /* Set node to something safe */

  /* FIXME : Improve error handling. Send a signal with error code/message ? */
  /* Importer must set title/description/filename/node of bookmarks */
  if ((mozilla_import (new_bookmarks, filename)) == FALSE) {
    gnobog_bookmarks_destroy (GTK_OBJECT (new_bookmarks));
    return NULL;
  }

  // FIXME : If file is readonly, change readonly bookmarks status to readonly
                
  /* Auto alias */
  gnobog_bookmarks_node_auto_alias (new_bookmarks, new_bookmarks->node);
        
  /* Tell everybody that the work has been done */
  /* FIXME : Seems Unuseful */
  list = gnobog_bookmarks_get_list (new_bookmarks);
  gtk_signal_emit (GTK_OBJECT (new_bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   NULL,
                   INSERT_INTO,
                   list);
  g_list_free (list);

  /* Log a little message for debugging */
  gnobog_debug ("Creating New Bookmarks Object From File %s ... DONE", filename);
        
  return new_bookmarks;
}

static gboolean
gnobog_bookmarks_destroy_name_hash_table (gpointer key, gpointer value, gpointer user_data)
{
  //  g_free (key);
  g_message ("Name Hash Table Residu %s", (gchar*) key);
  return TRUE;
}

static gboolean
gnobog_bookmarks_destroy_alias_hash_table (gpointer key, gpointer value, gpointer user_data)
{
  //  g_free (key);
  g_message ("Alias Hash Table Residu %s", (gchar*) key);
  /* The following is not needed if alias management is safe */
  /* FIXME : Print a message to know safety */
  (void)g_hash_table_foreach_remove (value,
                                     &gnobog_bookmarks_destroy_name_hash_table,
                                     NULL);
  /* Destroy hash table */
  g_hash_table_destroy (value);
        
  return TRUE;
}

void    
gnobog_bookmarks_destroy (GtkObject* object)
{
  GnobogBookmarks*  bookmarks;

  bookmarks = GNOBOG_BOOKMARKS (object);
        
  /* Log a little message for debugging */
  gnobog_debug ("Destroy Bookmarks Object : %s",
                (bookmarks->title == NULL) ? "(null)" : bookmarks->title);
        
  g_free (bookmarks->title);
  g_free (bookmarks->description);
  g_free (bookmarks->filename);
  
  if (bookmarks->node != NULL) { /* Prevent unuseful crash */
    gnobog_bookmarks_node_destroy (bookmarks, bookmarks->node);
  } 

  /* The following is here just to test alias management safety */
  if (bookmarks->alias_hash_table != NULL) { /* Prevent unuseful crash */
    if (g_hash_table_size (bookmarks->alias_hash_table) != 0) {
      g_message ("Arggghh !! Incoherences found in alias management :-(");
      (void)g_hash_table_foreach_remove (bookmarks->alias_hash_table,
                                         &gnobog_bookmarks_destroy_alias_hash_table,
                                         NULL);
    } else {
      g_message ("No incoherence found in alias management :-)");
    }
    /* Destroy hash table */
    g_hash_table_destroy (bookmarks->alias_hash_table);
  }
        
  /* FIXME : about functions working on list, do they have to destroy the list ? */
}

gboolean
gnobog_bookmarks_save_file (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, FALSE); 

  /* FIXME : Check if destination file is not read only */
        
  if (bookmarks->readonly == TRUE) {
    /* Log a little message for debugging */
    gnobog_debug ("Unable to save -READ ONLY- file %s",
                  bookmarks->filename);
    return FALSE;
  }
        
  /* Log a little message for debugging */
  gnobog_debug ("Saving file %s", bookmarks->filename);

  /* FIXME : Only Mozilla export is handled */
  /* Export to Mozilla format file */
  if (mozilla_export (bookmarks, bookmarks->filename) == TRUE) {       
    /* Tag bookmarks as not 'modified' */
    bookmarks->modified = FALSE;
    return TRUE;
  } else {     
    return FALSE;
  }
}

gboolean
gnobog_bookmarks_save_as_file (GnobogBookmarks* bookmarks,
                               gchar* filename,
                               GnobogBookmarksFileFormat format)
{
  g_return_val_if_fail (filename != NULL, FALSE);  
  g_return_val_if_fail (bookmarks != NULL, FALSE); 

  /* FIXME : Check if destination file is not read only */
        
  /* Log a little message for debugging*/
  gnobog_debug ("Saving file %s as %s",
                (bookmarks->filename == NULL) ? "(null)" : bookmarks->filename,
                filename);
        
  /* FIXME : Only Mozilla export is handled */
  /* Export to Mozilla format file */
  if (mozilla_export (bookmarks, filename) == TRUE) {
    /* Tag bookmarks as not 'modified' */
    bookmarks->modified = FALSE;
        
    /* Change current filename */
    /* TODO: save copy as */
    g_free (bookmarks->filename);
    bookmarks->filename = g_strdup (filename);
    return TRUE;
  } else {     
    return FALSE;
  }
}

gboolean
gnobog_bookmarks_save_copy_as_file (GnobogBookmarks* bookmarks,
                                    gchar* filename,
                                    GnobogBookmarksFileFormat format)
{
  g_return_val_if_fail (filename != NULL, FALSE);  
  g_return_val_if_fail (bookmarks != NULL, FALSE); 

  /* FIXME : 
   * Check if destination file is not read only
   * Check if destination file has a different name
   */   

  /* Log a little message for debugging */
  gnobog_debug ("Saving file %s as %s", bookmarks->filename, filename);
        
  /* FIXME : Only Mozilla export is handled */
  /* Export to Mozilla format file */
  if (mozilla_export (bookmarks, filename) == TRUE) {       
    return TRUE;
  } else {     
    return FALSE;
  }
}







gchar*
gnobog_bookmarks_get_title (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, "(null)");   
        
  return bookmarks->title;
}

void
gnobog_bookmarks_set_title (GnobogBookmarks* bookmarks, gchar* title)
{
  g_return_if_fail (bookmarks != NULL);   
        
  bookmarks->title = g_strdup (title);
  return;
}

gchar*
gnobog_bookmarks_get_description (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, "(null)");   
        
  return bookmarks->description;
}

void
gnobog_bookmarks_set_description (GnobogBookmarks* bookmarks, gchar* description)
{
  g_return_if_fail (bookmarks != NULL);   
        
  bookmarks->description = g_strdup (description);
  return;
}

gboolean
gnobog_bookmarks_is_modified (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, FALSE);      
        
  return bookmarks->modified;
}

gboolean
gnobog_bookmarks_is_readonly (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, FALSE);      
        
  return bookmarks->readonly;
}

void
gnobog_bookmarks_set_readonly (GnobogBookmarks* bookmarks, gboolean readonly)
{
  g_return_if_fail (bookmarks != NULL); 
        
  bookmarks->readonly = readonly;
  return;
}

GnobogBookmarksInsertMode
gnobog_bookmarks_get_insert_default_mode (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, INSERT_DEFAULT_MODE);        
        
  return bookmarks->insert_default_mode;
}

void
gnobog_bookmarks_set_insert_default_mode (GnobogBookmarks* bookmarks, GnobogBookmarksInsertMode mode)
{
  g_return_if_fail (bookmarks != NULL); 
        
  bookmarks->insert_default_mode = mode;
  return;
}

GnobogBookmarksInsertMode
gnobog_bookmarks_get_folder_insert_default_mode (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, INSERT_DEFAULT_MODE);        
        
  return bookmarks->folder_insert_default_mode;
}

void
gnobog_bookmarks_set_folder_insert_default_mode (GnobogBookmarks* bookmarks, GnobogBookmarksInsertMode mode)
{
  g_return_if_fail (bookmarks != NULL); 
        
  bookmarks->folder_insert_default_mode = mode;
  return;
}

gchar*
gnobog_bookmarks_get_filename (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, "(null)");   
        
  return bookmarks->filename;
}

void
gnobog_bookmarks_set_filename (GnobogBookmarks* bookmarks, gchar* filename)
{
  g_return_if_fail (bookmarks != NULL);   
        
  bookmarks->filename = g_strdup (filename);
  return;
}

GnobogBookmarksFileFormat
gnobog_bookmarks_get_file_format (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, UNKNOWN_FILEFORMAT);   
        
  return bookmarks->format;
}










GnobogBookmarksNodeType
gnobog_bookmarks_node_get_type (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, UNKNOWN_ENTRY);   
  return node_get_type (node);
}
                                
gchar*
gnobog_bookmarks_node_get_name (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, "");      
  return node_get_name (node);
}
                                
gchar*
gnobog_bookmarks_node_get_location (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, "");      
  return node_get_location (node);
}
                                
gchar*
gnobog_bookmarks_node_get_description (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, "");      
  return node_get_description (node);
}
                                
GTime
gnobog_bookmarks_node_get_creation_time (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, 0);       
  return node_get_creation_time (node);
}
                                
GTime
gnobog_bookmarks_node_get_modification_time (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, 0);       
  return node_get_modification_time (node);
}
                                
GTime
gnobog_bookmarks_node_get_visit_time (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, 0);       
  return node_get_visit_time (node);
}
                                
gboolean
gnobog_bookmarks_node_get_folder_state (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, FALSE);   
  return node_get_folder_state (node);
}
                                
GList*
gnobog_bookmarks_node_get_alias_list (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, NULL);    
  return node_get_alias_list (node);
}
                                
gboolean
gnobog_bookmarks_node_is_folder_open (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, FALSE);   
  return node_is_folder_open (node);
}

gboolean
gnobog_bookmarks_node_have_aliases (GnobogBookmarksNode node)
{
  g_return_val_if_fail (node != NULL, FALSE);   
  return node_have_aliases (node);
}









GnobogBookmarksNode
gnobog_bookmarks_node_import (GnobogBookmarksNodeType type,
                              gchar* name,
                              gchar* location,
                              gchar* description,
                              guint created,
                              guint visited,
                              guint modified,
                              gboolean open)
{
  GnobogBookmarksNode   node;
  GnobogBookmarksEntry  entry;
  GList*                alias_list;
        
  entry = gnobog_bookmarks_entry_import (type,
                                         name,
                                         location,
                                         description,
                                         created,
                                         visited,
                                         modified,
                                         open);  

  node = g_node_new (entry);

  /* Create alias_list and store it in node's entry */
  alias_list = g_list_prepend (NULL, node);
  node_set_alias_list (node, alias_list);

  return node;
}

GnobogBookmarksNode
gnobog_bookmarks_node_import_separator (void)
{
  GnobogBookmarksNode  node;
        
  node = gnobog_bookmarks_node_import (SEPARATOR,
                                       "",
                                       "",
                                       "",
                                       0,
                                       0,
                                       0,
                                       FALSE);  
  return node;
}

/* used when dropping from mozilla
 * and when inserting from popup menu */
GnobogBookmarksNode
gnobog_bookmarks_node_insert_new (GnobogBookmarks* bookmarks,
                                  GnobogBookmarksNode anchor_node,
                                  GnobogBookmarksInsertMode mode,
                                  GnobogBookmarksNodeType type,
                                  gchar* name,
                                  gchar* location,
                                  gchar* description)
{
  GList*                                        list;   
  GnobogBookmarksNode   new_node;
  GnobogBookmarksNode   real_anchor_node;
  GnobogBookmarksEntry  new_entry;
  GList*                                        alias_list;     
        
  g_return_val_if_fail (bookmarks != NULL, NULL);       
        
  /* Handle null anchor */
  if (anchor_node == NULL) {
    real_anchor_node = bookmarks->node;
    mode = INSERT_INTO;
  } else {
    real_anchor_node = anchor_node;
  }
        
  /* Get Insert Default Mode explicitly */
  if (mode == INSERT_DEFAULT_MODE)
    mode = gnobog_bookmarks_node_get_insert_default_mode (bookmarks, real_anchor_node);
        
  if (!gnome_bookmarks_list_is_insertion_valid (bookmarks, real_anchor_node, mode, NULL)) {
    return NULL;
  }
        
  /* Create new entry */
  new_entry = gnobog_bookmarks_entry_new (type, name, location, description);
        
  /* Create new node */
  new_node = g_node_new (new_entry);

  /* Create alias_list and store it in node's entry */
  alias_list = g_list_prepend (NULL, new_node);
  node_set_alias_list (new_node, alias_list);

  /* Link new node */
  gnobog_bookmarks_node_link (bookmarks, real_anchor_node, mode, new_node);
                
  /* Auto alias the new node */
  gnobog_bookmarks_node_auto_alias (bookmarks, new_node);
        
  /* Sanity Check : We should NOT emit a 'list_created' signal with mode set to INSERT_DEFAULT_MODE */
  g_assert (mode != INSERT_DEFAULT_MODE);
        
  /* Tell everybody that the work has been done */
  list = g_list_prepend (NULL, new_node);
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   anchor_node,
                   mode,
                   list);
  g_list_free (list);
        
  return new_node;
}

GnobogBookmarksNode
gnobog_bookmarks_node_insert_new_separator (GnobogBookmarks* bookmarks,
                                            GnobogBookmarksNode anchor_node,
                                            GnobogBookmarksInsertMode mode)
{
  return gnobog_bookmarks_node_insert_new (bookmarks,
                                           anchor_node,
                                           INSERT_DEFAULT_MODE,
                                           SEPARATOR,
                                           "",
                                           "",
                                           "");
}








void
gnobog_bookmarks_node_modify (GnobogBookmarks* bookmarks,
                              GnobogBookmarksNode node,
                              gchar* name,
                              gchar* location,
                              gchar* description)
{
  GList*  list;   
        
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);
        
  /* Update node entry */
  gnobog_bookmarks_node_update (bookmarks, node, name, location, description);
                
  /* If node has aliases, aliases have been modified too */
  if (gnobog_bookmarks_node_have_aliases (node) == TRUE) {
    list = g_list_copy (node_get_alias_list (node));
  } else {
    list = g_list_prepend (NULL, node);
  }
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_MODIFIED],
                   list);
  g_list_free (list);
}

void    
gnobog_bookmarks_list_copy_from (GnobogBookmarks* bookmarks, GnobogBookmarks* source_bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksInsertMode mode, GList* list)
{
  GList*               simplified_list;
  GList*               duplicated_nodes_list;
  GnobogBookmarksNode  real_anchor_node;
        
  g_return_if_fail (bookmarks != NULL);
  /* TODO: better handle this case */
  g_return_if_fail (list != NULL);
        
  /* Handle null anchor */
  if (anchor_node == NULL) {
    real_anchor_node = bookmarks->node;
    mode = INSERT_INTO;
  } else {
    real_anchor_node = anchor_node;
  }
        
  /* Get Insert Default Mode explicitly */
  /* First operation to perform (before testing insertion) */
  if (mode == INSERT_DEFAULT_MODE)
    mode = gnobog_bookmarks_node_get_insert_default_mode (bookmarks, real_anchor_node);
        
  if (!gnome_bookmarks_list_is_insertion_valid (bookmarks, real_anchor_node, mode, list))
    return;
        
  /* Make a simplified copy of the list so as to work only on folders and
   * orphans (that is, nodes in the list while their ancestors are not) */
  simplified_list = gnobog_bookmarks_list_simplify (source_bookmarks, list);
                
  /* duplicate the nodes */
  duplicated_nodes_list = gnobog_bookmarks_list_duplicate (source_bookmarks, simplified_list);
  g_list_free (simplified_list);
        
  /* Then insert them */
  gnobog_bookmarks_list_link (bookmarks, real_anchor_node, mode, duplicated_nodes_list);
        
  /* Auto alias the newly arrived nodes */
  gnobog_bookmarks_list_auto_alias (bookmarks, duplicated_nodes_list);
        
  /* Sanity Check : We should NOT emit a 'list_created' signal with mode set to INSERT_DEFAULT_MODE */
  g_assert (mode != INSERT_DEFAULT_MODE);
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   anchor_node, mode,
                   duplicated_nodes_list);
  g_list_free (duplicated_nodes_list);
}

void
gnobog_bookmarks_list_move_from (GnobogBookmarks* bookmarks, GnobogBookmarks* source_bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksInsertMode mode, GList* list)
{
  GList*               simplified_list;
  GList*               duplicated_nodes_list;
  GnobogBookmarksNode  real_anchor_node;
        
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (list != NULL);
        
  /* Handle null anchor */
  if (anchor_node == NULL) {
    real_anchor_node = bookmarks->node;
    mode = INSERT_INTO;
  } else {
    real_anchor_node = anchor_node;
  }
        
  /* Get Insert Default Mode explicitly */
  /* First operation to perform (before testing insertion) */
  if (mode == INSERT_DEFAULT_MODE)
    mode = gnobog_bookmarks_node_get_insert_default_mode (bookmarks, real_anchor_node);
        
  if (!gnome_bookmarks_list_is_insertion_valid (bookmarks, real_anchor_node, mode, list))
    return;
        
  /* Make a simplified copy of the list so as to work only on folders and
   * orphans (that is, nodes in the list while their ancestors are not) */
  simplified_list = gnobog_bookmarks_list_simplify (bookmarks, list);
                
  /* Duplicate the nodes */
  duplicated_nodes_list = gnobog_bookmarks_list_duplicate (source_bookmarks, simplified_list);
        
  /* Delete all nodes in list */
  gnobog_bookmarks_list_delete (source_bookmarks, simplified_list);
  g_list_free (simplified_list);
             
  /* FIXME : What if signals aren't catched in the good order (Multithread) ? */
        
  /* Then link them in the new bookmarks */
  gnobog_bookmarks_list_link (bookmarks, real_anchor_node, mode, duplicated_nodes_list);
        
  /* Auto alias the newly arrived nodes */
  gnobog_bookmarks_list_auto_alias (bookmarks, duplicated_nodes_list);
        
  /* Sanity Check : We should NOT emit a 'list_created' signal with mode set to INSERT_DEFAULT_MODE */
  g_assert (mode != INSERT_DEFAULT_MODE);
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   anchor_node, mode,
                   duplicated_nodes_list);

  g_list_free (duplicated_nodes_list);
}

void    
gnobog_bookmarks_list_copy (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksInsertMode mode, GList* list)
{
  GList*               simplified_list;
  GList*               duplicated_nodes_list;
  GnobogBookmarksNode  real_anchor_node;
        
  g_return_if_fail (bookmarks != NULL);
  /* TODO: better handle this case */
  g_return_if_fail (list != NULL);
        
  /* Handle null anchor */
  if (anchor_node == NULL) {
    real_anchor_node = bookmarks->node;
    mode = INSERT_INTO;
  } else {
    real_anchor_node = anchor_node;
  }
        
  /* Get Insert Default Mode explicitly */
  /* First operation to perform (before testing insertion) */
  if (mode == INSERT_DEFAULT_MODE)
    mode = gnobog_bookmarks_node_get_insert_default_mode (bookmarks, real_anchor_node);
        
  if (!gnome_bookmarks_list_is_insertion_valid (bookmarks, real_anchor_node, mode, list))
    return;
        
  /* Make a simplified copy of the list so as to work only on folders and
   * orphans (that is, nodes in the list while their ancestors are not) */
  simplified_list = gnobog_bookmarks_list_simplify (bookmarks, list);
                
  /* duplicate the nodes */
  duplicated_nodes_list = gnobog_bookmarks_list_duplicate (bookmarks, simplified_list);
  g_list_free (simplified_list);
        
  /* Then link them again */
  gnobog_bookmarks_list_link (bookmarks, real_anchor_node, mode, duplicated_nodes_list);
        
  /* Auto alias the newly created nodes */
  gnobog_bookmarks_list_auto_alias (bookmarks, duplicated_nodes_list);
        
  /* Sanity Check : We should NOT emit a 'list_created' signal with mode set to INSERT_DEFAULT_MODE */
  g_assert (mode != INSERT_DEFAULT_MODE);
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   anchor_node,
                   mode,
                   duplicated_nodes_list);

  g_list_free (duplicated_nodes_list);
}


void
gnobog_bookmarks_list_move (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksInsertMode mode, GList* list)
{
  GList*               simplified_list;
  GnobogBookmarksNode  real_anchor_node;
        
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (list != NULL);
        
  /* Handle null anchor */
  if (anchor_node == NULL) {
    real_anchor_node = bookmarks->node;
    mode = INSERT_INTO;
  } else {
    real_anchor_node = anchor_node;
  }
        
  /* Get Insert Default Mode explicitly */
  /* First operation to perform (before testing insertion) */
  if (mode == INSERT_DEFAULT_MODE)
    mode = gnobog_bookmarks_node_get_insert_default_mode (bookmarks, real_anchor_node);
        
  if (!gnome_bookmarks_list_is_insertion_valid (bookmarks, real_anchor_node, mode, list))
    return;
        
  /* Make a simplified copy of the list so as to work only on folders and
   * orphans (that is, nodes in the list while their ancestors are not) */
  simplified_list = gnobog_bookmarks_list_simplify (bookmarks, list);
                
  /* First unlink all nodes in list */
  gnobog_bookmarks_list_unlink (bookmarks, simplified_list);
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks), gnobog_bookmarks_signals[LIST_DELETED], simplified_list);
        
  /* FIXME : What if signals aren't catched in the good order (Multithread) ? */
        
  /* Then link them again */
  gnobog_bookmarks_list_link (bookmarks, real_anchor_node, mode, simplified_list);
        
  /* Sanity Check : We should NOT emit a 'list_created' signal with mode set to INSERT_DEFAULT_MODE */
  g_assert (mode != INSERT_DEFAULT_MODE);
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_CREATED],
                   anchor_node,
                   mode,
                   simplified_list);

  g_list_free (simplified_list);
}

void
gnobog_bookmarks_list_delete (GnobogBookmarks* bookmarks, GList* list)
{
  GList*  simplified_list;
        
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (list != NULL);

  /* Log a little message for debugging */
  gnobog_debug ("Deleting list of bookmarks");

  simplified_list = gnobog_bookmarks_list_simplify (bookmarks, list);
                
  /* First unlink nodes */
  gnobog_bookmarks_list_unlink (bookmarks, simplified_list);
        
  /* Then destroy nodes */
  gnobog_bookmarks_list_destroy (bookmarks, simplified_list);
        
  /* Tell everybody that the work has been done */
  gtk_signal_emit (GTK_OBJECT (bookmarks),
                   gnobog_bookmarks_signals[LIST_DELETED],
                   simplified_list);
        
  g_list_free (simplified_list);
}

void    
gnobog_bookmarks_clear (GnobogBookmarks* bookmarks)
{
  GList* list;
        
  g_return_if_fail (bookmarks != NULL);
        
  /* Log a little message for debugging */
  gnobog_debug ("Clearing Bookmarks Object : %s",
                (bookmarks->title == NULL) ? "(null)" : bookmarks->title);
        
  list = gnobog_bookmarks_get_list (bookmarks);
  gnobog_bookmarks_list_destroy (bookmarks, list);
  g_list_free (list);
        
  /* emit 'bookmarks_cleared' signal */
  gtk_signal_emit (GTK_OBJECT (bookmarks), gnobog_bookmarks_signals[BOOKMARKS_CLEARED]);
}

GList*
gnobog_bookmarks_get_redundant  (GnobogBookmarks* bookmarks)
{
  g_return_val_if_fail (bookmarks != NULL, NULL);
        
  return NULL;
}

gchar*
gnobog_bookmarks_node_get_path  (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{
  GnobogBookmarksNode iterator;
  GnobogBookmarksNode parent;
  gchar* new_path;
  gchar* current_path;

  g_return_val_if_fail (bookmarks != NULL, NULL);
  g_return_val_if_fail (node != NULL, NULL);
        
  current_path = g_strdup ("");
  iterator = node;
  while ((parent = g_node_parent (iterator)) != bookmarks->node) {
    new_path = g_strjoin ("/", node_get_name (parent), current_path, NULL);
    g_free (current_path);
    current_path = new_path;
    iterator = parent;
  }
                
  return current_path;
}

static void
gnobog_execute_command_line (const gchar* commandline)
{ /* This code is from gnome_url_show () in libgnome */
  int     argc;
  char**  argv;

  /* we use a popt function as it does exactly what we want to do and
   * gnome already uses popt */
  if (poptParseArgvString (commandline, &argc, &argv) != 0) {
    /* can't parse */
    g_warning ("Parse error of '%s'", commandline);
    return;
  }

  /* use execute async, and not the shell, shell is evil and a
   * security hole */
  gnome_execute_async (NULL, argc, argv);

  /* the way the poptParseArgvString works is that the entire thing
  * is allocated as one buffer, so just free will suffice, also
  * it must be free and not g_free */
  free(argv);
}


/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_netscape (gchar* location,
                               gboolean open_new_window)
{
  gchar* commandline;

  /* FIXME : Must we use gome-moz-remote ?? */
  if (open_new_window == TRUE) {
    commandline = g_strdup_printf (NETSCAPE_IN_NEW_WINDOW_CL, location);
  } else {
    commandline = g_strdup_printf (NETSCAPE_IN_EXISTING_WINDOW_CL, location);
  }
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_mozilla (gchar* location,
                              gboolean open_new_window)
{
  gchar* commandline;

  /* FIXME : Don't work at all if mozilla is not running */
  if (open_new_window == TRUE) {
    commandline = g_strdup_printf (MOZILLA_IN_NEW_WINDOW_CL, location);
  } else {
    commandline = g_strdup_printf (MOZILLA_IN_EXISTING_WINDOW_CL, location);
  }
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_galeon (gchar* location,
                             gboolean open_new_window)
{
  gchar* commandline;

  /* FIXME : I can't test these... */  
  if (open_new_window == TRUE) {
    commandline = g_strdup_printf (GALEON_IN_NEW_WINDOW_CL, location);
  } else {
    commandline = g_strdup_printf (GALEON_IN_EXISTING_WINDOW_CL, location);
  }
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_gnome_handler (gchar* location,
                                    gboolean open_new_window)
{
  /* Use what is configured in Gnome Control Center */
  /* FIXME : Does it really work ? */
  gnome_url_show (location);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_nautilus (gchar* location,
                               gboolean open_new_window)
{
  gchar* commandline;

  /* FIXME : I can't test these... */  
  if (open_new_window == TRUE) {
    commandline = g_strdup_printf (NAUTILUS_IN_NEW_WINDOW_CL, location);
  } else {
    commandline = g_strdup_printf (NAUTILUS_IN_EXISTING_WINDOW_CL, location);
  }
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_konqueror (gchar* location,
                                gboolean open_new_window)
{
  gchar* commandline;

  /* FIXME : Reuse of existing windows doesn't seem possible with Konqueror */
  commandline = g_strdup_printf (KONQUEROR_IN_NEW_WINDOW_CL, location);
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_opera (gchar* location,
                            gboolean open_new_window)
{
  gchar* commandline;

  /* FIXME : Nothing works with Opera 4.0 Beta 1 */
  if (open_new_window == TRUE) {
    commandline = g_strdup_printf (OPERA_IN_NEW_WINDOW_CL, location);
  } else {
    commandline = g_strdup_printf (OPERA_IN_EXISTING_WINDOW_CL, location);
  }
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_lynx (gchar* location,
                           gboolean open_new_window)
{
  gchar* commandline;

  commandline = g_strdup_printf (LYNX_IN_NEW_WINDOW_CL, location);
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_links (gchar* location,
                            gboolean open_new_window)
{
  gchar* commandline;

  commandline = g_strdup_printf (LINKS_IN_NEW_WINDOW_CL, location);
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_gftp (gchar* location,
                           gboolean open_new_window)
{
  gchar* commandline;

  commandline = g_strdup_printf (GFTP_IN_NEW_WINDOW_CL, location);
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_ncftp (gchar* location,
                            gboolean open_new_window)
{
  gchar* commandline;

  commandline = g_strdup_printf (NCFTP_IN_NEW_WINDOW_CL, location);
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

/* Do not call directly, use gnobog_bookmarks_node_view_in_browser () */
static void
gnobog_url_browse_in_lftp (gchar* location,
                           gboolean open_new_window)
{
  gchar* commandline;

  commandline = g_strdup_printf (LFTP_IN_NEW_WINDOW_CL, location);
  gnobog_execute_command_line (commandline);
  g_free (commandline);
}

void
gnobog_bookmarks_node_view_in_browser (GnobogBookmarks* bookmarks,
                                       GnobogBookmarksNode node,
                                       GnobogBrowser browser,
                                       gboolean open_new_window)
{
  gchar* location;

  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);

  /* Store current timestamp */
  node_touch_visit_time (node);
                
  /* Get location */
  location = node_get_location (node);

  if (node_get_type (node) == BOOKMARK) {
    switch (browser) {
    case DEFAULT_BROWSER:
      /* FIXME : Must be configurable but where ? */
      g_message ("Using default browser");
      browser = NETSCAPE_BROWSER;
    case NETSCAPE_BROWSER:
      g_message ("Launch URL in Netscape : %s", location);
      gnobog_url_browse_in_netscape (location, open_new_window);
      break;
    case MOZILLA_BROWSER:
      g_message ("Launch URL in Mozilla : %s", location);
      gnobog_url_browse_in_mozilla (location, open_new_window);
      break;
    case GALEON_BROWSER:
      g_message ("Launch URL in Galeon : %s", location);
      gnobog_url_browse_in_galeon (location, open_new_window);
      break;
    case GNOME_HANDLER_BROWSER:
      g_message ("Launch URL in Gnome URL Handler : %s", location);
      gnobog_url_browse_in_gnome_handler (location, open_new_window);
      break;
    case NAUTILUS_BROWSER:
      g_message ("Launch URL in Nautilus : %s", location);
      gnobog_url_browse_in_nautilus (location, open_new_window);
      break;
    case KONQUEROR_BROWSER:
      g_message ("Launch URL in Konqueror : %s", location);
      gnobog_url_browse_in_konqueror (location, open_new_window);
      break;
    case OPERA_BROWSER:
      g_message ("Launch URL in Opera : %s", location);
      gnobog_url_browse_in_opera (location, open_new_window);
      break;
    case LYNX_BROWSER:
      g_message ("Launch URL in Lynx : %s", location);
      gnobog_url_browse_in_lynx (location, open_new_window);
      break;
    case LINKS_BROWSER:
      g_message ("Launch URL in Links : %s", location);
      gnobog_url_browse_in_links (location, open_new_window);
      break;
    case GFTP_BROWSER:
      g_message ("Launch URL in gFTP : %s", location);
      gnobog_url_browse_in_gftp (location, open_new_window);
      break;
    case NCFTP_BROWSER:
      g_message ("Launch URL in NcFTP : %s", location);
      gnobog_url_browse_in_ncftp (location, open_new_window);
      break;
    case LFTP_BROWSER:
      g_message ("Launch URL in LFTP : %s", location);
      gnobog_url_browse_in_lftp (location, open_new_window);
      break;
    }
  } else {
    g_message ("You can only browse bookmarks...");
  }
        
  return;
}

/* Unref the node from the hash tables used in alias management */
static void
gnobog_bookmarks_node_unalias (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{       
  GHashTable*           alias_hash_table;
  GHashTable*           name_hash_table;
  GnobogBookmarksEntry  entry;
  GList*                alias_list;
  gchar*                orig_name;
  gchar*                orig_location;
  gboolean              found;

  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);
  
  /* If node is a FOLDER, return.
   * We don't do aliasing on FOLDER or TITLE */
  if ( (node_get_type (node) == FOLDER)
       || (node_get_type (node) == TITLE) ) return ;
        
  /* We *ARE* aliasing SEPARATOR, for now */
        
  alias_hash_table = bookmarks->alias_hash_table;
        
  /* First lookup in alias_hash_table with the node's location
   * to get the name_hash_table associated
   * If location is not in alias_hash_table, it's a bug :-) */
  found = g_hash_table_lookup_extended
                      (alias_hash_table,
                       node_get_location (node),
                       (gpointer*) &orig_location,
                       (gpointer*) &name_hash_table);
  
  if (found == FALSE) {
    /* Ok, ok, ok, there is a bug, no need to shout it out loudly.
     * Display something nice and return as if nothing had happened :-) */
    g_warning ("Theorically, this node %s %s *should* be in alias_hash_table. Too bad it wasn't",
               (node_get_name (node) == NULL) ? "(null)" : node_get_name (node),
               (node_get_location (node) == NULL) ? "(null)" : node_get_location (node));
    return;
  } else { /* This location is in alias_hash_table. Everything is under control :-) */
                
    /* Get back the entry associated with the node's name */
   found = g_hash_table_lookup_extended
                       (name_hash_table,
                        node_get_name (node),
                        (gpointer*) &orig_name,
                        (gpointer*) &entry);
    if (found == FALSE) {
      /* The node's name isn't in name_hashtable table :-( */
      /* Ok, ok, ok, there is a bug, no need to shout it out loudly.
       * Display something nice and return as if nothing had happened :-) */
      g_warning ("Theorically, this node %s %s *should* in name_hash_table. Too bad it wasn't",
                 node_get_name (node), node_get_location (node));
      return;
    } else { /* This name is in name_hash_table. Everything is under control :-) */
      alias_list = entry->alias_list;
      /* Check if this node is aliased */
      if (g_list_find (alias_list, node) == NULL) {
        /* This node is not aliased :-( */
        /* Ok, ok, ok, there is a bug, no need to shout it out loudly.
         * Display something nice and return as if nothing had happened :-) */
        g_warning ("This node %s %s is not aliased. Too bad it wasn't",
                   node_get_name (node), node_get_location (node));
        return;
      } else {
        g_hash_table_remove (name_hash_table,
                             node_get_name (node));
        g_free (orig_name);
        if (g_hash_table_size (name_hash_table) == 0) {
          /* No other node use this URL, destroy name_hash_table and clear alias_hash_table */
          g_hash_table_destroy (name_hash_table);
          g_hash_table_remove (alias_hash_table,
                               node_get_location (node));
          g_free (orig_location);
        }
      }
    }
  }

  return;
}

/* FIXME : This function has a bad prototype
 * TODO : Transform if into two functions :
 *         - a wrapper called by g_node_traverse ()
 *         - and a a function with a good prototype */
static gboolean
gnobog_bookmarks_node_auto_alias_entry (GnobogBookmarksNode node, gpointer data)
{       
  GHashTable*           alias_hash_table;
  GHashTable*           name_hash_table;
  GnobogBookmarksEntry  entry;
  GnobogBookmarksNode   node_to_alias;
  GnobogBookmarksEntry  entry_to_alias;
  GList*                list_of_node_to_alias;
  GList*                alias_list;
  GList*                iterator;

  /* If node is a FOLDER or TITLE, return FALSE so that g_node_traverse ()
   * which is calling us, keep going on.
   * We don't do aliasing on FOLDER or TITLE */
  if ( (node_get_type (node) == FOLDER)
       || (node_get_type (node) == TITLE) ) return FALSE;
        
  /* We *ARE* aliasing SEPARATOR, for now */
        
  alias_hash_table = data;
        
  /* First lookup in alias_hash_table if the node's location is in it,
   * and if it is in it, we get back the hash table containing all the names
   * associated with this location */
  name_hash_table = g_hash_table_lookup (alias_hash_table, node_get_location (node));

  if (name_hash_table == NULL) {
    /* This location isn't in alias_hash_table, so add it */

    name_hash_table = g_hash_table_new (g_str_hash, g_str_equal);
    g_hash_table_insert (name_hash_table, g_strdup (node_get_name (node)), node->data);
    g_hash_table_insert (alias_hash_table, g_strdup (node_get_location (node)), name_hash_table);
  } else {
    /* This location is in alias_hash_table */
                
    /* Get back the entry associated with the node's name */
    entry = g_hash_table_lookup (name_hash_table, node_get_name (node));
    if (entry == NULL) {
      /* The node's name isn't in name_hash_table table, add it */

      g_hash_table_insert (name_hash_table, g_strdup (node_get_name (node)), node->data);
    } else {
      /* This name is in name_hash_table */

      alias_list = entry->alias_list;

      /* Check if this node has not already been aliased */
      if (g_list_find (alias_list, node) == NULL) {
        entry_to_alias = node->data;
        list_of_node_to_alias = node_get_alias_list (node);
        /* Sanity Check */
        if (g_list_find (list_of_node_to_alias, node) == NULL)
          g_message ("Bug! Node should be in alias list");

        iterator = list_of_node_to_alias;
        do {
          node_to_alias = iterator->data;
          if (g_list_find (alias_list, node_to_alias) == NULL) {
            alias_list = g_list_prepend (alias_list, node_to_alias);
            node_to_alias->data = entry;
          } else {
            gnobog_debug ("Something bad has been avoided");
          }
        } while ((iterator = g_list_next (iterator)) != NULL);
        entry->alias_list = alias_list;
        gnobog_bookmarks_entry_destroy (entry_to_alias);
      }
    }
  }

  /* Return FALSE so that g_node_traverse() which is calling us, keep going on */
  return FALSE;
}

static void
gnobog_bookmarks_node_auto_alias (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);

  /* Run through node's descendants and auto alias everybody */
  g_node_traverse (node,
                   G_PRE_ORDER,
                   G_TRAVERSE_LEAFS,
                   -1,
                   &gnobog_bookmarks_node_auto_alias_entry,
                   bookmarks->alias_hash_table);
}

static void
gnobog_bookmarks_list_auto_alias (GnobogBookmarks* bookmarks,  GList* list)
{
  GList*               iterator;
  GnobogBookmarksNode  node;   
        
  g_return_if_fail (bookmarks != NULL);

  if (list == NULL) return;

  /* Run through list and destroy each node */
  iterator = list;
  do {
    node = iterator->data;
    gnobog_bookmarks_node_auto_alias (bookmarks, node);
  } while ((iterator = g_list_next (iterator)) != NULL);        
}

/*
** Private methods
*/

GnobogBookmarksEntry
gnobog_bookmarks_entry_new (GnobogBookmarksNodeType type,
                            gchar* name,
                            gchar* location,
                            gchar* description)
{
  GnobogBookmarksEntry  new_entry;

  new_entry               = g_new (struct _GnobogBookmarksEntry, 1);
  new_entry->type         = type;
  new_entry->name         = g_strdup (name);
  new_entry->location     = g_strdup (location);
  new_entry->description  = g_strdup (description);
        
  new_entry->created      = time (NULL);
  new_entry->visited      = 0;
  new_entry->modified     = 0;
        
  /* Folder State */
  new_entry->open         = FALSE;
        
  /* Alias list */
  new_entry->alias_list   = NULL;
        
  return new_entry;
}

GnobogBookmarksEntry
gnobog_bookmarks_entry_import (GnobogBookmarksNodeType type,
                               gchar* name,
                               gchar* location,
                               gchar* description,
                               guint created,
                               guint visited,
                               guint modified,
                               gboolean open)
{
  GnobogBookmarksEntry new_entry;

  new_entry               = g_new (struct _GnobogBookmarksEntry, 1);
  new_entry->type         = type;
  new_entry->name         = g_strdup (name);
  new_entry->location     = g_strdup (location);
  new_entry->description  = g_strdup (description);
  new_entry->created      = created;
  new_entry->visited      = visited;
  new_entry->modified     = modified;
        
  /* Folder State */
  new_entry->open         = open;
        
  /* Alias list */
  new_entry->alias_list   = NULL;
        
  return new_entry;
}

GnobogBookmarksEntry
gnobog_bookmarks_entry_duplicate (GnobogBookmarksEntry entry)
{
  GnobogBookmarksEntry  new_entry;

  g_return_val_if_fail (entry != NULL, NULL);
        
  new_entry               = g_new (struct _GnobogBookmarksEntry, 1);
  new_entry->type         = entry->type;
  new_entry->name         = g_strdup (entry->name);
  new_entry->location     = g_strdup (entry->location);
  new_entry->description  = g_strdup (entry->description);
  new_entry->created      = entry->created;
  new_entry->visited      = entry->visited;
  new_entry->modified     = entry->modified;      
        
  /* Folder State */
  new_entry->open         = entry->open;
        
  /* Alias list */
  new_entry->alias_list   = NULL;
        
  return new_entry;
}

void
gnobog_bookmarks_entry_destroy (GnobogBookmarksEntry entry)
{
  g_return_if_fail (entry != NULL);
        
  g_free (entry->name);
  g_free (entry->location);
  g_free (entry->description);
  g_list_free (entry->alias_list);
  g_free (entry);
}

GnobogBookmarksInsertMode
gnobog_bookmarks_node_get_insert_default_mode
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode node)
{
  GnobogBookmarksInsertMode  mode;
        
  /* Initialize with something bad to trigger assert if something goes wrong */
  mode = INSERT_DEFAULT_MODE;
        
  if (node == NULL) return INSERT_INTO; /* insert into root : invert the list */
        
  switch (gnobog_bookmarks_node_get_type (node)) {
  case TITLE:
    mode = bookmarks->folder_insert_default_mode;
    break;
  case FOLDER:
    mode = bookmarks->folder_insert_default_mode;
    break;
  case BOOKMARK:
    mode = bookmarks->insert_default_mode;
    break;
  case SEPARATOR:
    mode = bookmarks->insert_default_mode;
    break;
  default:
    /* If we are here it's not good at all ;-) */
    g_assert_not_reached();
    break;
  }
        
  /* Sanity Check : Never return INSERT_DEFAULT_MODE */
  g_assert (mode != INSERT_DEFAULT_MODE);
        
  return mode;  
}

gboolean
gnome_bookmarks_list_is_insertion_valid
               (GnobogBookmarks* bookmarks,
                GnobogBookmarksNode anchor_node,
                GnobogBookmarksInsertMode mode,
                GList* list)
{
  GnobogBookmarksNodeType  type;
  GList*                   iterator;
  GnobogBookmarksNode      node;

  g_return_val_if_fail (anchor_node != NULL, FALSE);
  g_return_val_if_fail (mode != INSERT_DEFAULT_MODE, FALSE);
        
  type = node_get_type (anchor_node);
  switch (mode) {
  case INSERT_AFTER:
  case INSERT_BEFORE:
    if (type == TITLE) {
      /* Log a little message for debugging */
      gnobog_debug ("Link After/Before is not allowed on TITLE");
      return FALSE;
    }                   
    break;
  case INSERT_INTO:
    if ((type != FOLDER) && (type != TITLE)) {
      /* Log a little message for debugging */
      gnobog_debug ("Link Into can only be performed on TITLE and FOLDER");
      return FALSE;
    }
    break;
  default:
    /* If we are here it's not good at all ;-) */
    g_assert_not_reached();
    break;
  }
        
  /* Check that operation isn't absurd */
  iterator = list;
  while (iterator != NULL) {
    node = iterator->data;
    if ((node == anchor_node) || (g_node_is_ancestor (node, anchor_node)))
      {
        /* Log a little message for debugging */
        gnobog_debug ("Impossible to perform move operation");
        return FALSE;
      }
    iterator = g_list_next (iterator);  
  };    
        
  return TRUE;
}

void
gnobog_bookmarks_node_link_after
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode anchor_node,
                 GnobogBookmarksNode node)
{
  GnobogBookmarksNode  node_parent;
  GnobogBookmarksNode  node_sibling;
        
  /* Get parent of anchor node */
  node_parent = g_node_parent (anchor_node);
        
  /* Get sibling of anchor node
   * In fact to paste after, we insert before the next sibling
   * FIXME : And if there is no next sibling ????
   *         In fact, in this case g_node_next_sibling return NULL,
   *         so everything goes like we want ;-) */
  /* WARNING : sibling changes each time we insert a node...
   * So we have to traverse the list in reverse order */
  node_sibling = g_node_next_sibling (anchor_node);
        
  /* Log a little message for debugging */
  gnobog_debug ("Link %s After Bookmark Node : %s, that is Before : %s",
                gnobog_bookmarks_node_get_name (node),
                gnobog_bookmarks_node_get_name (anchor_node),
                gnobog_bookmarks_node_get_name (node_sibling));
   
  /* Link node */
  g_node_insert_before (node_parent, node_sibling, node);
        
  /* Tag bookmarks as 'modified' */
  bookmarks->modified = TRUE;
}

void
gnobog_bookmarks_node_link_before
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode anchor_node,
                 GnobogBookmarksNode node)
{
  GnobogBookmarksNode  node_parent;
  GnobogBookmarksNode  node_sibling;
        
  /* Log a little message for debugging */
  gnobog_debug ("Link %s Before Bookmark Node : %s",
                gnobog_bookmarks_node_get_name (node),
                gnobog_bookmarks_node_get_name (anchor_node));
        
  /* Get parent of anchor node */
  node_parent = g_node_parent (anchor_node);
        
  /* Get sibling of anchor node
   * In fact to paste after, we insert before the next sibling
   * FIXME : And if there is no next sibling ????
   *         In fact, in this case g_node_next_sibling return NULL,
   *         so everything goes like we want ;-) */
  node_sibling = anchor_node;
        
  /* Link node */
  g_node_insert_before (node_parent, node_sibling, node);
        
  /* Tag bookmarks as 'modified' */
  bookmarks->modified = TRUE;
}

/* Beware when anchor_node is null, for example with an empty bookmarks object */
/* Test if bookmarks->node is null */
void
gnobog_bookmarks_node_link_into
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode anchor_node,
                 GnobogBookmarksNode node)
{
  GnobogBookmarksNode  node_parent;
  GnobogBookmarksNode  node_sibling;
  GnobogBookmarksNode  new_node;
        
  /* Log a little message for debugging */
  gnobog_debug ("Link %s Into Bookmark Node : %s",
                gnobog_bookmarks_node_get_name (node),
                gnobog_bookmarks_node_get_name (anchor_node));
        
  /* Get parent of anchor node */
  node_parent = anchor_node;
        
  /* Get sibling of anchor node
   * In fact to paste after, we insert before the next sibling
   * FIXME : And if there is no next sibling ????
   *         In fact, in this case g_node_next_sibling return NULL,
   *         so everything goes like we want ;-) */
  node_sibling = g_node_first_child (anchor_node);
        
  /* Link node */
  new_node = g_node_insert_before (node_parent, node_sibling, node);
        
  /* Tag bookmarks as 'modified' */
  bookmarks->modified = TRUE;
}

void
gnobog_bookmarks_node_link
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode anchor_node,
                 GnobogBookmarksInsertMode mode,
                 GnobogBookmarksNode node)
{
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);
  g_return_if_fail (mode != INSERT_DEFAULT_MODE);
        
  if (anchor_node == NULL) {
    gnobog_bookmarks_node_link_into (bookmarks, bookmarks->node, node);
  } else {
    switch (mode) {
    case INSERT_AFTER:
      gnobog_bookmarks_node_link_after (bookmarks, anchor_node, node);
      break;
    case INSERT_BEFORE:
      gnobog_bookmarks_node_link_before (bookmarks, anchor_node, node);
      break;
    case INSERT_INTO:
      gnobog_bookmarks_node_link_into (bookmarks, anchor_node, node);
      break;
    default:
      /* If we are here it's not good at all ;-) */
      g_assert_not_reached();
      break;
    }
  }
}

void
gnobog_bookmarks_list_link
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode anchor_node,
                 GnobogBookmarksInsertMode mode, 
                 GList* list)
{
  GnobogBookmarksNode  node;
  GList*               iterator;       
        
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (list != NULL);
                
  /* Run through list and link each node */
  /* 'Insert after' inserts before the next sibling, which
   * changes after each insertion, so we have to travese backwards.
   * 'Insert into' performs the same kind of operation with first child. */
  if (mode == INSERT_AFTER || mode == INSERT_INTO) {
    iterator = g_list_last (list);
    do {
      node = iterator->data;
      gnobog_bookmarks_node_link (bookmarks, anchor_node, mode, node);
    } while ((iterator = g_list_previous (iterator)) != NULL);  
  } else {
    /* Let's start with the first element :-) */
    iterator = list;
    do {
      node = iterator->data;
      gnobog_bookmarks_node_link (bookmarks, anchor_node, mode, node);
    } while ((iterator = g_list_next (iterator)) != NULL);      
  }
}

void
gnobog_bookmarks_node_update
                (GnobogBookmarks* bookmarks,
                 GnobogBookmarksNode node,
                 gchar* name,
                 gchar* location,
                 gchar* description)
{
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);
                
  /* If no modifications to do, return */
  if ( (name == NULL)
       || (location == NULL)
       || (description == NULL) ) return;
  
  /* Store current timestamp */
  node_touch_modification_time (node);  
        
  /* Before modifying name and location, unalias node */
  gnobog_bookmarks_node_unalias (bookmarks, node);

  /* Change 'name' if is given */
  if (name != NULL) {
    g_message("Updating name (%s) to (%s)", node_get_name (node), name);
    g_free (node_get_name (node));
    node_set_name (node, g_strdup (name)); 
  }
                
  /* Change 'location' if is given */
  if (location != NULL) {
    g_message("Updating location (%s) to (%s)", node_get_location (node), location);
    g_free (node_get_location (node));
    node_set_location (node, g_strdup (location)); 
  }
                
  /* Change 'description' if is given */
  if (description != NULL) {
    g_message("Updating description (%s) to (%s)", node_get_description (node), description);
    g_free (node_get_description (node));
    node_set_description (node, g_strdup (description));   
  }

  /* Tag bookmarks as 'modified' */
  bookmarks->modified = TRUE;

  /* Realias node */
  gnobog_bookmarks_node_auto_alias (bookmarks, node);
}

/* Just a wrapper to gnobog_bookmarks_entry_duplicate()
   used in gnobog_bookmarks_node_duplicate() */
static GNode*
gnobog_bookmarks_node_duplicate_entry (GnobogBookmarksNode node, gpointer data)
{
  GnobogBookmarksNode   new_node;       
  GnobogBookmarksEntry  entry;
  GList*                alias_list;

  entry = gnobog_bookmarks_entry_duplicate ((GnobogBookmarksEntry) (node->data));
  new_node = g_node_new (entry);
        
  /* Create alias_list and store it in node's entry */
  alias_list = g_list_prepend (NULL, new_node);
  node_set_alias_list (new_node, alias_list);

  return new_node;
}

GnobogBookmarksNode
gnobog_bookmarks_node_duplicate (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{
  GnobogBookmarksNode  new_node;
        
  g_return_val_if_fail (node != NULL, NULL);    

  /* Log a little message for debugging */
  gnobog_debug ("Duplicate Bookmark Node : %s",
                gnobog_bookmarks_node_get_name (node));
       
  /* Run through node's descendants and destroy everybody */
  new_node = g_node_copy (node,
                          (GNodeCopyData) &gnobog_bookmarks_node_duplicate_entry,
                          NULL);
                
  return new_node;
}

GList*
gnobog_bookmarks_list_duplicate (GnobogBookmarks* bookmarks, GList* list)
{
  GList*               new_list;
  GList*               iterator;
  GnobogBookmarksNode  new_node;       
  GnobogBookmarksNode  node;   
        
  g_return_val_if_fail (list != NULL, NULL);    

  /* Run through list, duplicate each node and build new list */
  new_list = NULL;
  iterator = list;
  do {
    node = iterator->data;
    new_node = gnobog_bookmarks_node_duplicate (bookmarks, node);
    new_list = g_list_prepend (new_list, new_node);
  } while ((iterator = g_list_next (iterator)) != NULL);
  /* Reverse new list as it was build with prepend */
  new_list = g_list_reverse (new_list);
        
  return new_list;
}

void
gnobog_bookmarks_node_unlink (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);
        
  /* Log a little message for debugging */
  gnobog_debug ("Unlink Bookmark Node : %s",
                gnobog_bookmarks_node_get_name (node));
        
  /* Unlink node */
  g_node_unlink (node);
        
  /* Tag bookmarks as 'modified' */
  bookmarks->modified = TRUE;
}

void
gnobog_bookmarks_list_unlink (GnobogBookmarks* bookmarks,  GList* list)
{
  GList*               iterator;       
  GnobogBookmarksNode  node;   
        
  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (list != NULL);
                
  /* Run through list and unlink each node */
  iterator = list;
  do {
    node = iterator->data;
    gnobog_bookmarks_node_unlink (bookmarks, node);
  } while ((iterator = g_list_next (iterator)) != NULL);        
        
  /* Tag bookmarks as 'modified' */
  bookmarks->modified = TRUE;
}

void
gnobog_bookmarks_node_remove_alias (GnobogBookmarksNode node)
{       
  GList*  alias_list;

  alias_list = node_get_alias_list (node);
  alias_list = g_list_remove (alias_list, node);
  node_set_alias_list (node, alias_list);
}

/* Just a wrapper to gnobog_bookmarks_entry_destroy()
   used in gnobog_bookmarks_node_destroy() */
static gboolean
gnobog_bookmarks_node_destroy_entry (GnobogBookmarksNode node, gpointer data)
{       
  GnobogBookmarks*  bookmarks;

  bookmarks = GNOBOG_BOOKMARKS (data);

  if (gnobog_bookmarks_node_have_aliases (node) == TRUE) {
    gnobog_bookmarks_node_remove_alias (node);
  } else {
    gnobog_bookmarks_node_unalias (bookmarks, node);
    gnobog_bookmarks_entry_destroy ((GnobogBookmarksEntry) (node->data));
  }
  /* Return FALSE so that g_node_traverse() which is calling us, keep going on */
  return FALSE;
}

void
gnobog_bookmarks_node_destroy (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{
  gchar*  name;

  g_return_if_fail (bookmarks != NULL);
  g_return_if_fail (node != NULL);      

  /* Log a little message for debugging */
  name = gnobog_bookmarks_node_get_name (node);
  gnobog_debug ("Destroy Bookmark Node : %s",
                (name == NULL) ? "(null)" : name);
        
  /* Run through node's descendants and destroy everybody */
  g_node_traverse (node,
                   G_PRE_ORDER,
                   G_TRAVERSE_ALL,
                   -1,
                   &gnobog_bookmarks_node_destroy_entry,
                   bookmarks);
        
  /* Destroy node */
  g_node_destroy (node);
}

void
gnobog_bookmarks_list_destroy (GnobogBookmarks* bookmarks,  GList* list)
{
  GList*               iterator;
  GnobogBookmarksNode  node;   
        
  g_return_if_fail (bookmarks != NULL);

  if (list == NULL) return;

  /* Run through list and destroy each node */
  iterator = list;
  do {
    node = iterator->data;
    gnobog_bookmarks_node_destroy (bookmarks, node);
  } while ((iterator = g_list_next (iterator)) != NULL);        
}

/* Note : if we suppose the list of nodes is ordered, we can
 * optimize this function (since folders will always preceed or
 * succeed their descendant, whether the list is ordered forwards
 * or backwards).
 * For now it doesn't make any assumption on the list. */
GList*
gnobog_bookmarks_list_simplify (GnobogBookmarks* bookmarks, GList* list)
{
  GList*                   purified_list;
  GList*                   folders_list;
  GList*                   list_iterator;
  GList*                   folders_list_iterator;
  gboolean                 ancestor_exists;
  GnobogBookmarksNode      bookmarks_node;
  GnobogBookmarksNode      folders_node;
  GnobogBookmarksNodeType  type;
                
  g_return_val_if_fail (bookmarks != NULL, NULL);
  g_return_val_if_fail (list != NULL, NULL);
  g_return_val_if_fail (GNOBOG_IS_BOOKMARKS(bookmarks), NULL);

  purified_list = NULL;
  folders_list = NULL;
  list_iterator = NULL;
  folders_list_iterator = NULL;
        
  /* build folders list */
  list_iterator = list;
  do {
    bookmarks_node = (GnobogBookmarksNode) list_iterator->data;
    type = node_get_type (bookmarks_node);
    if (type == FOLDER) {
      folders_list = g_list_prepend (folders_list, bookmarks_node);
    }
  } while ( (list_iterator = g_list_next(list_iterator)) != NULL );

  if (folders_list == NULL) {
    purified_list = g_list_copy (list);
    return purified_list;
  }
        
  /* Now traverse the list to simplify and build the purified list */
  list_iterator = g_list_last (list); /* should already be the case */
  do {
    bookmarks_node = (GnobogBookmarksNode) list_iterator->data;
    ancestor_exists = FALSE;
    /* traverse folders list and add current node to purified list */
    /* only if it has no ancestor */
    folders_list_iterator = folders_list;
    do {
      folders_node = (GnobogBookmarksNode) folders_list_iterator->data;
      if ( /*(folders_node != bookmarks_node)
             && (*/ g_node_is_ancestor(folders_node, bookmarks_node) /*)*/) {
        ancestor_exists = TRUE;
      } 
    } while ( (folders_list_iterator = g_list_next(folders_list_iterator)) != NULL );
    if (!ancestor_exists) {     
      purified_list = g_list_prepend (purified_list, bookmarks_node);
    }
  } while ( (list_iterator = g_list_previous(list_iterator)) != NULL );
        

  /* Check !! */
  list_iterator = purified_list;
  do {
    bookmarks_node = (GnobogBookmarksNode) list_iterator->data;
    type = node_get_type (bookmarks_node);
    if (type == FOLDER) {
      g_message ("Purified list folder : %s", gnobog_bookmarks_node_get_name (bookmarks_node) );
    } else {
      g_message ("Purified list bookmarks : %s", gnobog_bookmarks_node_get_name (bookmarks_node) );             
    }
  } while ( (list_iterator = g_list_next(list_iterator)) != NULL);
                
  return purified_list;
}


GList*  
gnobog_bookmarks_node_get_child_list (GnobogBookmarks* bookmarks, GnobogBookmarksNode node)
{
  GList*               child_list;
  GnobogBookmarksNode  node_child;

  g_return_val_if_fail (bookmarks != NULL, NULL);
  g_return_val_if_fail (GNOBOG_IS_BOOKMARKS(bookmarks), NULL);
        
  /* Handle null node */
  if (node == NULL) {
    node = bookmarks->node;
  }
        
  child_list = NULL;
        
  /* build the list of 1st level nodes */
  node_child = g_node_first_child (node);
  if (node_child == NULL) return NULL;
  do {
    child_list = g_list_prepend (child_list, node_child);
  } while ( (node_child = g_node_next_sibling (node_child)) != NULL );
  child_list = g_list_reverse (child_list);
        
  return child_list;
}

GList*  
gnobog_bookmarks_get_list (GnobogBookmarks* bookmarks)
{
  GList*               list_first_level_nodes;
  GnobogBookmarksNode  bookmarks_node;

  g_return_val_if_fail (bookmarks != NULL, NULL);
  g_return_val_if_fail (GNOBOG_IS_BOOKMARKS(bookmarks), NULL);
        
  list_first_level_nodes = NULL;
        
  /* build the list of 1st level nodes */
  bookmarks_node = g_node_first_child (bookmarks->node);
  while ( bookmarks_node != NULL ){
    g_message ("%s", node_get_name(bookmarks_node));
    list_first_level_nodes = g_list_prepend (list_first_level_nodes, bookmarks_node);
    bookmarks_node = g_node_next_sibling (bookmarks_node);
  }
  list_first_level_nodes = g_list_reverse (list_first_level_nodes);
        
  return list_first_level_nodes;
}

/* might be used as a generalized version of _get_list */
GList*  
gnobog_bookmarks_get_folder_content (GnobogBookmarks* bookmarks, GnobogBookmarksNode bookmarks_node)
{
  GList*               list_children;
  GnobogBookmarksNode  child;

  g_return_val_if_fail (bookmarks != NULL, NULL);
  g_return_val_if_fail (GNOBOG_IS_BOOKMARKS(bookmarks), NULL);
        
  list_children = NULL;
        
  /* build the list of nodes */
  if (bookmarks_node == NULL) {
    child = g_node_first_child (bookmarks->node);
  } else {
    child = g_node_first_child (bookmarks_node);
  }
  while (child != NULL) {
    list_children = g_list_prepend (list_children, child);
    child = g_node_next_sibling (child);
  }
  list_children = g_list_reverse (list_children);
        
  return list_children;
}

