/* Target-dependent code for the ALPHA architecture, for GDB, the GNU Debugger.
   Copyright 1993, 1994, 1995 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "frame.h"
#include "inferior.h"
#include "symtab.h"
#include "value.h"
#include "gdbcmd.h"
#include "gdbcore.h"
#include "dis-asm.h"
#include "symfile.h"
#include "objfiles.h"
#include "gdb_string.h"
#include <pdscdef.h>
#include <psigdef.h>

/* FIXME: Some of this code should perhaps be merged with mips-tdep.c.  */

/* FIXME: Put this declaration in frame.h.  */
extern struct obstack frame_cache_obstack;

extern int step_over_calls;


/* Forward declarations.  */

static CORE_ADDR read_next_frame_reg PARAMS ((struct frame_info *, int));

static CORE_ADDR heuristic_proc_start PARAMS ((CORE_ADDR));

static void reinit_frame_cache_sfunc PARAMS ((char *, int,
					      struct cmd_list_element *));

static CORE_ADDR after_prologue PARAMS ((CORE_ADDR pc,
					 alpha_extra_func_info_t proc_desc));

static int alpha_in_prologue PARAMS ((CORE_ADDR pc,
				alpha_extra_func_info_t proc_desc));

/* Heuristic_proc_start may hunt through the text section for a long
   time across a 2400 baud serial line.  Allows the user to limit this
   search.  */
static unsigned int heuristic_fence_post = 0;

/* Layout of a stack frame on the alpha:

                |				|
 pdr members:	|  7th ... nth arg,		|
                |  `pushed' by caller.		|
                |				|
----------------|-------------------------------|<--  old_sp == vfp
   ^  ^  ^  ^	|				|
   |  |  |  |	|				|
   |  |localoff	|  Copies of 1st .. 6th		|
   |  |  |  |	|  argument if necessary.	|
   |  |  |  v	|				|
   |  |  |  ---	|-------------------------------|<-- FRAME_LOCALS_ADDRESS
   |  |  |      |				|
   |  |  |      |  Locals and temporaries.	|
   |  |  |      |				|
   |  |  |      |-------------------------------|
   |  |  |      |				|
   |-fregoffset	|  Saved float registers.	|
   |  |  |      |  F9				|
   |  |  |      |   .				|
   |  |  |      |   .				|
   |  |  |      |  F2				|
   |  |  v      |				|
   |  |  -------|-------------------------------|
   |  |         |				|
   |  |         |  Saved registers.		|
   |  |         |  S6				|
   |-regoffset	|   .				|
   |  |         |   .				|
   |  |         |  S0				|
   |  |         |  pdr.pcreg			|
   |  v         |				|
   |  ----------|-------------------------------|
   |            |				|
 frameoffset    |  Argument build area, gets	|
   |            |  7th ... nth arg for any	|
   |            |  called procedure.		|
   v            |  				|
   -------------|-------------------------------|<-- sp
                |				|
*/

#define PROC_LOW_ADDR(proc) ((proc)->pdr.adr) /* least address */
#define PROC_HIGH_ADDR(proc) ((proc)->pdr.iline) /* upper address bound */
#define PROC_FRAME_OFFSET(proc) ((proc)->pdr.frameoffset)
#define PROC_FRAME_REG(proc) ((proc)->pdr.framereg)
#define PROC_REG_MASK(proc) ((proc)->pdr.regmask)
#define PROC_FREG_MASK(proc) ((proc)->pdr.fregmask)
#define PROC_REG_OFFSET(proc) ((proc)->pdr.regoffset)
#define PROC_FREG_OFFSET(proc) ((proc)->pdr.fregoffset)
#define PROC_PC_REG(proc) ((proc)->pdr.pcreg)
#define PROC_LOCALOFF(proc) ((proc)->pdr.localoff)
#define PROC_SYMBOL(proc) (*(struct symbol**)&(proc)->pdr.isym)
#define _PROC_MAGIC_ 0x0F0F0F0F

struct linked_proc_info
{
  struct alpha_extra_func_info info;
  struct linked_proc_info *next;
} *linked_proc_desc_table = NULL;

/* Non-zero if we just simulated a single-step */
int one_stepped;
static CORE_ADDR next_pc, branch_pc, jsr_return_pc;
char next_mem[4], branch_mem[4];


CORE_ADDR
branch (pc_instruction, pc)
     long pc_instruction;
     CORE_ADDR pc;
{
  int opcode = (pc_instruction >> 26) & 0x3f;

  if (opcode == 0x1a)
    {
      int rb = (pc_instruction >> 16) & 0x1f;
      int code = (pc_instruction >> 14) & 0x3;

      /* code 1 is jsr, code 3 is jsr_coroutine (whatever that is) */
      if ((step_over_calls == 1) && (code == 1 || code == 3))
	return (CORE_ADDR) 0;

      if (code == 1 || code == 3)
	jsr_return_pc = next_pc;

      return read_register (rb);
    }
  else if (opcode >= 0x30 && opcode <= 0x3f)
    {
      int disp = pc_instruction & 0x1fffff;

      /* 0x34 is bsr */
      if ((step_over_calls == 1) && (opcode == 0x34))
	return (CORE_ADDR) 0;

      if (opcode == 0x34)
	jsr_return_pc = next_pc;

      /* sign extend displacement and multiply by 4 */
      disp = (disp << 11) >> 9;

      /* The extra +4 is due to the fact the a bp trap report pc+4 as
	 the next instruction to execute, so the stub subtracts 4
	 before returning the pc to gdb.  Here we adjust for this. */
      return pc+disp+4;
    }
  else
    return (CORE_ADDR) 0;
}

void
single_step (pid)
     int pid; /* we don't need it */
{
  CORE_ADDR pc;
  long pc_instruction;

  if (!one_stepped)
    {
      pc = read_register (PC_REGNUM);
      next_pc = pc + 4; /* branch not taken */

      if (target_insert_breakpoint (next_pc, next_mem))
	{
	  error ("Memory protection violation at 0x%x while single stepping.",
		 next_pc);
	}

      pc_instruction = read_memory_integer (pc, 4);
      branch_pc = branch (pc_instruction, pc);
      if (branch_pc && (branch_pc != next_pc))
	{
	  if (target_insert_breakpoint (branch_pc, branch_mem))
	    {
	      /* Probably tried to insert a bp into protected memory.
		 Reset branch to the last jsr return pc */

	      branch_pc = jsr_return_pc;
	      target_insert_breakpoint (branch_pc, branch_mem);
	    }
	}

      /* We are ready to let it go */
      one_stepped = 1;
    }
  else
    {
      /* Remove breakpoints */
      if (branch_pc && (branch_pc != next_pc))
	  target_remove_breakpoint (branch_pc, branch_mem);
      target_remove_breakpoint (next_pc, next_mem);
      one_stepped = 0;
    }
}

void
alpha_find_saved_regs (frame)
     struct frame_info *frame;
{
  CORE_ADDR reg_address;
  int ireg, freg;

  frame->saved_regs = (struct frame_saved_regs *)
    obstack_alloc (&frame_cache_obstack, sizeof(struct frame_saved_regs));
  memset (frame->saved_regs, 0, sizeof (struct frame_saved_regs));

  if (frame->pd->pdsc$v_kind == PDSC$K_KIND_FP_STACK)
    {
      reg_address = frame->frame;
      frame->saved_regs->regs[PV_REGNUM] = reg_address;

      reg_address += frame->pd->pdsc$w_rsa_offset;
      frame->saved_regs->regs[RA_REGNUM] = reg_address;
      reg_address += 8;

      for (ireg = 0; ireg <= 31 ; ++ireg)
	if (frame->pd->pdsc$l_ireg_mask & (1 << ireg))
	  {
	    frame->saved_regs->regs[ireg] = reg_address;
	    reg_address += 8;
	  }

      for (freg = 0; freg <= 31 ; ++freg)
	if (frame->pd->pdsc$l_freg_mask & (1 << freg))
	  {
	    frame->saved_regs->regs[32 + freg] = reg_address;
	    reg_address += 8;
	  }

      frame->saved_regs->regs[FP_REGNUM] = frame->frame;
      frame->saved_regs->regs[PC_REGNUM] = frame->saved_regs->regs[RA_REGNUM];
    }
  else if (frame->pd->pdsc$v_kind == PDSC$K_KIND_FP_REGISTER)
    {
      frame->saved_regs->regs[RA_REGNUM] = frame->pd->pdsc$b_save_ra;
      frame->saved_regs->regs[FP_REGNUM] = frame->pd->pdsc$b_save_fp;
      frame->saved_regs->regs[PC_REGNUM] = frame->saved_regs->regs[RA_REGNUM];
    }

}

static CORE_ADDR
read_next_frame_reg(fi, regno)
     struct frame_info *fi;
     int regno;
{
  for (; fi; fi = fi->next)
    {
      if (regno == FP_REGNUM)
	{
	  return fi->frame;
	}
      else if (regno == PC_REGNUM)
	{
	  /* It's easier just to cheat and let VMS tell us what the
	     previous frame's PC is, because I can't figure out how to
	     find it in an interrupt stack */
	  return fi->prev_invo_pc;
	}
      else
	{
	  if (fi->saved_regs == NULL)
	    alpha_find_saved_regs (fi);
	  if (fi->saved_regs->regs[regno])
	    {
	      if ((fi->pd->pdsc$v_kind == PDSC$K_KIND_FP_REGISTER)
		  && (fi->saved_regs->regs[regno] < 31))
		{
		  /* This isn't going to work if it's saved in r0 */
		  return read_register (fi->saved_regs->regs[regno]);
		}
	      else
		{
		  return read_memory_integer(fi->saved_regs->regs[regno], 8);
		}
	    }
	}
    }
  return read_register(regno);
}

CORE_ADDR
alpha_frame_saved_pc(frame)
     struct frame_info *frame;
{
  /* We have to get the saved pc from the sigcontext
     if it is a signal handler frame.  VMS ??? */
  int pcreg = frame->signal_handler_caller ? PC_REGNUM : RA_REGNUM;

  return read_next_frame_reg(frame, pcreg);
}

CORE_ADDR
alpha_saved_pc_after_call (frame)
     struct frame_info *frame;
{
  return read_register (RA_REGNUM);
}


/* This fencepost looks highly suspicious to me.  Removing it also
   seems suspicious as it could affect remote debugging across serial
   lines.  */

static CORE_ADDR
heuristic_proc_start(pc)
    CORE_ADDR pc;
{
    CORE_ADDR start_pc = pc;
    CORE_ADDR fence = start_pc - heuristic_fence_post;

    if (start_pc == 0)	return 0;

    if (heuristic_fence_post == UINT_MAX
	|| fence < VM_MIN_ADDRESS)
      fence = VM_MIN_ADDRESS;

    /* search back for previous return */
    for (start_pc -= 4; ; start_pc -= 4)
	if (start_pc < fence)
	  {
	    /* It's not clear to me why we reach this point when
	       stop_soon_quietly, but with this test, at least we
	       don't print out warnings for every child forked (eg, on
	       decstation).  22apr93 rich@cygnus.com.  */
	    if (!stop_soon_quietly)
	      {
		static int blurb_printed = 0;

		if (fence == VM_MIN_ADDRESS)
		  warning("Hit beginning of text section without finding");
		else
		  warning("Hit heuristic-fence-post without finding");
		
		warning("enclosing function for address 0x%lx", pc);
		if (!blurb_printed)
		  {
		    printf_filtered ("\
This warning occurs if you are debugging a function without any symbols\n\
(for example, in a stripped executable).  In that case, you may wish to\n\
increase the size of the search with the `set heuristic-fence-post' command.\n\
\n\
Otherwise, you told GDB there was a function where there isn't one.\n");
		    blurb_printed = 1;
		  }
	      }

	    return 0; 
	  }
	else if (ABOUT_TO_RETURN(start_pc))
	    break;

    start_pc += 4; /* skip return */
    return start_pc;
}

/* This returns the PC of the first inst after the prologue.  If we can't
   find the prologue, then return 0.  */

static CORE_ADDR
after_prologue (pc, proc_desc)
     CORE_ADDR pc;
     alpha_extra_func_info_t proc_desc;
{
  return 0;
}

/* Return non-zero if we *might* be in a function prologue.  Return zero if we
   are definatly *not* in a function prologue.  */

static int
alpha_in_prologue (pc, proc_desc)
     CORE_ADDR pc;
     alpha_extra_func_info_t proc_desc;
{
  CORE_ADDR after_prologue_pc;

  after_prologue_pc = 0;

  if (after_prologue_pc == 0
      || pc < after_prologue_pc)
    return 1;
  else
    return 0;
}

alpha_extra_func_info_t cached_proc_desc;

CORE_ADDR
alpha_frame_chain(frame)
    struct frame_info *frame;
{
    CORE_ADDR saved_pc = FRAME_SAVED_PC(frame);

    if ((frame->frameflags & 0x4) /* bottom_of_stack flag */
	|| saved_pc == 0
	|| inside_entry_file (saved_pc))
      return 0;

    return read_next_frame_reg(frame, FP_REGNUM);
}

void
print_extra_frame_info (fi)
     struct frame_info *fi;
{
  if (fi && (fi->pd->pdsc$v_kind == PDSC$K_KIND_FP_STACK))
    {
      printf_filtered (" stack frame procedure\n");
      printf_filtered (" stack frame size is %d bytes\n", fi->pd->pdsc$l_size);
    }
  else if (fi && (fi->pd->pdsc$v_kind == PDSC$K_KIND_FP_REGISTER))
    {
      printf_filtered (" register frame procedure\n");
      printf_filtered (" stack frame size is %d bytes\n", fi->pd->pdsc$l_size);
    }
  else if (fi && (fi->pd->pdsc$v_kind == PDSC$K_KIND_NULL))
    printf_filtered (" null procedure\n");

  if (fi && fi->frameflags & 0x1)
    printf_filtered (" exception frame\n");
  if (fi && fi->frameflags & 0x2)
    printf_filtered (" ast frame\n");
  if (fi && fi->frameflags & 0x4)
    printf_filtered (" bottom of stack\n");
}

void
init_extra_frame_info (frame)
     struct frame_info *frame;
{
  frame->saved_regs = NULL;

  if (frame->next)
    frame->invo_handle
      = vmschild_get_prev_invo_handle(frame->next->invo_handle);
  else
    frame->invo_handle = vmschild_get_prev_invo_handle(0);

  frame->pd = (PDSCDEF *)
    obstack_alloc (&frame_cache_obstack, sizeof(PDSCDEF));
  memset (frame->pd, 0, sizeof (PDSCDEF));

  vmschild_read_pd_from_invo_handle
    (frame->invo_handle, &frame->frameflags, &frame->frame, frame->pd,
     &frame->prev_invo_pc);

  frame->signal_handler_caller = frame->frameflags & 0x2;
}

struct frame_info *
setup_arbitrary_frame (argc, argv)
     int argc;
     CORE_ADDR *argv;
{
  if (argc != 2)
    error ("ALPHA frame specifications require two arguments: sp and pc");

  return create_new_frame (argv[0], argv[1]);
}

#if 1
/* Maybe this should be added to inferior.h ??? FIXME */
extern int stop_after_trap;

void
alpha_pop_frame()
{
  struct frame_info *prev_frame = get_prev_frame (get_current_frame ());
  struct symtab_and_line sal;
  struct breakpoint *breakpoint;
  struct cleanup *old_chain;

  clear_proceed_status ();
  stop_after_trap = 1;

  sal = find_pc_line (prev_frame->pc, 0);
  sal.pc = prev_frame->pc;

  breakpoint = set_momentary_breakpoint (sal, prev_frame, bp_finish);
  insert_breakpoints ();

  old_chain = make_cleanup(delete_breakpoint, breakpoint);

  vmschild_goto_unwind (prev_frame->invo_handle);

  wait_for_inferior ();
  normal_stop();

  do_cleanups(old_chain);
}
#endif

#if 0
/* This version of pop frame won't work on VMS because you can't change
   the SP reliably.  Verified by Calling Standard doc and by test. */
void
alpha_pop_frame()
{
  register int regnum;
  struct frame_info *frame = get_current_frame ();
  CORE_ADDR new_sp;

  write_register (PC_REGNUM, FRAME_SAVED_PC(frame));
  if (frame->saved_regs == NULL)
    alpha_find_saved_regs (frame);

  new_sp = frame->frame + frame->pd->pdsc$l_size;

  for (regnum = 32; --regnum >= 0; )
    if (frame->pd->pdsc$l_ireg_mask & (1 << regnum))
      write_register
	(regnum, read_memory_integer (frame->saved_regs->regs[regnum], 8));

  for (regnum = 32; --regnum >= 0; )
    if (frame->pd->pdsc$l_freg_mask & (1 << regnum))
      write_register
	(regnum + 32,
	 read_memory_integer (frame->saved_regs->regs[regnum + 32], 8));

  write_register
    (RA_REGNUM, read_memory_integer (frame->saved_regs->regs[RA_REGNUM], 8));
  write_register (SP_REGNUM, new_sp);
  flush_cached_frames ();

}
#endif

/* To skip prologues, I use this predicate.  Returns either PC itself
   if the code at PC does not look like a function prologue; otherwise
   returns an address that (if we're lucky) follows the prologue.  If
   LENIENT, then we must skip everything which is involved in setting
   up the frame (it's OK to skip more, just so long as we don't skip
   anything which might clobber the registers which are being saved.
   Currently we must not skip more on the alpha, but we might the lenient
   stuff some day.  */

CORE_ADDR
alpha_skip_prologue (pc, lenient)
     CORE_ADDR pc;
     int lenient;
{
    unsigned long inst;
    int offset;
    char buf[8];
    int status;

#ifdef GDB_TARGET_HAS_SHARED_LIBS
    /* Silently return the unaltered pc upon memory errors.
       This could happen on OSF/1 if decode_line_1 tries to skip the
       prologue for quickstarted shared library functions when the
       shared library is not yet mapped in.
       Reading target memory is slow over serial lines, so we perform
       this check only if the target has shared libraries.  */
    if (target_read_memory (pc, buf, 4))
      return pc;
#endif

    /* Can't determine prologue from the symbol table, need to examine
       instructions.  */

    /* Skip the typical prologue instructions. These are the stack adjustment
       instruction and the instructions that save registers on the stack
       or in the gcc frame.  */
    for (offset = 0; offset < 100; offset += 4)
      {
	int status;

	status = read_memory_nobpt (pc + offset, buf, 4);
	if (status)
	  memory_error (status, pc + offset);
	inst = extract_unsigned_integer (buf, 4);

	/* The alpha has no delay slots. But let's keep the lenient stuff,
	   we might need it for something else in the future.  */
	if (lenient && 0)
	  continue;

	if ((inst & 0xffff0000) == 0x27bb0000)	/* ldah $gp,n($t12) */
	    continue;
	if ((inst & 0xffff0000) == 0x23bd0000)	/* lda $gp,n($gp) */
	    continue;
	if ((inst & 0xffff0000) == 0x23de0000)	/* lda $sp,n($sp) */
	    continue;
	else if ((inst & 0xfc1f0000) == 0xb41e0000)
	    continue;				/* stq reg,n($sp) */
	else if ((inst & 0xfc1f0000) == 0xb01e0000)
	    continue;				/* stl reg,n($sp) */
	else if ((inst & 0xfc1f0000) == 0x9c1e0000
		 && (inst & 0xffff0000) != 0x9ffe0000)
	    continue;				/* stt reg,n($sp) */
						/* reg != $zero */
	else if (inst == 0x47de041d)		/* bis sp,sp,fp (VMS) */
	    continue;
	else if ((inst & 0xffffff00) == 0x47fd0400) /* bis r31,fp,reg (VMS) */
	    continue;

        /* These next two were discovered in __gnat_break_start, which is a
           register frame procedure when compiled optimized. */
	else if ((inst & 0xffffff00) == 0x47bd0400) /* bis fp,fp,reg (VMS) */
	    continue;
	else if (inst == 0x477b041d)		/* bis pv,pv,fp (VMS) */
	    continue;
	else if (inst == 0x47fb041d)		/* bis r31,pv,fp (VMS) */
	    continue;
	else
	    break;
    }
    return pc + offset;
}

/* The alpha needs a conversion between register and memory format if
   the register is a floating point register and
      memory format is float, as the register format must be double
   or
      memory format is an integer with 4 bytes or less, as the representation
      of integers in floating point registers is different. */
void
alpha_register_convert_to_virtual (regnum, valtype, raw_buffer, virtual_buffer)
    int regnum;
    struct type *valtype;
    char *raw_buffer;
    char *virtual_buffer;
{
  if (TYPE_LENGTH (valtype) >= REGISTER_RAW_SIZE (regnum))
    {
      memcpy (virtual_buffer, raw_buffer, REGISTER_VIRTUAL_SIZE (regnum));
      return;
    }

  if (TYPE_CODE (valtype) == TYPE_CODE_FLT)
    {
      double d = extract_floating (raw_buffer, REGISTER_RAW_SIZE (regnum));
      store_floating (virtual_buffer, TYPE_LENGTH (valtype), d);
    }
  else if (TYPE_CODE (valtype) == TYPE_CODE_INT && TYPE_LENGTH (valtype) <= 4)
    {
      ULONGEST l;
      l = extract_unsigned_integer (raw_buffer, REGISTER_RAW_SIZE (regnum));
      l = ((l >> 32) & 0xc0000000) | ((l >> 29) & 0x3fffffff);
      store_unsigned_integer (virtual_buffer, TYPE_LENGTH (valtype), l);
    }
  else
    error ("Cannot retrieve value from floating point register");
}

void
alpha_register_convert_to_raw (valtype, regnum, virtual_buffer, raw_buffer)
    struct type *valtype;
    int regnum;
    char *virtual_buffer;
    char *raw_buffer;
{
  if (TYPE_LENGTH (valtype) >= REGISTER_RAW_SIZE (regnum))
    {
      memcpy (raw_buffer, virtual_buffer, REGISTER_RAW_SIZE (regnum));
      return;
    }

  if (TYPE_CODE (valtype) == TYPE_CODE_FLT)
    {
      double d = extract_floating (virtual_buffer, TYPE_LENGTH (valtype));
      store_floating (raw_buffer, REGISTER_RAW_SIZE (regnum), d);
    }
  else if (TYPE_CODE (valtype) == TYPE_CODE_INT && TYPE_LENGTH (valtype) <= 4)
    {
      ULONGEST l;
      if (TYPE_UNSIGNED (valtype))
	l = extract_unsigned_integer (virtual_buffer, TYPE_LENGTH (valtype));
      else
	l = extract_signed_integer (virtual_buffer, TYPE_LENGTH (valtype));
      l = ((l & 0xc0000000) << 32) | ((l & 0x3fffffff) << 29);
      store_unsigned_integer (raw_buffer, REGISTER_RAW_SIZE (regnum), l);
    }
  else
    error ("Cannot store value in floating point register");
}

/* Given a return value in `regbuf' with a type `valtype', 
   extract and copy its value into `valbuf'.  */

void
alpha_extract_return_value (valtype, regbuf, valbuf)
    struct type *valtype;
    char regbuf[REGISTER_BYTES];
    char *valbuf;
{
  if (TYPE_CODE (valtype) == TYPE_CODE_FLT)
    alpha_register_convert_to_virtual (FP0_REGNUM, valtype,
				       regbuf + REGISTER_BYTE (FP0_REGNUM),
				       valbuf);
  else
    memcpy (valbuf, regbuf + REGISTER_BYTE (R0_REGNUM), TYPE_LENGTH (valtype));
}

/* Given a return value in `regbuf' with a type `valtype', 
   write its value into the appropriate register.  */

void
alpha_store_return_value (valtype, valbuf)
    struct type *valtype;
    char *valbuf;
{
  char raw_buffer[MAX_REGISTER_RAW_SIZE];
  int regnum = R0_REGNUM;
  int length = TYPE_LENGTH (valtype);
  
  if (TYPE_CODE (valtype) == TYPE_CODE_FLT)
    {
      regnum = FP0_REGNUM;
      length = REGISTER_RAW_SIZE (regnum);
      alpha_register_convert_to_raw (valtype, regnum, valbuf, raw_buffer);
    }
  else
    memcpy (raw_buffer, valbuf, length);

  write_register_bytes (REGISTER_BYTE (regnum), raw_buffer, length);
}

/* Just like reinit_frame_cache, but with the right arguments to be
   callable as an sfunc.  */

static void
reinit_frame_cache_sfunc (args, from_tty, c)
     char *args;
     int from_tty;
     struct cmd_list_element *c;
{
  reinit_frame_cache ();
}

void
_initialize_alpha_tdep ()
{
  struct cmd_list_element *c;

  tm_print_insn = print_insn_alpha;

  /* Let the user set the fence post for heuristic_proc_start.  */

  /* We really would like to have both "0" and "unlimited" work, but
     command.c doesn't deal with that.  So make it a var_zinteger
     because the user can always use "999999" or some such for unlimited.  */
  c = add_set_cmd ("heuristic-fence-post", class_support, var_zinteger,
		   (char *) &heuristic_fence_post,
		   "\
Set the distance searched for the start of a function.\n\
If you are debugging a stripped executable, GDB needs to search through the\n\
program for the start of a function.  This command sets the distance of the\n\
search.  The only need to set it is when debugging a stripped executable.",
		   &setlist);
  /* We need to throw away the frame cache when we set this, since it
     might change our ability to get backtraces.  */
  c->function.sfunc = reinit_frame_cache_sfunc;
  add_show_from_set (c, &showlist);
}
