/* gnu.classpath.tools.gjdoc.Main
   Copyright (C) 2001 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
 
GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA. */

package gnu.classpath.tools.gjdoc;

import com.sun.javadoc.*;
import java.io.*;
import java.util.*;
import java.lang.reflect.*;

public final class Main extends DocImpl implements RootDoc {

   /**
    *  Do we load classes that are referenced as base
    *  class?
    */
   static final boolean DESCEND_SUPERCLASS = true;

   /**
    *  Do we load classes that are referenced as
    *  interface?
    */
   static final boolean DESCEND_INTERFACES = false;

   /**
    *  Do we load classes that are imported in a 
    *  source file?
    */
   static final boolean DESCEND_IMPORTED   = true;

   /**
    *  Document only public members.
    */
   static final int COVERAGE_PUBLIC    = 0;

   /**
    *  Document only public and protected members.
    */
   static final int COVERAGE_PROTECTED = 1;

   /**
    *  Document public, protected and package private members.
    */
   static final int COVERAGE_PACKAGE   = 2;

   /**
    *  Document all members.
    */
   static final int COVERAGE_PRIVATE   = 3;

   /**
    *  Grid for looking up whether a particular access level
    *  is included in the documentation.
    */
   static final boolean[][] coverageTemplates = new boolean[][] {
      new boolean[] { true, false, false, false },  // public
      new boolean[] { true, true,  false, false },  // protected
      new boolean[] { true, true,  true,  false },  // package
      new boolean[] { true, true,  true,  true  },  // private
   };

   /**
    *  Keeps track of the number of errors occured
    *  during generation.
    */
   private int errorCount=0;

   /**
    *  Keeps track of the number of warnings occured
    *  during generation.
    */
   private int warningCount=0;

   /**
    *  Holds the Singleton instance of this class.
    */
   private static final Main instance = new Main();

   /**
    *  Avoid re-instantiation of this class.
    */
   private Main() {}

   /**
    *  <code>false</code> during Phase I: preparation
    *  of the documentation data. <code>true</code>
    *  during Phase II: documentation output by doclet.
    */
   boolean docletRunning=false;

   //---- Command line options

   /**
    *  Option "-doclet": name of the Doclet class to use.
    */
   private String option_doclet;

   /**
    *  Option "-overview": path to the special overview file.
    */
   private String option_overview;

   /**
    *  Option "-coverage": which members to include in generated documentation.
    */
   private int option_coverage = COVERAGE_PROTECTED;

   /**
    *  Option "-help": display command line usage.
    */
   private boolean option_help;

   /**
    *  Option "-docletpath": path to doclet classes.
    */
   private String option_docletpath;
   
   /**
    *  Option "-classpath": path to additional classes.
    */
   private String option_classpath;

   /**
    *  Option "-sourcepath": path to the Java source files to be documented.
    *  FIXME: this should be a list of paths
    */
   private List   option_sourcepath = new ArrayList();

   /**
    *  Option "-bootclasspath": path to Java bootstrap classes.
    */
   private String option_bootclasspath;

   /**
    *  Option "-extdirs": path to Java extension files.
    */
   private String option_extdirs;

   /**
    *  Option "-verbose": Be verbose when generating documentation.
    */
   private boolean option_verbose;

   /**
    *  Option "-nowarn": Do not print warnings.
    */
   private boolean option_nowarn;

   /**
    *  Option "-locale:" Specify the locale charset of Java source files.
    */
   private String option_locale;

   /**
    *  Option "-encoding": Specify character encoding of Java source files.
    */
   private String option_encoding;

   /**
    *  Option "-J": Specify flags to be passed to Java runtime.
    */
   private List option_java_flags = new LinkedList(); //ArrayList();

   /**
    *  All options and their corresponding values which are not recognized
    *  by Gjdoc. These are passed to the Doclet as "custom options".
    *  Each element in this array is again a String array, with the 
    *  option name as first element (including prefix dash) and possible
    *  option values as following elements.
    */
   private String[][] customOptionArr;

   /**
    *  The names of all classes explicitly specified on the 
    *  command line.
    *
    *  @contains String
    */
   private List specifiedClassNames = new LinkedList();

   /**
    *  The names of all packages explicitly specified on the 
    *  command line.
    *
    *  @contains String
    */
   private List specifiedPackageNames = new LinkedList();

   /**
    *  Stores all classes specified by the user: those given by
    *  individual class names on the command line, and those
    *  contained in the packages given on the command line.
    *
    *  @contains ClassDocImpl
    */
   private List classesList = new LinkedList(); //new LinkedList();

   /**
    *  Stores all classes loaded in the course of preparing
    *  the documentation data. Maps the fully qualified name 
    *  of a class to its ClassDocImpl representation.
    *
    *  @contains String->ClassDocImpl
    */
   private Map classDocMap = new HashMap();

   /**
    *  Stores all packages loaded in the course of preparing
    *  the documentation data. Maps the package name 
    *  to its PackageDocImpl representation.
    *
    *  @contains String->PackageDocImpl
    */
   private Map packageDocMap = new HashMap();

   /**
    *  All classes specified by the user, both those explicitly
    *  individually specified on the command line and those contained
    *  in packages specified on the command line (as Array for quick
    *  retrieval by Doclet).  This is created from classesList after
    *  all classes have been loaded.  
    */
   private ClassDocImpl[] classes;

   /**
    *  All classes which were individually specified on the command 
    *  line (as Array for quick retrieval by Doclet). This is created 
    *  from specifiedClassNames after all classes have been loaded.
    */
   private ClassDocImpl[] specifiedClasses;

   /**
    *  All packages which were specified on the command line (as Array
    *  for quick retrieval by Doclet). This is created from
    *  specifiedPackageNames after all classes have been loaded.  
    */
   private PackageDocImpl[] specifiedPackages;

   /**
    *  Temporarily stores a list of classes which are referenced
    *  by classes already loaded and which still have to be
    *  resolved.
    */
   private List scheduledClasses=new LinkedList();
   
   /**
    *  Parse all source files/packages and subsequentially
    *  start the Doclet given on the command line.
    *
    *  @param customOptions  List of unrecognized command line tokens
    */
   private void startDoclet(List customOptions) {

      try {

	 //--- Fetch the Class object for the Doclet.

	 Debug.log(1, "loading doclet class...");

	 Class docletClass = Class.forName(option_doclet);
	 //Object docletInstance = docletClass.newInstance();

	 Debug.log(1, "doclet class loaded...");

	 Method startMethod=null;
	 Method optionLenMethod=null;
	 Method validOptionsMethod=null;

	 //--- Try to find the optionLength method in the Doclet class.

	 try {
	    optionLenMethod = docletClass.getMethod("optionLength", new Class[]{String.class});
	 }
	 catch (NoSuchMethodException e) {
	    // Ignore if not found; it's OK it the Doclet class doesn't define this method.
	 }

	 //--- Try to find the validOptions method in the Doclet class.

	 try {
	    validOptionsMethod = docletClass.getMethod("validOptions", new Class[]{String[][].class, DocErrorReporter.class});
	 }
	 catch (NoSuchMethodException e) {
	    // Ignore if not found; it's OK it the Doclet class doesn't define this method.
	 }

	 //--- Find the start method in the Doclet class; complain if not found

	 startMethod = docletClass.getMethod("start", new Class[]{RootDoc.class});

	 //--- Feed the custom command line tokens to the Doclet

	 // stores all recognized options
	 List options=new LinkedList();

	 // stores packages and classes defined on the command line
	 List packageAndClasses=new LinkedList(); 

	 for (Iterator it=customOptions.iterator(); it.hasNext(); ) {
	    String option=(String)it.next();

	    Debug.log(9, "parsing option '"+option+"'");

	    if (option.startsWith("-")) {

	       //--- Parse option
	       
	       int optlen=0;

	       //--- Try to get option length from Doclet class

	       if (optionLenMethod!=null) {
		  
		  optionLenMethod.invoke(null, new Object[]{option});

		  Debug.log(3, "invoking optionLen method");

		  optlen=((Integer)optionLenMethod.invoke(null, new Object[]{option})).intValue();

		  Debug.log(3, "done");
	       }

	       if (optlen==0) {

		  //--- Complain if not found

		  printError("Unknown option "+option);
		  shutdown();
	       }
	       else {

		  //--- Read option values

		  String[] optionAndValues=new String[optlen];
		  optionAndValues[0]=option;
		  for (int i=1; i<optlen; ++i) {
		     if (!it.hasNext()) {
			printError("Missing value for option "+option);
			shutdown();
		     }
		     else {
			optionAndValues[i]=(String)it.next();
		     }
		  }
		  
		  //--- Store option for processing later

		  options.add(optionAndValues);
	       }
	    }
	    else {

	       //--- Add to list of packages/classes if not option or option value

	       packageAndClasses.add(option);
	    }
	 }

	 Debug.log(9, "options parsed...");
	 

	 //--- Complain if no packages or classes specified

	 if (packageAndClasses.isEmpty()) {
	    printError("No packages or classes specified.");
	    usage();
	    shutdown();
	 }

	 //--- For each class or package specified on the command line,
	 //         check that it exists and find out whether it is a class
         //         or a package

	 for (Iterator it=packageAndClasses.iterator(); it.hasNext(); ) {

	    String classOrPackage=(String)it.next();

	    //--- Check for illegal name

	    if (classOrPackage.startsWith(".") 
		|| classOrPackage.endsWith(".") 
		|| classOrPackage.indexOf("..")>0 
		|| !checkCharSet(classOrPackage,"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890_.")) {
	       throw new ParseException("Illegal class or package name '"+classOrPackage+"'");
	    }

	    //--- Assemble absolute path to package

 	    String classOrPackageRelPath = classOrPackage.replace('.', File.separatorChar);

	    //--- Create one file object each for a possible package directory
	    //         and a possible class file, and find out if they exist.

	    File packageDir = findSourceFile(classOrPackageRelPath);
	    File sourceFile = findSourceFile(classOrPackageRelPath+".java");

	    boolean packageDirExists = packageDir!=null && packageDir.exists() 
	       && packageDir.getCanonicalFile().getAbsolutePath().endsWith(classOrPackageRelPath);

	    boolean sourceFileExists = sourceFile!=null && sourceFile.exists() 
	       && sourceFile.getCanonicalFile().getAbsolutePath().endsWith(classOrPackageRelPath+".java");

	    //--- Complain if neither exists: not found

	    if (!packageDirExists && !sourceFileExists) {
	       printError("Class or package "+classOrPackage+" not found.");
	       printError(packageDir.getAbsolutePath());
	       printError(sourceFile.getAbsolutePath());
	       shutdown();
	    }

	    //--- Complain if both exist: ambigious

	    else if (packageDirExists && sourceFileExists) {
	       printError("Ambigious class/package name "+classOrPackage+".");
	       shutdown();
	    }

	    //--- Otherwise, if the package directory exists, it is a package

	    else if (packageDirExists) {
	       if (!packageDir.isDirectory()) {
		  printError("File \""+packageDir+"\" doesn't have .java extension.");
		  shutdown();
	       }
	       else {
		  specifiedPackageNames.add(classOrPackage);
	       }
	    }

	    //--- Otherwise, it must be a Java source file file

	    else /* if (sourceFileExists) */ {
	       if (sourceFile.isDirectory()) {
		  printError("File \""+sourceFile+"\" is a directory!");
		  shutdown();
	       }
	       else {
		  specifiedClassNames.add(classOrPackage);
	       }
	    }
	 }

	 //--- Validate custom options passed on command line
	 //         by asking the Doclet if they are OK.

	 customOptionArr=(String[][])options.toArray(new String[0][0]);
	 if (validOptionsMethod!=null 
	     && !((Boolean)validOptionsMethod.invoke(null, 
						     new Object[]{customOptionArr,
								  this})).booleanValue()) {
	    // Not ok: shutdown system.
	    shutdown();
	 }

	 //--- Create a temporary random access file for caching comment text.

	 //File rawCommentCacheFile=File.createTempFile("gjdoc_rawcomment",".cache");
	 File rawCommentCacheFile = new File("gjdoc_rawcomment.cache");
	 rawCommentCacheFile.deleteOnExit();
	 rawCommentCache = new RandomAccessFile(rawCommentCacheFile, "rw");

	 //--- Parse all files in "java.lang".

	 File javaLangSources = findSourceFile("java/lang");
	 if (null!=javaLangSources) {
	    Parser.getInstance().processSourceDir(javaLangSources);
	 }
	 else {

	    printWarning("For now, please make sure that you include the core source directory\nin -sourcepath");
	    Debug.log(1, "Sourcepath is "+option_sourcepath);

	    // Core docs not included in source-path: 
	    // we need to gather the information about java.lang
	    // classes via reflection...

	 }

	 //--- Parse all files in explicitly specified package directories.
	 
	 for (Iterator it=specifiedPackageNames.iterator(); it.hasNext(); ) {

	    String specifiedPackageName = (String)it.next();
	    printNotice("Loading classes for package "+specifiedPackageName+" ...");
	    File sourceDir = findSourceFile(specifiedPackageName.replace('.',File.separatorChar));
	    if (null!=sourceDir) {
	       Parser.getInstance().processSourceDir(sourceDir);
	    }
	    else {
	       printError("Package '"+specifiedPackageName+"' not found.");
	    }
	 }

	 //--- Parse all explicitly specified class files.

	 for (Iterator it=specifiedClassNames.iterator(); it.hasNext(); ) {

	    String specifiedClassName = (String)it.next();
	    printNotice("Loading class "+specifiedClassName+" ...");
	    File sourceFile = findSourceFile(specifiedClassName.replace('.',File.separatorChar)+".java");
	    if (null!=sourceFile) {
	       Parser.getInstance().processSourceFile(sourceFile, true);
	    }
	    else {
	       printError("Class '"+specifiedClassName+"' not found.");
	    }
	 }

	 //--- Let the user know that all specified classes are loaded.

	 printNotice("Constructing Javadoc information...");

	 //--- Load all classes implicitly referenced by explicitly specified classes.

	 loadScheduledClasses();

	 //--- Resolve pending references in all ClassDocImpls

	 for (Iterator it=classDocMap.values().iterator(); it.hasNext(); ) {
	    ClassDocImpl cd=(ClassDocImpl)it.next();
	    cd.resolve();
	 }

	 //--- Resolve pending references in all PackageDocImpls

	 for (Iterator it=packageDocMap.values().iterator(); it.hasNext(); ) {
	    PackageDocImpl pd=(PackageDocImpl)it.next();
	    pd.resolve();
	 }

	 //--- Assemble the array with all specified classes

	 List specifiedClassesList=new LinkedList();
	 for (Iterator it=specifiedClassNames.iterator(); it.hasNext(); ) {
	    String specifiedClassName = (String)it.next();
	    ClassDocImpl specifiedClassDoc = (ClassDocImpl)classDocMap.get(specifiedClassName);
	    specifiedClassDoc.setIsIncluded(true);
	    specifiedClassesList.add(specifiedClassDoc);
	    classesList.add(specifiedClassDoc);
	 }
	 this.specifiedClasses=(ClassDocImpl[])specifiedClassesList.toArray(new ClassDocImpl[0]);

	 //--- Assemble the array with all specified packages

	 List specifiedPackageList=new LinkedList();
	 for (Iterator it=specifiedPackageNames.iterator(); it.hasNext(); ) {
	    String specifiedPackageName = (String)it.next();
	    PackageDoc specifiedPackageDoc = (PackageDoc)packageDocMap.get(specifiedPackageName);
	    if (null!=specifiedPackageDoc) {
	       ((PackageDocImpl)specifiedPackageDoc).setIsIncluded(true);
	       specifiedPackageList.add(specifiedPackageDoc);

	       ClassDoc[] packageClassDocs=specifiedPackageDoc.allClasses();
	       for (int i=0; i<packageClassDocs.length; ++i) {
		  ClassDocImpl specifiedPackageClassDoc=(ClassDocImpl)packageClassDocs[i];
		  specifiedPackageClassDoc.setIsIncluded(true);
		  classesList.add(specifiedPackageClassDoc);
	       }
	    }
	 }
	 this.specifiedPackages=(PackageDocImpl[])specifiedPackageList.toArray(new PackageDocImpl[0]);

	 //--- Resolve pending references in comment data of all classes

	 for (Iterator it=classDocMap.values().iterator(); it.hasNext(); ) {
	    ClassDocImpl cd=(ClassDocImpl)it.next();
	    cd.resolveComments();
	 }

	 //--- Resolve pending references in comment data of all packages

	 for (Iterator it=packageDocMap.values().iterator(); it.hasNext(); ) {
	    PackageDocImpl pd=(PackageDocImpl)it.next();
	    pd.resolveComments();
	 }

	 //--- Create array with all loaded classes

	 this.classes=(ClassDocImpl[])classesList.toArray(new ClassDocImpl[0]);

	 //--- Our work is done, tidy up memory

	 System.gc();

	 //--- Set flag indicating Phase II of documentation generation

	 docletRunning=true;

	 //--- Invoke the start method on the Doclet: produce output

	 startMethod.invoke(null, new Object[]{this});

	 //--- Close comment cache

	 rawCommentCache.close();

	 //--- Let the user know how many warnings/errors occured

	 if (warningCount>0) {
	    printNotice(warningCount+" warnings");
	 }
	 if (errorCount>0) {
	    printNotice(errorCount+" errors");
	 }

	 //--- Done.
      }
      catch (Exception e) {
	 e.printStackTrace();
      }
   }

   /**
    *
    */
   private static boolean validOptions(String options[][], 
				       DocErrorReporter reporter) {

      boolean foundDocletOption = false;
      for (int i = 0; i < options.length; i++) {
	 String[] opt = options[i];
	 if (opt[0].equals("-doclet")) {
	    if (foundDocletOption) {
	       reporter.printError("Only one -doclet option allowed.");
	       return false;
	    } else { 
	       foundDocletOption = true;
	    }
	 } 
      }
      if (!foundDocletOption) {
	 reporter.printError("You must specify a doclet using -doclet.");
	 usage();
      }
      return foundDocletOption;
   }

   /**
    *  Main entry point.
    *
    *  This is the method called when gjdoc is invoked
    *  from the command line.
    */
   public static void main(String[] args) {

      try {

	 //--- For testing purposes only

	 //System.err.println("getting locale...");
	 //java.util.Locale loc = java.util.Locale.getDefault();
	 //System.err.println("locale="+loc.getLanguage()+" ("+loc.getDefault()+"), "+loc.getLanguage()+", "+loc.getVariant()+", "+loc.getCountry());

	 //--- Remember current time for profiling purposes

	 Timer.setStartTime();

	 //--- Handle control to the Singleton instance of this class
	 
	 instance.start(args);
      }
      catch (Exception e) {

	 //--- Report any error

	 e.printStackTrace();
      }
   }

   /**
    *  Parses command line arguments and subsequentially
    *  handles control to the startDoclet() method
    *
    *  @param args Command line arguments, as passed to the main() method
    */
   public void start(String[] args) {

      //--- Initialize Map for option parsing

      initOptions();

      //--- This will hold all options recognized by gjdoc itself
      //         and their associated arguments.
      //         Contains objects of type String[], where each entry
      //         specifies an option along with its aguments.

      List options=new LinkedList();

      //--- This will hold all command line tokens not recognized
      //         to be part of a standard option.
      //         These options are intended to be processed by the doclet
      //         Contains objects of type String, where each entry is
      //         one unrecognized token.

      List customOptions=new LinkedList();


      //--- Iterate over all options given on the command line

      for (int i=0; i<args.length; ++i) {

	 //--- Check if gjdoc recognizes this option as a standard option
	 //         and remember the options' argument count
	 
	 int optlen = optionLength(args[i]);

	 //--- Argument count == 0 indicates that the option is not recognized.
	 //         Add it to the list of custom option tokens

	 if (optlen == 0) {
	    customOptions.add(args[i]);
	 }

	 //--- Otherwise the option is recognized as a standard option.
	 //         If the option requires more arguments than given on the
	 //         command line, issue a fatal error

	 else if (i+optlen>args.length) {
	    printFatal("Missing value for option "+args[i]+".");
	 }

	 //--- The option is recognized as standard option, and all
	 //         required arguments are supplied. Create a new String
	 //         array for the option and its arguments, and store it
	 //         in the options array.

	 //         FIXME: this does not deal well with omitted arguments
	 //         like such '-sourcepath -private': this would lead
	 //         to '-private' being silently accepted as an argument
	 //         to '-sourcepath'.

	 else {
	    String[] option=new String[optlen];
	    for (int j=0; j<optlen; ++j)
	       option[j]=args[i+j];
	    options.add(option);

	    // Skip to next option token
	    i += optlen-1;
	 }
      }

      //--- Create an array of String arrays from the dynamic array built above

      String[][] optionArr=(String[][])options.toArray(new String[options.size()][0]);

      //--- Validate all options and issue warnings/errors
      
      if (validOptions(optionArr, this)) {

	 //--- We got valid options; parse them and store the parsed values
	 //         in 'option_*' fields.

	 readOptions(optionArr);

	 // If we have an empty source path list, add the current directory ('.')

	 if (option_sourcepath.size()==0) option_sourcepath.add(new File("."));

	 //--- We have all information we need to start the doclet at this time

	 startDoclet(customOptions);
      }
   }

   /**
    *  Helper class for parsing command line arguments.
    *  An instance of this class represents a particular
    *  option accepted by gjdoc (e.g. '-sourcepath')
    *  along with the number of expected arguments and
    *  behavior to parse the arguments.
    */
   private abstract class OptionProcessor {

      /**
       *  Number of arguments expected by this option.
       */
      private int argCount;

      /**
       *  Initializes this instance.
       */
      public OptionProcessor(int argCount) { 
	 this.argCount=argCount;
      }

      /**
       *  Overridden by derived classes with behavior
       *  to parse the arguments specified with this
       *  option.
       */
      abstract void process(String[] args);
   }
   
   /**
    *  Maps option tags (e.g. '-sourcepath') to
    *  OptionProcessor objects. Initialized only once
    *  by method initOptions().
    *  FIXME: Rename to 'optionProcessors'.
    */
   private static Map options = null;

   /**
    *  Initialize all OptionProcessor objects needed
    *  to scan/parse command line options.
    *
    *  This cannot be done in a static initializer block
    *  because OptionProcessors need access to the
    *  Singleton instance of the Main class.
    */
   private void initOptions() {

      options = new HashMap();
	 
      //--- Put one OptionProcessor object into the map
      //         for each option recognized.

      options.put("-overview", new OptionProcessor(2) { 
	    void process(String[] args) { option_overview = args[0]; }
	 });
      options.put("-public", new OptionProcessor(1) { 
	    void process(String[] args) { option_coverage = COVERAGE_PUBLIC; }
	 });
      options.put("-protected", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_coverage = COVERAGE_PROTECTED; 
	    }
	 });
      options.put("-package", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_coverage = COVERAGE_PACKAGE; 
	    }
	 });
      options.put("-private", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_coverage = COVERAGE_PRIVATE; 
	    }
	 });
      options.put("-help", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_help = true; 
	    }
	 });
      options.put("-doclet", new OptionProcessor(2) { 
	    void process(String[] args) { 
	       option_doclet = args[0]; 
	    }
	 });
      options.put("-nowarn", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_nowarn = true; 
	    }
	 });
      options.put("-sourcepath", new OptionProcessor(2) { 
	    void process(String[] args) { 
	       Debug.log(1, "-sourcepath is '"+args[0]+"'");
	       for (StringTokenizer st=new StringTokenizer(args[0], File.pathSeparator); st.hasMoreTokens(); ) {
		  String path = st.nextToken();
		  File file = new File(path);
		  option_sourcepath.add(file);
	       }
	    }
	 });
      options.put("-verbose", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_verbose = true; 
	    }
	 });
      options.put("-locale", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_locale = args[0]; 
	    }
	 });
      options.put("-encoding", new OptionProcessor(1) { 
	    void process(String[] args) { 
	       option_encoding = args[0]; 
	    }
	 });
   }

   /**
    *  Determine how many arguments the given option
    *  requires.
    *
    *  @param option  The name of the option without leading dash.
    */
   private static int optionLength(String option) {

      OptionProcessor op=(OptionProcessor)options.get(option);
      if (op!=null)
	 return op.argCount;
      else
	 return 0;
   }

   /**
    *  Process all given options.
    *
    *  Assumes that the options have been validated before.
    *
    *  @param optionArr  Each element is a series of Strings where [0] is
    *                    the name of the option and [1..n] are the arguments
    *                    to the option.
    */
   private void readOptions(String[][] optionArr) {

      //--- For each option, find the appropriate OptionProcessor
      //        and call its process() method

      for (int i=0; i<optionArr.length; ++i) {
	 String[] opt = optionArr[i];
	 String[] args = new String[opt.length-1];
	 System.arraycopy(opt,1,args,0,opt.length-1);
	 OptionProcessor op=(OptionProcessor)options.get(opt[0]);
	 op.process(args);
      }
   }

   /**
    *  Print command line usage.
    *
    */
   private static void usage() {
      System.err.print("\n"+
		       "USAGE: gjdoc [options] [packagenames] "+/*"[sourcefiles] "+*/"[classnames]"+/*" [@files]"+*/"\n\n"+
		       "  -doclet <class>           Doclet class to use for generating output\n"+
		       "  -sourcepath <pathlist>    Where to look for source files\n"+
		       "  -overview <file>          Read overview documentation from HTML file\n"+
		       "  -public                   Include only public classes and members\n"+
		       "  -protected                Include protected/public classes and members (default)\n"+
		       "  -package                  Include package/protected/public classes and members\n"+
		       "  -private                  Include all classes and members\n"+
		       "  -verbose                  Output messages about what Gjdoc is doing\n"+
		       /*
		       "  -locale <name>            Locale to be used, e.g. en_US or en_US_WIN\n"+
		       "  -encoding <name>          Source file encoding name\n"+
		       */
		       "  -help                     Show this information\n"
	 );
   }

   /**
    *  Shutdown the generator.
    */
   public void shutdown() {
      System.exit(5);
   }

   public static Main getRootDoc() { return instance; }

   public void addClassDoc(ClassDoc cd) {
      classDocMap.put(cd.qualifiedName(), cd);
   }

   public void addPackageDoc(PackageDoc pd) {
      packageDocMap.put(pd.name(), pd);
   }

   public PackageDocImpl getPackageDoc(String name) {
      return (PackageDocImpl)packageDocMap.get(name);
   }

   public ClassDocImpl getClassDoc(String qualifiedName) {
      return (ClassDocImpl)classDocMap.get(qualifiedName);
   }

   class ScheduledClass {

      ClassDoc contextClass;
      String qualifiedName;
      ScheduledClass(ClassDoc contextClass, String qualifiedName) {
	 this.contextClass=contextClass;
	 this.qualifiedName=qualifiedName;
      }
      
      public String toString() { return "ScheduledClass{"+qualifiedName+"}"; }
   }

   public void scheduleClass(ClassDoc context, String qualifiedName) throws ParseException, IOException {

      if (classDocMap.get(qualifiedName)==null) {

	 //Debug.log(9,"Scheduling "+qualifiedName+", context "+context+".");

	 scheduledClasses.add(new ScheduledClass(context, qualifiedName));
      }
   }

   /**
    *  Load all classes that were implictly referenced by the classes
    *  (already loaded) that the user explicitly specified on the
    *  command line.
    *
    *  For example, if the user generates Documentation for his simple
    *  'class Test {}', which of course 'extends java.lang.Object',
    *  then 'java.lang.Object' is implicitly referenced because it is
    *  the base class of Test.
    *
    *  Gjdoc needs a ClassDocImpl representation of all classes
    *  implicitly referenced through derivation (base class),
    *  or implementation (interface), or field type, method argument
    *  type, or method return type.
    *
    *  The task of this method is to ensure that Gjdoc has all this
    *  information at hand when it exits.
    *
    * 
    */
   public void loadScheduledClasses() throws ParseException, IOException {

      // Because the referenced classes could in turn reference other
      // classes, this method runs as long as there are still unloaded
      // classes.

      while (!scheduledClasses.isEmpty()) {

	 // Make a copy of scheduledClasses and empty it. This 
	 // prevents any Concurrent Modification issues.
	 // As the copy won't need to grow (as it won't change)
	 // we make it an Array for performance reasons.

	 ScheduledClass[] scheduledClassesArr = (ScheduledClass[])scheduledClasses.toArray(new ScheduledClass[0]);
	 scheduledClasses.clear();

	 // Load each class specified in our array copy
	 
	 for (int i=0; i<scheduledClassesArr.length; ++i) {

	    // The name of the class we are looking for. This name
	    // needs not be fully qualified.
	    
	    String scheduledClassName=scheduledClassesArr[i].qualifiedName;

	    // The ClassDoc in whose context the scheduled class was looked for.
	    // This is necessary in order to resolve non-fully qualified 
	    // class names.
	    ClassDoc scheduledClassContext=scheduledClassesArr[i].contextClass;

	    // If there already is a class doc with this name, skip. There's
	    // nothing to do for us.
	    if (classDocMap.get(scheduledClassName)!=null) {
	       continue;
	    }

	    try {
	       // Try to load the class
	       loadScheduledClass(scheduledClassName, scheduledClassContext);
	    }
	    catch (ParseException e) {

	       /**********************************************************

               // Check whether the following is necessary at all.


	       if (scheduledClassName.indexOf('.')>0) {

		  // Maybe the dotted notation doesn't mean a package
		  // name but instead an inner class, as in 'Outer.Inner'.
		  // so let's assume this and try to load the outer class.

		  String outerClass="";
		  for (StringTokenizer st=new StringTokenizer(scheduledClassName,"."); st.hasMoreTokens(); ) {
		     if (outerClass.length()>0) outerClass+=".";
		     outerClass+=st.nextToken();
		     if (!st.hasMoreTokens()) break;
		     try {
			loadClass(outerClass);
			//FIXME: shouldn't this be loadScheduledClass(outerClass, scheduledClassContext); ???
			continue;
		     }
		     catch (Exception ee) {
			// Ignore: try next level
		     }
		  }
	       }

	       **********************************************************/

	       // If we arrive here, the class could not be found

	       printWarning("Couldn't load class "+scheduledClassName+" referenced by "+scheduledClassContext);

	       //FIXME: shouldn't this be throw new Error("cannot load: "+scheduledClassName);
	    }
	 }
      }
   }

   private void loadScheduledClass(String scheduledClassName, ClassDoc scheduledClassContext) throws ParseException, IOException {

      ClassDoc loadedClass=(ClassDoc)scheduledClassContext.findClass(scheduledClassName);

      if (loadedClass==null || loadedClass instanceof ClassDocProxy) {

	 File file=findScheduledClassFile(scheduledClassName, scheduledClassContext);
	 if (file!=null) {
	    Parser.getInstance().processSourceFile(file, false);
	 }
	 else {
	    // It might be an inner class of one of the outer/super classes.
	    // But we can only check that when they are all fully loaded.
	    boolean retryLater = false;

	    int numberOfProcessedFilesBefore = Parser.getNumberOfProcessedFiles();

	    ClassDoc cc = scheduledClassContext.containingClass();
	    while (cc != null && !retryLater) {
		ClassDoc sc = cc.superclass();
		while (sc != null && !retryLater) {
		    if (sc instanceof ClassDocProxy) {
			((ClassDocImpl)cc).resolve();
			retryLater = true;
		    }
		    sc = sc.superclass();
		}
		cc = cc.containingClass();
	    }

	    // Now that outer/super references have been resolved, try again
	    // to find the class.

	    loadedClass = (ClassDoc)scheduledClassContext.findClass(scheduledClassName);

	    int numberOfProcessedFilesAfter = Parser.getNumberOfProcessedFiles();

	    boolean filesWereProcessed = numberOfProcessedFilesAfter > numberOfProcessedFilesBefore;

	    // Only re-schedule class if additional files have been processed
	    // If there haven't, there's no point in re-scheduling.
	    // Will avoid infinite loops of re-scheduling
	    if (null == loadedClass && retryLater && filesWereProcessed)
		scheduleClass(scheduledClassContext, scheduledClassName);
	    else if (null == loadedClass)
		printWarning("Can't find scheduled class '"
			     + scheduledClassName
			     + "' in context '"
			     + scheduledClassContext.qualifiedName()
			     + "'");
	 }
      }
   }


   public File findScheduledClassFile(String scheduledClassName, ClassDoc scheduledClassContext) 

      throws ParseException, IOException {

      File rc;

      if (scheduledClassName.indexOf('.')<0) {

	 ClassDoc[] importedClasses=scheduledClassContext.importedClasses();
	 int j;
	 for (j=0; j<importedClasses.length; ++j) {
	    if (importedClasses[j].qualifiedName().endsWith("."+scheduledClassName)) {
	       rc = findClass(importedClasses[j].qualifiedName());
	       if (rc!=null) return rc;
	    }
	 }
	    
	 PackageDoc[] importedPackages=scheduledClassContext.importedPackages();

	 for (j=0; j<importedPackages.length; ++j) {
	    rc = findClass(importedPackages[j].name()+"."+scheduledClassName);
	    if (rc!=null) return rc;
	 }
      }

      rc = findClass(scheduledClassName);
      if (rc!=null) return rc;

      if (scheduledClassContext.containingPackage()!=null) {
	 rc = findClass(scheduledClassContext.containingPackage().name()+"."+scheduledClassName);
	 if (rc!=null) return rc;
      }

      return findClass("java.lang."+scheduledClassName);
   }

   public File findClass(String qualifiedName) throws IOException {

      if (docletRunning) return null;

      if (Main.getRootDoc().getClassDoc(qualifiedName)==null) {

	 String relPath=qualifiedName.replace('.',File.separatorChar)+".java";
	 //String filename=new File(Main.getRootDoc().getSourcePath()).getCanonicalFile().getAbsolutePath()+File.separatorChar+relPath;
	 for (Iterator it=option_sourcepath.iterator(); it.hasNext(); ) {

	    File sourcePath = (File)it.next();

	    String filename=sourcePath.getAbsolutePath()+File.separatorChar+relPath;
	    File file = null;

	    Debug.log(9, "loadClass: trying file "+filename);

	    // FIXME: the following can probably be done simpler and more elegant using File.getParent()

	    while ((!(file=new File(filename)).exists()) 
		   || (!file.getAbsolutePath().toLowerCase().endsWith(relPath.toLowerCase()))) {
	       int ndx=filename.lastIndexOf(File.separatorChar);
	       if (ndx>=0) {
		  filename=filename.substring(0,ndx)+".java";
		  //Debug.log(6,"loadClass: trying file "+filename);
	       }
	       else {
		  file = null;
		  break;
	       }
	    }

	    if (null!=file) return file;
	 }

	 return null;
      }
      else {
	 //Debug.log(9, "class \""+qualifiedName+"\" already loaded.");
	 return null;
      }
   }

   public static boolean recursiveClasses = false;

   public static boolean checkCharSet(String toCheck, String charSet) {
      for (int i=0; i<toCheck.length(); ++i) {
	 if (charSet.indexOf(toCheck.charAt(i))<0)
	    return false;
      }
      return true;
   }

   public boolean includeAccessLevel(int accessLevel) {
      return coverageTemplates[option_coverage][accessLevel];
   }

   PackageDocImpl findOrCreatePackageDoc(String packageName) {
      PackageDocImpl rc=(PackageDocImpl)getPackageDoc(packageName);
      if (null==rc) {
	 rc=new PackageDocImpl(packageName);
	 if (specifiedPackageNames.contains(packageName)) {
	    File packageDocFile=findSourceFile(packageName.replace('.',File.separatorChar)+File.separatorChar+"package.html");
	    if (null!=packageDocFile) {
	       try {
		  long packageDocSize=packageDocFile.length();
		  char[] packageDocBuf=new char[(int)(packageDocSize)];
		  FileReader fr=new FileReader(packageDocFile);
		  int index = 0;
		  int i = fr.read(packageDocBuf, index, (int)packageDocSize);
		  while (i > 0) {
		      index += i;
		      packageDocSize -= i;
		      i = fr.read(packageDocBuf, index, (int)packageDocSize);
		  }
		  fr.close();

		  // We only need the part between the begin and end body tag.
		  String html = new String(packageDocBuf);
		  int start = html.indexOf("<body");
		  if (start == -1)
		      start = html.indexOf("<BODY");
		  int end = html.indexOf("</body>");
		  if (end == -1)
		      end = html.indexOf("</BODY>");
		  if (start != -1 && end != -1) {
		      // Start is end of body tag.
		      start = html.indexOf('>', start) + 1;
		      if (start != -1 && start < end)
		          html = html.substring(start, end);
		  }

		  rc.setRawCommentText(html);
	       }
	       catch (IOException e) {
		  printWarning("Error while reading documentation for package "+packageName+": "+e.getMessage());
	       }
	    }
	    else {
	       printWarning("No description found for package "+packageName);
	    }
	 }
	 addPackageDoc(rc);
      }
      return rc;
   }

   public boolean isDocletRunning() { return docletRunning; }

   RandomAccessFile rawCommentCache;

   public long writeRawComment(String rawComment) {
      try {
	 long pos=rawCommentCache.getFilePointer();
	 rawCommentCache.writeUTF(rawComment);
	 return pos;
      }
      catch (IOException e) {
	 printFatal("Cannot write to comment cache: "+e.getMessage());
	 return -1;
      }
   }

   public String readRawComment(long pos) {
      try {
	 rawCommentCache.seek(pos);
	 return rawCommentCache.readUTF();
      }
      catch (IOException e) {
	 printFatal("Cannot read from comment cache: "+e.getMessage());
	 return null;
      }
   }

   private File findSourceFile(String relPath) {

      for (Iterator it = option_sourcepath.iterator(); it.hasNext(); ) {
	 File path = (File)it.next();
	 File file = new File(path, relPath);
	 if (file.exists()) return file;
      }

      return null;
   }

   //--------------------------------------------------------------------------
   //
   // Implementation of RootDoc interface
   //
   //--------------------------------------------------------------------------

   /**
    *  Return classes and interfaces to be documented. 
    */
   public ClassDoc[] classes() { return classes; } 

   /**
    *  Return a ClassDoc object for the specified class/interface 
    *  name.
    *
    *  @return a ClassDoc object describing the given class, or 
    *  <code>null</code> if no corresponding ClassDoc object
    *  has been constructed.
    */
   public ClassDoc classNamed(String qualifiedName) { 
      return (ClassDoc)classDocMap.get(qualifiedName); 
   } 

   /**
    *  Return an xxx
    */
   public String[][] options() { return customOptionArr; } 

   // Return a PackageDoc for the specified package name 
   public PackageDoc packageNamed(String name) { 
      return (PackageDoc)packageDocMap.get(name); 
   }


   // classes and interfaces specified on the command line. 
   public ClassDoc[] specifiedClasses() { return specifiedClasses; } 

   // packages specified on the command line. 
   public PackageDoc[] specifiedPackages() { return specifiedPackages; }

   // Print error message, increment error count. 
   public void printError(java.lang.String msg) {
      System.err.println("ERROR: "+msg);
      ++errorCount;
   }

   // Print error message, increment error count. 
   public void printFatal(java.lang.String msg) {
      System.err.println("FATAL: "+msg);
      System.exit(10);
   }

   // Print a message. 
   public void printNotice(java.lang.String msg) {
      System.err.println(msg);
   }
   
   // Print warning message, increment warning count. 
   public void printWarning(java.lang.String msg) {
      if (!option_nowarn) {
	 System.err.println("WARNING: "+msg);
	 ++warningCount;;
      }
   }

   public String name() {
      return "RootDoc";
   }

}
