(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Data class containing the descriptions of GUI elements. *)

open Zog_types

module C = Configwin

(** Create a new entity. *)
let entity () = 
  { en_name = "" ;
    en_ele = None ;
    en_params = [] ;
  }

(** Get the first value of a property, function of the property kind. *)
let default_prop_value kind =
  try
    let (_,_,values_kind,_) = Zog_types.get_prop_info kind in
    match values_kind with
      Bool -> 
	(
	 match kind with
	   Expand | Homogeneous | Right_justify -> "false"
	 | _ -> "true"
	)
    | PosInt -> ""
    | Float -> ""
    | Code -> ""
    | Code_list -> "[]"
    | Enum [] -> ""
    | Enum ((s,_) :: _) -> s
    | Enum_list [] -> ""
    | Enum_list ((s,_) :: _) -> s
    | Keysym -> ""
  with
    Failure s ->
      prerr_endline s ;
      ""

(** Return [true] if the given ele will be in the class interface. *)
let in_interface ele =
  match ele.name with
    "" -> false
  | s -> s.[0] <> '_'

(** Get the list of properties, according to the given class. *)
let properties_of_class cl =
  let base_props = [ Width ; Height ; Border_width ] in
  let base_no_window = [ Tab_label ; Expand ; Fill ; Padding ] in
  let prop_kinds =
    match cl with
    | Custom_box ->
	[ Function ]
    | Button -> [ ]
    | Toggle_button
    | Check_button -> [ Active ; Draw_indicator ]
    | Radio_button -> [ Group ; Active ; Draw_indicator ]	
    | Toolbar ->
	[ Orientation ; Toolbar_style ;
	  Toolbar_space_size ; Toolbar_space_style ;
	  Tooltips ; Button_relief_style ]
    | Hbutton_box | Vbutton_box ->
	[ Spacing ; Child_width ; Child_height ; 
	  Child_ipadx ; Child_ipady ; Button_box_style ]
    | Hbox | Vbox ->
	[ Homogeneous ; Spacing ]
    | Fixed -> []
    | Frame ->
	[ PLabel ; Label_xalign ; Label_yalign ;
	  Shadow_type ]
    | Aspect_frame ->
	[ X_align ; Y_align ; Ratio ; Obey_child ;
	  PLabel ; Label_xalign ; Label_yalign ;
	  Shadow_type ]
    | Scrolled_window ->
	(* A VOIR : hadjustment et vadjustment *)
	[ Hscrollbar_policy ; Vscrollbar_policy ; Placement ]
    | Event_box -> []
    | Handle_box -> 
	[ Handle_position ; Snap_edge ; Shadow_type ]
    | Viewport ->
        (* A VOIR : hadjustment et vadjustment *)
	[ Shadow_type ]
    | Hseparator
    | Vseparator -> []
    | Clist ->
        (* A VOIR : hadjustment et vadjustment *)
	(* A VOIR : button_actions et autres proprits manquantes *)
	[ Column_number ; Column_titles ;
	  Shadow_type ; Selection_mode ;
	  Show_titles ]
    | Label ->
	(* A VOIR : proprit "pattern" *)
	[ PText ; Justification ; Line_wrap ;
	  X_align ; Y_align ; X_pad ; Y_pad ;
	] 
    | Statusbar ->
	[]
    | Notebook ->
	[ Tab_pos ; Tab_border ;
	  Show_tabs ; Homogeneous_tabs ; 
	  Show_border ; Scrollable ;
	  Popup ]
    | Color_selection ->
	[ Update_policy ]
    | Pixmap ->
	(* unused : make the compiler happy *)
	[ ]
    | Pixmap_file ->
	[ PPixmap_file ; X_align ; Y_align ; X_pad ; Y_pad ]
    | Pixmap_data ->
	[ PPixmap_data ; X_align ; Y_align ; X_pad ; Y_pad ]
    | Pixmap_code ->
	[ PPixmap_code ; X_align ; Y_align ; X_pad ; Y_pad ]
    | Entry ->
	[Max_length ; PText ; Visibility ; Editable ]
    | Spin_button ->
	(* A VOIR adjustment et autres  proprits *)
	[ SBUpdate_policy ; Shadow_type ]
    | Combo ->
	[ Popdown_strings ; Use_arrows ; Case_sensitive ; Value_in_list ; Ok_if_empty ]
    | Tree ->
	[ Selection_mode ; View_mode ; View_lines ]
    | Text ->
	(* A VOIR : hadjustment et vadjustment *)
	[ Editable ; Word_wrap ; Line_wrap ]
    | Hpaned
    | Vpaned ->
	[ Handle_size ]
    | Window -> 
	(* A VOIR : autres proprits *)
	[ Title ; Allow_shrink ; Allow_grow ; Auto_shrink ;
	  Modal ; X_pos ; Y_pos ]
    | Menubar ->
	[ Accel_group_name ]
    | Menu_item ->
	[ PLabel ; Accel_modifier ; Accel_flags ; Accel_keysym ;
	  Show_toggle ; Show_indicator ; Right_justify ]
    | Tearoff_menu_item ->
	[]
    | Check_menu_item ->
	[ PLabel ; Accel_modifier ; Accel_flags ; Accel_keysym ;
	  Show_toggle ; Show_indicator ; Right_justify ; Active]
    | Radio_menu_item ->
	[ PLabel ; Accel_modifier ; Accel_flags ; Accel_keysym ;
	  Show_toggle ; Show_indicator ; Right_justify ; Active ; 
	] 
    | Menu_separator ->
	[]
    | Menu ->
	[]
    | Table -> (* A VOIR *)
	[]
    | Progress_bar -> (* A VOIR *)
	[]
    | HRuler | VRuler -> (* A VOIR *)
	[]
    | Arrow ->
	[ Arrow_type ; Shadow_type ;
	  X_align ; Y_align ; X_pad ; Y_pad ]
    | Calendar ->
	[ Calendar_options ]
    | Drawing_area ->
	[]
    | Font_selection ->
	(* A VOIR : ajouter preview_text et font_name en paramtre dans lablgtk *)
	[]

  in
  List.map
    (fun k ->
       { prop_kind = k ; prop_value = default_prop_value k ;
         prop_value_loc = 0, 0 }) 
    (match cl with 
      Window -> base_props @ prop_kinds
    | _ -> 
	let pl = base_props @ prop_kinds @ base_no_window in
	match cl with
	  Hseparator | Vseparator | Label 
	| Entry | Spin_button | Text | Arrow | Calendar
	| HRuler | VRuler | Drawing_area 
	| Color_selection | Font_selection -> (* A VOIR, et aussi Zog_gui_handler.set_border_width *)
	    List.filter (fun p -> p <> Border_width) pl
	| _ -> pl
    )

let element_counter = ref 0
  
(** Create a new gui element from its class and name. *)
let gui_element ?(ask_label=true) cl name () = 
  incr element_counter;
  let e = 
    { 
      name = Printf.sprintf "%s%d" name !element_counter;
      name_loc = 0, 0 ;
      classe = cl ;
      props = properties_of_class cl ;
      children = [] ;
      expanded = true ;
    } 
  in
  if ask_label then
    match cl with
      Menu_item | Radio_menu_item | Check_menu_item ->
	let item_name = C.string
	    ~f: (fun s -> e.name <- s)
            Zog_messages.name e.name
	in
	let item_label = C.string
	    ~f: (fun s -> Zog_types.set_prop_value e PLabel s)
            Zog_messages.item_label (Zog_types.get_prop_value e.props PLabel)
	in
	ignore (C.simple_get Zog_messages.menu_item [item_name ; item_label])
    | _ ->
	()
  else
    ();
  e

(** Add missing properties to the given gui element. *)
let add_missing_properties g_ele =
  let pred p1 p2 = p1.prop_kind = p2.prop_kind in
  let dummy = gui_element ~ask_label: false g_ele.classe g_ele.name () in
  List.iter 
    (fun p ->
      if not (List.exists (pred p) g_ele.props) then
	g_ele.props <- g_ele.props @ [p]
    )
    dummy.props

(** Add missing properties in the gui elements of the given entity. *)
let add_missing_properties_in_entity entity =
  let rec iter ele =
    add_missing_properties ele ;
    List.iter iter ele.children
  in
  match entity.en_ele with
    None -> ()
  | Some ele -> iter ele

(** Remove badly associated properties to the given gui element. *)
let remove_properties g_ele =
  let pred p1 p2 = p1.prop_kind = p2.prop_kind in
  let dummy = gui_element ~ask_label: false g_ele.classe g_ele.name () in
  g_ele.props <-
    List.filter
      (fun p -> (List.exists (pred p) dummy.props))
      g_ele.props

(** Remove badly associated properties in the gui elements
   of the given entity. *)
let remove_properties_in_entity entity =
  let rec iter ele =
    remove_properties ele ;
    List.iter iter ele.children
  in
  match entity.en_ele with
    None -> ()
  | Some ele -> iter ele




let blank = "[ \013\009\012]"

(** Return a correct OCaml id from the given string.
   Remove beginning and trailing blanks, replace
   all unauthorized chars by '_', add 'x' at the 
   beginning of the string if it begins with a digit,
   and eventually lowercase the first letter.*)
let get_correct_id s =
  let s2 = Str.global_replace 
      (Str.regexp ("^"^blank^"*")) "" s
  in
  let s3 = Str.global_replace 
      (Str.regexp (blank^"*$")) "" s2
  in
  let l = String.length s3 in
  if l > 1 then
    for i = 1 to l - 1 do
      match s3.[i] with
	'a'..'z' | 'A'..'Z' | '0'..'9' | '\'' | '_' -> ()
      |	_ -> s3.[i] <- '_'
    done;
  let s4 = 
    if l > 0 then
      match s3.[0] with
        'a'..'z' 
        (* BEGIN CDK *)
      | '_'
        (* END CDK *)
        -> s3 
      | 'A'..'Z' -> String.uncapitalize s3
      | '0'..'9' | '\''  ->           "x"^s3
      |	_ -> s3.[0] <- 'x' ; s3
    else
      "x"
  in
  s4

(** Take a [gui_element] and changes its name if it is incorrect,
   that is, if it is not a correct OCaml id. Do the same to the
   children of the element.*)
let rec correct_gui_element ele =
  ele.name <- get_correct_id ele.name ;
  List.iter correct_gui_element ele.children



class data ?(gui=true) file =
  object (self)
    val mutable entities = ([] : Zog_types.entity list)
    val mutable templates = ([] : Zog_types.entity list)
    val mutable changed = false

    method entities = entities
    method templates = templates
    method set_changed b = changed <- b
    method changed = changed

    method save =
      try
	(* make sure all ocaml ids are correct *)
	List.iter
	  (fun ent -> 
	    ent.en_name <- get_correct_id ent.en_name ;
	    Zog_misc.apply_opt correct_gui_element ent.en_ele 
	  )
	  entities ;
	Zog_misc.write_entity_list file entities ;
(*
	let chanout = open_out_bin file in
	output_value chanout entities ;
	close_out chanout ;
*)
	changed <- false
      with
	Failure s ->
	  if gui then
	    GToolbox.message_box Zog_messages.error s
	  else
	    prerr_endline s

    method private load_file ?(msg_if_error=true) f =
      try
	(* create the file if it doesn't exist *)
	(
	 try ignore (Unix.stat f)
	 with Unix.Unix_error _ -> let oc = open_out f in close_out oc
	);
	Zog_misc.line_number := 0;
	let chanin = open_in f in
	let lexbuf = Lexing.from_channel chanin in
	let l = Zog_parser.project Zog_lexer.token lexbuf in
	List.iter add_missing_properties_in_entity l ;
	List.iter remove_properties_in_entity l ;
	close_in chanin ;
	l
	
(*
	let chanin = open_in_bin file in
	(
	 try 
	   entities <- input_value chanin ;
	 with End_of_file -> entities <- []
	);
	close_in chanin ;
*)
      with
      |	Failure s | Sys_error s ->
	  if msg_if_error then
	    if gui then
	      GToolbox.message_box Zog_messages.error s
	    else
	      prerr_endline s;
	  []
      |	Parsing.Parse_error ->
	  let s = "Parse error line "^(string_of_int !Zog_misc.line_number) in
	  if gui then 
	    GToolbox.message_box Zog_messages.error s
	  else
	    prerr_endline s;
	  []

    method load =
      let l = self#load_file file in
      entities <- l ;
      changed <- false

    method load_templates =
      let l = self#load_file ~msg_if_error: false Zog_config.templates in
      templates <- l

    method file = (file : string)

    method add_entity ent = 
      entities <- entities @ [ent] ;
      self#set_changed true

    method remove_entity ent =
      let old = entities in
      entities <- List.filter (fun e -> e != ent) entities ;
      self#set_changed (old <> entities)

    method up_entity ent =
      let rec f = function
          ent1 :: ent2 :: q -> 
            if ent2 == ent then
              ent2 :: ent1 :: q
            else
              ent1 :: (f (ent2 :: q))
        | l -> l
      in
      let old = entities in
      entities <- f entities ;
      self#set_changed (old <> entities)

(** {2 Entity manipulation functions} *)

    (** Remove a gui element from the given parent, in the given entity. *)
    method entity_remove_from_parent entity ele parent_opt =
      match parent_opt with
	None ->
	  (match entity.en_ele with
	    Some e when e == ele -> 
	      entity.en_ele <- None ;
	      self#set_changed true 
	  | _ -> ()
	  )
      |	Some p ->
	  p.children <- (List.filter (fun e -> e != ele) p.children) ;
	  self#set_changed true 

    (** Return [true] if an element of the given class can be appended
       to the children of the given element. *)
    method can_append_ele parent cl =
      match parent.classe with
      | Clist
      | Label
      | Hseparator
      | Vseparator
      | Custom_box
      | Color_selection
      | Pixmap 
      | Pixmap_file
      | Pixmap_data
      | Pixmap_code
      | Entry
      | Spin_button
      | Combo
      | Statusbar
      | Tree
      | Text 
      |	Table (* Table : A VOIR ? *)
      |	Progress_bar
      |	HRuler
      |	VRuler
      |	Arrow
      |	Calendar
      |	Drawing_area
      |	Font_selection
	-> false

      |	Menubar ->
	  List.mem cl [ Menu_item; Tearoff_menu_item; Menu_separator]

      |	Menu ->
	  List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			Radio_menu_item ; Tearoff_menu_item]

      |	Menu_item -> 
	  cl = Menu && parent.children = []

      |	Tearoff_menu_item 
      | Check_menu_item
      | Radio_menu_item
      |	Menu_separator ->
	  false

      | Button
      | Toggle_button
      | Check_button
      | Radio_button
      | Frame
      | Aspect_frame
      | Scrolled_window
      | Event_box
      | Handle_box
      | Viewport
      | Fixed
      | Toolbar 
      |	Window ->
	  not (List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			     Radio_menu_item ; Tearoff_menu_item ; Menu]) &&
	  parent.children = []

      | Hpaned
      |	Vpaned ->
	  not (List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			     Radio_menu_item ; Tearoff_menu_item ; Menu]) &&
	  (List.length parent.children) < 2

      |	Notebook
      | Hbox
      | Vbox
      |	Vbutton_box
      | Hbutton_box ->
	  not (List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			     Radio_menu_item ; Tearoff_menu_item ; Menu])
	  

    (** Add a gui element to another one, in a given entity.*)
    method entity_append_in_parent entity ele_paste parent_opt =
      match parent_opt with
	None -> 
	  (
	   match entity.en_ele with
	     None ->
	       entity.en_ele <- Some ele_paste ;
	       self#set_changed true
	   | _ -> 
	       raise (Failure Zog_messages.only_one_root)
	  )
      |	Some p -> 
	  if ele_paste.classe <> Window &
	    self#can_append_ele p ele_paste.classe then
	    (
	     p.children <- p.children @ [ele_paste] ;
	     self#set_changed true 
	    )
	  else
	    raise (Failure (Zog_messages.cant_add_element p.name))

    method up_element ele parent_opt =
      let rec f = function
          ele1 :: ele2 :: q -> 
            if ele2 == ele then
              ele2 :: ele1 :: q
            else
              ele1 :: (f (ele2 :: q))
        | l -> l
      in
      match parent_opt with
        None -> ()
      | Some p_ele ->
          p_ele.children <- f p_ele.children ;
          self#set_changed true 

    initializer
      self#load ;
      self#load_templates
  end
